pico-8 cartridge // http://www.pico-8.com
version 16
__lua__

cls(0)
--#region Utils (math, misc)
function debug_print(a) cls();print(a);for i = 1,300 do flip() end end

function dot2d(x,y,x2,y2)
	return x*x2+y*y2;
end

function insert(list, pos, value)
  if pos and not value then
   value=pos
   pos=#list+1
  end
  if pos <= #list then
   for i=#list, pos,-1 do
    list[i+1] = list[i]
   end
  end
  list[pos] = value
end

function sort (arr, comp)
  if not comp then
   comp = function (a, b)
    return a < b
   end
  end
  local function partition (a, lo, hi)
    pivot = a[hi]
    i = lo - 1
    for j = lo, hi - 1 do
    if comp(a[j], pivot) then
    i = i + 1
    a[i], a[j] = a[j], a[i]
    end
    end
    a[i + 1], a[hi] = a[hi], a[i + 1]
    return i + 1
   end
  local function quicksort (a, lo, hi)
   if lo < hi then
    p = partition(a, lo, hi)
    quicksort(a, lo, p - 1)
    return quicksort(a, p + 1, hi)
   end
  end
  return quicksort(arr, 1, #arr)
end

function sign(a)
	if a<0 then return -1 else return 1 end
end
function randr(a,b)
	return a+rnd(b-a)
end

function clamp(v,mn,mx)
	return max(mn,min(v,mx))
end

function strong(x)
	if (x==0) then
		return 0
	elseif (x==1) then
		return 1
	else 
		return x*strong(x-1)
	end
end

function clone(t)
  local t2 = {}
  for k,v in pairs(t) do
    t2[k] = v
  end
  return t2
end

do
 local f={}
 local p={}
 local permutation={151,160,137,91,90,15,
  131,13,201,95,96,53,194,233,7,225,140,36,103,30,69,142,8,99,37,240,21,10,23,
  190,6,148,247,120,234,75,0,26,197,62,94,252,219,203,117,35,11,32,57,177,33,
  88,237,149,56,87,174,20,125,136,171,168,68,175,74,165,71,134,139,48,27,166,
  77,146,158,231,83,111,229,122,60,211,133,230,220,105,92,41,55,46,245,40,244,
  102,143,54,65,25,63,161,1,216,80,73,209,76,132,187,208,89,18,169,200,196,
  135,130,116,188,159,86,164,100,109,198,173,186,3,64,52,217,226,250,124,123,
  5,202,38,147,118,126,255,82,85,212,207,206,59,227,47,16,58,17,182,189,28,42,
  223,183,170,213,119,248,152,2,44,154,163,70,221,153,101,155,167,43,172,9,
  129,22,39,253,19,98,108,110,79,113,224,232,178,185,112,104,218,246,97,228,
  251,34,242,193,238,210,144,12,191,179,162,241,81,51,145,235,249,14,239,107,
  49,192,214,31,181,199,106,157,184,84,204,176,115,121,50,45,127,4,150,254,
  138,236,205,93,222,114,67,29,24,72,243,141,128,195,78,66,215,61,156,180
 }

 for i=0,255 do
  local t=shr(i,8)
  f[t]=t*t*t*(t*(t*6-15)+10)

  p[i]=permutation[i+1]
  p[256+i]=permutation[i+1]
 end

 function lerp(t,a,b)
  return a+t*(b-a)
 end

 local function grad(hash,x,y,z)
  local h=band(hash,15)
  local u,v,r

  if h<8 then u=x else u=y end
  if h<4 then v=y elseif h==12 or h==14 then v=x else v=z end
  if band(h,1)==0 then r=u else r=-u end
  if band(h,2)==0 then r=r+v else r=r-v end

  return r
 end

 function noise(x)
  local xi=band(x,255)

  x=band(x,0x0.ff)

  local u=f[x]
  local v=f[1]
  local w=f[1]

  local a =p[xi  ]
  local aa=p[a   ]
  local ab=p[a+1 ]
  local b =p[xi+1]
  local ba=p[b   ]
  local bb=p[b+1 ]

  return lerp(w,lerp(v,lerp(u,grad(p[aa  ],x  ,0  ,0  ),
                              grad(p[ba  ],x-1,0  ,0  )),
                       lerp(u,grad(p[ab  ],x  ,-1,0  ),
                              grad(p[bb  ],x-1,-1,0  ))),
                lerp(v,lerp(u,grad(p[aa+1],x  ,0  ,-1),
                              grad(p[ba+1],x-1,0  ,-1)),
                       lerp(u,grad(p[ab+1],x  ,-1,-1),
                              grad(p[bb+1],x-1,-1,-1))))
 end
end

--#endregion

--#region Render utils
function spr_r(sx,sy,sw,sh,x,y,x0,y0,a)
	local sa=sin(a)
	local ca=cos(a)
	for ix=0,sw-1 do
		for iy=0,sh-1 do
			local dx=ix-x0
			local dy=iy-y0
			local xx=flr(dx*ca-dy*sa+x0)
			local yy=flr(dx*sa+dy*ca+y0)
			if (xx>=0 and xx<sw and yy>=0 and yy<=sh) then
				local p=sget(sx+xx,sy+yy)
				if p!=0 then pset(x+ix,y+iy,p) end
			end
		end
	end
end

function bham(x0, y0, x1, y1)
	local d_x, d_y, delta, t = 
		abs(x1 - x0), 
		abs(y1 - y0), 
		(x0 < x1) and 1 or -1
	t = d_y / 2

	return function ()
		while t > d_y do x0, t = x0 + delta, t - d_y end
		t = t + d_x
		return x0
	end
end

function trifill(x0, y0, x1, y1, x2, y2, color)
	if y1 < y0 then x0, y0, x1, y1 = x1, y1, x0, y0 end
	if y2 < y0 then x0, y0, x2, y2 = x2, y2, x0, y0 end
	if y2 < y1 then x2, y2, x1, y1 = x1, y1, x2, y2 end

	local a, b, c = 
		bham(x0, y0, x1, y1),
		bham(x1, y1, x2, y2),
		bham(x0, y0, x2, y2)

	for y =   y0, y1 do rectfill(a(), y, c(), y, color) end
	for y = 1+y1, y2 do rectfill(b(), y, c(), y, color) end
end

function rotate(cx, cy, x, y, r)
	return flr(cx+(x-cx)*cos(r)-(y-cy)*sin(r)), flr(cy+(y-cy)*cos(r)+(x-cx)*sin(r))
end

--#endregion

--#region object utils-----------------
function Point(x,y)
	local p = {}
	p.x = x
	p.y = y
	p.mag = function()
		return sqrt(p.x*p.x+p.y*p.y)
	end
	return p
end
function List(initial)
	local l={}
	l.add=function(el)
		add(l,el)
	end
	if initial then
		foreach(initial,function(o) l.add(o) end)
	end
	l.rem=function(el)
		del(l,el)
	end
	l.rand=function()
		return l[flr(randr(1,#l+1))]
	end
	return l
end
--#endregion
--#region useful gameob+jects---------------------------

function Camera()
 local c=Tran()
	c.lp=Point(0,0)
	--shakes
	c.sh=Point(0,0)
	c.sh_pow=0
	c.sh_dur=0
	c.sh_stime=0
	c.shake=function(pow,dur)
		c.sh_pow=pow;
		c.sh_dur=dur
		c.sh_stime=time()
	end
	c.up=function()
		local f=(time()-c.sh_stime)/c.sh_dur
		if (f>1) then
			c.sh.x=0
			c.sh.y=0
		else
			c.sh.x=rnd((1-f)*c.sh_pow)
			c.sh.y=rnd((1-f)*c.sh_pow)
		end
	end
	c.dr=function()
		camera(c.lp.x+c.sh.x, c.lp.y+c.sh.y)
	end
	return c
end

-- transforms tree (uses up and draw recursively)
local zSortedTransforms={}
function registerDrawable(e)
	for i=1,#zSortedTransforms do
		if zSortedTransforms[i].z>e.z then
			insert(zSortedTransforms,i,e)
			return
		end
	end
	insert(zSortedTransforms,1,e)
end
function unregisterDrawable(e)
	del(zSortedTransforms,e)
end
function sortAllDrawable()
	sort(zSortedTransforms,function(a,b) return a.z<b.z end)
end
function drawAllSorted()
	for i=1,#zSortedTransforms do
		zSortedTransforms[i].dr();
	end
end

function Tran()
	local o={}
	o.order=0
	o.lp=Point(0,0)
	o.z=0
	o.lr=0
	o.parent=null
	o.p=Point(0,0)
	o.r=0
	o.child=List()
	o.refresh=function()
		if (o.parent==null) then
			o.p.x=o.lp.x
			o.p.y=o.lp.y
			o.r=o.lr
		else
			if o.r==0 then
				o.p.x=o.parent.p.x+o.lp.x
				o.p.y=o.parent.p.y+o.lp.y
			else
				local an=atan2(o.lp.x,o.lp.y)
				local d=o.lp.mag()
				o.p.x=o.parent.p.x+cos(an+o.parent.r)*d
				o.p.y=o.parent.p.y+sin(an+o.parent.r)*d
			end
			o.r=o.lr+o.parent.r
		end
		for child in all(o.child) do
			child.refresh()
		end
	end
	o.up_tree=function()
		for child in all(o.child) do
			child.up()
			child.up_tree()
		end
	end
	o.up=function() end
	--[[o.dr_tree=function()
		o.dr()
		for child in all(o.child) do
			child.dr_tree()
		end
	end]]--
	o.dr=function() end
	o.add=function(ch)
		ch.parent=o
		--ch.z+=o.z+2
		o.child.add(ch)
	end
	o.rem=function(ch)
		if (ch!=null) then
			ch.parent=null
			o.child.rem(ch)
		end
	end
	registerDrawable(o)
	o.remall=function()
		o.child=List()
	end
	o.destroy=function()
		if o.parent!=null then o.parent.rem(o) end
		foreach(o.child,function(ch) ch.destroy() end)
		unregisterDrawable(o)
	end
	return o
end

-- particles!
-- config:
-- {
--	 rate:num
--   rot,power,size,life: {min,max}
--   col:list of numbers to randomize from
--   slowby:speed of slowing down (multiplier)
-- }
function ParticleSystem(cnf)
	local o=Tran()
	o.cnf=cnf
	o.nextsp=time()
	o.parts=List()
	o.enabled=true
	o.up=function()
		while (time()>o.nextsp) do
			o.nextsp+=1.0/o.cnf.rate
			if o.enabled then
				local part=Particle(o.cnf,o)
				o.parts.add(part)
			end
		end
		for part in all(o.parts) do
			part.up()
			if (time()>part.stime+part.ltime) then
				o.parts.rem(part)
				o.parent.rem(part)
			end
		end
	end
	o.dr=function()
		for part in all(o.parts) do
			part.dr()
		end
	end
	return o
end

function Particle(cnf,parent)
	local p={}
	local power=randr(cnf.power[1],cnf.power[2])
	local r=(randr(cnf.rot[1],cnf.rot[2])+parent.r)
	if r<0 then r+=1 end
	--say(r)
	p.x=parent.p.x;p.y=parent.p.y;
	p.v=Point(cos(r)*power,sin(r)*power)
	p.rad=flr(randr(cnf.size[1],cnf.size[2]+0.5))
	p.stime=time()
	p.ltime=randr(cnf.life[1],cnf.life[2])
	p.col=cnf.col.rand()
	p.up=function()
		p.x+=p.v.x
		p.y+=p.v.y
		p.v.x*=1-cnf.slowby
		p.v.y*=1-cnf.slowby
	end
	p.dr=function()
		if p.rad==1 then pset(p.x,p.y,p.col)
		elseif p.rad==2 then rectfill(p.x,p.y,p.x+1,p.y+1,p.col)
		else circfill(p.x,p.y,p.rad/2,p.col) end
	end
	return p
end

flashdur=0
function Sprite(sid,sw,sh,pivotx,pivoty,scale)
	local s=Tran()
	s.sw=sw*8;s.sh=sh*8;s.pivotx=pivotx*scale/2;s.pivoty=pivoty*scale/2
	s.scale=scale
	s.sp_x=(sid*8)%128
	s.sp_y=(flr(sid/16)) * 8
	s.flashstime=time()-5
	s.glitch=false
	s.enable_rot=false
	s.dr=function()
		if (s.glitch) then
			for x=s.sp_x,s.sp_x+s.sw-1 do
				for y=s.sp_y,s.sp_y+s.sh-1 do
					sset(x,y,rand(15))
				end
			end
		end
		if (time()<s.flashstime+flashdur) then
			local c=8+flr(rnd(2))
			for i=0,15 do
				pal(i,c)
			end
		end
		if not s.enable_rot then
			sspr(s.sp_x,s.sp_y,s.sw,s.sh,s.p.x+s.pivotx,s.p.y+s.pivoty,s.sw*s.scale,s.sh*s.scale)
		else
			spr_r(s.sp_x,s.sp_y,s.sw,s.sh,s.p.x-s.sw/2,s.p.y-s.sw/2,flr(s.sw/2),flr(s.sh/2),-s.r-0.25)
		end
		pal()
	end
	return s
end

-- c1=border color, c2=fill color
function Circle(x,y,r,c1,c2)
	local o=Tran()
	o.lp.x=x; o.lp.y=y
	o.rad=r
	o.rad2=r*r
	o.c1=c1;o.c2=c2
	o.dr=function()
		if o.c2 then
		circfill(o.p.x,o.p.y,o.rad,o.c2) end
		if o.c1 then circ(o.p.x,o.p.y,o.rad,o.c1) end
	end
	return o
end

-- offset_x, offset_y, width, height
function Rectangle(off_x,off_y,w,h,c1,c2,pat)
	local o=Tran()
	o.off_x=off_x;o.off_y=off_y
	o.w=w;o.h=h;o.c1=c1;o.c2=c2
	o.pat=pat
	o.dr=function()
		if o.pat then fillp(o.pat) end
		if o.r==0 then
			local blx,bly=o.p.x+o.off_x, o.p.y+o.off_y
			if o.c2 then rectfill(blx,bly,blx+o.w,bly+o.h,o.c2) end
			if o.c1 then rect(blx,bly,blx+o.w,bly+o.h,o.c1) end
		else
			local blx,bly=o.p.x+o.off_x, o.p.y+o.off_y
			local brx,bry=rotate(o.p.x,o.p.y, blx+o.w,bly,o.r)
			local tlx,tly=rotate(o.p.x,o.p.y, blx,bly+o.h, o.r )
			local trx,try=rotate(o.p.x,o.p.y, blx+o.w,bly+o.h, o.r )
			blx,bly=rotate(o.p.x,o.p.y,blx,bly,o.r)

			if o.c2 then 
				trifill(blx,bly,brx,bry,trx,try,o.c2)
				trifill(blx,bly,trx,try,tlx,tly,o.c2)
			end
			if o.c1 then
				line(blx,bly,brx,bry,o.c1)
				line(blx,bly,tlx,tly,o.c1)
				line(trx,try,tlx,tly,o.c1)
				line(trx,try,brx,bry,o.c1)
			end
		end
		if o.pat then fillp() end
	end
	return o
end

function Line(c)
	local o=Tran()
	o.c=c
	o.dr=function()
		local sx=0
		local sy=0
		if (o.parent!=null) then
			sx=o.parent.p.x
			sy=o.parent.p.y
		end
		line(sx,sy,o.p.x,o.p.y,o.c)
	end
	return o
end

function Text(text,c)
	local o=Tran()
	o.t=text;o.c=c
	o.dr=function()
		print(o.t,o.p.x,o.p.y,o.c)
	end
	return o
end

--#endregion

--#region Game config
zHoles=1
zPlayer=2 --3,4
zExplosion=5
zBorder=8
zUI=10
shrinkEvery=3

powerupPower=0.1
depowerupPower=0.15
maxPowerup=2.2
maxRotPowerup=1.2

pConfig={ -- 0=no,1=player,2=bot
	0,0,0,0,0,0
}

btnToStrTable={"<-","->","\\/","/\\","a","b"}
function btnToStr(btn)
	return btnToStrTable[btn]
end
local expl_psc={};
expl_psc["rate"]=100
expl_psc["rot"]={0.25,0.5}
expl_psc["slowby"]=0
expl_psc["grav"]=0
expl_psc["power"]={0,0}
expl_psc["size"]={6,12}
expl_psc["life"]={0.1,0.1}
expl_psc["col"]=List({8,9,7})
local dirt_conf={};
dirt_conf["rate"]=50
dirt_conf["rot"]={0.4,0.6}
dirt_conf["slowby"]=0.1
dirt_conf["grav"]=0
dirt_conf["power"]={1,5}
dirt_conf["size"]={1,2}
dirt_conf["life"]={0.1,0.2}
dirt_conf["col"]=List({9})
--#endregion

--#region Game objects
function Hole(x,y,r)
	local o=Circle(x,y,r,9,1)
	o.dead=false
	o.z=zHoles
	return o
end

local mapHolesToRemove={}
function Map()
	local o=Tran()
	o.holes=List()
	o.plays=List()
	local mapI=flr(randr(0,3.99))--flr(randr(0,0))
	o.sX=mapI*32
	o.cacheX=64
	o.cacheY=64
	o.justShrinked=false
	for x=o.sX,o.sX+31 do
		for y=32,63 do
			local c=sget(x,y)
			if c!=10 then
				local h=Hole(4*(x-o.sX)-64,4*(y-32)-64,c-5)
				o.add(h)
				o.holes.add(h)
				c=10
			end
			sset(o.cacheX+(x-o.sX)*2,o.cacheY+(y-32)*2,c)
			sset(o.cacheX+(x-o.sX)*2+1,o.cacheY+(y-32)*2,c)
			sset(o.cacheX+(x-o.sX)*2+1,o.cacheY+(y-32)*2+1,c)
			sset(o.cacheX+(x-o.sX)*2,o.cacheY+(y-32)*2+1,c)
		end
	end
	sort(o.holes,function(a,b)
		return rnd(1)>0.5
	end)
	local c=1
	o.tunnel=function(x,y,pc)
		x=o.cacheX+x/2;y=o.cacheY+y/2
		sset(x-1,y,c)
		sset(x+1,y,c)
		sset(x,y-1,c)
		sset(x,y+1,c)
		sset(x,y,c)
		return
		--[[stop(key)
		if rnd(10)<3 then
			add(mapHolesToRemove,Point(x,y))
			sset(x,y,pc)
		end]]--
	end

	o.shrink=0
	o.last_shrink=0
	o.enable_shrink=false
	o.up=function()
		if #mapHolesToRemove>10 then
			sset(mapHolesToRemove[1].x,mapHolesToRemove[1].y,c)
			del(mapHolesToRemove,mapHolesToRemove[0])
		end
		if o.enable_shrink then
			if time()>o.last_shrink+shrinkEvery then
				o.last_shrink=time()
				if abs(tele_top-tele_bottom)>48 then
					o.shrink+=0.5
					o.justShrinked=true
					tele_top-=1
					tele_bottom+=1
					tele_left+=1
					tele_right-=1
					--remove holes
					for i=#o.holes,1,-1 do
						local h=o.holes[i]
						if h.p.x<=tele_left or h.p.x>=tele_right or h.p.y>=tele_top or h.p.y<=tele_bottom then
							h.dead=true
							h.destroy()
							o.holes.rem(h)
						end
					end
				end
			end
		end
	end
	o.dr=function()
		palt(0, false)
		sspr(o.cacheX+o.shrink,o.cacheY+o.shrink,64-o.shrink*2,64-o.shrink*2,
		o.shrink-4,o.shrink-4,128-o.shrink*2+8,128-o.shrink*2+8)
		--sspr((i-1)*8,64,8,8,frX-8,frY,8,8)
		palt(0, true)
	end
	return o
end

local deadPlayers={}
function registerDeath(key)
	for i=1,#deadPlayers do
		if deadPlayers[i]==key then return end
	end
	add(deadPlayers,key)
end

function Flash()
	local o=Tran()
	o.starttime=-1
	o.endtime=-1
	o.flash=function()
		o.starttime=time()
		o.endtime=time()+0.25
	end
	o.dr=function()
		local t=time()
		if t>o.starttime and t<o.endtime then
			t=(t-o.starttime)/(o.endtime-o.starttime)
			if ( t<0.25 ) then fillp(0b1111111111111111.1)
			elseif ( t<0.5 ) then fillp(0b1100110000000000.1)
			elseif ( t<0.75 ) then fillp(0b1010010110100101.1)
			else fillp(0b0000000000000000.1) end
			palt(0,true)
			rectfill(-5,-5,132,132,0x07)
			palt(0,false)
			fillp()
		end
	end
	o.z=zUI+100
	return o
end

local playerSprites={128,129,130,131,132,133}--{128,129,130,131,132,133,134}
local playerSkinColors={6,8,15,12,13,14}
function Player(key,hole)
	local o=Tran()
	o.enable_rot=true
	--representation
	o.radius=5
	local head=Sprite(playerSprites[key+1],1,1,-6,-7,1)
	--head.enable_rot=true
	o.add(head)
	local skin=playerSkinColors[key+1]
	local c=Circle(0,0,o.radius*4,skin,skin)--4
	o.key=key
	c.lr=randr(0,1)
	o.add(c)
	o.circ=c
	local arr=Rectangle(0,-0.5253*o.radius,4,0.5253*o.radius*2,skin,skin)
	c.add(arr)
	local arr2=Rectangle(0,-0.2653*o.radius,8,0.2653*o.radius*2,skin,skin)
	arr.add(arr2)
	o.setWidth=function(x)
		arr.w=x/2
		arr2.w=x
	end
	o.z=zPlayer+3
	head.z=o.z
	c.z=o.z-1
	arr.z=c.z-1
	arr2.z=arr.z
	--logic
	o.interactable=false
	o.speed=2.5
	o.v=Point(0,0)
	o.rot_speed=0.01
	o.state="aim"
	o.last_hole=hole
	o.last_go_time=time()
	o.bot=(pConfig[key+1]==2)
	o.dead=false
	o.deathslide=Point(0,0)
	o.powerup=0
	o.particles=ParticleSystem(dirt_conf)
	o.particles.z=zPlayer
	c.add(o.particles)

	o.up=function()
		o.particles.enabled=(o.state=="go")
		--head.sp_y=72+8*(flr(time()*10)%4)
		--if head.sp_y==96 then head.sp_y-=16 end
		if c.rad != o.radius then
			c.rad=(c.rad*0.7+o.radius*0.3)
			if abs(c.rad-o.radius)<1 then
				c.rad=o.radius
				c.rad2=c.rad*c.rad
			end
		end
		if o.state=="aim" then
			if time()<o.lastbounceTime+0.4 then
				return
			end
			if o.powerup==0 then
				o.setWidth( 7 )
			end
			local targetX=o.last_hole.lp.x-cos(c.lr)*o.powerup*2;
			local targetY=o.last_hole.lp.y-sin(c.lr)*o.powerup*2;
			o.lp.x-=(o.lp.x-targetX)*0.5
			o.lp.y-=(o.lp.y-targetY)*0.5
			if o.powerup>0 then
				c.lr=(c.lr+o.rot_speed*(1+maxRotPowerup)) % 1
			else
				c.lr=(c.lr+o.rot_speed) % 1
			end
			if o.interactable then
				local press=(btn(o.key) or ( o.bot and rnd(10)<9 ))
				if press then
					o.powerup=min(maxPowerup,o.powerup+powerupPower)
					o.setWidth( 7+o.powerup/maxPowerup*7 )
					sfx(16)
				elseif ( not press ) and o.powerup>0 then
					sfx(3)
					-- shoot!
					o.state="go"
					c.lp.x=0
					o.v.x=cos(c.lr);o.v.y=sin(c.lr)
				end
				if o.last_hole.dead then
					sfx(3)
					-- hole is dead - move on
					o.state="go"
					c.lp.x=0
					o.v.x=cos(c.lr);o.v.y=sin(c.lr)
				end
			end
		elseif o.state=="go" then
			o.setWidth(7)
			local f=o.speed*(1+o.powerup)
			o.lp.x+=o.v.x*f
			o.lp.y+=o.v.y*f
			if o.powerup > 0 then
				o.powerup-=depowerupPower
			end

			local b=3
			--[[if o.p.x<tele_left-b then o.lp.x+=(tele_right-tele_left)+b*2; o.last_hole=nil end
			if o.p.x>tele_right+b then o.lp.x-=(tele_right-tele_left)+b*2; o.last_hole=nil end
			if o.p.y<tele_bottom-b then o.lp.y+=(tele_top-tele_bottom)+b*2; o.last_hole=nil end
			if o.p.y>tele_top+b then o.lp.y-=(tele_top-tele_bottom)+b*2; o.last_hole=nil end]]--
			if o.p.x<tele_left-b or o.p.x>tele_right+b then
				if o.p.x<tele_left-b then o.v.x=sign(o.v.x)*o.v.x
				else o.v.x=-sign(o.v.x)*o.v.x end
				o.lp.x+=o.v.x*f;
				c.lr=atan2(o.v.x,o.v.y)
				o.last_hole=nil
				cam.shake(2,0.1)
				sfx(4)
			end
			if o.p.y<tele_bottom-b or o.p.y>tele_top+b then
				if o.p.y<tele_bottom-b then o.v.y=-sign(o.v.y)*o.v.y
				else o.v.y=sign(o.v.y)*o.v.y end
				o.v.y=-sign(o.v.y);o.lp.y+=o.v.y*f;
				c.lr=atan2(o.v.x,o.v.y)
				o.last_hole=nil
				cam.shake(2,0.1)
				sfx(4)
			end
		elseif o.state=="dying" then
			if time()>o.last_go_time+0.5 then
				o.destroy()
			else
				o.lp.x+=o.deathslide.x
				o.lp.y+=o.deathslide.y
				o.deathslide.x*=0.9
				o.deathslide.y*=0.9
			end
		end
		--say(tostr(stat(9)).." "..tostr(counter))
		head.lr=c.lr
	end
	o.hit_hole=function(hole)
		if time()<o.lastbounceTime+0.4 then return true end
		if hole==o.last_hole then return true end
		o.last_hole=hole
		o.state="aim"
		o.rot_speed=-o.rot_speed
		o.last_go_time=time()
		return false
	end
	o.die=function(frPos)
		sfx(1)
		o.dead=true
		if o.state=="dying" then return end
		o.state="dying"
		o.flashdur=130000
		o.flashstime=time()+130
		local ps=ParticleSystem(expl_psc)
		o.add(ps)
		ps.z=zExplosion
		o.last_go_time=time()
		o.deathslide=Point((o.p.x-frPos.x)/5,(o.p.y-frPos.y)/5)
		sortAllDrawable()
	end
	o.levelup=function()
		if o.rot_speed>0 then
			--o.rot_speed+=0.001
		else
			--o.rot_speed-=0.001
		end
		--o.speed+=0.25
		cam.shake(4,0.35)
	end
	o.lastbounceTime=0
	o.bounce=function(from)
		if time()<o.lastbounceTime+0.2 then
			o.v.x*=1.1
			o.v.y*=1.1
			return
		end
		local dx=from.p.x-o.p.x
		local dy=from.p.y-o.p.y
		local angle = atan2(dx, dy);
        local targetX = o.p.x + cos(angle) * (o.circ.rad2+from.circ.rad2);
        local targetY = o.p.y + sin(angle) * (o.circ.rad2+from.circ.rad2);
        local ax = (targetX - from.p.x) * 1;
		local ay = (targetY - from.p.y) * 1;--dumb
		
		c.lr=atan2(o.v.x-ax,o.v.y-ay)
		o.lp.x -= cos(c.lr+0.5)*o.v.x*2 --rozsuniecie
		o.lp.y -= sin(c.lr+0.5)*o.v.y*2
		o.v.x=cos(c.lr)
		o.v.y=sin(c.lr)
		o.lastbounceTime=time()
		counter+=1
		o.last_hole=nil
		o.state="go"
	end
	return o
end
counter=0

function circles_col(c1,c2)
	return (c1.p.x-c2.p.x)*(c1.p.x-c2.p.x)+(c1.p.y-c2.p.y)*(c1.p.y-c2.p.y) < c1.rad2+c2.rad2
end
function Collisions(map)
	local o=Tran()
	o.map=map
	o.up=function()
		local new_dead={}
		for i=1,#(o.map.plays) do
			local p=o.map.plays[i]
			if not p.dead and p.interactable then
				if p.state=="go" then
					o.map.tunnel(p.p.x,p.p.y,p.circ.c1)
				end
				for j=i+1,#(o.map.plays) do --cols with others
					if circles_col(p.circ,o.map.plays[j].circ) then
						local pp=o.map.plays[j]
						if not p.dead and not pp.dead then
							if p.state=="aim" then
								p.die(pp.p)
								pp.levelup()
								add(new_dead,p)
							elseif pp.state=="aim" then
								pp.die(p.p)
								p.levelup()
								add(new_dead,pp)
							else
								--shouldBounce
								local a=atan2(p.p.x-pp.p.x,p.p.y-pp.p.y)
								local invDx=cos(a)
								local invDy=sin(a)
								local d1=dot2d(p.v.x,p.v.y,-invDx,-invDy);
								local d2=dot2d(pp.v.x,pp.v.y,invDx,invDy);
								counter+=1
								local a=abs(d1)>0.5
								local b=abs(d2)>0.5
								--debug_print(tostr(d1).." "..tostr(d2))
								if a and b then
									p.bounce(pp)
									pp.bounce(p)
								elseif b then
									p.die(pp.p)
									pp.levelup()
									add(new_dead,p)
								elseif a then
									pp.die(p.p)
									p.levelup()
									add(new_dead,pp)
								end
							end
						end
					end
				end
				for j=1,#(o.map.holes) do --cols with holes
					local h=o.map.holes[j]
					if circles_col(p.circ,h) then
						if p.hit_hole(h) then
							for z=1,#(o.map.plays) do
								local g=o.map.plays[z]
								if z!=i and g.state=="aim" and g.last_hole==h then
									if p.last_go_time<g.last_go_time then
										p.die(g.p)
										g.levelup()
										add(new_dead,p)
									end
								end
							end
						end
					end
				end
			end--if !dead
		end
		foreach(new_dead,function(pp)
			registerDeath(pp.key+1)
			o.map.plays.rem(pp)
			pConfig[pp.key+1]=-1
		end)
		--say(tostr(#o.map.plays))
	end
	return o
end

function UI()
	local h=16
	local o=Rectangle(0,0,128,h,0,0)
	o.lp.y=64-h;o.lp.x=-64
	o.t=Text("",7)
	o.add(o.t)
	local ppanel=function()
		local pp=Tran()
		pp.dr=function()
			local drawn=false
			for i=1,6 do
				local frX=o.p.x+i*16+(i-1)*4
				local frY=o.p.y+1
				local bp=0
				if btn(i-1) then bp=2 end
				if pConfig[i]==-1 then
					rectfill(frX-8,frY,frX+8,frY+8,8)
				end
				if pConfig[i]!=0 then
					sspr((i-1)*8,64,8,8,frX-8-bp/2,frY-bp/2,8+bp,8+bp)
					drawn=true
				end
				if pConfig[i]==1 then
					sspr((i-1)*8,0,8,8,frX-bp/2,frY-bp/2,8+bp,8+bp)
				elseif pConfig[i]==2 then
					sspr(6*8,0,8,8,frX,frY,8,8)
				end
			end
			if not drawn then
				if flr(time()*3)%3!=0 then
					print("PRESS BUTTON FOR EACH PLAYER", 8, 116,7)
				end
			end
		end
		return pp
	end
	o.pp=ppanel()
	o.add(o.pp);
	o.z=zUI;o.t.z=zUI+1;o.pp.z=zUI+1
	return o
end
ui={}
function say(t)
	ui.t.t=t
end
tele_left=-0
tele_right=128
tele_bottom=0
tele_top=128

function Borders(map)
	local o=Tran()
	o.c=0
	o.m=map
	o.drawRed=0;
	o.up=function()
		if o.m.justShrinked then
			o.m.justShrinked=false
			o.drawRed=2
		end
	end
	o.dr=function()
		if o.drawRed>0 then
			rectfill(-20,0,tele_left,148-ui.h,8)
			rectfill(tele_right,0,148,148-ui.h,8)
			rectfill(tele_left,-20,tele_right,tele_bottom,8)
			rectfill(tele_left,tele_top,tele_right,148-ui.h,8)

			rectfill(-20,0,tele_left-1,148-ui.h,o.c)
			rectfill(tele_right+1,0,148,148-ui.h,o.c)
			rectfill(tele_left-1,-20,tele_right+1,tele_bottom-1,o.c)
			rectfill(tele_left-1,tele_top+1,tele_right+1,148-ui.h,o.c)
			o.drawRed-=1
		else
			rectfill(-20,0,tele_left,148-ui.h,o.c)
			rectfill(tele_right,0,148,148-ui.h,o.c)
			rectfill(tele_left,-20,tele_right,tele_bottom,o.c)
			rectfill(tele_left,tele_top,tele_right,148-ui.h,o.c)
		end
	end
	o.z=zBorder
	return o
end
--#endregionroot=Tran()
local root=nil
state=nil
nextState=nil
stateChangeTime=0

function setState(new,instant)
	if state!=nil then
		state.stop()
	end
	if instant then
		state=new
		state.start()
	else
		state=nil
		nextState=new
		stateChangeTime=time()
	end
end

function _init()
	setState(Title(),true)
end

transCheeseSize=2
function _update()
	if root then
		root.refresh()
		root.up_tree()
	end
	if nextState then
		local f=max(0,min(1,(time()-stateChangeTime)))
		transCheeseSize=f
		if f==1 then
			state=nextState
			nextState=nil
			state.start()
		end
	end
	if state then
		state.up()
	end
end

pattern=0b1111111111111111--000001100110000
function _draw()
	if state then
		state.dr()
	end
	if transCheeseSize<1 then
		local realSize=transCheeseSize*32
		realSize+=3
		for i=1,15 do pal(i,0); end
		sspr(32,72,16,16,64-realSize*6,50-realSize*6,realSize*16,realSize*16)
		realSize-=3
		for i=1,15 do pal(i,i); end
		sspr(32,72,16,16,64-realSize*6,50-realSize*6,realSize*16,realSize*16)
		lastRealSize=realSize
	end
end

--#region States
function newRoot()
	root=Tran()
	root.lp.x=64; root.lp.y=64
	cam=Camera()
	root.add(cam)
	return root
end
function State()
	local o={}
	o.start=function() end
	o.up=function() end
	o.dr=function() end
	o.stop=function() end
	return o
end

howManyPlayers=2
howManyBots=0

howManyPlayersLastTime=2
howManyBotsLastTime=0

function ScrolledBg()
	local o=Tran()
	o.holes={}
	for i=1,10 do
		local h=Circle(randr(-64,64),randr(-32,32),randr(2,12),9,9)
		if h.lp.y<0 then h.lp.y-=32 else h.lp.y+=32 end
		h.z=zHoles
		add(o.holes,h)
		o.add(h)
	end
	o.z=zHoles-1
	sortAllDrawable()
	o.up=function()
		for i=1,10 do
			if o.holes[i].p.x+o.holes[i].rad<0 then
				o.holes[i].lp.x=128+o.holes[i].rad+1
			else
				o.holes[i].lp.x-=2
				--stop(tostr(o.holes[i].lp.x))
			end
		end
	end
	o.dr=function()
		cls(10)
	end
	return o
end

function Title()
	local o=State()
	o.choosed=0
	o.options=0
	o.start=function()
		music(2)
		newRoot()
		root.add(ScrolledBg())
		howManyPlayers=howManyPlayersLastTime
		howManyBots=howManyBotsLastTime
	end
	o.up=function()
		if btnp(2) then
			sfx(2)
			o.choosed-=1
			if o.choosed<0 then o.choosed+=o.options+1 end
		end
		if btnp(3) then
			sfx(2)
			o.choosed+=1
			if o.choosed>o.options then o.choosed=0 end
		end
		if o.choosed==1 then
			if btnp(0) then
				sfx(2)
				howManyPlayers=max(0,howManyPlayers-1)
			end
			if btnp(1) then
				sfx(2)
				howManyPlayers=min(6,howManyPlayers+1)
			end
			if howManyBots+howManyPlayers>6 then howManyBots-=1 end
		end
		if o.choosed==2 then
			if btnp(0) then
				sfx(2)
				howManyBots=max(0,howManyBots-1)
			end
			if btnp(1) then
				sfx(2)
				howManyBots=min(6,howManyBots+1)
			end
			if howManyBots+howManyPlayers>6 then howManyPlayers-=1 end
		end
		if howManyBots+howManyPlayers==0 then howManyPlayers=1 end
		if btnp(4) or btnp(5) then
			sfx(13)
			if o.choosed==0 then
				setState(Pregameplay())
			end
			if o.choosed==3 then
				setState(Tutorial())
			end
		end
	end
	o.text=function(text,index,x,y)
		y+=32
		if index==o.choosed then
			rectfill(x-40,y-1,x+36,y+7,0)
			rectfill(x-40+2,y-1+2,x+36-2,y+7-2,1)
			print(text,x-(#text/2*4),y,10)--flr(8+flr(time()*10)%3))
		else
			rectfill(x-40,y-1,x+36,y+7,0)
			print(text,x-(#text/2*4),y,7)
		end
		o.options=max(o.options,index)
	end
	o.dr=function()
		drawAllSorted()
		--o.text("CHEESE ROYALE BATTLE 1x5",-1,64,16)
		--rectfill(14,24,124,40,0)
		sspr(0,72,32,16,32,20,64,32)
		--print(text,64-(#text/2*4),30,1+flr(time()*10)%15)
		o.text("EAT CHEESE!",0,64,32)
		o.text("< PLAYERS: "..howManyPlayers.." >",1,64,40)
		o.text("< BOTS: "..howManyBots.." >",2,64,48)
		o.text("INFO",3,64,56)
	end
	o.stop=function()
		howManyPlayersLastTime=howManyPlayers
		howManyBotsLastTime=howManyBots
		zSortedTransforms={}
	end
	return o
end

function Pregameplay()
	local o=State()
	o.start=function()
		music(3)
		deadPlayers={}
		pConfig={0,0,0,0,0,0}
		newRoot()
		o.map=Map()
		root.add(Collisions(o.map))
		root.add(o.map)
		root.add(Borders(o.map))
		ui=UI()
		tele_left=0
		tele_right=128
		tele_bottom=0
		tele_top=128-ui.h
		root.add(ui)
		sortAllDrawable()
		o.endtime=-1
	end
	o.up=function()
		if o.endtime==-1 then
			for i=0,5 do
				if btnp(i) and pConfig[i+1]!=1 then
					sfx(0)
					pConfig[i+1]=1
					local p=Player(i,o.map.holes[i+1])
					p.lp.x=o.map.holes[i+1].lp.x
					p.lp.y=o.map.holes[i+1].lp.y
					root.add(p)
					p.refresh()
					_draw()
					o.map.plays.add(p)
					p.interactable=false
					sortAllDrawable()
					howManyPlayers-=1
				end
			end
			if howManyPlayers==0 then
				o.endtime=0
			end
		elseif o.endtime==0 then
			if #o.map.plays==0 or o.map.plays[#o.map.plays].circ.rad==o.map.plays[#o.map.plays].radius then
				for i=0,5 do
					for i=5,0,-1 do
						if howManyBots>0 and pConfig[i+1]!=1 then
							pConfig[i+1]=2
							local p=Player(i,o.map.holes[i+1])
							--stop(tostr(#o.map.holes))
							p.lp.x=o.map.holes[i+1].lp.x
							p.lp.y=o.map.holes[i+1].lp.y
							root.add(p)
							p.refresh()
							_draw()
							o.map.plays.add(p)
							howManyBots-=1
						end
					end
				end
				o.endtime=time()+1.5

				o.counter=Sprite(8,2,2,-18,-19,2)
				o.counter.lp=Point(0,0)
				o.counter.z=zUI+10
				root.add(o.counter)
				sfx(14)
				sortAllDrawable()
			end
		else
			if time()>o.endtime-1.0 and o.counter.sp_x!=80 and o.counter.sp_x!=96 then
				o.counter.sp_x=80
				sfx(14)
			end
			if time()>o.endtime-0.5 and o.counter.sp_x!=96 then
				o.counter.sp_x=96
				sfx(14)
			end
			if time()>o.endtime then
				sfx(15)
				o.counter.destroy()
				setState(Gameplay(),true)
			end
		end
	end
	o.dr=function()
		fillp(pattern)
		rectfill(-5,-5,132,132,0x90)
		fillp()
		drawAllSorted()
	end
	o.stop=function()
		o.map.enable_shrink=true
		for i=1,#o.map.plays do
			o.map.plays[i].interactable=true
		end
	end
	return o
end

function Gameplay()
	local o=State()
	o.start=function()
		music(4)
		o.flash=Flash()
		root.add(o.flash)
		o.flash.flash()
		sortAllDrawable()
		o.endtime=-1
	end
	o.up=function()
		if o.endtime==-1 then
			local count=6
			local living=nil
			for i=1,#pConfig do
				if pConfig[i]<=0 then
					count-=1
				else
					living=i
				end
			end
			if count==1 then
				registerDeath(living)
				o.endtime=time()+1
				o.flash.flash()
			end
		else
			if time()>o.endtime then
				setState(WhosBest(),true)
			end
		end
		--say(count)
		--fps.t=stat(9).." fps"
	end
	o.dr=function()
		fillp(pattern)
		rectfill(-5,-5,132,132,0x90)
		fillp()
		drawAllSorted()
	end
	o.stop=function()
		zSortedTransforms={}
	end
	return o
end

function WhosBest()
	o=State()
	o.start=function()
		music(0)
		newRoot()
		local r1=Rectangle(-16,0,32,32,10,10)
		root.add(r1)
		local r2=Rectangle(-48,32,96,32,10,10)
		root.add(r2)
		local c=Circle(-5,32,5,9,9)
		c.z=1
		r1.add(c)
		c=Circle(16,64,6,9,9)
		c.z=1
		r1.add(c)
		c=Circle(-24,64,3,9,9)
		c.z=1
		r1.add(c)
		c=Circle(-5,16,3,9,9)
		c.z=1
		r2.add(c)
		sortAllDrawable()
		o.starttime=time()
	end
	o.up=function()
		if time()>o.starttime+1 then
			if btnp(0) or btnp(1) or btnp(2) or btnp(3) or btnp(4) or btnp(5) then
				setState(Title(),true)
			end
		end
	end
	o.drspr=function(j,x,y,bump)
		local i=deadPlayers[j]
		--stop(i)
		if bump and flr((time()*9)%3)==0 then
			sspr((i-1)*8,64,8,8,x-16,y-34,32,32)
		else
			sspr((i-1)*8,64,8,8,x-16,y-32,32,32)
		end
		--[[if pConfig[i]==1 then
			sspr((i-1)*8,0,8,8,x,y,8,8)
		elseif pConfig[i]==2 then
			sspr(6*8,0,8,8,frX,frY,8,8)
		end	]]--
	end
	o.dr=function()
		cls(2)
		drawAllSorted()
		o.drspr(#deadPlayers,64,64,true)
		if #deadPlayers>1 then o.drspr(#deadPlayers-1,32,96,false) end
		if #deadPlayers>2 then o.drspr(#deadPlayers-2,96,96,false) end
		print("WINNER!",52,26,7)
	end
	o.stop=function()
		zSortedTransforms={}
	end
	return o
end

function Tutorial()
	local o=State()
	o.start=function()
		o.stt=time()
	end
	o.up=function()
		if btn(0) or btn(1) or btn(2) or btn(3) or btn(4) or btn(5) then
			o.stt-=0.1
		end
	end
	o.text=function(text,x,y)
		rectfill(x-(#text/2*4)-4,y-2,x+(#text/2*4)+4,y+7,0)
		print(text,x-(#text/2*4),y,7)
	end
	o.dr=function()
		cls(9)
		local t=time()-o.stt
		if t>0.0 then o.text("1 huge cheese",64,6) end
		if t>1.0 then o.text("6 hungry characters",64,22) end
		if t>2.0 then
			for i=1,6 do
				if t>2.0+i*0.2 then
					sspr((i-1)*8,64,8,8,13+(i-1)*17,36,16,16)
				end
			end
		end
		if t>4.0 then o.text("6 furious buttons",64,66) end
		if t>5.0 then
			for i=1,6 do
				if t>5.0+i*0.2 then
					sspr((i-1)*8,0,8,8,13+(i-1)*17,76,16,16)
				end
			end
		end
		if t>7.0 then o.text("who's going to survive",64,100) end
		if t>8.0 then o.text("this cheesy rampage?",64,116) end
		if t>10.0 then setState(Title()) end
	end
	o.stop=function()
	end
	return o
end
--#endregion

--#region rest

__gfx__
00000000000000000000000000000000000000000000000007770777000000000097777799999900009999999999990000999999999999000099999999999900
000000000000000000000000000cc000000000000000000007070070000000000997777777799990099999999999999009999999999999900999999999999990
0006000000008000000ff000000cc000000dd00000eee00007070070000000009997788877779999999999777777999999999997799999999999999999999999
006600000000880000ffff00000cc00000d00d0000e00e0007070070000000009998889987779999999977777777999999999977799999999999999999999999
06666666888888800ffffff0000cc00000d00d0000eee00007770070000000009999999988779999999777788877799999999777799999999977779999999999
0666666688888880000ff0000cccccc000dddd0000e00e0007070070000000009999999999779999999777899987799999999777779999999777779999977799
0066000000008800000ff00000cccc0000d00d0000e00e0007070070000000009999999999779999999888899997799999997778779999997777889997777799
0006000000008000000ff000000cc00000d00d0000eee00007070777000000009999997777777779999999999977799999977779779999997788999977777779
0000000000000000000ff00000000000000000000000000000000000000000009999999977777777999999999777799999977789779999997797777977887779
00000000000000000000000000000000000000000000000000000000000000009999999988888777999999997777899999988899779999997797777977998779
00000000000000000007700000000000000000000000000000000000000000009999779999999777999999777778999999999999779999997778877977997779
00000000000770000077770000000000000000000000000000000000000000009999777999997778999977777889999999999997779999997777777977977779
00077000007777000777777000000000000000000000000000000000000000009999777799777779999777777777799999999997779999998777778977777789
00077000007777000777777000000000000000000000000000000000000000009999877777777789999777777777779999999997789999999888889977777899
00000000000770000077770000000000000000000000000000000000000000000999987777788890099888888877779009999998899999900999999988888990
00000000000000000007700000000000000000000000000000000000000000000099998888899900009999999988880000999999999999000099999999999900
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaacaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaeaaaaaaaaaadaaaaaaaaaaaeaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaadaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaacaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaeaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaacaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaeaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaeaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaadaaaaaaaaaaaaaaaaadaaaaadaaaaaeaaaadaaaaadaaaaaaaaaacaaaaaaaaafaaaaaaaaaacaaaaaaaadaaaaaaaaaaaaaaaaaaaaaadaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaeaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaadaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaacaaaaaaaaaaaaaacaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaafaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaeaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaacaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaadaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaeaaaaaaaaaadaaaaaaaaaaaeaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa
000bb000000000000000000000000000004004000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
00bbbb00000fcf0005555550008888000074470000cccc0000000000000000008888888888888888888888888888888888888888888888888888888888888888
0bbbbbb0000fff0005cccc5008888880004444000cccc7c000000000000000008888888888888888888888888888888888888888888888888888888888888888
0bb4bbb000ff000005cccc5008588880004554000ccc7cc000000000000000008888888888888888888888888888888888888888888888888888888888888888
0b4444b000fff00005cccc50075588800ffeeff00cc777c000000000000000008888888888888888888888888888888888888888888888888888888888888888
0004400000ffee00055555500777777000eeee000ccc7cc000000000000000008888888888888888888888888888888888888888888888888888888888888888
000440000000f0e000055000007777000eeeeee000cc7c0000000000000000008888888888888888888888888888888888888888888888888888888888888888
0004400000000000005555000000000000f00f000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
00222222222229922299222222002222000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
22299299292999929999229992222992000000aa0000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
29999299292992229922299999299922000000aaaaa0000000000000000000008888888888888888888888888888888888888888888888888888888888888888
2992229999299922999229922229922000000aaaaaaaa00000000000000000008888888888888888888888888888888888888888888888888888888888888888
998888999989998899988899998999800000aa99aaaaaa0000000000000000008888888888888888888888888888888888888888888888888888888888888888
999888aa8a89888898888988998aa888000aaa00aaaaaaa000000000000000008888888888888888888888888888888888888888888888888888888888888888
8aaaa8aa8a8aaaa8aaaa8aaa000aaaa800aaaaaaaaaaa99000000000000000008888888888888888888888888888888888888888888888888888888888888888
888aa888888aaaa8aaaa88aa0008888800aaaaaaaaa9999000000000000000008888888888888888888888888888888888888888888888888888888888888888
008888880088888888888888888888800aaaaaaa9999999000000000000000008888888888888888888888888888888888888888888888888888888888888888
000089980089988998998899988999800aaaaa999009999000000000000000008888888888888888888888888888888888888888888888888888888888888888
00088998889989889898899888998880aaaa99999aa9900000000000000000008888888888888888888888888888888888888888888888888888888888888888
00089999889889888998899998999980aa9999999990000000000000000000008888888888888888888888888888888888888888888888888888888888888888
000898898899988889888888a8888a80990999990000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
0008aaaa88a88a88aa888aaaa8aaaa8099a999000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
0008a88a88aaaa88a888aa888aa88880999900000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
00088888888888888808888088880000990000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
00bbbb0000ffff00005555000088880000eeee0000cccc0000000000000000008888888888888888888888888888888888888888888888888888888888888888
0bbbbbb00ffffff005555550088888800eeeeee00cccccc000000000000000008888888888888888888888888888888888888888888888888888888888888888
bbbbbbbbffffffff5555555588888888eeeeeeeecccccccc00000000000000008888888888888888888888888888888888888888888888888888888888888888
bbbbbbbbffffffff5555555588888888eeeeeeeecccccccc00000000000000008888888888888888888888888888888888888888888888888888888888888888
bbb00bbbfff00fff5550055588800888eee00eeeccc00ccc00000000000000008888888888888888888888888888888888888888888888888888888888888888
bb0000bbff0000ff5500005588000088ee0000eecc0000cc00000000000000008888888888888888888888888888888888888888888888888888888888888888
0b0000b00f0000f005000050080000800e0000e00c0000c000000000000000008888888888888888888888888888888888888888888888888888888888888888
00000000000000000000000000000000000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
11111111111111111111111111111111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
11111111111111111111111111111111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
11111111111111111111111111111111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
11111111111aaaaaa111111111111111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
1111111aaaaaaaaaaaa1111111111111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
111111aaaaaaaaaaaaaaa11111111111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
11111aaaaaaaaaaaaaaaaaaaa1111111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
11111aaaaaaaaaaaaaaaaaaaaa111111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
1111aaaaaaaaaaaaaaaaaaaaaaa11111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
1111aaaa1111aaaaaaaaaaaaaaaa1111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
111aaaa111111aaaaaaaaaaaaaaa1111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
111aaaa111111aaaaaaaaaaaaaaa1111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
111aaaaa1111aaaaaaaaaaaaaaaaa111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
11aaaaaaaaaaaaaaaaaaaaaaaaaaa111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
11aaaaaaaaaaaaaaaaaaaaaaaaaaaa11000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
11aaaaaaaaaaaaaaaaaaaaaaaaaaaa11000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
1aaaaaaaaaaaaaaaaaaaaaaaaaaaaa11000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
1aaaaaaaaaaaaaaaaaaaaaaaaaaaaa11000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
1aaaaaaaaaaaaaaaaaaaaaa11aaaaa11000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
1aaaaaaaaaaaaaaaaaaaaaa11aaaaa11000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
1aaaaaaaaaaaaaaaaaaaaaaaaaaaaaa1000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
1aaaaaaaaaaaa11aaaaaaaaaaaaaaaa1000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
1aaaaaaaaaaa1111aaaaaaaaaaaaaaa1000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
11aaaaaaaaaaa11aaaaaaaaaaaaaaaa1000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
11aaaaaaaaaaaaaaaaaaaaaaaaaaaa11000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
11aaaaaaaaaaaaaaaaaaaaaaaaaaaa11000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
111aaaaaaaaaaaaaaaaaaaaaaaaaaa11000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
1111aaaaaaaaaaaaaaaaaaaaaaaaa111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
11111aaaaaaaaaa1aaaaaaaaaaaaa111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
111111111111111111111aaaaaaa1111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
11111111111111111111111111111111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
11111111111111111111111111111111000000000000000000000000000000008888888888888888888888888888888888888888888888888888888888888888
__label__
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
cccccccc777ccccccccccccccccccccccccccccccccccccccccccccc777ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
ccccc77777777cccccccccccccccccccccccccccccccccccccccc77777777ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
ccc777777777777cccccccccccccccccccccccccccccccccccc777777777777ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
ccc777777777777cccccccccccccccccccccccccccccccccccc777777777777ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccceecceeccceecceeccceecceeccceecceecc
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccceee88e88ceee88e88ceee88e88ceee88e88c
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccee888888cee888888cee888888cee888888c
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccce88888887e8888888ce8888888ce8888888c
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc788888888788888888c88888888c88888888c
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc77888888777888888ccc888888ccc888888cc
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc8888ccccc8888ccccc8888ccccc8888ccc
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc88ccccccc88ccccccc88ccccccc88cccc
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
ccccccc999999999999999999999999ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
ccccccc999999999999999999999999ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
ccccccc999999999999999999999999ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
ccccccc999999999999999999999999ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
ccc999999999999ffff9999ffff9999ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
ccc999999999999ffff9999ffff9999ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
ccc999999999999ffff9999ffff9999ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccffffffffccccffffffffccccccccccc
ccc999999999999ffff9999ffff9999ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccffffffffccccffffffffccccccccccc
ccc99999999ffffffffffffffffffffccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccffffffffccccffffffffccccccccccc
ccc99999999ffffffffffffffffffffccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccffffffffccccffffffffccccccccccc
ccc99999999ffffffffffffffffffffccbccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc444444444444ffffffffccccccccccc
ccc99999999ffffffffffffffffffffbcbccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc444444444444ffffffffccccccccccc
ccc9999ffffffffeeeeddddffffeeeeddddcccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc444444444444ffffffffccccccccccc
ccc9999ffffffffeeeeddddffffeeeeddddcccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc444444444444ffffffffccccccccccc
ccc9999ffffffffeeeeddddffffeeeeddddccccccccccccccccccccccccccccccccccccccccccccccccccccccccccffff4444ffffffff44444444ccccccccccc
ccc9999ffffffffeeeeddddffffeeeeddddccccccccccccccccccccccccccccccccccccccccccccccccccccccccccffff4444ffffffff44444444ccccccccccc
ccc9999ffffffffffffffffffffffffffffccccccccccccccccccccccccccccccccccccccccccccccccccccccccccffff4444ffffffff44444444ccccccccccc
ccc9999ffffffffffffffffffffffffffffccccccccccccccccccccccccccccccccccccccccccccccccccccccccccffff4444ffffffff44444444ccccccccccc
cbc9999ffffffffffffffffffffffffffffcbccccbccbccccbccccccccccccccccccccccccccccccccccccccccccc8888ffff8888ffff44444444ccccbccbccc
cbc9999ffffffffffffffffffffffffffffcbbcbcbccbbcbcbccccccccccccccccccccccccccccccccccccccccccc8888ffff8888ffff44444444bcbcbccbbcb
bbcffffffffffffffff88888888ffffffffbbbcbbbcbbbcbbbccccccccccccccccccccccccccccccccccccccccccc8888ffff8888ffff44444444bcbbbcbbbcb
bbbffffffffffffffff88888888ffffffffbbbbbbbbbbbbbbbbcccccccccccccccccccccccccccccccccccccccccc8888ffff8888ffff44444444bbbbbbbbbbb
bbbffffffffffffffff88888888ffffffffbbbbbbbbbbbbbbbbccccccccccccccccccccccccccccccccccccccffffffffffffffff444444444444bbbbbbbbbbb
bbbffffffffffffffff88888888ffffffffbbbbbbbbbbbbbbbbccccccccccccccccccccccccccccccccccccccffffffffffffffff444444444444bbbbbbbbbbb
bbbffffffffffffffff8888ffffffffffffbbbbbbbbbbbbbbbbccccccccccccccccccccccccccccccccccccccffffffffffffffff444444444444bbbbbbbbbbb
bbbffffffffffffffff8888ffffffffffffbbbbbbbbbbbbbbbbccccccccccccccccccccccccccccccccccccccffffffffffffffff444444444444bbbbbbbbbbb
bbbffffffffffffffff8888ffffffffffffbbbbbbbbbbbbbbbbcbccccbccbccccbccbccccbccbccccbccbffffffffffffffffffffffff44444444bbbbbbbbbbb
bbbffffffffffffffff8888ffffffffffffbbbbbbbbbbbbbbbbcbbcbcbccbbcbcbccbbcbcbccbbcbcbccbffffffffffffffffffffffff44444444bbbbbbbbbbb
bbbbbbbffffffffffffffffffffffffffffbbbbbbbbbbbbbbbbbbbcbbbcbbbcbbbcbbbcbbbcbbbcbbbcbbffffffffffffffffffffffff44444444bbbbbbbbbbb
bbb1111ffffffffffffffffffffffffffffbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffffffffffffffff44444444bbbbbbbbbbb
bbb1111ffffffffffffffffffffffffffffbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffff7777ffff7777ffffffff44444444bbbbbbbbbbb
bbb1111ffffffffffffffffffffffffffffbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffff7777ffff7777ffffffff44444444bbbbbbbbbbb
bbb1111111111111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffff7777ffff7777ffffffff44444444bbbbbbbbbbb
111111111111111111111111111fffffffffffffffbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffff7777ffff7777ffffffff44444444bbbbbbbbbbb
111111111111111111111111111fffffffffffffffffbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb
111111111111111111111111111ffffffffffffffffffffbbbbbffffffffffffbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb
111111111111111111111111111ffffffffffffffffffffffbbbffffffffffffbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb
111111111111111111111111111ffffffffffffffffffffffffbffffff5555ff556666666666bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb
111111111111111111111111111fffffffffffffffffffffffffffffff5555ff556666666666bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb
111111111111111111111111111fffffffffffffffffffffffffffff55555555556666bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb
111111111111111111111111111ffffbbbbbbbbbbbffffffffffffff55555555556666bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbffffffffffffffff55ffbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbfffffffffffff55ffbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbffffbbbffffffffbbbbbbbbbbbbbbbbbbbbb7777ffff7777ffff7777ffff44444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbffbbbffffffffbbbbbbbbbbbbbbbbbbbbb7777ffff7777ffff7777ffff44444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb7777ffff7777ffff7777ffff44444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb7777ffff7777ffff7777ffff44444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffffffff444444444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffffffff444444444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffffffff444444444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffffffff444444444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb44444444444444444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb444444444444444444444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb444444444444444444444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb444444444444444444444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb444444444444444444444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb444444444444444444444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb444444444444444444444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb444444444444444444444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb444444444444444444444444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffff444444444444ffffffff4444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffff444444444444ffffffff4444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffff444444444444ffffffff4444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffff444444444444ffffffff4444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffff444444444444ffffffff4444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffff444444444444ffffffff4444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffff444444444444ffffffff4444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffff444444444444ffffffff4444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffff44444444ffffffff4444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffff44444444ffffffff4444bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffff44444444fffffff88844bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffff44444444ffffff888884bbbbbbbbbbb
111ffffffff11111111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffff44444444ffff8f888884bbbbbbbbbbb
111ffffffffffff1111111111111111bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffff44444444fff888888884bbbbbbbbbbb
111ffffffffffff1111111111111111bb3bb3bbbb3bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffff44444444fff888888844bbbb3bb3bbb
111ffffffffffff11111111111111113b3bb33b3b3bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffff44444444fff8888884443b3b3bb33b3
111ffffffffffff1111111111111111333b333b333bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffff44444444fff8888844448b333b333b3
111ffffffffffff1111111111111111333333333333bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffff44444444fff88888844888333333333
111ffffffffffff1111111111111111333333333333bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffff44444444ffff8888884483333333333
111ffffffffffff1111111111111111335333333333bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffff44444444ffff4444888833333333333
111ffffffffffff1111111111111111335333355333bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffff4444444444444444888883333333333
111ffffffffffff1111111111115555535333355333b55bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbffffffffffff4444444444444444888883333333333
111ffffffffffff1111111111115555555555555333b55bbb3553bbbb3bb3bbbb3bb3bbbb3bb3bbbb3bb3ffffffffffff4444444444444444888883333333333
111ffffffffffff111111111111555555555555555555555535533b3555533b3b3bb33b3b3bb33b3b3bb3ffffffffffff4444444444444444488833333333333
111ffffffffffff11111111111155553353335555555555555555555555533b333b333b333b333b333b33ffffffffffff4444444444444444444433333333333
1111111111111111111111111111111335333355333355355555555555555533333333333333333333333ffffffffffff4444444444444444444433333333333
1111111111111111111111111111111335333355333355333355333355555533333333333333333333333ffffffffffff4444444444444444444833333333333
1111111111111111111111111111111335333355333355333355333355553333333333333333333333333ffffffffffff4444444444444444448883333333333
1111111111111111111111111111111335333355333355333355333355553333333333333333333333333ffffffffffff4400000000000000000000000000333
1111111111111111111111111111111335333355333355333355333355553333333333333333333333333ffffffffffff4407770077077007070777070700333
1111111111111111111111111111111335333355333355333355333355553333333333333333333333333ffffffffffff4407770707070707070700070700333
1111111111111111111111111111111335333355333355333355333355553333333333333333333333333ffffffffffff4407070707070707700770077700333
1111111111111111111111111111111335333355333355333355333355553333333333333333333333333ffffffff44444407070707070707070700000700333
3335555555555555555555555555555335333355333355333355333355553333333333333333333333333ffffffff44444407070770070707070777077700333
3335555555555555555555555555555335333355333355333355333355553333333333333333333333333ffffffff44444400000000000000000000000000333
3335555555555555555555555555555335333355333355333355333355553333333333333333333333333ffffffff44444444444444444444484433333333333
33355555555555555555555555555553353333553333553333553333555533333330000000000000000000000000000000000000000000000000000000000333
33355555555555555555555555555553353333553333553333553333555533333330707077707000700077707700000000007770000000700000777077700333
33355555555555555555555555555553353333553333553333553333555533333330707007007000700070007070070000000070000007000000007070700333
33355555555555555555555555555553353333553333553333553333555533333330770007007000700077007070000000007770000007000000777070700333
33355555555555555555555555555553353333553333553333553333555533333330707007007000700070007070070000007000000007000000700070700333
33355555555555555555555555555553353333553333553333553333555533333330707077707770777077707770000000007770000070000000777077700333
33355555555555555555555555555553353333553333553333553333555533333330000000000000000000000000000000000000000000000000000000000333
33355555555555555555555555555553353333553333553333553333555533333333333333333333333333333333344444444444444444444444433333333333
33355555555555555555555555555553353333553333553333553333555533333333333333333333333333333333344444444ffffffff4444444433333333333

__map__
3030303030303030303030303030303030300000000000000000000000000022222222222222222222000000000080808080808080808080808080808080808000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303030303030303030303030303030300000000039000039000000002222372237222237222222220000000080808181818181808080808080808080808000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303030303030303030303030303132300000000037370037003900002237373737373737372237220000000080818080808080818180808080808081818100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030313230303030303030303030303030300000000037370037003700222237373722373722373737222200000081818080808080808081818180818181808000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303030303030303030303030303030300000000037373737373700223737372222373722223737373700000080808080808080808080808081818080808000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303030303030303030303030303030300000393737373737373700373737223030373730302237373700000080808080808080808080808080808080808000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303030303030303030303030303030300000373737373737373700373722373737373737373722373700000080808080808080808080808080808080808000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303030313230303030303030303030300000003737373737383700373737373737373737373737373700000080818181818181818181808080808080808000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303030303030303030303030303030300000000037373737373700003737373730303037373737373700000081818080808080808081818080808080818100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303030303030303030303030303030300000000000363636363600003637373730373730373737360000000080808080808080808080818181818181818000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303030303030303030303031323030300000000000363636363600363636363737373737373736363600000080808080808080808080808080808080808000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303030303030303030303030303030300000000000333333333300363636363636363636363636363600000080808080808080808080808080808080808000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3031323030303030303030303030303030300000000000333333333300363636363636363636363636363600000080808080818181818080808080808080808100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303030303030303030303030303030300000000000333333333300003636363636363636363636360000000081818181808080818181808080808080818100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303030303031323030303030303030300000000000333333333300000000000000000000000000000000000081818080808080808080818181818181818000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303030303030303030303030303030300000000000333333333300000000000000000000000000000000000080808080808080808080808080808080808000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303132303030303030303030303132300000000000000000000000000000000000000000000000000000000080808080808080808080808080808080808000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303030303030313230303030303030300000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303030303030303030303030303030300000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3030303030303030303030303030303030300000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3530303030303030303030303030303035350000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3335353030303030303535353535353533330000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3333333535353535353333333333333333330000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3333333333333333333333333333333333330000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3333333333333333333333333333333333330000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3333333333333333333333333333333333330000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3333333333333333333333333333333334340000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3333333333333333333333333334343436360000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3434333333333333343433343436363636360000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3636343434343434363634363636363636360000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3636363636363636363636363636363636360000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
3636363636363636363636363636363636360000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
__sfx__
000100000a3700b3700b3700c3700e3701037014370183701d3702335028350061001510000000000001210000000000001010000000000000e100000000d1000b1000b1000a1000a1000a1000a1000b1000b100
000100002d250312502e2502d2503075029250272502525023250202501e2501865017650106500a6500365001650016500165001650016500165001650016500165001650016500165001650016500165001650
010200002175023750257502775021700227002270022700237002470000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000200002775027750257502575025750247502475024750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000100001f5501d55019550185501555013550115500e5500c5500955006550035500155000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
011000000455303603045002d503075530000307500046030455303603045532d503075530755307553046030455303603045002d503075530000307500046030455303603045532d50307553075530755307500
011000001855018550185501b5501c5501f5501b5501c5501855018550185501b5501c5501f5501b5501c5501855018550185501b5501c5501f5501b5501c5501855018550185501b5501c5501f5501b5501c550
001000001c5001c5001c5001c5001c5501c5501c5501c5501855018550185501b5501c5501f5501b5501c5501855018550185501b5501c5501f5501b5501c5501855018550185501b5501c5501f5501b5501c550
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000c00000372503725037250372503725037250372503725037250372503725037250372503725037250372503725037250372503725037250372503725037250372503725037250372503725037250372503725
010c0000165510000115551165511555116551165011655116401165511a1011655100001165010000116401165510000015500165001555116551165001650016400165001a1001650000000165010000116401
010c0000021500560005600125001350013500031500a600021501350013500135001250012500021501250008600000000315000000021500000003150000000215000000000000000000000000000000000000
000c00001015010150121001215012150071001210014100121000320010200032001215012150121501210010150101501210012150121501210014100121000320012100141001210000000000001010012100
0001000004350063501135009350193500b3501f3502135024350283001430015300183001a3001b3001d3001e3001f3000000000000000000000000000000000000000000000000000000000000000000000000
001000002105000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000002d0502d0502d0502d05000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000100001c7501d7501e7501f75000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
__music__
01 07054344
02 06054344
03 090a4344
03 0b424344
03 0b0c4344

