//	Altirra - Atari 800/800XL emulator
//	Copyright (C) 2008 Avery Lee
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "stdafx.h"
#include "cpu.h"
#include "console.h"
#include "disasm.h"
#include "simulator.h"

namespace {
	enum {
		kStateReadOpcode,
		kStateReadOpcodeNoBreak,
		kStateReadDummyOpcode,
		kStateAddToHistory,
		kStateAddAsPathStart,
		kStateAddToPath,
		kStateBreakOnUnsupportedOpcode,
		kStateReadImm,
		kStateReadAddrL,
		kStateReadAddrH,
		kStateReadAddrHX,
		kStateReadAddrHY,
		kStateRead,
		kStateReadAddX,
		kStateReadAddY,
		kStateReadCarry,
		kStateReadAbsIndAddrBroken,
		kStateReadIndAddr,
		kStateReadIndYAddr,
		kStateWrite,
		kStateWriteZpX,
		kStateWriteZpY,
		kStateWriteAbsX,
		kStateWriteAbsY,
		kStateWait,
		kStateAtoD,
		kStateXtoD,
		kStateYtoD,
		kStateStoD,
		kStatePtoD,
		kStatePtoD_B0,
		kStatePtoD_B1,
		kStateDtoA,
		kStateDtoX,
		kStateDtoY,
		kStateDtoS,
		kStateDtoP,
		kStateDSetSZ,
		kStateDSetSV,
		kStateAddrToPC,
		kStateNMIVecToPC,
		kStateIRQVecToPC,
		kStatePush,
		kStatePushPCL,
		kStatePushPCH,
		kStatePushPCLM1,
		kStatePushPCHM1,
		kStatePop,
		kStatePopPCL,
		kStatePopPCH,
		kStatePopPCHP1,
		kStateAdc,
		kStateSbc,
		kStateCmp,
		kStateCmpX,
		kStateCmpY,
		kStateInc,
		kStateDec,
		kStateDecC,
		kStateAnd,
		kStateAnc,
		kStateOr,
		kStateXor,
		kStateAsl,
		kStateLsr,
		kStateRol,
		kStateRor,
		kStateBit,
		kStateSEI,
		kStateCLI,
		kStateSEC,
		kStateCLC,
		kStateSED,
		kStateCLD,
		kStateCLV,
		kStateJs,
		kStateJns,
		kStateJc,
		kStateJnc,
		kStateJz,
		kStateJnz,
		kStateJo,
		kStateJno,
		kStateJsAddToPath,
		kStateJnsAddToPath,
		kStateJcAddToPath,
		kStateJncAddToPath,
		kStateJzAddToPath,
		kStateJnzAddToPath,
		kStateJoAddToPath,
		kStateJnoAddToPath,
		kStateJccFalseRead,
		kStateInvokeHLE,
		kStateHLEDelay,
		kStateCount
	};
}

using namespace AT6502;

ATCPUEmulator::ATCPUEmulator()
	: mpHLE(NULL)
{
	mSBrk = 0x100;

	memset(mInsnFlags, 0, sizeof mInsnFlags);

	mbHistoryEnabled = false;
	mbPathfindingEnabled = false;

	RebuildDecodeTables();
}

ATCPUEmulator::~ATCPUEmulator() {
}

bool ATCPUEmulator::Init(ATCPUEmulatorMemory *mem) {
	mpMemory = mem;

	ColdReset();

	memset(mHistory, 0, sizeof mHistory);
	mHistoryIndex = 0;

	return true;
}

void ATCPUEmulator::SetHLE(IATCPUHighLevelEmulator *hle) {
	mpHLE = hle;
}

void ATCPUEmulator::ColdReset() {
	mbStep = false;
	mbTrace = false;
	WarmReset();
}

void ATCPUEmulator::WarmReset() {
	mpNextState = mStates;
	mbIRQPending = false;
	mbNMIPending = false;
	mPC = 0xFFFC;
	mP = 0x20 | kFlagI;
	mStates[0] = kStateReadAddrL;
	mStates[1] = kStateReadAddrH;
	mStates[2] = kStateAddrToPC;
	mStates[3] = kStateReadOpcode;
}

void ATCPUEmulator::SetPC(uint16 pc) {
	mPC = pc;
	mpDstState = mStates;
	mpNextState = mStates;
	*mpDstState++ = kStateReadOpcodeNoBreak;
}

void ATCPUEmulator::SetHook(uint16 pc, bool enable) {
	if (enable)
		mInsnFlags[pc] |= kInsnFlagHook;
	else
		mInsnFlags[pc] &= ~kInsnFlagHook;
}

sint32 ATCPUEmulator::GetNextBreakpoint(sint32 last) const {
	if ((uint32)last >= 0x10000)
		last = -1;

	for(++last; last < 0x10000; ++last) {
		if (mInsnFlags[last] & kInsnFlagBreakPt)
			return last;
	}

	return -1;
}

void ATCPUEmulator::SetHistoryEnabled(bool enable) {
	if (mbHistoryEnabled != enable) {
		mbHistoryEnabled = enable;
		RebuildDecodeTables();
	}
}

void ATCPUEmulator::SetPathfindingEnabled(bool enable) {
	if (mbPathfindingEnabled != enable) {
		mbPathfindingEnabled = enable;
		RebuildDecodeTables();
	}
}

bool ATCPUEmulator::IsBreakpointSet(uint16 addr) const {
	return 0 != (mInsnFlags[addr] & kInsnFlagBreakPt);
}

void ATCPUEmulator::SetBreakpoint(uint16 addr) {
	mInsnFlags[addr] |= kInsnFlagBreakPt;
}

void ATCPUEmulator::SetOneShotBreakpoint(uint16 addr) {
	mInsnFlags[addr] |= kInsnFlagBreakPt | kInsnFlagOneShotBreakPt;
}

void ATCPUEmulator::ClearBreakpoint(uint16 addr) {
	mInsnFlags[addr] &= ~(kInsnFlagBreakPt | kInsnFlagOneShotBreakPt);
}

void ATCPUEmulator::ClearAllBreakpoints() {
	for(uint32 i=0; i<65536; ++i)
		mInsnFlags[i] &= ~(kInsnFlagBreakPt | kInsnFlagOneShotBreakPt);
}

void ATCPUEmulator::ResetAllPaths() {
	for(uint32 i=0; i<65536; ++i)
		mInsnFlags[i] &= ~(kInsnFlagPathStart | kInsnFlagPathExecuted);
}

sint32 ATCPUEmulator::GetNextPathInstruction(sint32 last) const {
	if ((uint32)last >= 0x10000)
		last = -1;

	for(++last; last < 0x10000; ++last) {
		if (mInsnFlags[last] & (kInsnFlagPathStart | kInsnFlagPathExecuted))
			return last;
	}

	return -1;
}

bool ATCPUEmulator::IsPathStart(uint16 addr) const {
	return 0 != (mInsnFlags[addr] & kInsnFlagPathStart);
}

bool ATCPUEmulator::IsInPath(uint16 addr) const {
	return 0 != (mInsnFlags[addr] & (kInsnFlagPathStart | kInsnFlagPathExecuted));
}

void ATCPUEmulator::DumpStatus() {
	ATConsoleTaggedPrintf("PC=%04X A=%02X X=%02X Y=%02X S=%02X P=%02X (%c%c%c%c%c%c%c%c)  "
		, mPC
		, mA
		, mX
		, mY
		, mS
		, mP
		, mP & 0x80 ? 'N' : ' '
		, mP & 0x40 ? 'V' : ' '
		, mP & 0x20 ? '1' : ' '
		, mP & 0x10 ? 'B' : ' '
		, mP & 0x08 ? 'D' : ' '
		, mP & 0x04 ? 'I' : ' '
		, mP & 0x02 ? 'Z' : ' '
		, mP & 0x01 ? 'C' : ' '
		);

	ATDisassembleInsn(mPC);
}

void ATCPUEmulator::InjectOpcode(uint8 op) {
	mOpcode = op;
	mpNextState = mpDecodePtrs[mOpcode];
}

void ATCPUEmulator::Push(uint8 v) {
	mpMemory->WriteByte(0x100 + mS, v);
	--mS;
}

uint8 ATCPUEmulator::Pop() {
	return mpMemory->ReadByte(0x100 + (uint8)++mS);
}

void ATCPUEmulator::Jump(uint16 pc) {
	mPC = pc;
	mpDstState = mStates;
	mpNextState = mStates;
	*mpDstState++ = kStateReadOpcode;
}

void ATCPUEmulator::AssertIRQ() {
	mbIRQPending = true;
//	ATConsoleTaggedPrintf("CPU: Raising IRQ\n");
}

void ATCPUEmulator::NegateIRQ() {
	mbIRQPending = false;
//	ATConsoleTaggedPrintf("CPU: Lowering IRQ\n");
}

void ATCPUEmulator::AssertNMI() {
	mbNMIPending = true;
}

//#include <vd2/system/file.h>
//VDFile f("e:\\boot.txt", nsVDFile::kWrite | nsVDFile::kDenyRead | nsVDFile::kCreateAlways);

int ATCPUEmulator::Advance(bool busLocked) {
	if (busLocked)
		return kATSimEvent_None;

	for(;;) {
		switch(*mpNextState++) {
			case kStateReadOpcode:
				{
					uint8 iflags = mInsnFlags[mPC];

					if (iflags & (kInsnFlagBreakPt | kInsnFlagSpecialTracePt)) {
						if (mInsnFlags[mPC] & kInsnFlagOneShotBreakPt)
							mInsnFlags[mPC] &= ~(kInsnFlagBreakPt | kInsnFlagOneShotBreakPt);

						RedecodeInsnWithoutBreak();
						return kATSimEvent_CPUPCBreakpoint;
					}
				}

				if (mbStep) {
					mbStep = false;
					RedecodeInsnWithoutBreak();
					return kATSimEvent_CPUSingleStep;
				}

				if (mS >= mSBrk) {
					mSBrk = 0x100;
					RedecodeInsnWithoutBreak();
					return kATSimEvent_CPUStackBreakpoint;
				}

				if (mbTrace)
					DumpStatus();

#if 0
				{
					static bool foo = false;

					if (mPC == 0xf1e1)
						foo = true;
					
					if (foo) {
						static char buf[256];
						sprintf(buf, "PC=%04X A=%02X X=%02X Y=%02X S=%02X P=%02X (%c%c%c%c%c%c%c%c)  "
							, mPC
							, mA
							, mX
							, mY
							, mS
							, mP
							, mP & 0x80 ? 'N' : ' '
							, mP & 0x40 ? 'V' : ' '
							, mP & 0x20 ? '1' : ' '
							, mP & 0x10 ? 'B' : ' '
							, mP & 0x08 ? 'D' : ' '
							, mP & 0x04 ? 'I' : ' '
							, mP & 0x02 ? 'Z' : ' '
							, mP & 0x01 ? 'C' : ' '
							);
						f.write(buf, strlen(buf));
						ATDisassembleInsn(buf, mPC);
						f.write(buf, strlen(buf));
					}
				}
#endif
			case kStateReadOpcodeNoBreak:
				{
					uint8 iflags = mInsnFlags[mPC];

					if (iflags & kInsnFlagHook) {
						uint8 op = mpMemory->CPUHookHit(mPC);

						if (op) {
							++mPC;
							mOpcode = op;
							mpNextState = mpDecodePtrs[mOpcode];
							return kATSimEvent_None;
						}
					}
				}

				if (mbNMIPending) {
					if (mbTrace)
						ATConsoleWrite("CPU: Jumping to NMI vector\n");

					mbNMIPending = false;

					mpNextState = mStates;
					mpDstState = mStates;
					*mpDstState++ = kStateReadDummyOpcode;
					*mpDstState++ = kStateReadDummyOpcode;
					*mpDstState++ = kStatePushPCH;
					*mpDstState++ = kStatePushPCL;
					*mpDstState++ = kStatePtoD_B0;
					*mpDstState++ = kStatePush;
					*mpDstState++ = kStateSEI;
					*mpDstState++ = kStateNMIVecToPC;
					*mpDstState++ = kStateReadAddrL;
					*mpDstState++ = kStateReadAddrH;
					*mpDstState++ = kStateAddrToPC;

					if (mbPathfindingEnabled)
						*mpDstState++ = kStateAddAsPathStart;

					*mpDstState++ = kStateReadOpcode;
					continue;
				}

				if (mbIRQPending && !(mP & kFlagI)) {
					if (mbTrace)
						ATConsoleWrite("CPU: Jumping to IRQ vector\n");

					mpNextState = mStates;
					mpDstState = mStates;
					*mpDstState++ = kStateReadDummyOpcode;
					*mpDstState++ = kStateReadDummyOpcode;
					*mpDstState++ = kStatePushPCH;
					*mpDstState++ = kStatePushPCL;
					*mpDstState++ = kStatePtoD_B0;
					*mpDstState++ = kStatePush;
					*mpDstState++ = kStateSEI;
					*mpDstState++ = kStateIRQVecToPC;
					*mpDstState++ = kStateReadAddrL;
					*mpDstState++ = kStateReadAddrH;
					*mpDstState++ = kStateAddrToPC;

					if (mbPathfindingEnabled)
						*mpDstState++ = kStateAddAsPathStart;

					*mpDstState++ = kStateReadOpcode;
					continue;
				}

				mOpcode = mpMemory->ReadByte(mPC++);
				mpNextState = mpDecodePtrs[mOpcode];
				return kATSimEvent_None;

			case kStateReadDummyOpcode:
				mpMemory->ReadByte(mPC);
				return kATSimEvent_None;

			case kStateAddToHistory:
				{
					HistoryEntry& he = mHistory[mHistoryIndex++ & 131071];

					he.mTimestamp = mpMemory->GetTimestamp();
					he.mPC = mPC - 1;
					he.mS = mS;
					he.mP = mP;
					he.mA = mA;
					he.mX = mX;
					he.mY = mY;
				}
				break;

			case kStateAddAsPathStart:
				if (!(mInsnFlags[mPC] & kInsnFlagPathStart)) {
					mInsnFlags[mPC] |= kInsnFlagPathStart;
				}
				break;

			case kStateAddToPath:
				{
					uint16 adjpc = mPC - 1;
					if (!(mInsnFlags[adjpc] & kInsnFlagPathExecuted))
						mInsnFlags[adjpc] |= kInsnFlagPathExecuted;
				}
				break;

			case kStateBreakOnUnsupportedOpcode:
				--mPC;
				if (!mbTrace)
					DumpStatus();
				return kATSimEvent_CPUPCBreakpoint;

			case kStateReadImm:
				mData = mpMemory->ReadByte(mPC++);
				return kATSimEvent_None;

			case kStateReadAddrL:
				mAddr = mpMemory->ReadByte(mPC++);
				return kATSimEvent_None;

			case kStateReadAddrH:
				mAddr += mpMemory->ReadByte(mPC++) << 8;
				return kATSimEvent_None;

			case kStateReadAddrHX:
				mAddr += mpMemory->ReadByte(mPC++) << 8;
				mAddr2 = (mAddr & 0xff00) + ((mAddr + mX) & 0x00ff);
				mAddr = mAddr + mX;
				return kATSimEvent_None;

			case kStateReadAddrHY:
				mAddr += mpMemory->ReadByte(mPC++) << 8;
				mAddr2 = (mAddr & 0xff00) + ((mAddr + mY) & 0x00ff);
				mAddr = mAddr + mY;
				return kATSimEvent_None;

			case kStateRead:
				mData = mpMemory->ReadByte(mAddr);
				return kATSimEvent_None;

			case kStateReadAddX:
				mData = mpMemory->ReadByte(mAddr);
				mAddr = (uint8)(mAddr + mX);
				return kATSimEvent_None;

			case kStateReadAddY:
				mData = mpMemory->ReadByte(mAddr);
				mAddr = (uint8)(mAddr + mY);
				return kATSimEvent_None;

			case kStateReadCarry:
				mData = mpMemory->ReadByte(mAddr2);
				if (mAddr == mAddr2)
					++mpNextState;
				return kATSimEvent_None;

			case kStateWrite:
				mpMemory->WriteByte(mAddr, mData);
				return kATSimEvent_None;

			case kStateReadAbsIndAddrBroken:
				mAddr = mData + ((uint16)mpMemory->ReadByte((mAddr & 0xff00) + ((mAddr + 1) & 0xff)) << 8);
				return kATSimEvent_None;

			case kStateReadIndAddr:
				mAddr = mData + ((uint16)mpMemory->ReadByte(0xff & (mAddr + 1)) << 8);
				return kATSimEvent_None;

			case kStateReadIndYAddr:
				mAddr = mData + ((uint16)mpMemory->ReadByte(0xff & (mAddr + 1)) << 8);
				mAddr2 = (mAddr & 0xff00) + ((mAddr + mY) & 0x00ff);
				mAddr = mAddr + mY;
				return kATSimEvent_None;

			case kStateWait:
				return kATSimEvent_None;

			case kStateAtoD:
				mData = mA;
				break;

			case kStateXtoD:
				mData = mX;
				break;

			case kStateYtoD:
				mData = mY;
				break;

			case kStateStoD:
				mData = mS;
				break;

			case kStatePtoD:
				mData = mP;
				break;

			case kStatePtoD_B0:
				mData = mP & ~kFlagB;
				break;

			case kStatePtoD_B1:
				mData = mP | kFlagB;
				break;

			case kStateDtoA:
				mA = mData;
				break;

			case kStateDtoX:
				mX = mData;
				break;

			case kStateDtoY:
				mY = mData;
				break;

			case kStateDtoS:
				mS = mData;
				break;

			case kStateDtoP:
				mP = mData | 0x20;
				break;

			case kStateDSetSZ:
				mP &= ~(kFlagN | kFlagZ);
				if (mData & 0x80)
					mP |= kFlagN;
				if (!mData)
					mP |= kFlagZ;
				break;

			case kStateDSetSV:
				mP &= ~(kFlagN | kFlagV);
				mP |= (mData & 0xC0);
				break;

			case kStateAddrToPC:
				mPC = mAddr;
				break;

			case kStateNMIVecToPC:
				mPC = 0xFFFA;
				break;

			case kStateIRQVecToPC:
				mPC = 0xFFFE;
				break;

			case kStatePush:
				mpMemory->WriteByte(0x100 + (uint8)mS--, mData);
				return kATSimEvent_None;

			case kStatePushPCL:
				mpMemory->WriteByte(0x100 + (uint8)mS--, mPC & 0xff);
				return kATSimEvent_None;

			case kStatePushPCH:
				mpMemory->WriteByte(0x100 + (uint8)mS--, mPC >> 8);
				return kATSimEvent_None;

			case kStatePushPCLM1:
				mpMemory->WriteByte(0x100 + (uint8)mS--, (mPC - 1) & 0xff);
				return kATSimEvent_None;

			case kStatePushPCHM1:
				mpMemory->WriteByte(0x100 + (uint8)mS--, (mPC - 1) >> 8);
				return kATSimEvent_None;

			case kStatePop:
				mData = mpMemory->ReadByte(0x100 + (uint8)++mS);
				return kATSimEvent_None;

			case kStatePopPCL:
				mPC = mpMemory->ReadByte(0x100 + (uint8)++mS);
				return kATSimEvent_None;

			case kStatePopPCH:
				mPC += mpMemory->ReadByte(0x100 + (uint8)++mS) << 8;
				return kATSimEvent_None;

			case kStatePopPCHP1:
				mPC += mpMemory->ReadByte(0x100 + (uint8)++mS) << 8;
				++mPC;
				return kATSimEvent_None;

			case kStateAdc:
				if (mP & kFlagD) {
					uint32 lowResult = (mA & 15) + (mData & 15) + (mP & kFlagC);
					if (lowResult >= 10)
						lowResult += 6;

					uint32 highResult = (mA & 0xf0) + (mData & 0xf0) + lowResult;

					if (highResult >= 0xA0)
						highResult += 0x60;

					mP &= ~kFlagC;

					if (highResult >= 0x100)
						mP |= kFlagC;

					mA = (uint8)highResult;
				} else {
					uint32 carry7 = (mA & 0x7f) + (mData & 0x7f) + (mP & kFlagC);
					uint32 result = carry7 + (mA & 0x80) + (mData & 0x80);

					mP &= ~(kFlagC | kFlagN | kFlagZ | kFlagV);

					if (result & 0x80)
						mP |= kFlagN;

					if (result >= 0x100)
						mP |= kFlagC;

					if (!(result & 0xff))
						mP |= kFlagZ;

					mP |= ((result >> 2) ^ (carry7 >> 1)) & kFlagV;

					mA = (uint8)result;
				}
				break;

			case kStateSbc:
				if (mP & kFlagD) {
					// Pole Position needs N set properly here for its passing counter
					// to stop correctly!

					mData ^= 0xff;

					// Flags set according to binary op
					uint32 carry7 = (mA & 0x7f) + (mData & 0x7f) + (mP & kFlagC);
					uint32 result = carry7 + (mA & 0x80) + (mData & 0x80);

					// BCD
					uint32 lowResult = (mA & 15) + (mData & 15) + (mP & kFlagC);
					if (lowResult < 0x10)
						lowResult -= 6;

					uint32 highResult = (mA & 0xf0) + (mData & 0xf0) + lowResult;

					if (highResult < 0x100)
						highResult -= 0x60;

					mP &= ~(kFlagC | kFlagN | kFlagZ | kFlagV);

					if (result & 0x80)
						mP |= kFlagN;

					if (result >= 0x100)
						mP |= kFlagC;

					if (!(result & 0xff))
						mP |= kFlagZ;

					mP |= ((result >> 2) ^ (carry7 >> 1)) & kFlagV;

					mA = (uint8)highResult;
				} else {
					mData ^= 0xff;
					uint32 carry7 = (mA & 0x7f) + (mData & 0x7f) + (mP & kFlagC);
					uint32 result = carry7 + (mA & 0x80) + (mData & 0x80);

					mP &= ~(kFlagC | kFlagN | kFlagZ | kFlagV);

					if (result & 0x80)
						mP |= kFlagN;

					if (result >= 0x100)
						mP |= kFlagC;

					if (!(result & 0xff))
						mP |= kFlagZ;

					mP |= ((result >> 2) ^ (carry7 >> 1)) & kFlagV;

					mA = (uint8)result;
				}
				break;

			case kStateCmp:
				{
					mData ^= 0xff;
					uint32 result = mA + mData + 1;

					mP &= ~(kFlagC | kFlagN | kFlagZ);

					if (result & 0x80)
						mP |= kFlagN;

					if (result >= 0x100)
						mP |= kFlagC;

					if (!(result & 0xff))
						mP |= kFlagZ;
				}
				break;

			case kStateCmpX:
				{
					mData ^= 0xff;
					uint32 result = mX + mData + 1;

					mP &= ~(kFlagC | kFlagN | kFlagZ);

					if (result & 0x80)
						mP |= kFlagN;

					if (result >= 0x100)
						mP |= kFlagC;

					if (!(result & 0xff))
						mP |= kFlagZ;
				}
				break;

			case kStateCmpY:
				{
					mData ^= 0xff;
					uint32 result = mY + mData + 1;

					mP &= ~(kFlagC | kFlagN | kFlagZ);

					if (result & 0x80)
						mP |= kFlagN;

					if (result >= 0x100)
						mP |= kFlagC;

					if (!(result & 0xff))
						mP |= kFlagZ;
				}
				break;

			case kStateInc:
				++mData;
				mP &= ~(kFlagN | kFlagZ);
				if (mData & 0x80)
					mP |= kFlagN;
				if (!mData)
					mP |= kFlagZ;
				break;

			case kStateDec:
				--mData;
				mP &= ~(kFlagN | kFlagZ);
				if (mData & 0x80)
					mP |= kFlagN;
				if (!mData)
					mP |= kFlagZ;
				break;

			case kStateDecC:
				--mData;
				mP |= kFlagC;
				if (!mData)
					mP &= ~kFlagC;
				break;

			case kStateAnd:
				mData &= mA;
				mP &= ~(kFlagN | kFlagZ);
				if (mData & 0x80)
					mP |= kFlagN;
				if (!mData)
					mP |= kFlagZ;
				break;

			case kStateAnc:
				mData &= mA;
				mP &= ~(kFlagN | kFlagZ | kFlagC);
				if (mData & 0x80)
					mP |= kFlagN | kFlagC;
				if (!mData)
					mP |= kFlagZ;
				break;

			case kStateOr:
				mA |= mData;
				mP &= ~(kFlagN | kFlagZ);
				if (mA & 0x80)
					mP |= kFlagN;
				if (!mA)
					mP |= kFlagZ;
				break;

			case kStateXor:
				mA ^= mData;
				mP &= ~(kFlagN | kFlagZ);
				if (mA & 0x80)
					mP |= kFlagN;
				if (!mA)
					mP |= kFlagZ;
				break;

			case kStateAsl:
				mP &= ~(kFlagN | kFlagZ | kFlagC);
				if (mData & 0x80)
					mP |= kFlagC;
				mData += mData;
				if (mData & 0x80)
					mP |= kFlagN;
				if (!mData)
					mP |= kFlagZ;
				break;

			case kStateLsr:
				mP &= ~(kFlagN | kFlagZ | kFlagC);
				if (mData & 0x01)
					mP |= kFlagC;
				mData >>= 1;
				if (!mData)
					mP |= kFlagZ;
				break;

			case kStateRol:
				{
					uint32 result = (uint32)mData + (uint32)mData + (mP & kFlagC);
					mP &= ~(kFlagN | kFlagZ | kFlagC);
					if (result & 0x100)
						mP |= kFlagC;
					mData = (uint8)result;
					if (mData & 0x80)
						mP |= kFlagN;
					if (!mData)
						mP |= kFlagZ;
				}
				break;

			case kStateRor:
				{
					uint32 result = (mData >> 1) + ((mP & kFlagC) << 7);
					mP &= ~(kFlagN | kFlagZ | kFlagC);
					if (mData & 0x1)
						mP |= kFlagC;
					mData = (uint8)result;
					if (mData & 0x80)
						mP |= kFlagN;
					if (!mData)
						mP |= kFlagZ;
				}
				break;

			case kStateBit:
				{
					uint8 result = mData & mA;
					mP &= ~kFlagZ;
					if (!result)
						mP |= kFlagZ;
				}
				break;

			case kStateSEI:
				mP |= kFlagI;
				break;

			case kStateCLI:
				mP &= ~kFlagI;
				break;

			case kStateSEC:
				mP |= kFlagC;
				break;

			case kStateCLC:
				mP &= ~kFlagC;
				break;

			case kStateSED:
				mP |= kFlagD;
				break;

			case kStateCLD:
				mP &= ~kFlagD;
				break;

			case kStateCLV:
				mP &= ~kFlagV;
				break;

			case kStateJs:
				if (!(mP & kFlagN)) {
					++mpNextState;
					break;
				}

				mpMemory->ReadByte(mPC);
				mAddr = mPC & 0xff00;
				mPC += (sint16)(sint8)mData;
				mAddr += mPC & 0xff;
				if (mAddr == mPC)
					++mpNextState;
				return kATSimEvent_None;

			case kStateJns:
				if (mP & kFlagN) {
					++mpNextState;
					break;
				}

				mpMemory->ReadByte(mPC);
				mAddr = mPC & 0xff00;
				mPC += (sint16)(sint8)mData;
				mAddr += mPC & 0xff;
				if (mAddr == mPC)
					++mpNextState;
				return kATSimEvent_None;

			case kStateJc:
				if (!(mP & kFlagC)) {
					++mpNextState;
					break;
				}

				mpMemory->ReadByte(mPC);
				mAddr = mPC & 0xff00;
				mPC += (sint16)(sint8)mData;
				mAddr += mPC & 0xff;
				if (mAddr == mPC)
					++mpNextState;
				return kATSimEvent_None;

			case kStateJnc:
				if (mP & kFlagC) {
					++mpNextState;
					break;
				}

				mpMemory->ReadByte(mPC);
				mAddr = mPC & 0xff00;
				mPC += (sint16)(sint8)mData;
				mAddr += mPC & 0xff;
				if (mAddr == mPC)
					++mpNextState;
				return kATSimEvent_None;

			case kStateJz:
				if (!(mP & kFlagZ)) {
					++mpNextState;
					break;
				}

				mpMemory->ReadByte(mPC);
				mAddr = mPC & 0xff00;
				mPC += (sint16)(sint8)mData;
				mAddr += mPC & 0xff;
				if (mAddr == mPC)
					++mpNextState;
				return kATSimEvent_None;

			case kStateJnz:
				if (mP & kFlagZ) {
					++mpNextState;
					break;
				}

				mpMemory->ReadByte(mPC);
				mAddr = mPC & 0xff00;
				mPC += (sint16)(sint8)mData;
				mAddr += mPC & 0xff;
				if (mAddr == mPC)
					++mpNextState;
				return kATSimEvent_None;

			case kStateJo:
				if (!(mP & kFlagV)) {
					++mpNextState;
					break;
				}

				mpMemory->ReadByte(mPC);
				mAddr = mPC & 0xff00;
				mPC += (sint16)(sint8)mData;
				mAddr += mPC & 0xff;
				if (mAddr == mPC)
					++mpNextState;
				return kATSimEvent_None;

			case kStateJno:
				if (mP & kFlagV) {
					++mpNextState;
					break;
				}

				mpMemory->ReadByte(mPC);
				mAddr = mPC & 0xff00;
				mPC += (sint16)(sint8)mData;
				mAddr += mPC & 0xff;
				if (mAddr == mPC)
					++mpNextState;
				return kATSimEvent_None;

			/////////
			case kStateJsAddToPath:
				if (!(mP & kFlagN)) {
					++mpNextState;
					break;
				}

				mpMemory->ReadByte(mPC);
				mAddr = mPC & 0xff00;
				mPC += (sint16)(sint8)mData;
				mAddr += mPC & 0xff;
				mInsnFlags[mPC] |= kInsnFlagPathStart;
				if (mAddr == mPC)
					++mpNextState;
				return kATSimEvent_None;

			case kStateJnsAddToPath:
				if (mP & kFlagN) {
					++mpNextState;
					break;
				}

				mpMemory->ReadByte(mPC);
				mAddr = mPC & 0xff00;
				mPC += (sint16)(sint8)mData;
				mAddr += mPC & 0xff;
				mInsnFlags[mPC] |= kInsnFlagPathStart;
				if (mAddr == mPC)
					++mpNextState;
				return kATSimEvent_None;

			case kStateJcAddToPath:
				if (!(mP & kFlagC)) {
					++mpNextState;
					break;
				}

				mpMemory->ReadByte(mPC);
				mAddr = mPC & 0xff00;
				mPC += (sint16)(sint8)mData;
				mAddr += mPC & 0xff;
				mInsnFlags[mPC] |= kInsnFlagPathStart;
				if (mAddr == mPC)
					++mpNextState;
				return kATSimEvent_None;

			case kStateJncAddToPath:
				if (mP & kFlagC) {
					++mpNextState;
					break;
				}

				mpMemory->ReadByte(mPC);
				mAddr = mPC & 0xff00;
				mPC += (sint16)(sint8)mData;
				mAddr += mPC & 0xff;
				mInsnFlags[mPC] |= kInsnFlagPathStart;
				if (mAddr == mPC)
					++mpNextState;
				return kATSimEvent_None;

			case kStateJzAddToPath:
				if (!(mP & kFlagZ)) {
					++mpNextState;
					break;
				}

				mpMemory->ReadByte(mPC);
				mAddr = mPC & 0xff00;
				mPC += (sint16)(sint8)mData;
				mAddr += mPC & 0xff;
				mInsnFlags[mPC] |= kInsnFlagPathStart;
				if (mAddr == mPC)
					++mpNextState;
				return kATSimEvent_None;

			case kStateJnzAddToPath:
				if (mP & kFlagZ) {
					++mpNextState;
					break;
				}

				mpMemory->ReadByte(mPC);
				mAddr = mPC & 0xff00;
				mPC += (sint16)(sint8)mData;
				mAddr += mPC & 0xff;
				mInsnFlags[mPC] |= kInsnFlagPathStart;
				if (mAddr == mPC)
					++mpNextState;
				return kATSimEvent_None;

			case kStateJoAddToPath:
				if (!(mP & kFlagV)) {
					++mpNextState;
					break;
				}

				mpMemory->ReadByte(mPC);
				mAddr = mPC & 0xff00;
				mPC += (sint16)(sint8)mData;
				mAddr += mPC & 0xff;
				mInsnFlags[mPC] |= kInsnFlagPathStart;
				if (mAddr == mPC)
					++mpNextState;
				return kATSimEvent_None;

			case kStateJnoAddToPath:
				if (mP & kFlagV) {
					++mpNextState;
					break;
				}

				mpMemory->ReadByte(mPC);
				mAddr = mPC & 0xff00;
				mPC += (sint16)(sint8)mData;
				mAddr += mPC & 0xff;
				mInsnFlags[mPC] |= kInsnFlagPathStart;
				if (mAddr == mPC)
					++mpNextState;
				return kATSimEvent_None;

			case kStateJccFalseRead:
				mpMemory->ReadByte(mAddr);
				return kATSimEvent_None;

			case kStateInvokeHLE:
				if (mpHLE) {
					mHLEDelay = 0;
					int r = mpHLE->InvokeHLE(*this, *mpMemory, mPC - 1, mAddr);

					if (r == -2) {
						mpNextState += 2;
						break;
					}

					if (r >= 0)
						return r;
				}
				break;

			case kStateHLEDelay:
				if (mHLEDelay) {
					--mHLEDelay;
					return kATSimEvent_None;
				}
				break;
		}
	}
}

void ATCPUEmulator::RedecodeInsnWithoutBreak() {
	mpNextState = mStates;
	mStates[0] = kStateReadOpcodeNoBreak;
}

void ATCPUEmulator::RebuildDecodeTables() {
	mpDstState = mDecodeHeap;
	for(int i=0; i<256; ++i) {
		mpDecodePtrs[i] = mpDstState;
		Decode((uint8)i);
	}
}

void ATCPUEmulator::Decode(uint8 opcode) {
	if (mbHistoryEnabled)
		*mpDstState++ = kStateAddToHistory;

	if (mbPathfindingEnabled)
		*mpDstState++ = kStateAddToPath;

	switch(opcode) {
#if 1
		case 0x00:	// BRK
			*mpDstState++ = kStateReadAddrL;	// 1
			*mpDstState++ = kStateReadAddrH;	// 2
			*mpDstState++ = kStatePushPCHM1;	// 3
			*mpDstState++ = kStatePushPCLM1;	// 4
			*mpDstState++ = kStatePtoD_B1;
			*mpDstState++ = kStatePush;			// 5
			*mpDstState++ = kStateSEI;
			*mpDstState++ = kStateIRQVecToPC;
			*mpDstState++ = kStateReadAddrL;	// 6
			*mpDstState++ = kStateReadAddrH;	// 7
			*mpDstState++ = kStateAddrToPC;
			*mpDstState++ = kStateReadOpcode;
			break;
#endif

		case 0x01:	// ORA (zp,X)
			DecodeReadIndX();
			*mpDstState++ = kStateOr;
			break;

		case 0x02:
			*mpDstState++ = kStateReadAddrL;
			*mpDstState++ = kStateReadAddrH;
			*mpDstState++ = kStateInvokeHLE;
			*mpDstState++ = kStateHLEDelay;
			*mpDstState++ = kStateReadOpcode;
			*mpDstState++ = kStateBreakOnUnsupportedOpcode;
			*mpDstState++ = kStateReadOpcode;
			break;

		case 0x03:	// SLO (zp,X)
			DecodeReadIndX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateAsl;
			*mpDstState++ = kStateOr;
			*mpDstState++ = kStateWrite;
			break;

		case 0x04:	// NOP zp
			DecodeReadZp();
			*mpDstState++ = kStateWait;
			break;

		case 0x05:	// ORA zp
			DecodeReadZp();
			*mpDstState++ = kStateOr;
			break;

		case 0x06:	// ASL zp
			DecodeReadZp();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateAsl;
			*mpDstState++ = kStateWrite;
			break;

		case 0x07:	// SLO zp
			DecodeReadZp();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateAsl;
			*mpDstState++ = kStateOr;
			*mpDstState++ = kStateWrite;
			break;

		case 0x08:	// PHP
			*mpDstState++ = kStatePtoD;
			*mpDstState++ = kStateWait;
			*mpDstState++ = kStatePush;
			break;

		case 0x09:	// ORA imm
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = kStateOr;
			break;

		case 0x0A:	// ASL A
			*mpDstState++ = kStateAtoD;
			*mpDstState++ = kStateAsl;
			*mpDstState++ = kStateWait;
			*mpDstState++ = kStateDtoA;
			break;

		case 0x0B:	// AAC imm
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = kStateAnc;
			*mpDstState++ = kStateDtoA;
			break;

		case 0x0C:	// NOP abs
			DecodeReadAbs();
			break;

		case 0x0D:	// ORA abs
			DecodeReadAbs();
			*mpDstState++ = kStateOr;
			break;

		case 0x0E:	// ASL abs
			DecodeReadAbs();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateAsl;
			*mpDstState++ = kStateWrite;
			break;

		case 0x0F:	// SLO abs
			DecodeReadAbs();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateAsl;
			*mpDstState++ = kStateOr;
			*mpDstState++ = kStateWrite;
			break;

		case 0x10:	// BPL rel
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = mbPathfindingEnabled ? kStateJnsAddToPath : kStateJns;
			*mpDstState++ = kStateJccFalseRead;
			break;

		case 0x11:	// ORA (zp),Y
			DecodeReadIndY();
			*mpDstState++ = kStateOr;
			break;

		case 0x14:	// NOP zp,X
			DecodeReadZpX();
			break;

		case 0x15:	// ORA zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateOr;
			break;

		case 0x16:	// ASL zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateAsl;
			*mpDstState++ = kStateWrite;
			break;

		case 0x17:	// SLO zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateAsl;
			*mpDstState++ = kStateOr;
			*mpDstState++ = kStateWrite;
			break;

		case 0x18:	// CLC
			*mpDstState++ = kStateCLC;
			*mpDstState++ = kStateWait;
			break;

		case 0x19:	// ORA abs,Y
			DecodeReadAbsY();
			*mpDstState++ = kStateOr;
			break;

		case 0x1A:	// NOP*
			*mpDstState++ = kStateWait;
			break;

		case 0x1C:	// NOP abs,X
			DecodeReadAbsX();
			break;

		case 0x1D:	// ORA abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateOr;
			break;

		case 0x1E:	// ASL abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateAsl;
			*mpDstState++ = kStateWrite;
			break;

		case 0x1F:	// SLO abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateAsl;
			*mpDstState++ = kStateOr;
			*mpDstState++ = kStateWrite;
			break;

		case 0x20:	// JSR abs
			*mpDstState++ = kStateReadAddrL;
			*mpDstState++ = kStateReadAddrH;
			*mpDstState++ = kStatePushPCHM1;
			*mpDstState++ = kStatePushPCLM1;
			*mpDstState++ = kStateAddrToPC;
			*mpDstState++ = kStateWait;

			if (mbPathfindingEnabled)
				*mpDstState++ = kStateAddAsPathStart;
			break;

		case 0x21:	// AND (zp,X)
			DecodeReadIndX();
			*mpDstState++ = kStateAnd;
			*mpDstState++ = kStateDtoA;
			break;

		case 0x24:	// BIT zp
			DecodeReadZp();
			*mpDstState++ = kStateDSetSV;
			*mpDstState++ = kStateBit;
			break;

		case 0x25:	// AND zp
			DecodeReadZp();
			*mpDstState++ = kStateAnd;
			*mpDstState++ = kStateDtoA;
			break;

		case 0x26:	// ROL zp
			DecodeReadZp();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateRol;
			*mpDstState++ = kStateWrite;
			break;

		case 0x28:	// PLP
			*mpDstState++ = kStatePop;
			*mpDstState++ = kStateDtoP;
			*mpDstState++ = kStateWait;
			*mpDstState++ = kStateWait;
			break;

		case 0x29:	// AND imm
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = kStateAnd;
			*mpDstState++ = kStateDtoA;
			break;

		case 0x2A:	// ROL A
			*mpDstState++ = kStateAtoD;
			*mpDstState++ = kStateRol;
			*mpDstState++ = kStateWait;
			*mpDstState++ = kStateDtoA;
			break;

		case 0x2B:	// AAC imm
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = kStateAnc;
			*mpDstState++ = kStateDtoA;
			break;

		case 0x2C:	// BIT abs
			DecodeReadAbs();
			*mpDstState++ = kStateDSetSV;
			*mpDstState++ = kStateBit;
			break;

		case 0x2D:	// AND abs
			DecodeReadAbs();
			*mpDstState++ = kStateAnd;
			*mpDstState++ = kStateDtoA;
			break;

		case 0x2E:	// ROL abs
			DecodeReadAbs();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateRol;
			*mpDstState++ = kStateWrite;
			break;

		case 0x30:	// BMI rel
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = mbPathfindingEnabled ? kStateJsAddToPath : kStateJs;
			*mpDstState++ = kStateJccFalseRead;
			break;

		case 0x31:	// AND (zp),Y
			DecodeReadIndY();
			*mpDstState++ = kStateAnd;
			*mpDstState++ = kStateDtoA;
			break;

		case 0x35:	// AND zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateAnd;
			*mpDstState++ = kStateDtoA;
			break;

		case 0x36:	// ROL zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateRol;
			*mpDstState++ = kStateWrite;
			break;

		case 0x37:	// RLA Zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateRol;
			*mpDstState++ = kStateAnd;
			*mpDstState++ = kStateWrite;
			break;

		case 0x38:	// SEC
			*mpDstState++ = kStateSEC;
			*mpDstState++ = kStateWait;
			break;

		case 0x39:	// AND abs,Y
			DecodeReadAbsY();
			*mpDstState++ = kStateAnd;
			*mpDstState++ = kStateDtoA;
			break;

		case 0x3A:	// NOP*
			*mpDstState++ = kStateWait;
			break;

		case 0x3C:	// NOP abs,X
			DecodeReadAbsX();
			break;

		case 0x3D:	// AND abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateAnd;
			*mpDstState++ = kStateDtoA;
			break;

		case 0x3E:	// ROL abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateRol;
			*mpDstState++ = kStateWrite;
			break;

		case 0x40:	// RTI
			*mpDstState++ = kStatePop;
			*mpDstState++ = kStateDtoP;
			*mpDstState++ = kStatePopPCL;
			*mpDstState++ = kStatePopPCH;
			*mpDstState++ = kStateWait;
			*mpDstState++ = kStateWait;
			break;

		case 0x41:	// EOR (zp,X)
			DecodeReadIndX();
			*mpDstState++ = kStateXor;
			break;

		case 0x43:	// SRE (zp,X)
			DecodeReadIndX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateLsr;
			*mpDstState++ = kStateXor;
			*mpDstState++ = kStateWrite;
			break;

		case 0x44:	// NOP zp
			DecodeReadZp();
			break;

		case 0x45:	// EOR zp
			DecodeReadZp();
			*mpDstState++ = kStateXor;
			break;

		case 0x46:	// LSR zp
			DecodeReadZp();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateLsr;
			*mpDstState++ = kStateWrite;
			break;

		case 0x47:	// SRE zp
			DecodeReadZp();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateLsr;
			*mpDstState++ = kStateXor;
			*mpDstState++ = kStateWrite;
			break;

		case 0x48:	// PHA
			*mpDstState++ = kStateAtoD;
			*mpDstState++ = kStateWait;
			*mpDstState++ = kStatePush;
			break;

		case 0x49:	// EOR imm
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = kStateXor;
			break;

		case 0x4A:	// LSR A
			*mpDstState++ = kStateAtoD;
			*mpDstState++ = kStateLsr;
			*mpDstState++ = kStateWait;
			*mpDstState++ = kStateDtoA;
			break;

		case 0x4B:	// ASR imm
			*mpDstState++ = kStateAtoD;
			*mpDstState++ = kStateAnd;
			*mpDstState++ = kStateLsr;
			*mpDstState++ = kStateDtoA;
			break;

		case 0x4C:	// JMP abs
			*mpDstState++ = kStateReadAddrL;
			*mpDstState++ = kStateReadAddrH;
			*mpDstState++ = kStateAddrToPC;
			break;

		case 0x4D:	// EOR abs
			DecodeReadAbs();
			*mpDstState++ = kStateXor;
			break;

		case 0x4E:	// LSR abs
			DecodeReadAbs();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateLsr;
			*mpDstState++ = kStateWrite;
			break;

		case 0x4F:	// SRE abs
			DecodeReadAbs();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateLsr;
			*mpDstState++ = kStateXor;
			*mpDstState++ = kStateWrite;
			break;

		case 0x50:	// BVC
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = mbPathfindingEnabled ? kStateJnoAddToPath : kStateJno;
			*mpDstState++ = kStateJccFalseRead;
			break;

		case 0x51:	// EOR (zp),Y
			DecodeReadIndY();
			*mpDstState++ = kStateXor;
			break;

		case 0x53:	// SRE (zp),Y
			DecodeReadIndY();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateLsr;
			*mpDstState++ = kStateXor;
			*mpDstState++ = kStateWrite;
			break;

		case 0x54:	// NOP zp,X
			DecodeReadZpX();
			break;

		case 0x55:	// EOR zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateXor;
			break;

		case 0x56:	// LSR zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateLsr;
			*mpDstState++ = kStateWrite;
			break;

		case 0x57:	// SRE zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateLsr;
			*mpDstState++ = kStateXor;
			*mpDstState++ = kStateWrite;
			break;

		case 0x58:	// CLI
			*mpDstState++ = kStateCLI;
			*mpDstState++ = kStateWait;
			break;

		case 0x59:	// EOR abs,Y
			DecodeReadAbsY();
			*mpDstState++ = kStateXor;
			break;

		case 0x5A:	// NOP*
			*mpDstState++ = kStateWait;
			break;

		case 0x5B:	// SRE abs,Y
			DecodeReadAbsY();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateLsr;
			*mpDstState++ = kStateXor;
			*mpDstState++ = kStateWrite;
			break;

		case 0x5C:	// NOP abs,X
			DecodeReadAbsX();
			break;

		case 0x5D:	// EOR abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateXor;
			break;

		case 0x5E:	// LSR abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateLsr;
			*mpDstState++ = kStateWrite;
			break;

		case 0x5F:	// SRE abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateLsr;
			*mpDstState++ = kStateXor;
			*mpDstState++ = kStateWrite;
			break;

		case 0x60:	// RTS
			*mpDstState++ = kStatePopPCL;
			*mpDstState++ = kStatePopPCHP1;
			*mpDstState++ = kStateWait;
			*mpDstState++ = kStateWait;
			*mpDstState++ = kStateWait;
			break;

		case 0x61:	// ADC (zp,X)
			DecodeReadIndX();
			*mpDstState++ = kStateAdc;
			break;

		case 0x65:	// ADC zp
			DecodeReadZp();
			*mpDstState++ = kStateAdc;
			break;

		case 0x66:	// ROR zp
			DecodeReadZp();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateRor;
			*mpDstState++ = kStateWrite;
			break;

		case 0x68:	// PLA
			*mpDstState++ = kStatePop;
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoA;
			*mpDstState++ = kStateWait;
			*mpDstState++ = kStateWait;
			break;

		case 0x69:	// ADC imm
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = kStateAdc;
			break;

		case 0x6A:	// ROR A
			*mpDstState++ = kStateAtoD;
			*mpDstState++ = kStateRor;
			*mpDstState++ = kStateWait;
			*mpDstState++ = kStateDtoA;
			break;

		case 0x6C:	// JMP (abs)
			*mpDstState++ = kStateReadAddrL;
			*mpDstState++ = kStateReadAddrH;
			*mpDstState++ = kStateRead;
			*mpDstState++ = kStateReadAbsIndAddrBroken;
			*mpDstState++ = kStateAddrToPC;
			break;

		case 0x6D:	// ADC abs
			DecodeReadAbs();
			*mpDstState++ = kStateAdc;
			break;

		case 0x6E:	// ROR abs
			DecodeReadAbs();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateRor;
			*mpDstState++ = kStateWrite;
			break;

		case 0x70:	// BVS
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = mbPathfindingEnabled ? kStateJoAddToPath : kStateJo;
			*mpDstState++ = kStateJccFalseRead;
			break;

		case 0x71:	// ADC (zp),Y
			DecodeReadIndY();
			*mpDstState++ = kStateAdc;
			break;

		case 0x75:	// ADC zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateAdc;
			break;

		case 0x76:	// ROR zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateRor;
			*mpDstState++ = kStateWrite;
			break;

		case 0x78:	// SEI
			*mpDstState++ = kStateSEI;
			*mpDstState++ = kStateWait;
			break;

		case 0x79:	// ADC abs,Y
			DecodeReadAbsY();
			*mpDstState++ = kStateAdc;
			break;

		case 0x7A:	// NOP*
			*mpDstState++ = kStateWait;
			break;

		case 0x7C:	// NOP abs,X
			DecodeReadAbsX();
			break;

		case 0x7D:	// ADC abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateAdc;
			break;

		case 0x7E:	// ROR abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateRor;
			*mpDstState++ = kStateWrite;
			break;

		case 0x80:	// NOP
			*mpDstState++ = kStateWait;
			break;

		case 0x81:	// STA (zp,X)
			*mpDstState++ = kStateReadAddrL;
			*mpDstState++ = kStateReadAddX;
			*mpDstState++ = kStateRead;
			*mpDstState++ = kStateReadIndAddr;
			*mpDstState++ = kStateAtoD;
			*mpDstState++ = kStateWrite;
			break;

		case 0x84:	// STY zp
			*mpDstState++ = kStateReadAddrL;		
			*mpDstState++ = kStateYtoD;
			*mpDstState++ = kStateWrite;
			break;

		case 0x85:	// STA zp
			*mpDstState++ = kStateReadAddrL;		
			*mpDstState++ = kStateAtoD;
			*mpDstState++ = kStateWrite;	
			break;

		case 0x86:	// STX zp
			*mpDstState++ = kStateReadAddrL;		
			*mpDstState++ = kStateXtoD;
			*mpDstState++ = kStateWrite;	
			break;

		case 0x88:	// DEY
			*mpDstState++ = kStateYtoD;
			*mpDstState++ = kStateDec;
			*mpDstState++ = kStateDtoY;
			*mpDstState++ = kStateWait;
			break;

		case 0x8A:	// TXA
			*mpDstState++ = kStateXtoD;
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoA;
			*mpDstState++ = kStateWait;
			break;

		case 0x8C:	// STY abs
			*mpDstState++ = kStateReadAddrL;		
			*mpDstState++ = kStateReadAddrH;
			*mpDstState++ = kStateYtoD;
			*mpDstState++ = kStateWrite;
			break;

		case 0x8D:	// STA abs
			*mpDstState++ = kStateReadAddrL;		
			*mpDstState++ = kStateReadAddrH;
			*mpDstState++ = kStateAtoD;
			*mpDstState++ = kStateWrite;	
			break;

		case 0x8E:	// STX abs
			*mpDstState++ = kStateReadAddrL;		
			*mpDstState++ = kStateReadAddrH;
			*mpDstState++ = kStateXtoD;
			*mpDstState++ = kStateWrite;
			break;

		case 0x8F:	// SAX abs
			*mpDstState++ = kStateReadAddrL;		
			*mpDstState++ = kStateReadAddrH;
			*mpDstState++ = kStateXtoD;
			*mpDstState++ = kStateAnd;
			*mpDstState++ = kStateWrite;
			break;

		case 0x90:	// BCC rel8
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = mbPathfindingEnabled ? kStateJncAddToPath : kStateJnc;
			*mpDstState++ = kStateJccFalseRead;
			break;

		case 0x91:	// STA (zp),Y
			*mpDstState++ = kStateReadAddrL;
			*mpDstState++ = kStateRead;
			*mpDstState++ = kStateReadIndYAddr;
			*mpDstState++ = kStateAtoD;
			*mpDstState++ = kStateWait;
			*mpDstState++ = kStateWrite;
			break;

		case 0x94:	// STY zp,X
			*mpDstState++ = kStateReadAddrL;
			*mpDstState++ = kStateReadAddX;
			*mpDstState++ = kStateYtoD;
			*mpDstState++ = kStateWrite;
			break;

		case 0x95:	// STA zp,X
			*mpDstState++ = kStateReadAddrL;
			*mpDstState++ = kStateReadAddX;
			*mpDstState++ = kStateAtoD;
			*mpDstState++ = kStateWrite;	
			break;

		case 0x96:	// STX zp,Y
			*mpDstState++ = kStateReadAddrL;
			*mpDstState++ = kStateReadAddY;
			*mpDstState++ = kStateXtoD;
			*mpDstState++ = kStateWrite;
			break;

		case 0x98:	// TYA
			*mpDstState++ = kStateYtoD;
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoA;
			*mpDstState++ = kStateWait;
			break;

		case 0x99:	// STA abs,Y
			*mpDstState++ = kStateReadAddrL;
			*mpDstState++ = kStateReadAddrHY;
			*mpDstState++ = kStateAtoD;
			*mpDstState++ = kStateWait;
			*mpDstState++ = kStateWrite;
			break;

		case 0x9A:	// TXS
			*mpDstState++ = kStateXtoD;
			*mpDstState++ = kStateDtoS;
			*mpDstState++ = kStateWait;
			break;

		case 0x9D:	// STA abs,X
			*mpDstState++ = kStateReadAddrL;
			*mpDstState++ = kStateReadAddrHX;
			*mpDstState++ = kStateAtoD;
			*mpDstState++ = kStateWait;
			*mpDstState++ = kStateWrite;
			break;

		case 0x9E:	// STX abs,Y
			*mpDstState++ = kStateReadAddrL;
			*mpDstState++ = kStateReadAddrHY;
			*mpDstState++ = kStateXtoD;
			*mpDstState++ = kStateWait;
			*mpDstState++ = kStateWrite;
			break;

		case 0xA0:	// LDY imm
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoY;
			break;

		case 0xA1:	// LDA (zp,X)
			DecodeReadIndX();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoA;
			break;

		case 0xA2:	// LDX imm
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoX;
			break;

		case 0xA4:	// LDY zp
			DecodeReadZp();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoY;
			break;

		case 0xA5:	// LDA zp
			DecodeReadZp();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoA;
			break;

		case 0xA6:	// LDX zp
			DecodeReadZp();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoX;
			break;

		case 0xA8:	// TAY
			*mpDstState++ = kStateAtoD;
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoY;
			*mpDstState++ = kStateWait;
			break;

		case 0xA9:	// LDA imm
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoA;
			break;

		case 0xAA:	// TAX
			*mpDstState++ = kStateAtoD;
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoX;
			*mpDstState++ = kStateWait;
			break;

		case 0xAC:	// LDY abs
			DecodeReadAbs();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoY;
			break;

		case 0xAD:	// LDA abs
			DecodeReadAbs();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoA;
			break;

		case 0xAE:	// LDX abs
			DecodeReadAbs();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoX;
			break;

		case 0xAF:	// LAX abs
			DecodeReadAbs();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoX;
			*mpDstState++ = kStateDtoA;
			break;

		case 0xB0:	// BCS rel8
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = mbPathfindingEnabled ? kStateJcAddToPath : kStateJc;
			*mpDstState++ = kStateJccFalseRead;
			break;

		case 0xB1:	// LDA (zp),Y
			DecodeReadIndY();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoA;
			break;

		case 0xB3:	// LAX (zp),Y
			DecodeReadIndY();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoX;
			*mpDstState++ = kStateDtoA;
			break;

		case 0xB4:	// LDY zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoY;
			break;

		case 0xB5:	// LDA zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoA;
			break;

		case 0xB6:	// LDX zp,Y
			DecodeReadZpY();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoX;
			break;

		case 0xB8:	// CLV
			*mpDstState++ = kStateCLV;
			*mpDstState++ = kStateWait;
			break;

		case 0xB9:	// LDA abs,Y
			DecodeReadAbsY();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoA;
			break;

		case 0xBA:	// TSX
			*mpDstState++ = kStateStoD;
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoX;
			*mpDstState++ = kStateWait;
			break;

		case 0xBC:	// LDY abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoY;
			break;

		case 0xBD:	// LDA abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoA;
			break;

		case 0xBE:	// LDX abs,Y
			DecodeReadAbsY();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoX;
			break;

		case 0xBF:	// LAX abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateDSetSZ;
			*mpDstState++ = kStateDtoX;
			*mpDstState++ = kStateDtoA;
			break;

		case 0xC0:	// CPY imm
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = kStateCmpY;
			break;

		case 0xC1:	// CMP (zp,X)
			DecodeReadIndX();
			*mpDstState++ = kStateCmp;
			break;

		case 0xC3:	// DCP (zp,X)
			DecodeReadIndX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateDecC;
			*mpDstState++ = kStateWrite;
			break;

		case 0xC4:	// CPY zp
			DecodeReadZp();
			*mpDstState++ = kStateCmpY;
			break;

		case 0xC5:	// CMP zp
			DecodeReadZp();
			*mpDstState++ = kStateCmp;
			break;

		case 0xC6:	// DEC zp
			DecodeReadZp();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateDec;
			*mpDstState++ = kStateWrite;
			break;

		case 0xC8:	// INY
			*mpDstState++ = kStateYtoD;
			*mpDstState++ = kStateInc;
			*mpDstState++ = kStateDtoY;
			*mpDstState++ = kStateWait;
			break;

		case 0xC9:	// CMP imm
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = kStateCmp;
			break;

		case 0xCA:	// DEX
			*mpDstState++ = kStateXtoD;
			*mpDstState++ = kStateDec;
			*mpDstState++ = kStateDtoX;
			*mpDstState++ = kStateWait;
			break;

		case 0xCC:	// CPY abs
			DecodeReadAbs();
			*mpDstState++ = kStateCmpY;
			break;

		case 0xCD:	// CMP abs
			DecodeReadAbs();
			*mpDstState++ = kStateCmp;
			break;

		case 0xCE:	// DEC abs
			DecodeReadAbs();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateDec;
			*mpDstState++ = kStateWrite;
			break;

		case 0xCF:	// DCP abs
			DecodeReadAbs();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateDecC;
			*mpDstState++ = kStateWrite;
			break;

		case 0xD0:	// BNE rel8
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = mbPathfindingEnabled ? kStateJnzAddToPath : kStateJnz;
			*mpDstState++ = kStateJccFalseRead;
			break;

		case 0xD1:	// CMP (zp),Y
			DecodeReadIndY();
			*mpDstState++ = kStateCmp;
			break;

		case 0xD3:	// DCP (zp),Y
			DecodeReadIndY();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateDecC;
			*mpDstState++ = kStateWrite;
			break;

		case 0xD5:	// CMP zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateCmp;
			break;

		case 0xD6:	// DEC zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateDec;
			*mpDstState++ = kStateWrite;
			break;

		case 0xD8:	// CLD
			*mpDstState++ = kStateCLD;
			*mpDstState++ = kStateWait;
			break;

		case 0xD9:	// CMP abs,Y
			DecodeReadAbsY();
			*mpDstState++ = kStateCmp;
			break;

		case 0xDA:	// NOP*
			*mpDstState++ = kStateWait;
			break;

		case 0xDB:	// DCP abs,Y
			DecodeReadAbsY();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateDecC;
			*mpDstState++ = kStateWrite;
			break;

		case 0xDC:	// NOP abs,X
			DecodeReadAbsX();
			break;

		case 0xDD:	// CMP abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateCmp;
			break;

		case 0xDE:	// DEC abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateDec;
			*mpDstState++ = kStateWrite;
			break;

		case 0xDF:	// DCP abs,X
			DecodeReadIndX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateDecC;
			*mpDstState++ = kStateWrite;
			break;

		case 0xE0:	// CPX imm
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = kStateCmpX;
			break;

		case 0xE1:	// SBC (zp,X)
			DecodeReadIndX();
			*mpDstState++ = kStateSbc;
			break;

		case 0xE3:	// ISB zp
			DecodeReadZp();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateInc;
			*mpDstState++ = kStateSbc;
			*mpDstState++ = kStateWrite;
			break;

		case 0xE4:	// CPX zp
			DecodeReadZp();
			*mpDstState++ = kStateCmpX;
			break;

		case 0xE5:	// SBC zp
			DecodeReadZp();
			*mpDstState++ = kStateSbc;
			break;

		case 0xE6:	// INC zp
			DecodeReadZp();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateInc;
			*mpDstState++ = kStateWrite;
			break;

		case 0xE7:	// ISB zp
			DecodeReadAbsX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateInc;
			*mpDstState++ = kStateSbc;
			*mpDstState++ = kStateWrite;
			break;

		case 0xE8:	// INX
			*mpDstState++ = kStateXtoD;
			*mpDstState++ = kStateInc;
			*mpDstState++ = kStateDtoX;
			*mpDstState++ = kStateWait;
			break;

		case 0xE9:	// SBC imm
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = kStateSbc;
			break;

		case 0xEA:	// NOP
			*mpDstState++ = kStateWait;
			break;

		case 0xEC:	// CPX abs
			DecodeReadAbs();
			*mpDstState++ = kStateCmpX;
			break;

		case 0xED:	// SBC abs
			DecodeReadAbs();
			*mpDstState++ = kStateSbc;
			break;

		case 0xEE:	// INC abs
			DecodeReadAbs();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateInc;
			*mpDstState++ = kStateWrite;
			break;

		case 0xEF:	// ISB abs
			DecodeReadAbs();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateInc;
			*mpDstState++ = kStateSbc;
			*mpDstState++ = kStateWrite;
			break;

		case 0xF0:	// BEQ rel8
			*mpDstState++ = kStateReadImm;
			*mpDstState++ = mbPathfindingEnabled ? kStateJzAddToPath : kStateJz;
			*mpDstState++ = kStateJccFalseRead;
			break;

		case 0xF1:	// SBC (zp),Y
			DecodeReadIndY();
			*mpDstState++ = kStateSbc;
			break;

		case 0xF3:	// ISB zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateInc;
			*mpDstState++ = kStateSbc;
			*mpDstState++ = kStateWrite;
			break;

		case 0xF4:	// NOP zp,X
			DecodeReadZpX();
			break;

		case 0xF5:	// SBC zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateSbc;
			break;

		case 0xF6:	// INC zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateInc;
			*mpDstState++ = kStateWrite;
			break;

		case 0xF7:	// ISB zp,X
			DecodeReadZpX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateInc;
			*mpDstState++ = kStateSbc;
			*mpDstState++ = kStateWrite;
			break;

		case 0xF8:	// SED
			*mpDstState++ = kStateSED;
			*mpDstState++ = kStateWait;
			break;

		case 0xF9:	// SBC abs,Y
			DecodeReadAbsY();
			*mpDstState++ = kStateSbc;
			break;

		case 0xFA:	// NOP*
			*mpDstState++ = kStateWait;
			break;

		case 0xFB:	// ISB abs,Y
			DecodeReadAbsY();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateInc;
			*mpDstState++ = kStateSbc;
			*mpDstState++ = kStateWrite;
			break;

		case 0xFC:	// NOP abs,X
			DecodeReadAbsX();
			break;

		case 0xFD:	// SBC abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateSbc;
			break;

		case 0xFE:	// INC abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateInc;
			*mpDstState++ = kStateWrite;
			break;

		case 0xFF:	// ISB abs,X
			DecodeReadAbsX();
			*mpDstState++ = kStateWrite;
			*mpDstState++ = kStateInc;
			*mpDstState++ = kStateSbc;
			*mpDstState++ = kStateWrite;
			break;

		default:
			*mpDstState++ = kStateBreakOnUnsupportedOpcode;
			break;
	}

	*mpDstState++ = kStateReadOpcode;
}

void ATCPUEmulator::DecodeReadZp() {
	*mpDstState++ = kStateReadAddrL;
	*mpDstState++ = kStateRead;
}

void ATCPUEmulator::DecodeReadZpX() {
	*mpDstState++ = kStateReadAddrL;		// 2
	*mpDstState++ = kStateReadAddX;			// 3
	*mpDstState++ = kStateRead;				// 4
}

void ATCPUEmulator::DecodeReadZpY() {
	*mpDstState++ = kStateReadAddrL;		// 2
	*mpDstState++ = kStateReadAddY;			// 3
	*mpDstState++ = kStateRead;				// 4
}

void ATCPUEmulator::DecodeReadAbs() {
	*mpDstState++ = kStateReadAddrL;		// 2
	*mpDstState++ = kStateReadAddrH;		// 3
	*mpDstState++ = kStateRead;				// 4
}

void ATCPUEmulator::DecodeReadAbsX() {
	*mpDstState++ = kStateReadAddrL;		// 2
	*mpDstState++ = kStateReadAddrHX;		// 3
	*mpDstState++ = kStateReadCarry;		// 4
	*mpDstState++ = kStateRead;				// (5)
}

void ATCPUEmulator::DecodeReadAbsY() {
	*mpDstState++ = kStateReadAddrL;		// 2
	*mpDstState++ = kStateReadAddrHY;		// 3
	*mpDstState++ = kStateReadCarry;		// 4
	*mpDstState++ = kStateRead;				// (5)
}

void ATCPUEmulator::DecodeReadIndX() {
	*mpDstState++ = kStateReadAddrL;		// 2
	*mpDstState++ = kStateReadAddX;			// 3
	*mpDstState++ = kStateRead;				// 4
	*mpDstState++ = kStateReadIndAddr;		// 5
	*mpDstState++ = kStateRead;				// 6
}

void ATCPUEmulator::DecodeReadIndY() {
	*mpDstState++ = kStateReadAddrL;		// 2
	*mpDstState++ = kStateRead;				// 3
	*mpDstState++ = kStateReadIndYAddr;		// 4
	*mpDstState++ = kStateReadCarry;		// 5
	*mpDstState++ = kStateRead;				// (6)
}
