/****************************************************************************
File    : sound_win.c
/*
@(#) #SY# Atari800Win PLus
@(#) #IS# Sound implementation for Win32 platforms
@(#) #BY# Richard Lawrence, Tomasz Szymankowski
@(#) #LM# 07.03.2002
*/

#define DIRECTSOUND_VERSION		0x0500

#include <stdio.h>
#include <windows.h>
#include <mmsystem.h>
#include <limits.h>
#include <crtdbg.h>
#include <dsound.h>
#include "WinConfig.h"
#include "Resource.h"
#include "Helpers.h"
#include "Debug.h"
#include "atari800.h"
#include "globals.h"
#include "macros.h"
#include "registry.h"
#include "timing.h"
#include "avisave.h"
#include "display_win.h"
#include "misc_win.h"
#include "sound_win.h"


#define MM_NUMBER_OF_FRAGS			8
#define MM_FRAMES_PER_FRAG			2

#define DS_NUMBER_OF_FRAGS			2 /* Do not change the value! */
#define DS_FRAMES_PER_FRAG			3
/* TIP:
   Frames per fragment = sound latency in frames at present,
   because the interval between play and save cursors equals 1
*/
#define MM_PLAY_FRAG_NO				0
#define MM_SAVE_FRAG_NO				MM_PLAY_FRAG_NO
#define DS_PLAY_FRAG_NO				0
#define DS_SAVE_FRAG_NO				DS_PLAY_FRAG_NO

#define NUMBER_OF_EVENTS			2
#define POS_EVENT					0
#define END_EVENT					1

/* Public objects */

struct SoundCtrl_t g_Sound =
{
	DEF_SOUND_STATE,
	DEF_SOUND_RATE,
	DEF_SOUND_VOL,
	DEF_SKIP_UPDATE,
	DEF_SOUND_LATENCY,
	DEF_USE_VOLUME_ONLY,
	NULL
};

/* Private objects */

static WAVEHDR      s_arrWaveHDR[ MM_NUMBER_OF_FRAGS ];
static WAVEFORMATEX s_wfxWaveFormat;
static HWAVEOUT     s_hWaveOut              = 0;
static char        *s_pSoundBuffer          = NULL;
static char        *s_pPlayCursor           = NULL;
static char        *s_pSaveCursor           = NULL;
static volatile int s_nPlayFragNo           = MM_PLAY_FRAG_NO;
static volatile int s_nSaveFragNo           = MM_SAVE_FRAG_NO;
static int          s_nNumberOfFrags        = MM_NUMBER_OF_FRAGS;
static int          s_nFramesPerFrag        = MM_FRAMES_PER_FRAG;
static int          s_nFrameCnt             = 1;
static int          s_nUpdatesPerFrag       = TV_PAL * MM_FRAMES_PER_FRAG / DEF_SKIP_UPDATE;
static int          s_nUpdateCnt            = 1;
static int          s_nSkipUpdate           = DEF_SKIP_UPDATE;
static DWORD        s_dwBufferSize          = 0;
static DWORD        s_dwFragSize            = 0;
static DWORD        s_dwFragPos             = 0;
static DWORD        s_dwStartVolume         = 0;
static BOOL         s_bSoundIsPaused        = TRUE;

static const GUID IID_IDirectSoundNotify = { 0xb0210783, 0x89cd, 0x11d0, { 0xaf, 0x08, 0x00, 0xa0, 0xc9, 0x25, 0xcd, 0x16 } };

static LPDIRECTSOUND       s_lpDirectSound  = NULL;
static LPDIRECTSOUNDBUFFER s_lpDSBuffer     = NULL;
static LPDIRECTSOUNDNOTIFY s_lpDSNotify     = NULL;
static LPDSBPOSITIONNOTIFY s_lpDSNotifyPos  = NULL;
static HANDLE              s_hNotifyThread  = NULL;
static DWORD               s_dwThreadID     = 0;
static HANDLE              s_arrEvents[ NUMBER_OF_EVENTS ];

static void  SndPlay_NoSound( void );
static void  SndPlay_MMSound( void );
static void  SndPlay_DSSound( void );

DWORD WINAPI DSSoundThreadProc( LPVOID lpParam );

void (*Atari_PlaySound)(void) = SndPlay_NoSound;

static BOOL GetDSErrorString( HRESULT  hResult,  LPSTR lpszErrorBuff, DWORD dwError );
static BOOL GetMMErrorString( MMRESULT mmResult, LPSTR lpszErrorBuff, DWORD dwError );

#ifdef _DEBUG
#define ServeMMError( nUID, hResult, bQuit ) \
		ShowMMError( nUID, hResult, bQuit, __FILE__, __LINE__ )
#define ServeDSError( nUID, hResult, bQuit ) \
		ShowDSError( nUID, hResult, bQuit, __FILE__, __LINE__ )
#else /*_DEBUG*/
#define ServeMMError( nUID, hResult, bQuit ) \
		ShowMMError( nUID, hResult, bQuit )
#define ServeDSError( nUID, hResult, bQuit ) \
		ShowDSError( nUID, hResult, bQuit )
#endif /*_DEBUG*/


/*========================================================
Function : ShowMMError
=========================================================*/
/* #FN#
   Displays a multimedia sound error description */
static
void
/* #AS#
   Nothing */
ShowMMError(
	UINT     nUID,
	MMRESULT mmResult,
	BOOL     bQuit
#ifdef _DEBUG
  , char   *pszFile,
	DWORD   dwLine
#endif /*_DEBUG*/
)
{
	char szError [ LOADSTRING_SIZE_S ];
	char szAction[ LOADSTRING_SIZE_L ];

#ifdef _DEBUG
	Aprint( "Multimedia Sound error: %s@%ld", pszFile, dwLine );
#endif /*_DEBUG*/

	/* Get us back to a GDI display and disable sound */
	Screen_SetSafeDisplay( FALSE );
	Sound_Disable( TRUE );

	/* Get the error string and present it to the user */
	GetMMErrorString( mmResult, szError, LOADSTRING_SIZE_S );
	DisplayMessage( NULL, IDS_MMERR_PROMPT, IDS_MMERR_HDR, MB_ICONSTOP | MB_OK, _LoadStringLx( nUID, szAction ), szError );

	/* Start a quit (this will end up in Atari_Exit()) */
	if( bQuit )
	{
		/* Make sure the atari is turned off */
		g_ulAtariState = ATARI_UNINITIALIZED;
		PostMessage( g_hMainWnd, WM_CLOSE, 0, 0L );
	}
} /* #OF# ShowMMError */

/*========================================================
Function : ShowDSError
=========================================================*/
/* #FN#
   Displays a direct sound error description */
static
void
/* #AS#
   Nothing */
ShowDSError(
	UINT    nUID,
	HRESULT hResult,
	BOOL    bQuit
#ifdef _DEBUG
  , char   *pszFile,
	DWORD   dwLine
#endif /*_DEBUG*/
)
{
	char szError [ LOADSTRING_SIZE_S ];
	char szAction[ LOADSTRING_SIZE_L ];

#ifdef _DEBUG
	Aprint( "DirectSound error: %s@%ld", pszFile, dwLine );
#endif /*_DEBUG*/

	/* Get us back to a GDI display and disable sound */
	Screen_SetSafeDisplay( FALSE );
	Sound_Disable( TRUE );

	/* Get the error string and present it to the user */
	GetDSErrorString( hResult, szError, LOADSTRING_SIZE_S );
	DisplayMessage( NULL, IDS_DSERR_PROMPT, IDS_DSERR_HDR, MB_ICONSTOP | MB_OK, _LoadStringLx( nUID, szAction ), szError );

	/* Start a quit (this will end up in Atari_Exit()) */
	if( bQuit )
	{
		/* Make sure the atari is turned off */
		g_ulAtariState = ATARI_UNINITIALIZED;
		PostMessage( g_hMainWnd, WM_CLOSE, 0, 0L );
	}
} /* #OF# ShowDSError */

/*========================================================
Function : Sound_GetInterParms
=========================================================*/
/* #FN#
   Returns a sound information structure */
void
/* #AS#
   Nothing */
Sound_GetInterParms(
	struct SoundInterParms_t *pInfo
)
{
	/* Make copy of an used WAVEFORMATEX structure */ 
	if( pInfo->dwMask & SIP_WAVEFORMAT )
		CopyMemory( &pInfo->wfxFormat, &s_wfxWaveFormat, sizeof(WAVEFORMATEX) );

	/* Make copy of sample size value */ 
	if( pInfo->dwMask & SIP_SAMPLESIZE )
		pInfo->dwSampleSize = s_dwFragSize;
} /* #OF# Sound_GetInterParms */

/*========================================================
Function : DetermineHardwareCaps
=========================================================*/
/* #FN#
   Determines sound hardware capabilities */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
DetermineHardwareCaps(
	LPDIRECTSOUND lpDirectSound,
	BOOL          bPrimary
)
{
	DSCAPS  dscaps;
	char	szFailed[ LOADSTRING_SIZE_L * 2 ];
	char	szError [ LOADSTRING_SIZE_M ];
	HRESULT hResult;

	dscaps.dwSize = sizeof(DSCAPS);
	*szFailed = '\0';

	hResult = IDirectSound_GetCaps( lpDirectSound, &dscaps );

	if( SUCCEEDED(hResult) )
	{
		if( bPrimary )
		{
			if( !(dscaps.dwFlags & DSCAPS_PRIMARY8BIT) )
				strcat( szFailed, _LoadStringMx( IDS_DSERR_NO_PRIMARY, szError ) );

			if( !(dscaps.dwFlags & DSCAPS_PRIMARYMONO) )
				strcat( szFailed, _LoadStringMx( IDS_DSERR_NO_MONO, szError ) );

			if( stereo_enabled && !(dscaps.dwFlags & DSCAPS_PRIMARYSTEREO) )
				strcat( szFailed, _LoadStringMx( IDS_DSERR_NO_STEREO, szError ) );
		}
		if( dscaps.dwFlags & DSCAPS_EMULDRIVER  )
			strcat( szFailed, _LoadStringMx( IDS_DSERR_NO_DRIVER, szError ) );

		if( *szFailed )
		{
			Sound_Disable( TRUE );
			DisplayMessage( NULL, IDS_DSERR_INIT, IDS_DSERR_HDR, MB_ICONEXCLAMATION | MB_OK, szFailed );
			return FALSE;
		}
	}
	else
	{
		ServeDSError( IDS_DSERR_QUERY, hResult, FALSE );
		return FALSE;
	}
	return TRUE;
} /* #OF# DetermineHardwareCaps */

/*========================================================
Function : Sound_Initialise
=========================================================*/
/* #FN#
   Initializes the Waveout/DirectSound playback */
int
/* #AS#
   1 if succeeded, otherwise 0 */
Sound_Initialise( void )
{
	DWORD dwBufferSize    = s_dwBufferSize;
	DWORD dwFragSize      = 0;
	int   nNumberOfFrags  = 0;
	int   nFramesPerFrag  = 0;
	int   nUpdatesPerFrag = 0;
	int   nSkipUpdate     = g_Sound.nSkipUpdate;
	int   nChannels       = (stereo_enabled ? 2 : 1);
	int   i; /* Loop counter */

	_ASSERT(g_Sound.ulState & (SOUND_MMSOUND | SOUND_DSSOUND));

	/* We have to clear sound resources first */
	if( g_Sound.ulState & SOUND_MMSOUND )
	{
		nNumberOfFrags = MM_NUMBER_OF_FRAGS;
		nFramesPerFrag = MM_FRAMES_PER_FRAG;
	}
	else
	if( g_Sound.ulState & SOUND_DSSOUND )
	{
		nNumberOfFrags = DS_NUMBER_OF_FRAGS;
		nFramesPerFrag = g_Sound.nLatency;//DS_FRAMES_PER_FRAG;

		_ASSERT(2 == nNumberOfFrags);
	}
	/* Compute the values related to sound buffer size */
	if( TV_PAL == tv_mode )
	{
		dwFragSize = (g_Sound.nRate / g_Timer.nPalFreq) * nFramesPerFrag * nChannels;
		nUpdatesPerFrag = TV_PAL * nFramesPerFrag / nSkipUpdate;
	}
	else
	{
		dwFragSize = (g_Sound.nRate / g_Timer.nNtscFreq) * nFramesPerFrag * nChannels;
		nUpdatesPerFrag = TV_NTSC * nFramesPerFrag / nSkipUpdate;
	}
	/* If size of the sound buffer has been not changed, there is no need to realloc it */
	Sound_Clear( TRUE, !(nNumberOfFrags * dwFragSize == dwBufferSize) );

	s_dwFragSize      = dwFragSize;
	s_nNumberOfFrags  = nNumberOfFrags;
	s_nFramesPerFrag  = nFramesPerFrag;
	s_nUpdatesPerFrag = nUpdatesPerFrag;
	s_dwFragPos       = 0;
	s_nSkipUpdate     = nSkipUpdate;
	s_nUpdateCnt      = 1;
	
	/* Calculate and save the sound buffer size */
	s_dwBufferSize = s_nNumberOfFrags * s_dwFragSize;

	if( !s_pSoundBuffer &&
		!(s_pSoundBuffer = calloc( 1, s_dwBufferSize )) )
	{
		Sound_Disable( TRUE );
		return 0;
	}

	if( g_Sound.ulState & SOUND_MMSOUND )
	{
		s_nPlayFragNo = MM_PLAY_FRAG_NO;
		s_nSaveFragNo = MM_SAVE_FRAG_NO;
	}
	else
	if( g_Sound.ulState & SOUND_DSSOUND )
	{
		s_nPlayFragNo = DS_PLAY_FRAG_NO;
		s_nSaveFragNo = DS_SAVE_FRAG_NO;
	}
	_ASSERT(s_nSaveFragNo < s_nNumberOfFrags);

	for( i = s_nPlayFragNo * s_dwFragSize; i < s_nSaveFragNo * (int)s_dwFragSize; i++ )
		s_pSoundBuffer[ i ] = (char)0x80;

	s_pPlayCursor = &s_pSoundBuffer[ s_nPlayFragNo * s_dwFragSize ];
	s_pSaveCursor = &s_pSoundBuffer[ s_nSaveFragNo * s_dwFragSize ];

	/* Initialize the kernel sound machine */
	Pokey_sound_init( FREQ_17_EXACT, (short)g_Sound.nRate, (char)nChannels );

	/* Set this up for PCM, 1/2 channels, 8 bits unsigned samples */
	ZeroMemory( &s_wfxWaveFormat, sizeof(WAVEFORMATEX) );
	s_wfxWaveFormat.wFormatTag      = WAVE_FORMAT_PCM; /* The only tag valid with DirectSound */
	s_wfxWaveFormat.nChannels       = nChannels;
	s_wfxWaveFormat.nSamplesPerSec  = g_Sound.nRate;
	s_wfxWaveFormat.wBitsPerSample  = 8;
	s_wfxWaveFormat.nBlockAlign     = s_wfxWaveFormat.wBitsPerSample / 8 * s_wfxWaveFormat.nChannels;
	s_wfxWaveFormat.nAvgBytesPerSec = s_wfxWaveFormat.nSamplesPerSec * s_wfxWaveFormat.nBlockAlign;
	s_wfxWaveFormat.cbSize          = 0; /* This member is always zero for PCM formats */

	/* CAUTION:
	   It is necessary to initialise WAVEFORMATEX structure even if the sound
	   has been muted! (vide Sound_VolumeCapable() function)
	*/
	if( g_Sound.ulState & SOUND_NOSOUND )
	{
		Atari_PlaySound = SndPlay_NoSound;
		return 1; /* Exit if SOUND_NOSOUND flag is set */
	}

	if( g_Sound.ulState & SOUND_MMSOUND )
	{
		MMRESULT mmResult = MMSYSERR_NOERROR;

		for( i = 0; i < s_nNumberOfFrags; i++ )
			ZeroMemory( &s_arrWaveHDR[ i ], sizeof(WAVEHDR) );

		if( !s_hWaveOut )
		{
			mmResult = waveOutOpen( &s_hWaveOut, WAVE_MAPPER, &s_wfxWaveFormat, 0, 0, CALLBACK_NULL );
			if( mmResult != MMSYSERR_NOERROR )
			{
				ServeMMError( IDS_MMERR_OPEN, mmResult, FALSE );
				return 0;
			}
		}
		for( i = 0; i < s_nNumberOfFrags; i++ )
		{
			s_arrWaveHDR[ i ].lpData          = &s_pSoundBuffer[ i * s_dwFragSize ];
			s_arrWaveHDR[ i ].dwBufferLength  = s_dwFragSize;
			s_arrWaveHDR[ i ].dwBytesRecorded = 0;
			s_arrWaveHDR[ i ].dwUser          = 0;
			s_arrWaveHDR[ i ].dwFlags         = 0;
			s_arrWaveHDR[ i ].dwLoops         = 1;

			mmResult = waveOutPrepareHeader( s_hWaveOut, &s_arrWaveHDR[ i ], sizeof(WAVEHDR) );
			if( mmResult != MMSYSERR_NOERROR )
			{
				ServeMMError( IDS_MMERR_PREP_HDR, mmResult, FALSE );
				return 0;
			}
		}
		Atari_PlaySound = SndPlay_MMSound;
	}
	else
	if( g_Sound.ulState & SOUND_DSSOUND )
	{
		DSBUFFERDESC dsbdesc;
		HRESULT	hResult;

		/* Create IDirectSound using the primary sound device */
		if( !s_lpDirectSound && FAILED(
			hResult = DirectSoundCreate( NULL, &s_lpDirectSound, NULL )) )
		{
			ServeDSError( IDS_DSERR_CREATE_OBJ, hResult, FALSE );
			return 0;
		}
		_ASSERT(s_lpDirectSound);

		if( !DetermineHardwareCaps( s_lpDirectSound, g_Sound.ulState & SOUND_CUSTOM_RATE ) )
			return 0;

		/* Set up DSBUFFERDESC structure */
		ZeroMemory( &dsbdesc, sizeof(DSBUFFERDESC) );
		dsbdesc.dwSize  = sizeof(DSBUFFERDESC);
		dsbdesc.dwFlags = DSBCAPS_PRIMARYBUFFER;

		if( g_Sound.ulState & SOUND_CUSTOM_RATE )
		{
			LPDIRECTSOUNDBUFFER lpDSBPrimary;

			/* Obtain priority cooperative level */
			hResult = IDirectSound_SetCooperativeLevel( s_lpDirectSound, g_hMainWnd, DSSCL_PRIORITY );
			if( FAILED(hResult) )
			{
				ServeDSError( IDS_DSERR_COOP_PRIORITY, hResult, FALSE );
				return 0;
			}
			/* The DirectSound mixer produces the best sound quality if all
			   application's sounds use the same wave format and the hardware
			   output format is matched to the format of the sounds. If this
			   is done, the mixer doesn't need perform any format conversion.
			   Note that this primary buffer is for control purposes only;
			   creating it is not the same as obtaining write access to the
			   primary buffer */

			/* Try to create the primary buffer */
			hResult = IDirectSound_CreateSoundBuffer( s_lpDirectSound, &dsbdesc, &lpDSBPrimary, NULL );
			if( FAILED(hResult) )
			{
				lpDSBPrimary = NULL;
				ServeDSError( IDS_DSERR_CREATE_PRIMARY, hResult, FALSE );
				return 0;
			}
			/* Set primary buffer to desired format */
			hResult = IDirectSoundBuffer_SetFormat( lpDSBPrimary, &s_wfxWaveFormat );
			if( FAILED(hResult) )
			{
				ServeDSError( IDS_DSERR_FORMAT_PRIMARY, hResult, FALSE );
				return 0;
			}
			/* Release primary buffer */
			IDirectSoundBuffer_Release( lpDSBPrimary );
		}
		else
		{
			/* Obtain normal cooperative level (22kHz) */
			hResult = IDirectSound_SetCooperativeLevel( s_lpDirectSound, g_hMainWnd, DSSCL_NORMAL );
			if( FAILED(hResult) )
			{
				ServeDSError( IDS_DSERR_COOP_NORMAL, hResult, FALSE );
				return 0;
			}
		}
		/* Create secondary DirectSound buffers */

		/* Set up DSBUFFERDESC structure */
		ZeroMemory( &dsbdesc, sizeof(DSBUFFERDESC) );
		dsbdesc.dwSize = sizeof(DSBUFFERDESC);
		/* Need default controls (volume, frequency) */
		dsbdesc.dwFlags = DSBCAPS_CTRLPOSITIONNOTIFY |	/* Needed for notification */
			              DSBCAPS_CTRLVOLUME |
						  DSBCAPS_CTRLPAN |
						  DSBCAPS_CTRLFREQUENCY |
						  DSBCAPS_GETCURRENTPOSITION2 | /* Always a good idea */
						  DSBCAPS_GLOBALFOCUS;			/* Allows background playing */
		dsbdesc.dwBufferBytes = s_dwBufferSize;
		dsbdesc.lpwfxFormat   = &s_wfxWaveFormat;

		hResult = IDirectSound_CreateSoundBuffer( s_lpDirectSound, &dsbdesc, &s_lpDSBuffer, NULL );
		if( FAILED(hResult) )
		{
			s_lpDSBuffer = NULL;
			ServeDSError( IDS_DSERR_CREATE_BUFF, hResult, FALSE );
			return 0;
		}

		/* Check if IDirectSoundNotify interface is available. It should be
		   there if DirectX 5.0 or higher is installed, otherwise we use a
		   periodic timer event callback */

		hResult = IDirectSoundNotify_QueryInterface( s_lpDSBuffer, (GUID *)&IID_IDirectSoundNotify, (LPVOID FAR *)&s_lpDSNotify );
		if( FAILED(hResult) )
		{
			s_lpDSNotify = NULL;
			ServeDSError( IDS_DSERR_NO_NOTIFY, hResult, FALSE );
			return 0;
		}
		/* Set up notification events for a playback buffer */
		s_lpDSNotifyPos = calloc( 1, s_nNumberOfFrags * sizeof(DSBPOSITIONNOTIFY) );
		if( !s_lpDSNotifyPos )
		{
			ServeDSError( IDS_DSERR_SET_NOTIFY, DSERR_OUTOFMEMORY, FALSE );
			return 0;
		}
		if( !(s_arrEvents[ POS_EVENT ] = CreateEvent( NULL, FALSE, FALSE, NULL )) ||
			!(s_arrEvents[ END_EVENT ] = CreateEvent( NULL, FALSE, FALSE, NULL )) )
		{
			ServeDSError( IDS_DSERR_CREATE_THREAD, 0 - GetLastError(), FALSE );
			return 0;
		}
		for( i = 0; i < s_nNumberOfFrags; i++ )
		{
			s_lpDSNotifyPos[ i ].dwOffset = s_dwFragSize * i;
			s_lpDSNotifyPos[ i ].hEventNotify = s_arrEvents[ POS_EVENT ];
		}
		hResult = IDirectSoundNotify_SetNotificationPositions( s_lpDSNotify, s_nNumberOfFrags, s_lpDSNotifyPos );
		if( FAILED(hResult) )
		{
			ServeDSError( IDS_DSERR_SET_NOTIFY, hResult, FALSE );
			return 0;
		}
		/* Spawn a secondary sound thread */
		if( !(s_hNotifyThread = _CreateThreadEx( NULL,
												 0,
												 DSSoundThreadProc,
												 NULL,
												 0,
												 &s_dwThreadID )) )
		{
			ServeDSError( IDS_DSERR_CREATE_THREAD, 0 - GetLastError(), FALSE );
			return 0;
		}
		/* Set priority of the secondary thread */
		SetThreadPriority( s_hNotifyThread, THREAD_PRIORITY_ABOVE_NORMAL );

		Atari_PlaySound = SndPlay_DSSound;
	}
	else
		return 0;

	/* Set sound volume */
	Sound_SetVolume();

	if( !(g_Misc.ulState & MS_FULL_SPEED) )
		Sound_Restart();

	return 1;
} /* #OF# Sound_Initialise */

/*========================================================
Function : pokey_update
=========================================================*/
/* #FN#
   This function is called by an Atari800 kernel */
void
/* #AS#
   Nothing */
pokey_update( void )
{
	if( --s_nSkipUpdate )
		return;

	s_nSkipUpdate = g_Sound.nSkipUpdate;

	if( !s_bSoundIsPaused )
	{
		DWORD dwFragPos = (s_dwFragSize * s_nUpdateCnt / s_nUpdatesPerFrag) & 0xfffffffe;
		if( dwFragPos > s_dwFragSize )
			dwFragPos = s_dwFragSize;

		/* Write the part of audio data to the buffer */
		Pokey_process( s_pSaveCursor + s_dwFragPos, (short)(dwFragPos - s_dwFragPos) );

		s_dwFragPos = dwFragPos;
		s_nUpdateCnt++;
	}
} /* #OF# pokey_update */

/*========================================================
Function : SndPlay_NoSound
=========================================================*/
/* #FN#
   Generates sound stream but doesn't play the sound */
static
void
SndPlay_NoSound( void )
{
	if( ++s_nFrameCnt > s_nFramesPerFrag )
	{
		_ASSERT(s_dwFragPos <= s_dwFragSize);

		if( s_dwFragPos < s_dwFragSize )
			Pokey_process( s_pSaveCursor + s_dwFragPos, (short)(s_dwFragSize - s_dwFragPos) );

		if( g_Sound.pfOutput )
			fwrite( s_pSaveCursor, s_dwFragSize, 1, g_Sound.pfOutput );

		Video_SaveFrame( NULL, 0, s_pSaveCursor, s_dwFragSize );

		if( ++s_nPlayFragNo == s_nNumberOfFrags )
			s_nPlayFragNo = 0;

		if( ++s_nSaveFragNo == s_nNumberOfFrags )
			s_nSaveFragNo = 0;

		s_pPlayCursor = &s_pSoundBuffer[ s_nPlayFragNo * s_dwFragSize ];
		s_pSaveCursor = &s_pSoundBuffer[ s_nSaveFragNo * s_dwFragSize ];

		s_dwFragPos  = 0;
		s_nUpdateCnt = 1;
		s_nFrameCnt  = 1;
	}
} /* #OF# SndPlay_NoSound */

static
void
SndPlay_MMSound( void )
{
	if( ++s_nFrameCnt > s_nFramesPerFrag )
	{
		_ASSERT(s_dwFragPos <= s_dwFragSize);

		if( s_dwFragPos < s_dwFragSize )
			/* Write the audio data to the buffer if it is not full */
			Pokey_process( s_pSaveCursor + s_dwFragPos, (short)(s_dwFragSize - s_dwFragPos) );

		if( g_Sound.pfOutput )
			fwrite( s_pSaveCursor, s_dwFragSize, 1, g_Sound.pfOutput );

		Video_SaveFrame( NULL, 0, s_pSaveCursor, s_dwFragSize );

		/* SoundIsPaused indicates full speed mode in this case; if
		   the emulated Atari is paused, this routine is not invoked */
		if( !s_bSoundIsPaused )
			waveOutWrite( s_hWaveOut, &s_arrWaveHDR[ s_nPlayFragNo ], sizeof(WAVEHDR) );

		if( ++s_nPlayFragNo == s_nNumberOfFrags )
			s_nPlayFragNo = 0;

		if( ++s_nSaveFragNo == s_nNumberOfFrags )
			s_nSaveFragNo = 0;

		s_pPlayCursor = &s_pSoundBuffer[ s_nPlayFragNo * s_dwFragSize ];
		s_pSaveCursor = &s_pSoundBuffer[ s_nSaveFragNo * s_dwFragSize ];

		s_dwFragPos  = 0;
		s_nUpdateCnt = 1;
		s_nFrameCnt  = 1;
	}
} /* #OF# SndPlay_MMSound */

/*========================================================
Function : SndPlay_DSSound
=========================================================*/
/* #FN#
   Generates sound stream but doesn't play the sound */
static
void
SndPlay_DSSound( void )
{
	if( ++s_nFrameCnt > s_nFramesPerFrag )
	{
		int nDelay = 40;

		_ASSERT(s_dwFragPos <= s_dwFragSize);

		if( s_dwFragPos < s_dwFragSize )
			Pokey_process( s_pSaveCursor + s_dwFragPos, (short)(s_dwFragSize - s_dwFragPos) );

		if( g_Sound.pfOutput )
			fwrite( s_pSaveCursor, s_dwFragSize, 1, g_Sound.pfOutput );

		Video_SaveFrame( NULL, 0, s_pSaveCursor, s_dwFragSize );

		/* SoundIsPaused indicates full speed mode in this case; if
		   the emulated Atari is paused, this routine is not invoked */
		if( !s_bSoundIsPaused )
		{
//			_TRACE2("PrimaryThread.SndPlay_DSSound: s_nSaveFragNo: %d, s_nPlayFragNo: %d\n", s_nSaveFragNo, s_nPlayFragNo);

			/* There is spinlock used to synchronize the threads. We should
			   avoid this solution but it's rather fast and we don't really
			   need the interlocked functions or critical sections here */
			while( s_nSaveFragNo != s_nPlayFragNo && nDelay-- )
				Sleep( 1 );
		}
		/* The PlayFragNo indicator is incremented by a secondary thread so we can
		   synchronize the tread with primary thread's writes to a stream buffer */

		if( ++s_nSaveFragNo == s_nNumberOfFrags )
			s_nSaveFragNo = 0;

		s_pSaveCursor = &s_pSoundBuffer[ s_nSaveFragNo * s_dwFragSize ];

		s_dwFragPos  = 0;
		s_nUpdateCnt = 1;
		s_nFrameCnt  = 1;
	}
} /* #OF# SndPlay_DSSound */

/*========================================================
Function : LockSoundBuffer
=========================================================*/
/* #FN#
   Locks the DirectSound buffer for direct accessing to be safe */
static
_inline
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
LockSoundBuffer(
	BOOL     bLock,
	DWORD    dwBufferOffset,
	DWORD    dwFragSize,
	LPLPVOID lppvPtr1,
	PDWORD   pBytes1,
	LPLPVOID lppvPtr2,
	PDWORD   pBytes2,
	DWORD    dwFlags
)
{
	if( bLock )
	{
		HRESULT
		/* Obtain a valid write pointer to the sound buffer's audio data */
		hResult = IDirectSoundBuffer_Lock( s_lpDSBuffer, dwBufferOffset, dwFragSize, lppvPtr1, pBytes1, lppvPtr2, pBytes2, dwFlags );
		/* If DSERR_BUFFERLOST is returned, restore and retry lock */
		if( DSERR_BUFFERLOST == hResult )
		{
			hResult = IDirectSoundBuffer_Restore( s_lpDSBuffer );
			if( FAILED(hResult) )
				return FALSE;

			hResult = IDirectSoundBuffer_Lock( s_lpDSBuffer, dwBufferOffset, s_dwFragSize, lppvPtr1, pBytes1, lppvPtr2, pBytes2, dwFlags );
			if( FAILED(hResult) )
				return FALSE;
		}
	}
	else
		IDirectSoundBuffer_Unlock( s_lpDSBuffer, *lppvPtr1, *pBytes1, *lppvPtr2, *pBytes2 );

	return TRUE;
} /* #OF# LockSoundBuffer */

/*========================================================
Function : DSSoundThreadProc
=========================================================*/
/* #FN#
   The DirectSound notification thread procedure */
DWORD WINAPI
DSSoundThreadProc(
	LPVOID lpParam
)
{
	DWORD   dwPlayCursor   = 0;
	DWORD   dwSaveCursor   = 0;
	DWORD   dwBufferOffset = 0;
	BOOL    bBufferCleared = FALSE;
	LPVOID  lpvPtr1        = NULL;
	LPVOID  lpvPtr2        = NULL;
	DWORD   dwBytes1       = 0;
	DWORD   dwBytes2       = 0;
	DWORD   dwEventResult  = 0;
	HRESULT hResult;
	int     i;

	while( WAIT_OBJECT_0 + 1 != dwEventResult )
	{
		dwEventResult = WaitForMultipleObjects( NUMBER_OF_EVENTS, s_arrEvents, FALSE, INFINITE );
		if( WAIT_OBJECT_0 == dwEventResult )
		{
//			_TRACE2("SecondaryThread.DSSoundThreadProc: s_nSaveFragNo: %d, s_nPlayFragNo: %d\n", s_nSaveFragNo, s_nPlayFragNo);
			if( s_nPlayFragNo == s_nSaveFragNo )
			{
//				_TRACE0("SecondaryThread.DSSoundThreadProc: s_nPlayFragNo == s_nSaveFragNo\n");
				if( !bBufferCleared )
				{
					if( LockSoundBuffer( TRUE, 0, 0, &lpvPtr1, &dwBytes1, &lpvPtr2, &dwBytes2, DSBLOCK_ENTIREBUFFER ) )
					{
						for( i = 0; i < (int)dwBytes1; i++ )
							((UBYTE*)lpvPtr1)[ i ] = (UBYTE)0x80;

						if( lpvPtr2 != NULL )
						{
							for( i = 0; i < (int)dwBytes2; i++ )
								((UBYTE*)lpvPtr2)[ i ] = (UBYTE)0x80;
						}
						LockSoundBuffer( FALSE, 0, 0, &lpvPtr1, &dwBytes1, &lpvPtr2, &dwBytes2, 0 );

						bBufferCleared = TRUE;
					}
				}
			}
			else /* s_nPlayFragNo != s_nSaveFragNo */
			{
				hResult = IDirectSoundBuffer_GetCurrentPosition( s_lpDSBuffer, &dwPlayCursor, &dwSaveCursor );
				if( SUCCEEDED(hResult) )
				{
					dwBufferOffset = s_nPlayFragNo * s_dwFragSize;

					if( !(dwPlayCursor >= dwBufferOffset && dwPlayCursor < dwBufferOffset + s_dwFragSize) )
					{
						if( LockSoundBuffer( TRUE, dwBufferOffset, s_dwFragSize, &lpvPtr1, &dwBytes1, &lpvPtr2, &dwBytes2, 0 ) )
						{
							s_pPlayCursor = &s_pSoundBuffer[ dwBufferOffset ];

							CopyMemory( lpvPtr1, s_pPlayCursor, dwBytes1 );
							if( NULL != lpvPtr2 )
								CopyMemory( lpvPtr2, s_pPlayCursor + dwBytes1, dwBytes2 );

							LockSoundBuffer( FALSE, 0, 0, &lpvPtr1, &dwBytes1, &lpvPtr2, &dwBytes2, 0 );

							if( ++s_nPlayFragNo == s_nNumberOfFrags )
								s_nPlayFragNo = 0;

							bBufferCleared = FALSE;
						}
					}
				}
			} /* s_nPlayFragNo == s_nSaveFragNo */
		}
	}
	return 0;
} /* #OF# DSSoundThreadProc */

/*========================================================
Function : Sound_VolumeCapable
=========================================================*/
/* #FN#
   Checks if a sound output device supports a volume control */
BOOL
/* #AS#
   TRUE if volume control is supported, otherwise FALSE */
Sound_VolumeCapable( void )
{
	BOOL bReturn = TRUE;

	if( g_Sound.ulState & SOUND_MMSOUND )
	{
		WAVEOUTCAPS woc;
		ZeroMemory( &woc, sizeof(WAVEOUTCAPS) );

		if( s_hWaveOut )
		{
			waveOutGetDevCaps( (unsigned int)s_hWaveOut, &woc, sizeof(WAVEOUTCAPS) );
		}
		else
		{
			HWAVEOUT hLocalWave;
			MMRESULT mmResult;

			mmResult = waveOutOpen( &hLocalWave, WAVE_MAPPER, &s_wfxWaveFormat, 0, 0, CALLBACK_NULL );
			if( mmResult == MMSYSERR_NOERROR )
			{
				waveOutGetDevCaps( (unsigned int)hLocalWave, &woc, sizeof(WAVEOUTCAPS) );
				waveOutClose( hLocalWave );
			}
		}
		bReturn = (woc.dwSupport & WAVECAPS_VOLUME ? TRUE : FALSE);
	}
	return bReturn;
} /* #OF# Sound_VolumeCapable */

/*========================================================
Function : Sound_SetVolume
=========================================================*/
/* #FN#
   Sets volume of a sound playback */
void
/* #AS#
   Nothing */
Sound_SetVolume( void )
{
	if( g_Sound.ulState & SOUND_MMSOUND )
	{
		if( s_hWaveOut && Sound_VolumeCapable() )
		{
			MMRESULT mmResult;
			DWORD    dwVolume, dwTempVolume;

			/* Save the volume setting so we can put it back later */
			if( !s_dwStartVolume )
				mmResult = waveOutGetVolume( s_hWaveOut, &s_dwStartVolume );

			/* Hiword is the right channel, low word is the left channel */
			dwVolume = (HIWORD(s_dwStartVolume) + HIWORD(s_dwStartVolume) / 100 * g_Sound.nVolume) * 65536;
			dwVolume += LOWORD(s_dwStartVolume) + LOWORD(s_dwStartVolume) / 100 * g_Sound.nVolume;

			mmResult = waveOutSetVolume( s_hWaveOut, dwVolume );
			if( mmResult != MMSYSERR_NOERROR )
			{
				ServeMMError( IDS_MMERR_SET_VOLUME, mmResult, FALSE );
				return;
			}

			/* It's possible this wave device doesn't support 16 bits of volume control,
			   so we'll check the result of the set with waveOutGetVolume, if it's less
			   than what we set, we know the new max and we'll scale to that */

			waveOutGetVolume( s_hWaveOut, &dwTempVolume );
			if( dwTempVolume < dwVolume )
			{
				float fPercentage = ((float)dwTempVolume / dwVolume);
				dwVolume = (unsigned long)(dwVolume * fPercentage);
				waveOutSetVolume( s_hWaveOut, dwVolume );
			}
		}
	}
	if( g_Sound.ulState & SOUND_DSSOUND )
	{
		if( s_lpDSBuffer )
			IDirectSoundBuffer_SetVolume( s_lpDSBuffer, g_Sound.nVolume * 22 );
	}
} /* #OF# Sound_SetVolume */

/*========================================================
Function : Sound_Clear
=========================================================*/
/* #FN#
   Closes/mutes a sound output device */
void
/* #AS#
   Nothing */
Sound_Clear(
	BOOL bPermanent,
	BOOL bFreeBuffer
)
{
	/* Are we shutting down everything, or just pausing the
	   sound that is playing currently? */
	if( bPermanent )
	{
		DWORD dwExitCode = 0;

		Atari_PlaySound = SndPlay_NoSound;

		if( g_Sound.pfOutput )
		{
			/* Close Sound Output file */
			/* CAUTION:
			   After closing the info box the main window receives the
			   OnActivate(::RestartSound) message; that's why we set
			   the SoundIsPaused to TRUE after calling CloseOutput */
			Sound_CloseOutput();
			/* The tray bar has to be redrawn */
			UpdateIndicator( ID_INDICATOR_WAV );
		}
		s_bSoundIsPaused = TRUE;

		/* Clear Multimedia stuff */
		if( s_hWaveOut )
		{
			int i;
			if( s_dwStartVolume )
			{
				waveOutSetVolume( s_hWaveOut, s_dwStartVolume );
				s_dwStartVolume = 0;
			}
			waveOutReset( s_hWaveOut );
			for( i = 0; i < s_nNumberOfFrags; i++ )
			{
				waveOutUnprepareHeader( s_hWaveOut, &s_arrWaveHDR[ i ], sizeof(WAVEHDR) );
				s_arrWaveHDR[ i ].lpData = NULL;
			}
			waveOutClose( s_hWaveOut );
			s_hWaveOut = 0;
		}

		/* Clear DirectSound stuff */
		if( s_lpDSBuffer )
			/* Stop buffer play */
			IDirectSoundBuffer_Stop( s_lpDSBuffer );

		if( s_hNotifyThread )
		{
			/* Stop & Kill streaming thread */
			SetEvent( s_arrEvents[ END_EVENT ] );
			/*  Wait for thread termination */
			do
			{
				Sleep( 1 ); /* Wait a moment, please */
				GetExitCodeThread( s_hNotifyThread, &dwExitCode );
			}
			while( STILL_ACTIVE == dwExitCode );

			CloseHandle( s_hNotifyThread );
			s_hNotifyThread = NULL;

			CloseHandle( s_arrEvents[ POS_EVENT ] );
			s_arrEvents[ POS_EVENT ] = NULL;

			CloseHandle( s_arrEvents[ END_EVENT ] );
			s_arrEvents[ END_EVENT ] = NULL;
		}
		if( s_lpDSNotify )
		{
			IDirectSoundNotify_Release( s_lpDSNotify );
			s_lpDSNotify = NULL;
		}
		if( s_lpDSNotifyPos )
		{
			free( s_lpDSNotifyPos );
			s_lpDSNotifyPos = NULL;
		}
		if( s_lpDSBuffer )
		{
			IDirectSoundBuffer_Release( s_lpDSBuffer );
			s_lpDSBuffer = NULL;
		}
		if( s_lpDirectSound )
		{
			IDirectSound_Release( s_lpDirectSound );
			s_lpDirectSound = NULL;
		}
		if( s_pSoundBuffer && bFreeBuffer )
		{
			free( s_pSoundBuffer );
			s_pSoundBuffer = NULL;
		}
		s_pPlayCursor = NULL;
		s_pSaveCursor = NULL;
	}
	else
	{
		s_bSoundIsPaused = TRUE;

		/* Stop playback and clear waveOut buffer */
		if( s_hWaveOut )
			waveOutPause( s_hWaveOut );

		/* Clear DirectSound buffer and stop playback */
		/* CAUTION:
		   It seems to be not really needed since the
		   secondary buffer clears sound playback (?) */
		if( s_lpDSBuffer )
		{
			IDirectSoundBuffer_Stop( s_lpDSBuffer );
			/* Clear the DirectSound sound buffer
			   CAUTION:
			   We assume that there are only two buffers for
			   DirectSound! */
			SetEvent( s_arrEvents[ POS_EVENT ] );
			SetEvent( s_arrEvents[ POS_EVENT ] );
		}
//		if( s_pSoundBuffer )
//		{
//			int i;
//			for( i = 0; i < s_dwBufferSize; i++ )
//				s_pSoundBuffer[ i ] = (char)0x80;
//		}
	}
} /* #OF# Sound_Clear */

/*========================================================
Function : Sound_Restart
=========================================================*/
/* #FN#
   Resumes playback on a paused sound output device */
void
/* #AS#
   Nothing */
Sound_Restart( void )
{
	if( g_Misc.ulState & MS_FULL_SPEED )
		return;

	if( s_bSoundIsPaused )
	{
		if( s_hWaveOut )
		{
			waveOutRestart( s_hWaveOut );
			s_bSoundIsPaused = FALSE;
		}
		if( s_lpDSBuffer )
		{
			IDirectSoundBuffer_Play( s_lpDSBuffer, 0, 0, DSBPLAY_LOOPING );
			s_bSoundIsPaused = FALSE;
		}
	}
} /* #OF# Sound_Restart */

/*========================================================
Function : Sound_OpenOutput
=========================================================*/
/* #FN#
   Opens a sound file */
void
/* #AS#
   Nothing */
Sound_OpenOutput( char *pszOutFileName )
{
	if( g_Sound.pfOutput )
		/* Close Sound Output file */
		Sound_CloseOutput();

	_ASSERT(stereo_enabled == 0 || stereo_enabled == 1);

	if( (g_Sound.pfOutput = fopen( pszOutFileName, "wb" )) )
	{
		WAVEFORMAT wf;

//		wf.wFormatTag      = WAVE_FORMAT_PCM;
		wf.nChannels       = stereo_enabled + 1;
		wf.nSamplesPerSec  = g_Sound.nRate;
		wf.nBlockAlign     = wf.nChannels;
		wf.nAvgBytesPerSec = wf.nSamplesPerSec * wf.nBlockAlign;
		/*
		  Offset  Length   Contents

		The RIFF header:
		  0       4 bytes  'RIFF'
		  4       4 bytes  <file length - 8>
		  8       4 bytes  'WAVE'

		The format (fmt) chunk:
		  12      4 bytes  'fmt '
		  16      4 bytes  0x00000010     // Length of the fmt data (16 bytes)
		  20      2 bytes  0x0001         // Format tag: 1 = PCM
		  22      2 bytes  <channels>     // Channels: 1 = mono, 2 = stereo
		  24      4 bytes  <sample rate>  // Samples per second: e.g., 44100
		  28      4 bytes  <bytes/second> // sample rate * block align
		  32      2 bytes  <block align>  // channels * bits/sample / 8
		  34      2 bytes  <bits/sample>  // 8 or 16

		The data chunk:
		  36      4 bytes  'data'
		  40      4 bytes  <length of the data block>
		  44        bytes  <sample data>
		*/
		fwrite( "RIFF\000\000\000\000WAVEfmt\040\020\000\000\000\001\000", 22, 1, g_Sound.pfOutput );
		fwrite( &wf.nChannels,       2, 1, g_Sound.pfOutput );
		fwrite( &wf.nSamplesPerSec,  4, 1, g_Sound.pfOutput );
		fwrite( &wf.nAvgBytesPerSec, 4, 1, g_Sound.pfOutput );
		fwrite( &wf.nBlockAlign,     2, 1, g_Sound.pfOutput );
		fwrite( "\010\000data\000\000\000\000", 10, 1, g_Sound.pfOutput );
	}
	else
		DisplayMessage( NULL, IDS_ERROR_FILE_OPEN, 0, MB_ICONEXCLAMATION | MB_OK, pszOutFileName );
} /* #OF# Sound_OpenOutput */

/*========================================================
Function : Sound_CloseOutput
=========================================================*/
/* #FN#
   Closes a sound file */
void
/* #AS#
   Nothing */
Sound_CloseOutput( void )
{
	if( g_Sound.pfOutput )
	{
		UINT unPos = 0;

		/* Sound file is finished, so modify header and close it */
		unPos = ftell( g_Sound.pfOutput ) - 8;
		fseek ( g_Sound.pfOutput, 4, SEEK_SET );	// Seek past RIFF
		fwrite( &unPos, 4, 1, g_Sound.pfOutput );	// Write out size of entire data chunk
		fseek ( g_Sound.pfOutput, 40, SEEK_SET );
		unPos -= 36;
		fwrite( &unPos, 4, 1, g_Sound.pfOutput );	// Write out size of just sample data
		fclose( g_Sound.pfOutput );

		g_Sound.pfOutput = NULL;

		DisplayMessage( NULL, IDS_SFX_FILE_CLOSED, 0, MB_ICONINFORMATION | MB_OK );
	}
} /* #OF# Sound_CloseOutput */

/*========================================================
Function : Sound_Disable
=========================================================*/
/* #FN#
   Turns off a sound playback (disable sound) */
int
/* #AS#
   A status returned by Sound_Initialise function */
Sound_Disable(
	BOOL bClearSound
)
{
	if( bClearSound )
		Sound_Clear( TRUE, FALSE );

	/* Modify sound state of emulator */
//	g_Sound.ulState &= ~(SOUND_DSSOUND | SOUND_MMSOUND); /* Sound/Mute needs these flags */
	g_Sound.ulState |= SOUND_NOSOUND;
	WriteRegDWORD( NULL, REG_SOUND_STATE, g_Sound.ulState );

	/* It always is succeeded for SOUND_NOSOUND */
	return Sound_Initialise();
} /* #OF# Sound_Disable */

/*========================================================
Function : GetDSErrorString
=========================================================*/
/* #FN#
   Outputs a debug string to debugger */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
GetDSErrorString( HRESULT hResult,
				  LPSTR   lpszErrorBuff,
				  DWORD   dwError )
{
	DWORD cLen;
	LPSTR lpszError;
	TCHAR szMsg[ 256 ];

	/* Check parameters */
	if( !lpszErrorBuff || !dwError )
	{
		/* Error, invalid parameters */
		return FALSE;
	}

	switch( hResult )
	{
		/* The request completed successfully */
		case DS_OK:
			lpszError = TEXT("DS_OK");
			break;

#ifdef WIN_TRANSLATE_ERRORS

		/* The request failed because resources, such as a priority level, were already in use by another caller */
		case DSERR_ALLOCATED:
			lpszError = TEXT("DSERR_ALLOCATED");
			break;

		/* The object is already initialized */
		case DSERR_ALREADYINITIALIZED:
			lpszError = TEXT("DSERR_ALREADYINITIALIZED");
			break;

		/* The specified wave format is not supported */
		case DSERR_BADFORMAT:
			lpszError = TEXT("DSERR_BADFORMAT");
			break;

		/* The buffer memory has been lost and must be restored */
		case DSERR_BUFFERLOST:
			lpszError = TEXT("DSERR_BUFFERLOST");
			break;

		/* The control (volume, pan, and so forth) requested by the caller is not available */
		case DSERR_CONTROLUNAVAIL:
			lpszError = TEXT("DSERR_CONTROLUNAVAIL");
			break;

		/* An undetermined error occurred inside the DirectSound subsystem */
		case DSERR_GENERIC:
			lpszError = TEXT("DSERR_GENERIC");
			break;

		/* This function is not valid for the current state of this object */
		case DSERR_INVALIDCALL:
			lpszError = TEXT("DSERR_INVALIDCALL");
			break;

		/* An invalid parameter was passed to the returning function */
		case DSERR_INVALIDPARAM:
			lpszError = TEXT("DSERR_INVALIDPARAM");
			break;

		/* The object does not support aggregation */
		case DSERR_NOAGGREGATION:
			lpszError = TEXT("DSERR_NOAGGREGATION");
			break;

		/* No sound driver is available for use */
		case DSERR_NODRIVER:
			lpszError = TEXT("DSERR_NODRIVER");
			break;

		/* The requested COM interface is not available */
		case DSERR_NOINTERFACE:
			lpszError = TEXT("DSERR_NOINTERFACE");
			break;

		/* Another application has a higher priority level, preventing this call from succeeding */
		case DSERR_OTHERAPPHASPRIO:
			lpszError = TEXT("DSERR_OTHERAPPHASPRIO");
			break;

		/* The DirectSound subsystem could not allocate sufficient memory to complete the caller's request */
		case DSERR_OUTOFMEMORY:
			lpszError = TEXT("DSERR_OUTOFMEMORY");
			break;

		/* The caller does not have the priority level required for the function to succeed */
		case DSERR_PRIOLEVELNEEDED:
			lpszError = TEXT("DSERR_PRIOLEVELNEEDED");
			break;

		/* The IDirectSound::Initialize method has not been called or has not been called successfully before other methods were called */
		case DSERR_UNINITIALIZED:
			lpszError = TEXT("DSERR_UNINITIALIZED");
			break;

		/* The function called is not supported at this time */
		case DSERR_UNSUPPORTED:
			lpszError = TEXT("DSERR_UNSUPPORTED");
			break;

#endif /*WIN_TRANSLATE_ERRORS*/

		/* Unknown DS Error */
		default:
			wsprintf( szMsg, "Error #%ld", (DWORD)(hResult & 0x0000FFFFL) );
			lpszError = szMsg;
			break;
	}
	/* Copy DS Error string to buff */
	cLen = strlen( lpszError );
	if (cLen >= dwError)
	{
		cLen = dwError - 1;
	}
	if( cLen )
	{
		strncpy( lpszErrorBuff, lpszError, cLen );
		lpszErrorBuff[ cLen ] = 0;
	}
	return TRUE;
} /* #OF# GetDSErrorString */

/*========================================================
Function : GetMMErrorString
=========================================================*/
/* #FN#
   Outputs a debug string to debugger */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
GetMMErrorString( MMRESULT mmResult,
				  LPSTR    lpszErrorBuff,
				  DWORD    dwError )
{
	DWORD cLen;
	LPSTR lpszError;
	TCHAR szMsg[ 256 ];

	/* Check parameters */
	if( !lpszErrorBuff || !dwError )
	{
		/* Error, invalid parameters */
		return FALSE;
	}

	switch( mmResult )
	{
		/* No error */
		case MMSYSERR_NOERROR:
			lpszError = TEXT("MMSYSERR_NOERROR");
			break;

#ifdef WIN_TRANSLATE_ERRORS

		/* Wave header is not prepared for output */
		case WAVERR_UNPREPARED:
			lpszError = TEXT( "WAVERR_UNPREPARED");
			break;

		/* Unspecified error */
		case MMSYSERR_ERROR:
			lpszError = TEXT("MMSYSERR_ERROR");
			break;

		/* Device ID out of range */
		case MMSYSERR_BADDEVICEID:
			lpszError = TEXT("MMSYSERR_BADDEVICEID");
			break;

		/* Driver failed enable */
		case MMSYSERR_NOTENABLED:
			lpszError = TEXT("MMSYSERR_NOTENABLED");
			break;

		/* Device already allocated */
		case MMSYSERR_ALLOCATED:
			lpszError = TEXT("MMSYSERR_ALLOCATED");
			break;

		/* Device handle is invalid */
		case MMSYSERR_INVALHANDLE:
			lpszError = TEXT("MMSYSERR_INVALHANDLE");
			break;

		/* No device driver present */
		case MMSYSERR_NODRIVER:
			lpszError = TEXT("MMSYSERR_NODRIVER");
			break;

		/* Memory allocation error */
		case MMSYSERR_NOMEM:
			lpszError = TEXT("MMSYSERR_NOMEM");
			break;

		/* Function isn't supported */
		case MMSYSERR_NOTSUPPORTED:
			lpszError = TEXT("MMSYSERR_NOTSUPPORTED");
			break;

		/* Error value out of range */
		case MMSYSERR_BADERRNUM:
			lpszError = TEXT("MMSYSERR_BADERRNUM");
			break;

		/* Invalid flag passed */
		case MMSYSERR_INVALFLAG:
			lpszError = TEXT("MMSYSERR_INVALFLAG");
			break;

		/* Invalid parameter passed */
		case MMSYSERR_INVALPARAM:
			lpszError = TEXT("MMSYSERR_INVALPARAM");
			break;

		/* Handle being used */
		case MMSYSERR_HANDLEBUSY:
			lpszError = TEXT("MMSYSERR_HANDLEBUSY");
			break;

		/* Specified alias not found */
		case MMSYSERR_INVALIDALIAS:
			lpszError = TEXT("MMSYSERR_INVALIDALIAS");
			break;

		/* Bad registry database */
		case MMSYSERR_BADDB:
			lpszError = TEXT("MMSYSERR_BADDB");
			break;

		/* Registry key not found */
		case MMSYSERR_KEYNOTFOUND:
			lpszError = TEXT("MMSYSERR_KEYNOTFOUND");
			break;

		/* Registry read error */
		case MMSYSERR_READERROR:
			lpszError = TEXT("MMSYSERR_READERROR");
			break;

		/* Registry write error */
		case MMSYSERR_WRITEERROR:
			lpszError = TEXT("MMSYSERR_WRITEERROR");
			break;

		/* Registry delete error */
		case MMSYSERR_DELETEERROR:
			lpszError = TEXT("MMSYSERR_DELETEERROR");
			break;

		/* Registry value not found */
		case MMSYSERR_VALNOTFOUND:
			lpszError = TEXT("MMSYSERR_VALNOTFOUND");
			break;

		/* Driver does not call DriverCallback */
		case MMSYSERR_NODRIVERCB:
			lpszError = TEXT("MMSYSERR_NODRIVERCB");
			break;

#endif /*WIN_TRANSLATE_ERRORS*/

		/* Unknown MM Error */
		default:
			wsprintf( szMsg, "Unknown Error #%ld", (DWORD)(mmResult) );
			lpszError = szMsg;
			break;
	}
	/* Copy MM Error string to buff */
	cLen = strlen (lpszError);
	if( cLen >= dwError )
	{
		cLen = dwError - 1;
	}
	if( cLen )
	{
		strncpy( lpszErrorBuff, lpszError, cLen );
		lpszErrorBuff[ cLen ] = 0;
	}
	return TRUE;
} /* #OF# GetMMErrorString */
