/****************************************************************************
File    : registry.c
/*
@(#) #SY# Atari800Win PLus
@(#) #IS# Registry service implementation for Win32 platforms
@(#) #BY# Richard Lawrence, Tomasz Szymankowski
@(#) #LM# 10.03.2002
*/

#include <stdio.h>
#include <crtdbg.h>
#include <stdlib.h>
#include <io.h>
#include <fcntl.h>
#include <windows.h>
#include "WinConfig.h"
#include "Resource.h"
#include "atari800.h"
#include "globals.h"
#include "rdevice.h"
#include "display_win.h"
#include "misc_win.h"
#include "input_win.h"
#include "sound_win.h"
#include "registry.h"


#define PLUS_REGKEY		"Atari800WinPLus"

/* Private objects */

static int s_nVersion = CURRENT_REV;

/* Macros */

#define RegReadString(key, sz, def, len) \
	if( ReadRegBinary( hkKey, key, sz, len, FALSE ) == READ_REG_FAIL ) { \
		strcpy( sz, def ); bRegFail = TRUE; }

#define RegWriteString(key, val) \
	WriteRegString( hkKey, key, val )

#define RegReadNumber(key, val, def) \
	if( ReadRegDWORD( hkKey, key, (DWORD*)&(val), TRUE ) == READ_REG_FAIL ) { \
		val = def; bRegFail = TRUE;	}

#define RegWriteNumber(key, val) \
	WriteRegDWORD( hkKey, key, val )


/*========================================================
Function : DeleteAllRegKeys
=========================================================*/
/* #FN#
   DeleteAllRegKeys will recursively delete everything from a supplied initial
   Key. All subkeys are enumerated and deleted as found. Note that ALL values
   underneath a key are deleted when that key is deleted. */
void
/* #AS#
   Nothing */
DeleteAllRegKeys(
	HKEY  hkInput,
    LPSTR pszName
)
{
	HKEY     hkKey     = hkInput;
	DWORD    dwIndex   = 0;
	DWORD    dwBufSize = 256;
	FILETIME ftDummy;
	char     szSubKeyName[ 256 ];

	if( NULL == hkKey )
	{
		if( RegOpenKeyEx( HKEY_CURRENT_USER, pszName, 0, KEY_ALL_ACCESS, &hkKey ) != ERROR_SUCCESS )
			return;
	}
	while( RegEnumKeyEx( hkKey, dwIndex++, szSubKeyName, &dwBufSize, NULL, NULL, NULL, &ftDummy ) == ERROR_SUCCESS )
		DeleteAllRegKeys( hkKey, szSubKeyName );

	RegDeleteKey( HKEY_CURRENT_USER, pszName );
} /* #OF# DeleteAllRegKeys */

/*========================================================
Function : WriteRegDWORD
=========================================================*/
/* #FN#
   WriteRegDWORD writes out an int to the preset Registry key HKEY_CURRENT_USER\REGNAME.
   If the HKEY passed in is valid it is used, otherwise the key is grabbed and released
   within the function. Note that RegOpenKey is used here instead of RegCreateKey, which
   is only used at init time. No calls should be made to this prior to HandleRegistry().
   Any write to the Registry that doesn't work is skipped with user notification. */
void
/* #AS#
   Nothing */
WriteRegDWORD(
	HKEY  hkInput,
	LPSTR pszItem,
	DWORD dwValue
)
{
	HKEY hkKey = hkInput;

	if( NULL == hkKey )
	{
		if( RegOpenKeyEx( HKEY_CURRENT_USER, REGNAME, 0, KEY_WRITE, &hkKey ) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	if( RegSetValueEx( hkKey, pszItem, 0, REG_DWORD, (UCHAR *)&dwValue, sizeof(DWORD) ) != ERROR_SUCCESS )
		DisplayMessage( NULL, IDS_REG_WRITE_ERROR, 0, MB_ICONSTOP | MB_OK );

	if( NULL == hkInput )
		RegCloseKey( hkKey );
} /* #OF# WriteRegDWORD */

/*========================================================
Function : ReadRegDWORD
=========================================================*/
/* #FN#
   ReadRegDWORD retrieves an existing value. To make it bulletproof the
   calling routine can request to display errors or not, depending on how
   fatal they are considered. */
int
/* #AS#
   The value was read */
ReadRegDWORD(
	HKEY   hkInput,
	LPSTR  pszItem,
	DWORD *pdwData,
	BOOL   bShowError
)
{
	HKEY  hkKey = hkInput;
	DWORD dwType;
	DWORD dwSize;
	DWORD dwValue;
	
	if( NULL == hkKey )
	{
		if( RegOpenKeyEx( HKEY_CURRENT_USER, REGNAME, 0, KEY_READ, &hkKey ) != ERROR_SUCCESS )
		{
			if( bShowError )
			{
				DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
				return READ_REG_FAIL;
			}
		}
	}
	/* Win95 is really picky about having this size set; WinNT doesn't care. Go figure. */
	dwSize = sizeof(DWORD);

	if( RegQueryValueEx( hkKey, pszItem, 0, &dwType, (UCHAR *)&dwValue, &dwSize ) != ERROR_SUCCESS )
	{
		if( bShowError )
		{
			DisplayMessage( NULL, IDS_REG_LOAD_ERROR, 0, MB_ICONEXCLAMATION | MB_OK, pszItem );
		}
		else if( NULL == hkInput )
			RegCloseKey( hkKey );

		if( NULL == hkInput )
			RegCloseKey( hkKey );

		return READ_REG_FAIL;
	}
	
	if( dwType != REG_DWORD || dwSize != sizeof(DWORD) )
	{
		DisplayMessage( NULL, IDS_REG_WRONG_SIZE, 0, MB_ICONSTOP | MB_OK );
		if( NULL == hkInput )
			RegCloseKey( hkKey );

		return READ_REG_FAIL;
	}

	if( NULL == hkInput )
		RegCloseKey( hkKey );
	
	*pdwData = dwValue;

	return TRUE;
} /* #OF# ReadRegDWORD */

/*========================================================
Function : ReadRegBinary
=========================================================*/
/* #FN#
   Retrieves an existing value */
DWORD
/* #AS#
   The value was read */
ReadRegBinary(
	HKEY  hkInput,
	LPSTR pszItem,
	LPSTR pszBuffer,
	ULONG ulMaxSize,
	BOOL  bShowError
)
{
	HKEY  hkKey     = hkInput;
	UINT  nFullSize = ulMaxSize;
	DWORD dwType    = 0;
	
	if( NULL == hkKey )
	{
		if( RegOpenKeyEx( HKEY_CURRENT_USER, REGNAME, 0, KEY_READ, &hkKey ) != ERROR_SUCCESS )
		{
			if( bShowError )
			{
				DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
				return READ_REG_FAIL;
			}
		}
	}
	if( RegQueryValueEx( hkKey, pszItem, 0, &dwType, (UCHAR *)pszBuffer, &ulMaxSize ) != ERROR_SUCCESS )
	{
		if( bShowError )
		{
			DisplayMessage( NULL, IDS_REG_KEY_ERROR, 0, MB_ICONSTOP | MB_OK, pszItem );
		}
		else
		{
			if( NULL == hkInput )
				RegCloseKey( hkKey );

			*pszBuffer = '\0';
		}
		if( NULL == hkInput )
			RegCloseKey( hkKey );

		return READ_REG_FAIL;
	}

	if( nFullSize > ulMaxSize + 1 )
		pszBuffer[ ulMaxSize + 1 ] = '\0';
	
	if( NULL == hkInput )
		RegCloseKey( hkKey );
	
	return dwType;
} /* #OF# ReadRegBinary */

/*========================================================
Function : WriteRegBinary
=========================================================*/
/* #FN#
   WriteRegBinary is similar to WriteRegDWORD except it dumps an arbitrary
   binary section of data */
void
/* #AS#
   Nothing */
WriteRegBinary(
	HKEY   hkInput,
	LPSTR  pszItem,
	UCHAR *pszData,
	int    nSize
)
{
	HKEY hkKey = hkInput;
	
	if( NULL == hkKey )
	{
		if( RegOpenKeyEx( HKEY_CURRENT_USER, REGNAME, 0, KEY_WRITE, &hkKey ) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	if( RegSetValueEx( hkKey, pszItem, 0, REG_BINARY, pszData, nSize ) != ERROR_SUCCESS )
		DisplayMessage( NULL, IDS_REG_WRITE_ERROR, 0, MB_ICONSTOP | MB_OK );

	if( NULL == hkInput )
		RegCloseKey( hkKey );
} /* #OF# WriteRegBinary */

/*========================================================
Function : WriteRegString
=========================================================*/
/* #FN#
   WriteRegString is similar to WriteRegBinary except it writes a null-terminated
   string */
void
/* #AS#
   Nothing */
WriteRegString(
	HKEY  hkInput,
	LPSTR pszItem,
	LPSTR pszData
)
{
	HKEY hkKey = hkInput;
	
	if( NULL == hkKey )
	{
		if( RegOpenKeyEx( HKEY_CURRENT_USER, REGNAME, 0, KEY_WRITE, &hkKey ) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	if( RegSetValueEx( hkKey, pszItem,	0, REG_SZ, (const UCHAR *)pszData, strlen( pszData ) ) != ERROR_SUCCESS )
		DisplayMessage( NULL, IDS_REG_WRITE_ERROR, 0, MB_ICONSTOP | MB_OK );

	if( NULL == hkInput )
		RegCloseKey( hkKey );
} /* #OF# WriteRegString */

/*========================================================
Function : ReadRegKeyset
=========================================================*/
/* #FN#
   */
int
/* #AS#
   */
ReadRegKeyset(
	HKEY hkInput,
	int  nKeyset
)
{
	HKEY  hkKey        = hkInput;
	ULONG ulKeysetReg1 = 0L;
	ULONG ulKeysetReg2 = 0L;
	ULONG ulKeysetReg3 = 0L;
	BOOL  bRegFail     = FALSE;
	int   i;

	if( NULL == hkKey )
	{
		if( RegOpenKeyEx( HKEY_CURRENT_USER, REGNAME, 0, KEY_READ, &hkKey ) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return FALSE;
		}
	}
	if( KEYS_A_JOYSTICK == nKeyset )
	{
		RegReadNumber( REG_KEYSET_A1, ulKeysetReg1, 0L );
		RegReadNumber( REG_KEYSET_A2, ulKeysetReg2, 0L );
		RegReadNumber( REG_KEYSET_A3, ulKeysetReg3, 0L );
	}
	else
	{
		RegReadNumber( REG_KEYSET_B1, ulKeysetReg1, 0L );
		RegReadNumber( REG_KEYSET_B2, ulKeysetReg2, 0L );
		RegReadNumber( REG_KEYSET_B3, ulKeysetReg3, 0L );
	}
	_ASSERT(NUM_KBJOY_KEYS - 2 <= sizeof(ULONG) * 2);

	i = NUM_KBJOY_KEYS - 1;

	g_Input.Joy.anKeysets[ nKeyset + NUM_KBJOY_DEVICES ][ i-- ] = (WORD)(ulKeysetReg3 & 0x000000ffL);
	ulKeysetReg3 = ulKeysetReg3 >> 8;
	g_Input.Joy.anKeysets[ nKeyset + NUM_KBJOY_DEVICES ][ i-- ] = (WORD)(ulKeysetReg3 & 0x000000ffL);

	for( ; i > -1; )
	{
		g_Input.Joy.anKeysets[ nKeyset + NUM_KBJOY_DEVICES ][ i-- ] = (WORD)(ulKeysetReg2 & 0x000000ffL);
		ulKeysetReg2 = ulKeysetReg2 >> 8;
		g_Input.Joy.anKeysets[ nKeyset + NUM_KBJOY_DEVICES ][ i-- ] = (WORD)(ulKeysetReg1 & 0x000000ffL);
		ulKeysetReg1 = ulKeysetReg1 >> 8;
	}

	if( NULL == hkInput )
		RegCloseKey( hkKey );

	return !bRegFail;
} /* #OF# ReadRegKeyset */

/*========================================================
Function : WriteRegKeyset
=========================================================*/
/* #FN#
   */
void
/* #AS#
   Nothing */
WriteRegKeyset(
	HKEY hkInput,
	int  nKeyset
)
{
	HKEY  hkKey        = hkInput;
	ULONG ulKeysetReg1 = 0L;
	ULONG ulKeysetReg2 = 0L;
	ULONG ulKeysetReg3 = 0L;
	int   i;

	if( NULL == hkKey )
	{
		if( RegOpenKeyEx( HKEY_CURRENT_USER, REGNAME, 0, KEY_WRITE, &hkKey ) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	/* The key values will be packed into DWORDs */
	_ASSERT(NUM_KBJOY_KEYS - 2 <= sizeof(ULONG) * 2);

	for( i = 0; i < NUM_KBJOY_KEYS - 2; )
	{
		ulKeysetReg1 = ulKeysetReg1 << 8;
		ulKeysetReg1 |= g_Input.Joy.anKeysets[ nKeyset + NUM_KBJOY_DEVICES ][ i++ ] & 0x000000ffL;
		ulKeysetReg2 = ulKeysetReg2 << 8;
		ulKeysetReg2 |= g_Input.Joy.anKeysets[ nKeyset + NUM_KBJOY_DEVICES ][ i++ ] & 0x000000ffL;
	}
	ulKeysetReg3 |= g_Input.Joy.anKeysets[ nKeyset + NUM_KBJOY_DEVICES ][ i++ ] & 0x000000ffL;
	ulKeysetReg3 = ulKeysetReg3 << 8;
	ulKeysetReg3 |= g_Input.Joy.anKeysets[ nKeyset + NUM_KBJOY_DEVICES ][ i ] & 0x000000ffL;

	if( KEYS_A_JOYSTICK == nKeyset )
	{
		RegWriteNumber( REG_KEYSET_A1, ulKeysetReg1 );
		RegWriteNumber( REG_KEYSET_A2, ulKeysetReg2 );
		RegWriteNumber( REG_KEYSET_A3, ulKeysetReg3 );
	}
	else
	{
		RegWriteNumber( REG_KEYSET_B1, ulKeysetReg1 );
		RegWriteNumber( REG_KEYSET_B2, ulKeysetReg2 );
		RegWriteNumber( REG_KEYSET_B3, ulKeysetReg3 );
	}
} /* #OF# WriteRegKeyset */

/*========================================================
Function : WriteRegDrives
=========================================================*/
/* #FN#
   */
void
/* #AS#
   Nothing */
WriteRegDrives(
	HKEY hkInput
)
{
	HKEY hkKey = hkInput;

	if( NULL == hkKey )
	{
		if( RegOpenKeyEx( HKEY_CURRENT_USER, REGNAME, 0, KEY_WRITE, &hkKey ) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	RegWriteString( REG_HD1, atari_h1_dir );
	RegWriteString( REG_HD2, atari_h2_dir );
	RegWriteString( REG_HD3, atari_h3_dir );
	RegWriteString( REG_HD4, atari_h4_dir );

	RegWriteString( REG_DRIVE1, sio_filename[ 0 ] );
	RegWriteString( REG_DRIVE2, sio_filename[ 1 ] );
	RegWriteString( REG_DRIVE3, sio_filename[ 2 ] );
	RegWriteString( REG_DRIVE4, sio_filename[ 3 ] );
	RegWriteString( REG_DRIVE5, sio_filename[ 4 ] );
	RegWriteString( REG_DRIVE6, sio_filename[ 5 ] );
	RegWriteString( REG_DRIVE7, sio_filename[ 6 ] );
	RegWriteString( REG_DRIVE8, sio_filename[ 7 ] );

	RegWriteString( REG_EXE_PATH, atari_exe_dir );
	RegWriteString( REG_FILE_STATE, atari_state_dir );

	if( NULL == hkInput )
		RegCloseKey( hkKey );
} /* #OF# WriteRegDrives */

/*========================================================
Function : WriteAtari800Registry
=========================================================*/
/* #FN#
   */
void
/* #AS#
   Nothing */
WriteAtari800Registry(
	HKEY hkInput
)
{
	HKEY hkKey = hkInput;

	if( NULL == hkKey )
	{
		if( RegOpenKeyEx( HKEY_CURRENT_USER, REGNAME, 0, KEY_ALL_ACCESS, &hkKey ) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	RegWriteNumber( REG_REFRESH_RATE,      refresh_rate                  );
	RegWriteNumber( REG_MACHINE_TYPE,      machine_type                  );
	RegWriteNumber( REG_TV_MODE,           tv_mode                       );
	RegWriteNumber( REG_HD_READ_ONLY,      hd_read_only                  );
	RegWriteNumber( REG_DISABLE_BASIC,     disable_basic                 );
	RegWriteNumber( REG_ENABLE_SIO_PATCH,  enable_sio_patch              );
	RegWriteNumber( REG_ENABLE_H_PATCH,    enable_h_patch                );
	RegWriteNumber( REG_ENABLE_P_PATCH,    enable_p_patch                );
	RegWriteNumber( REG_ENABLE_R_PATCH,    enable_r_patch                );
	RegWriteNumber( REG_ENABLE_RTIME,      rtime_enabled                 );
	RegWriteNumber( REG_ENABLE_STEREO,     stereo_enabled                );
	RegWriteNumber( REG_ARTIF_MODE,        global_artif_mode             );
	RegWriteNumber( REG_CART_TYPE,         cart_type                     );
	RegWriteString( REG_ROM_OSA,           atari_osa_filename            );
	RegWriteString( REG_ROM_OSB,           atari_osb_filename            );
	RegWriteString( REG_ROM_XLXE,          atari_xlxe_filename           );
	RegWriteString( REG_ROM_5200,          atari_5200_filename           );
	RegWriteString( REG_ROM_BASIC,         atari_basic_filename          );
	RegWriteString( REG_FILE_TAPE,         cassette_filename             );
	RegWriteString( REG_PRINT_COMMAND,     print_command                 );
	RegWriteNumber( REG_MOUSE_MODE,        mouse_mode                    );
	RegWriteNumber( REG_MOUSE_PORT,        mouse_port                    );
	RegWriteNumber( REG_MOUSE_SPEED,       mouse_speed                   );
	RegWriteNumber( REG_POT_MIN,           mouse_pot_min                 );
	RegWriteNumber( REG_POT_MAX,           mouse_pot_max                 );
	RegWriteNumber( REG_JOY_INERTIA,       mouse_joy_inertia             );
	RegWriteNumber( REG_PEN_XOFFSET,       mouse_pen_ofs_h               );
	RegWriteNumber( REG_PEN_YOFFSET,       mouse_pen_ofs_v               );
	RegWriteNumber( REG_BLOCK_OPPOSITE,    joy_block_opposite_directions );
	RegWriteString( REG_ROM_CURRENT,       g_szCurrentRom                );
	RegWriteString( REG_ROM_OTHER,         g_szOtherRom                  );
	RegWriteString( REG_FILE_TEMPLATE,     g_szTemplateFile              );
	RegWriteString( REG_FILE_PALETTE,      g_szPaletteFile               );
	RegWriteString( REG_FILE_AUTOBOOT,     g_szAutobootFile              );
	RegWriteNumber( REG_START_XPOS,        g_nStartX                     );
	RegWriteNumber( REG_START_YPOS,        g_nStartY                     );
	RegWriteNumber( REG_SCREEN_STATE,      g_Screen.ulState              );
	RegWriteNumber( REG_STRETCH_MODE,      g_Screen.nStretchMode         );
	RegWriteNumber( REG_MEMORY_TYPE,       g_Screen.nMemoryType          );
	RegWriteNumber( REG_COLOR_BLACK,       g_Screen.Pal.nBlackLevel      );
	RegWriteNumber( REG_COLOR_WHITE,       g_Screen.Pal.nWhiteLevel      );
	RegWriteNumber( REG_COLOR_SATURATION,  g_Screen.Pal.nSaturation      );
	RegWriteNumber( REG_COLOR_SHIFT,       g_Screen.Pal.nColorShift      );
	RegWriteNumber( REG_USE_VOLUME_ONLY,   g_Sound.nDigitized            );
	RegWriteNumber( REG_SOUND_STATE,       g_Sound.ulState               );
	RegWriteNumber( REG_SOUND_RATE,        g_Sound.nRate                 );
	RegWriteNumber( REG_SOUND_VOLUME,      g_Sound.nVolume               );
	RegWriteNumber( REG_SOUND_UPDATE,      g_Sound.nSkipUpdate           );
	RegWriteNumber( REG_SOUND_LATENCY,     g_Sound.nLatency              );
	RegWriteNumber( REG_INPUT_STATE,       g_Input.ulState               );
	RegWriteNumber( REG_JOYSTICKS,         g_Input.Joy.ulSelected        );
	RegWriteNumber( REG_AUTOFIRE_MODE,     g_Input.Joy.nAutoMode         );
	RegWriteNumber( REG_AUTOFIRE_STICKS,   g_Input.Joy.ulAutoSticks      );
	RegWriteNumber( REG_ARROWS_MODE,       g_Input.Key.nArrowsMode       );
	RegWriteNumber( REG_MISC_STATE,        g_Misc.ulState                );
	RegWriteNumber( REG_DONT_SHOW,         g_Misc.ulDontShow             );
	RegWriteNumber( REG_FILE_ASSOCIATIONS, g_Misc.ulFileAssociations     );
	RegWriteNumber( REG_CHEAT_COLLISIONS,  g_Misc.Cheat.ulCollisions     );
	RegWriteNumber( REG_CHEAT_MEMO,        g_Misc.Cheat.nMemo            );
	RegWriteNumber( REG_CHEAT_SEARCH,      g_Misc.Cheat.nSearch          );
	RegWriteNumber( REG_CHEAT_LOCK,        g_Misc.Cheat.nLock            );
	RegWriteNumber( REG_DOUBLE_RATE,       g_nDoubleRate                 );
	RegWriteNumber( REG_SPEED_PERCENT,     g_nSpeedPercent               );
	RegWriteNumber( REG_RAMSIZE_OSA,       g_anRamSize[ MACHINE_OSA  ]   );
	RegWriteNumber( REG_RAMSIZE_OSB,       g_anRamSize[ MACHINE_OSB  ]   );
	RegWriteNumber( REG_RAMSIZE_XLXE,      g_anRamSize[ MACHINE_XLXE ]   );
	RegWriteNumber( REG_CURRENT_REV,       s_nVersion                    );

	WriteRegDrives( hkKey );

	WriteRegKeyset( hkKey, KEYS_A_JOYSTICK );
	WriteRegKeyset( hkKey, KEYS_B_JOYSTICK );

	if( NULL == hkInput )
		RegCloseKey( hkKey );
} /* #OF# WriteAtari800Registry */

/*========================================================
Function : InitialiseRegistry
=========================================================*/
/* #FN#
   */
void
/* #AS#
   Nothing */
InitialiseRegistry(
	HKEY hkInput,
	BOOL bErasePaths
)
{
	HKEY hkKey = hkInput;
	int  i;

	if( NULL == hkKey )
	{
		DWORD dwDisposition = REG_OPENED_EXISTING_KEY;
		
		if( RegOpenKeyEx( HKEY_CURRENT_USER, REGNAME, 0, KEY_ALL_ACCESS, &hkKey ) != ERROR_SUCCESS )
		{
			if( RegCreateKeyEx( HKEY_CURRENT_USER, REGNAME, 0, PLUS_REGKEY, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hkKey, &dwDisposition ) != ERROR_SUCCESS )
			{
				DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
				/* Probably should make this fault in a more elegant manner */
				Atari_Exit( FALSE );
			}
		}
	}
	global_artif_mode           = 0;
	machine_type                = MACHINE_XLXE;
	tv_mode                     = TV_PAL;
	refresh_rate                = 1;
	hd_read_only                = 1;
	disable_basic               = 1;
	rtime_enabled               = 1;
	enable_sio_patch            = 1;
	enable_h_patch              = 0;
	enable_p_patch              = 0;
	enable_r_patch              = 0;
	stereo_enabled              = 0;
	cart_type                   = CART_NONE;
	mouse_mode                  = DEF_MOUSE_MODE;
	mouse_port                  = DEF_MOUSE_PORT;
	mouse_speed                 = DEF_MOUSE_SPEED;
	mouse_pot_min               = DEF_POT_MIN;
	mouse_pot_max               = DEF_POT_MAX;
	mouse_joy_inertia           = DEF_JOY_INERTIA;
	mouse_pen_ofs_h             = DEF_PEN_OFFSET_X;
	mouse_pen_ofs_v             = DEF_PEN_OFFSET_Y;
	g_nDoubleRate               = DEF_DOUBLE_RATE;
	g_nSpeedPercent             = DEF_SPEED_PERCENT;
	g_anRamSize[ MACHINE_OSA  ] = DEF_RAMSIZE_OSA;
	g_anRamSize[ MACHINE_OSB  ] = DEF_RAMSIZE_OSB;
	g_anRamSize[ MACHINE_XLXE ] = DEF_RAMSIZE_XLXE;
	g_Misc.ulState              = DEF_MISC_STATE;
	g_Misc.ulDontShow           = DEF_DONT_SHOW_FLAGS;
	g_Misc.Cheat.ulCollisions   = DEF_CHEAT_COLLISIONS;
	g_Misc.Cheat.nMemo          = DEF_CHEAT_MEMO;
	g_Misc.Cheat.nSearch        = DEF_CHEAT_SEARCH;
	g_Misc.Cheat.nLock          = DEF_CHEAT_LOCK;
	g_Sound.ulState             = DEF_SOUND_STATE;
	g_Sound.nDigitized          = DEF_USE_VOLUME_ONLY;
	g_Sound.nRate               = DEF_SOUND_RATE;
	g_Sound.nVolume             = DEF_SOUND_VOL;
	g_Sound.nSkipUpdate         = DEF_SKIP_UPDATE;
	g_Sound.nLatency            = DEF_SOUND_LATENCY;
	g_Input.ulState             = DEF_INPUT_STATE;
	g_Input.Joy.ulSelected      = DEF_JOY_SELECTS;
	g_Input.Joy.nAutoMode       = DEF_AUTOFIRE_MODE;
	g_Input.Joy.ulAutoSticks    = DEF_AUTOFIRE_STICKS;
	g_Input.Key.nArrowsMode     = DEF_ARROWS_MODE;
	g_Screen.ulState            = DEF_SCREEN_STATE;
	g_Screen.nStretchMode       = DEF_STRETCH_MODE;
	g_Screen.nMemoryType        = DEF_MEMORY_TYPE;
	g_Screen.Pal.nBlackLevel    = DEF_CLR_BLACK_LEVEL;
	g_Screen.Pal.nWhiteLevel    = DEF_CLR_WHITE_LEVEL;
	g_Screen.Pal.nSaturation    = DEF_CLR_SATURATION;
	g_Screen.Pal.nColorShift    = DEF_CLR_SHIFT;

	/* Clear keysets A & B */
	for( i = 0; i < NUM_KBJOY_KEYS; i++ )
	{
		g_Input.Joy.anKeysets[ KEYS_A_JOYSTICK + NUM_KBJOY_DEVICES ][ i ] = 0;
		g_Input.Joy.anKeysets[ KEYS_B_JOYSTICK + NUM_KBJOY_DEVICES ][ i ] = 0;
	}
	
	if( bErasePaths || *atari_osa_filename == '\0' ) /* WinNT wants this */
		strcpy( atari_osa_filename, DEFAULT_OSA );
	if( bErasePaths || *atari_osb_filename == '\0' )
		strcpy( atari_osb_filename, DEFAULT_OSB );
	if( bErasePaths || *atari_xlxe_filename == '\0' )
		strcpy( atari_xlxe_filename, DEFAULT_OXL );
	if( bErasePaths || *atari_5200_filename == '\0' )
		strcpy( atari_5200_filename, DEFAULT_O52 );
	if( bErasePaths || *atari_basic_filename == '\0' )
		strcpy( atari_basic_filename, DEFAULT_BAS );

	if( bErasePaths || *g_szTemplateFile == '\0' )
		strcpy( g_szTemplateFile, DEFAULT_A8K );
	if( bErasePaths || *g_szPaletteFile == '\0' )
		strcpy( g_szPaletteFile, DEFAULT_ACT );
	if( bErasePaths || *atari_state_dir == '\0' )
		strcpy( atari_state_dir, DEFAULT_A8S );
	if( bErasePaths || *cassette_filename == '\0' )
		strcpy( cassette_filename, DEFAULT_TAP );
	if( bErasePaths || *g_szAutobootFile == '\0' )
		strcpy( g_szAutobootFile, "None" );

	if( bErasePaths || *atari_h1_dir == '\0' )
		strcpy( atari_h1_dir, DEFAULT_HDD );
	if( bErasePaths || *atari_h2_dir == '\0' )
		strcpy( atari_h2_dir, DEFAULT_HDD );
	if( bErasePaths || *atari_h3_dir == '\0' )
		strcpy( atari_h3_dir, DEFAULT_HDD );
	if( bErasePaths || *atari_h4_dir == '\0' )
		strcpy( atari_h4_dir, DEFAULT_HDD );

	if( bErasePaths || *atari_exe_dir == '\0' )
		strcpy( atari_exe_dir, "." );

	strcpy( g_szCurrentRom, "None" );
	strcpy( g_szOtherRom,   "None" );

	strcpy( print_command, DEF_PRINT_COMMAND );

	for( i = 0; i < MAX_DRIVES; i++ )
	{
		strcpy( sio_filename[ i ], "Off" ) ;
		drive_status[ i ] = Off;
	}

	WriteAtari800Registry( hkKey );

	if( NULL == hkInput )
		RegCloseKey( hkKey );
} /* #OF# InitialiseRegistry */

/*========================================================
Function : ReadRegPaths
=========================================================*/
/* #FN#
   */
BOOL
/* #AS#
   */
ReadRegPaths(
	HKEY hkInput
)
{
	HKEY hkKey    = hkInput;
	BOOL bRegFail = FALSE;

	if( NULL == hkKey )
	{
		if( RegOpenKeyEx( HKEY_CURRENT_USER, REGNAME, 0, KEY_READ, &hkKey ) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return FALSE;
		}
	}
	RegReadString( REG_ROM_BASIC,     atari_basic_filename, DEFAULT_BAS, MAX_PATH );
	RegReadString( REG_ROM_OSA,       atari_osa_filename,   DEFAULT_OSA, MAX_PATH );
	RegReadString( REG_ROM_OSB,       atari_osb_filename,   DEFAULT_OSB, MAX_PATH );
	RegReadString( REG_ROM_XLXE,      atari_xlxe_filename,  DEFAULT_OXL, MAX_PATH );
	RegReadString( REG_ROM_5200,      atari_5200_filename,  DEFAULT_O52, MAX_PATH );
	RegReadString( REG_FILE_TAPE,     cassette_filename,    DEFAULT_TAP, MAX_PATH );
	RegReadString( REG_FILE_STATE,    atari_state_dir,      DEFAULT_A8S, MAX_PATH );
	RegReadString( REG_EXE_PATH,      atari_exe_dir,        ".",         MAX_PATH );
	RegReadString( REG_HD1,           atari_h1_dir,         DEFAULT_HDD, MAX_PATH );
	RegReadString( REG_HD2,           atari_h2_dir,         DEFAULT_HDD, MAX_PATH );
	RegReadString( REG_HD3,           atari_h3_dir,         DEFAULT_HDD, MAX_PATH );
	RegReadString( REG_HD4,           atari_h4_dir,         DEFAULT_HDD, MAX_PATH );
	RegReadString( REG_FILE_TEMPLATE, g_szTemplateFile,     DEFAULT_A8K, MAX_PATH );
	RegReadString( REG_FILE_PALETTE,  g_szPaletteFile,      DEFAULT_ACT, MAX_PATH );
	RegReadString( REG_FILE_AUTOBOOT, g_szAutobootFile,     "None",      MAX_PATH );
	RegReadString( REG_ROM_CURRENT,   g_szCurrentRom,       "None",      MAX_PATH );
	RegReadString( REG_ROM_OTHER,     g_szOtherRom,         "None",      MAX_PATH );

	if( NULL == hkInput )
		RegCloseKey( hkKey );

	return !bRegFail;
} /* #OF# ReadRegPaths */

/*========================================================
Function : ReadRegDrives
=========================================================*/
/* #FN#
   */
void
/* #AS#
   Nothing */
ReadRegDrives(
	HKEY hkInput
)
{
	HKEY hkKey    = hkInput;
	BOOL bRegFail = FALSE;
	int  i;

	if( NULL == hkKey )
	{
		if( RegOpenKeyEx( HKEY_CURRENT_USER, REGNAME, 0, KEY_READ, &hkKey ) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	RegReadString( REG_DRIVE1, sio_filename[ 0 ], "Off", MAX_PATH );
	RegReadString( REG_DRIVE2, sio_filename[ 1 ], "Off", MAX_PATH );
	RegReadString( REG_DRIVE3, sio_filename[ 2 ], "Off", MAX_PATH );
	RegReadString( REG_DRIVE4, sio_filename[ 3 ], "Off", MAX_PATH );
	RegReadString( REG_DRIVE5, sio_filename[ 4 ], "Off", MAX_PATH );
	RegReadString( REG_DRIVE6, sio_filename[ 5 ], "Off", MAX_PATH );
	RegReadString( REG_DRIVE7, sio_filename[ 6 ], "Off", MAX_PATH );
	RegReadString( REG_DRIVE8, sio_filename[ 7 ], "Off", MAX_PATH );

	for( i = 0; i < MAX_DRIVES; i++ )
	{
		if( *sio_filename[ i ] == '\0' )
			strcpy( sio_filename[ i ], "Off" );

		if( strcmp( sio_filename[ i ], "Off" ) == 0 )
			drive_status[ i ] = Off;
		else
		if( strcmp( sio_filename[ i ], "Empty" ) == 0 )
			drive_status[ i ] = NoDisk;
	}

	if( NULL == hkInput )
		RegCloseKey( hkKey );

	if( bRegFail )
		WriteAtari800Registry( NULL );
} /* #OF# ReadRegDrives */

/*========================================================
Function : HandleRegistry
=========================================================*/
/* #FN#
   Creates the Registry entries if they don't exist and read all
   the defaults in at runtime (this is called from MainFrame) */
BOOL
/* #AS#
   TRUE if the Registry has been initialized, otherwise FALSE */
HandleRegistry( void )
{
	HKEY  hkKey         = NULL;
	DWORD dwDisposition = REG_OPENED_EXISTING_KEY;
	BOOL  bRegFail      = FALSE;
	BOOL  bInitialReg   = FALSE;

	if( RegOpenKeyEx( HKEY_CURRENT_USER, REGNAME, 0, KEY_ALL_ACCESS, &hkKey ) != ERROR_SUCCESS )
	{
		if( RegCreateKeyEx( HKEY_CURRENT_USER, REGNAME, 0, PLUS_REGKEY, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hkKey, &dwDisposition ) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			/* Probably should make this fault in a more elegant manner */
			Atari_Exit( FALSE );
		}
	}
	/* If the key doesn't exist, fill in defaults. This is the only time these will
	   be written all at once. From here on out we trust the Registry to hold them
	   (yeah, right) and update when key dialogs are used to configure them */
	if( dwDisposition == REG_CREATED_NEW_KEY )
	{
		bInitialReg = TRUE;
		InitialiseRegistry( hkKey, !Misc_TestRomPaths( NULL, NULL ) );
	}
	else
	{
		/* Read in the values from the Registry. Only fail and return error when it is
		   REALLY fatal (you never know what somebody might do with their registry)
		   For most of these an error will result in the value being the default run-time */
		if( ReadRegDWORD( hkKey, REG_CURRENT_REV, (DWORD *)&s_nVersion, FALSE ) == READ_REG_FAIL )
			s_nVersion = CURRENT_REV - 1;
		
		if( s_nVersion != CURRENT_REV )
		{
			DisplayMessage( NULL, IDS_WARN_OUTDATED, 0, MB_ICONINFORMATION | MB_OK );
			s_nVersion = CURRENT_REV;

			ReadRegPaths( hkKey );	/* Since we already have a registry, read the paths + filenames at least */
									/* Note that this will have to change if I ever invalidate the path system (unlikely) */
			DeleteAllRegKeys( hkKey, REGNAME );

			RegCloseKey( hkKey );
			if( RegCreateKeyEx( HKEY_CURRENT_USER, REGNAME, 0, PLUS_REGKEY, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hkKey, &dwDisposition ) != ERROR_SUCCESS )
			{
				DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
				/* Probably should make this fault in a more elegant manner */
				Atari_Exit( FALSE );
			}
			InitialiseRegistry( hkKey, FALSE );
			bInitialReg = TRUE;
		}
		else
		{
			RegReadString( REG_PRINT_COMMAND,     print_command,                 DEF_PRINT_COMMAND,  PRINT_CMD_LENGTH );
			RegReadNumber( REG_MACHINE_TYPE,      machine_type,                  MACHINE_XLXE          );
			RegReadNumber( REG_TV_MODE,           tv_mode,                       TV_PAL                );
			RegReadNumber( REG_CART_TYPE,         cart_type,                     CART_NONE             );
			RegReadNumber( REG_DISABLE_BASIC,     disable_basic,                 1                     );
			RegReadNumber( REG_REFRESH_RATE,      refresh_rate,                  1                     );
			RegReadNumber( REG_HD_READ_ONLY,      hd_read_only,                  1                     );
			RegReadNumber( REG_ENABLE_RTIME,      rtime_enabled,                 1                     );
			RegReadNumber( REG_ENABLE_SIO_PATCH,  enable_sio_patch,              1                     );
			RegReadNumber( REG_ENABLE_H_PATCH,    enable_h_patch,                0                     );
			RegReadNumber( REG_ENABLE_P_PATCH,    enable_p_patch,                0                     );
			RegReadNumber( REG_ENABLE_R_PATCH,    enable_r_patch,                0                     );
			RegReadNumber( REG_ENABLE_STEREO,     stereo_enabled,                0                     );
			RegReadNumber( REG_ARTIF_MODE,        global_artif_mode,             0                     );
			RegReadNumber( REG_BLOCK_OPPOSITE,    joy_block_opposite_directions, 1                     );
			RegReadNumber( REG_MOUSE_MODE,        mouse_mode,                    DEF_MOUSE_MODE        );
			RegReadNumber( REG_MOUSE_PORT,        mouse_port,                    DEF_MOUSE_PORT        );
			RegReadNumber( REG_MOUSE_SPEED,       mouse_speed,                   DEF_MOUSE_SPEED       );
			RegReadNumber( REG_POT_MIN,           mouse_pot_min,                 DEF_POT_MIN           );
			RegReadNumber( REG_POT_MAX,           mouse_pot_max,                 DEF_POT_MAX           );
			RegReadNumber( REG_JOY_INERTIA,       mouse_joy_inertia,             DEF_JOY_INERTIA       );
			RegReadNumber( REG_PEN_XOFFSET,       mouse_pen_ofs_h,               DEF_PEN_OFFSET_X      );
			RegReadNumber( REG_PEN_YOFFSET,       mouse_pen_ofs_v,               DEF_PEN_OFFSET_Y      );
			RegReadNumber( REG_DOUBLE_RATE,       g_nDoubleRate,                 DEF_DOUBLE_RATE       );
			RegReadNumber( REG_SPEED_PERCENT,     g_nSpeedPercent,               DEF_SPEED_PERCENT     );
			RegReadNumber( REG_RAMSIZE_OSA,       g_anRamSize[ MACHINE_OSA ],    DEF_RAMSIZE_OSA       );
			RegReadNumber( REG_RAMSIZE_OSB,       g_anRamSize[ MACHINE_OSB ],    DEF_RAMSIZE_OSB       );
			RegReadNumber( REG_RAMSIZE_XLXE,      g_anRamSize[ MACHINE_XLXE ],   DEF_RAMSIZE_XLXE      );
			RegReadNumber( REG_ARROWS_MODE,       g_Input.Key.nArrowsMode,       DEF_ARROWS_MODE       );
			RegReadNumber( REG_SCREEN_STATE,      g_Screen.ulState,              DEF_SCREEN_STATE      );
			RegReadNumber( REG_STRETCH_MODE,      g_Screen.nStretchMode,         DEF_STRETCH_MODE      );
			RegReadNumber( REG_MEMORY_TYPE,       g_Screen.nMemoryType,          DEF_MEMORY_TYPE       );
			RegReadNumber( REG_COLOR_BLACK,       g_Screen.Pal.nBlackLevel,      DEF_CLR_BLACK_LEVEL   );
			RegReadNumber( REG_COLOR_WHITE,       g_Screen.Pal.nWhiteLevel,      DEF_CLR_WHITE_LEVEL   );
			RegReadNumber( REG_COLOR_SATURATION,  g_Screen.Pal.nSaturation,      DEF_CLR_SATURATION    );
			RegReadNumber( REG_COLOR_SHIFT,       g_Screen.Pal.nColorShift,      DEF_CLR_SHIFT         );
			RegReadNumber( REG_SOUND_RATE,        g_Sound.nRate,                 DEF_SOUND_RATE        );
			RegReadNumber( REG_SOUND_STATE,       g_Sound.ulState,               DEF_SOUND_STATE       );
			RegReadNumber( REG_SOUND_VOLUME,      g_Sound.nVolume,               DEF_SOUND_VOL         );
			RegReadNumber( REG_SOUND_UPDATE,      g_Sound.nSkipUpdate,           DEF_SKIP_UPDATE       );
			RegReadNumber( REG_SOUND_LATENCY,     g_Sound.nLatency,              DEF_SOUND_LATENCY     );
			RegReadNumber( REG_USE_VOLUME_ONLY,   g_Sound.nDigitized,            DEF_USE_VOLUME_ONLY   );
			RegReadNumber( REG_INPUT_STATE,       g_Input.ulState,               DEF_INPUT_STATE       );
			RegReadNumber( REG_AUTOFIRE_MODE,     g_Input.Joy.nAutoMode,         DEF_DONT_SHOW_FLAGS   );
			RegReadNumber( REG_AUTOFIRE_STICKS,   g_Input.Joy.ulAutoSticks,      DEF_AUTOFIRE_STICKS   );
			RegReadNumber( REG_JOYSTICKS,         g_Input.Joy.ulSelected,        DEF_JOY_SELECTS       );
			RegReadNumber( REG_MISC_STATE,        g_Misc.ulState,                DEF_MISC_STATE        );
			RegReadNumber( REG_DONT_SHOW,         g_Misc.ulDontShow,             DEF_DONT_SHOW_FLAGS   );
			RegReadNumber( REG_FILE_ASSOCIATIONS, g_Misc.ulFileAssociations,     DEF_FILE_ASSOCIATIONS );
			RegReadNumber( REG_CHEAT_COLLISIONS,  g_Misc.Cheat.ulCollisions,     DEF_CHEAT_COLLISIONS  );
			RegReadNumber( REG_CHEAT_MEMO,        g_Misc.Cheat.nMemo,            DEF_CHEAT_MEMO        );
			RegReadNumber( REG_CHEAT_SEARCH,      g_Misc.Cheat.nSearch,          DEF_CHEAT_SEARCH      );
			RegReadNumber( REG_CHEAT_LOCK,        g_Misc.Cheat.nLock,            DEF_CHEAT_LOCK        );
			RegReadNumber( REG_START_XPOS,        g_nStartX,                     DEF_START_X           );
			RegReadNumber( REG_START_YPOS,        g_nStartY,                     DEF_START_Y           );

			if( g_nStartX > GetSystemMetrics( SM_CXFULLSCREEN ) )
				g_nStartX = GetSystemMetrics( SM_CXFULLSCREEN ) - (g_Screen.ulState & SM_WRES_DOUBLE ? ATARI_DOUBLE_VIS_WIDTH : ATARI_VIS_WIDTH);
			if( g_nStartX < 0 )
				g_nStartX = 0;
			
			if( g_nStartY > GetSystemMetrics( SM_CYFULLSCREEN ) )
				g_nStartY = GetSystemMetrics( SM_CYFULLSCREEN ) - (g_Screen.ulState & SM_WRES_DOUBLE ? ATARI_DOUBLE_HEIGHT : ATARI_HEIGHT);
			if( g_nStartY < 0 )
				g_nStartY = 0;

			if( 22050 == g_Sound.nRate )
				g_Sound.ulState &= ~SOUND_CUSTOM_RATE;

			if( !ReadRegPaths( NULL ) )
				bRegFail = TRUE;

			if( !ReadRegKeyset( hkKey, KEYS_A_JOYSTICK ) )
				bRegFail = TRUE;
			if( !ReadRegKeyset( hkKey, KEYS_B_JOYSTICK ) )
				bRegFail = TRUE;
		}
	}
	RegCloseKey( hkKey );

	if( bRegFail )
		WriteAtari800Registry( NULL );

	return bInitialReg;
} /* #OF# HandleRegistry */

/*========================================================
Function : GetRegKeyHandle
=========================================================*/
/* #FN#
   Opens/Creates registry key pszKeyName in HKEY_CLASSES_ROOT section */
HKEY
/* #AS#
   Opened/Created registry key handle */
GetRegKeyHandle(
	LPCSTR pszKeyName, /* #IN# Name of registry key */
	BOOL   bCreateKey  /* #IN# Create the specified key if the key does not exist in the registry */ )
{
	DWORD dwDisposition = REG_OPENED_EXISTING_KEY;
	BOOL  bRegFail      = TRUE;
	HKEY  hkKey         = NULL;

	if( RegOpenKeyEx( HKEY_CLASSES_ROOT, pszKeyName, 0, KEY_ALL_ACCESS, &hkKey ) != ERROR_SUCCESS )
	{
		if( bCreateKey )
			if( RegCreateKeyEx(
					HKEY_CLASSES_ROOT,			/* Handle of an open key               */
					pszKeyName,					/* Address of subkey name              */
					0,							/* Reserved                            */
					PLUS_REGKEY,				/* Address of class string             */
					REG_OPTION_NON_VOLATILE,	/* Special options flag                */
					KEY_ALL_ACCESS,				/* Desired security access             */
					NULL,						/* Address of key security structure   */
					&hkKey,						/* Address of buffer for opened handle */  
					&dwDisposition 				/* Address of disposition value buffer */
				) != ERROR_SUCCESS )
			{
				DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			}
			else
				bRegFail = FALSE;
	}
	else
		bRegFail = FALSE;

	return (bRegFail ? NULL : hkKey);
} /* #OF# GetRegKeyHandle */
