/****************************************************************************
File    : registry.c
/*
@(#) #SY# Atari800Win PLus
@(#) #IS# Registry service implementation for Win32 platforms
@(#) #BY# Richard Lawrence, Tomasz Szymankowski
@(#) #LM# 24.03.2001
*/

#include <stdio.h>
#include <crtdbg.h>
#include <stdlib.h>
#include <io.h>
#include <fcntl.h>
#include <windows.h>
#include "WinConfig.h"
#include "Resource.h"
#include "atari800.h"
#include "globals.h"
#include "display_win.h"
#include "misc_win.h"
#include "input_win.h"
#include "sound_win.h"
#include "registry.h"


#define PLUS_REGKEY		"Atari800WinPLus"

/* Private objects */

static int s_nVersion = CURRENT_REV;


/*========================================================
Function : DeleteAllRegKeys
=========================================================*/
/* #FN#
   DeleteAllRegKeys will recursively delete everything from a supplied initial
   Key. All subkeys are enumerated and deleted as found. Note that ALL values
   underneath a key are deleted when that key is deleted. */
void
/* #AS#
   Nothing */
DeleteAllRegKeys( HKEY  hkInput,
                  char *pszName )
{
	HKEY     hkKey     = NULL;
	DWORD    dwIndex   = 0;
	DWORD    dwBufSize = 256;
	FILETIME dummy;
	char     szSubKeyName[ 256 ];

	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,	/* handle of open key                */
			pszName,			/* address of name of subkey to open */
			0,					/* reserved                          */
			KEY_ALL_ACCESS,		/* security access mask              */
			&hkKey 				/* address of handle of open key     */
			) != ERROR_SUCCESS )
		{
			return;
		}
	}
	else
		hkKey = hkInput;

	while( RegEnumKeyEx( 
				hkKey,			/* handle of key to enumerate           */
				dwIndex++,		/* index of subkey to enumerate         */
				szSubKeyName,	/* address of buffer for subkey name    */
				&dwBufSize,		/* address for size of subkey buffer    */
				NULL,			/* reserved                             */
				NULL,			/* address of buffer for class string   */
				NULL,			/* address for size of class buffer     */
				&dummy			/* address for time key last written to */
		 ) == ERROR_SUCCESS )
	{
		DeleteAllRegKeys( hkKey, szSubKeyName );
	}
	RegDeleteKey( HKEY_CURRENT_USER, pszName );
} /* #OF# DeleteAllRegKeys */

/*========================================================
Function : WriteRegDWORD
=========================================================*/
/* #FN#
   WriteRegDWORD writes out an int to the preset Registry key HKEY_CURRENT_USER\REGNAME.
   If the HKEY passed in is valid it is used, otherwise the key is grabbed and released
   within the function. Note that RegOpenKey is used here instead of RegCreateKey, which
   is only used at init time. No calls should be made to this prior to HandleRegistry().
   Any write to the Registry that doesn't work is skipped with user notification. */
void
/* #AS#
   Nothing */
WriteRegDWORD( HKEY  hkInput,
			   char *item,
			   DWORD value)
{
	HKEY hkKey = NULL;

	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_WRITE,				// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	else
		hkKey = hkInput;
	
	if( RegSetValueEx(
		hkKey,						// handle of key to set value for  
		item,						// address of value to set 
		0,							// reserved 
		REG_DWORD,					// flag for value type 
		(unsigned char *)&value,	// address of value data 
		sizeof( DWORD )				// data buffer size
		) != ERROR_SUCCESS )
	{
		DisplayMessage( NULL, IDS_REG_WRITE_ERROR, 0, MB_ICONSTOP | MB_OK );
	}	
	if( hkInput == NULL )
		RegCloseKey( hkKey );
} /* #OF# WriteRegDWORD */

/*========================================================
Function : ReadRegDWORD
=========================================================*/
/* #FN#
   ReadRegDWORD retrieves an existing value. To make it bulletproof the
   calling routine can request to display errors or not, depending on how
   fatal they are considered. */
int
/* #AS#
   The value was read */
ReadRegDWORD( HKEY   hkInput,
			  char  *item,
			  DWORD *data,
			  BOOL   bShowError )
{
	DWORD type, size, value;
	HKEY  hkKey = NULL;
	
	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_READ,				// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			if( bShowError )
			{
				DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
				return READ_REG_FAIL;
			}
		}
	}
	else
		hkKey = hkInput;
	
	// Win95 is really picky about having this size set; WinNT doesn't care. Go figure.
	size = sizeof( DWORD );
	if( RegQueryValueEx(
		hkKey,						// handle of key to query 
		item,						// address of name of value to query 
		0,							// reserved 
		&type,						// address of buffer for value type 
		(unsigned char *)&value,	// address of data buffer 
		&size			 			// address of data buffer size 
		) != ERROR_SUCCESS )
	{
		if( bShowError )
		{
			DisplayMessage( NULL, IDS_REG_LOAD_ERROR, 0, MB_ICONEXCLAMATION | MB_OK, item );
		}
		else
		{
			if( hkInput == NULL )
				RegCloseKey( hkKey );
		}
		if( hkInput == NULL )
			RegCloseKey( hkKey );
		return READ_REG_FAIL;
	}
	
	if( type != REG_DWORD || size != sizeof( DWORD ) )
	{
		DisplayMessage( NULL, IDS_REG_WRONG_SIZE, 0, MB_ICONSTOP | MB_OK );
		if( hkInput == NULL )
			RegCloseKey( hkKey );
		return READ_REG_FAIL;
	}

	if( hkInput == NULL )
		RegCloseKey( hkKey );
	
	*data = value;
	return TRUE;
} /* #OF# ReadRegDWORD */

/*========================================================
Function : ReadRegBinary
=========================================================*/
/* #FN#
   Retrieves an existing value */
DWORD
/* #AS#
   The value was read */
ReadRegBinary( HKEY  hkInput,
			   char *item,
			   char *buffer,
			   ULONG maxsize,
			   BOOL  bShowError )
{
	UINT  iFullSize = maxsize;
	DWORD type  = 0;
	HKEY  hkKey = NULL;
	
	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_READ,				// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			if( bShowError )
			{
				DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
				return READ_REG_FAIL;
			}
		}
	}
	else
		hkKey = hkInput;
	
	if( RegQueryValueEx(
		hkKey,						// handle of key to query 
		item,						// address of name of value to query 
		0,							// reserved 
		&type,						// address of buffer for value type 
		(unsigned char *)buffer,	// address of data buffer 
		&maxsize			 		// address of data buffer size 
		) != ERROR_SUCCESS )
	{
		if( bShowError )
		{
			DisplayMessage( NULL, IDS_REG_KEY_ERROR, 0, MB_ICONSTOP | MB_OK, item );
		}
		else
		{
			if( hkInput == NULL )
				RegCloseKey( hkKey );
			*buffer = 0;
		}
		if( hkInput == NULL )
			RegCloseKey( hkKey );
		return READ_REG_FAIL;
	}

	if( iFullSize > maxsize + 1 )
		buffer[ maxsize + 1 ] = 0;
	
	if( hkInput == NULL )
		RegCloseKey( hkKey );
	
	return type;
} /* #OF# ReadRegBinary */

/*========================================================
Function : WriteRegBinary
=========================================================*/
/* #FN#
   WriteRegBinary is similar to WriteRegDWORD except it dumps an arbitrary
   binary section of data */
void
/* #AS#
   Nothing */
WriteRegBinary( HKEY hkInput,
			    char *item,
				unsigned char *data,
				int size )
{
	HKEY hkKey = NULL;
	
	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,	// handle of open key 
			REGNAME,			// address of name of subkey to open 
			0,					// reserved 
			KEY_WRITE,			// security access mask 
			&hkKey 				// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	else
		hkKey = hkInput;
	
	if( RegSetValueEx(
		hkKey,					// handle of key to set value for  
		item,					// address of value to set 
		0,						// reserved 
		REG_BINARY,				// flag for value type 
		data,					// address of value data 
		size					// data buffer size
		) != ERROR_SUCCESS )
	{
		DisplayMessage( NULL, IDS_REG_WRITE_ERROR, 0, MB_ICONSTOP | MB_OK );
	}	
	if( hkInput == NULL )
		RegCloseKey( hkKey );
} /* #OF# WriteRegBinary */

/*========================================================
Function : WriteRegString
=========================================================*/
/* #FN#
   WriteRegString is similar to WriteRegBinary except it writes a null-terminated
   string */
void
/* #AS#
   Nothing */
WriteRegString(
	HKEY hkInput,
	char *item,
	char *data
)
{
	HKEY hkKey = NULL;
	
	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_WRITE,				// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	else
		hkKey = hkInput;
	
	if( RegSetValueEx(
		hkKey,						// handle of key to set value for  
		item,						// address of value to set 
		0,							// reserved 
		REG_SZ,						// flag for value type 
		(const unsigned char *)data,// address of value data 
		strlen( data )				// data buffer size
		) != ERROR_SUCCESS )
	{
		DisplayMessage( NULL, IDS_REG_WRITE_ERROR, 0, MB_ICONSTOP | MB_OK );
	}	
	if( hkInput == NULL )
		RegCloseKey( hkKey );
} /* #OF# WriteRegString */

int
ReadRegKeyset( HKEY hkInput,
			   int  nKeyset )
{
	ULONG ulKeysetReg1 = 0L;
	ULONG ulKeysetReg2 = 0L;
	ULONG ulKeysetReg3 = 0L;
	BOOL  bRegFail     = FALSE;
	HKEY  hkKey        = NULL;
	int   i;

	if( !hkInput )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_READ,				// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return FALSE;
		}
	}
	else
		hkKey = hkInput;

	if( KEYS_A_JOYSTICK == nKeyset )
	{
		if( ReadRegDWORD( hkKey, REG_KEYSET_A1, (DWORD *)&ulKeysetReg1, TRUE ) == READ_REG_FAIL )
		{
			ulKeysetReg1 = 0L;
			bRegFail = TRUE;
		}
		if( ReadRegDWORD( hkKey, REG_KEYSET_A2, (DWORD *)&ulKeysetReg2, TRUE ) == READ_REG_FAIL )
		{
			ulKeysetReg2 = 0L;
			bRegFail = TRUE;
		}
		if( ReadRegDWORD( hkKey, REG_KEYSET_A3, (DWORD *)&ulKeysetReg3, TRUE ) == READ_REG_FAIL )
		{
			ulKeysetReg3 = 0L;
			bRegFail = TRUE;
		}
	}
	else
	{
		if( ReadRegDWORD( hkKey, REG_KEYSET_B1, (DWORD *)&ulKeysetReg1, TRUE ) == READ_REG_FAIL )
		{
			ulKeysetReg1 = 0L;
			bRegFail = TRUE;
		}
		if( ReadRegDWORD( hkKey, REG_KEYSET_B2, (DWORD *)&ulKeysetReg2, TRUE ) == READ_REG_FAIL )
		{
			ulKeysetReg2 = 0L;
			bRegFail = TRUE;
		}
		if( ReadRegDWORD( hkKey, REG_KEYSET_B3, (DWORD *)&ulKeysetReg3, TRUE ) == READ_REG_FAIL )
		{
			ulKeysetReg3 = 0L;
			bRegFail = TRUE;
		}
	}
	_ASSERT(NUM_KBJOY_KEYS - 2 <= sizeof(ULONG) * 2);

	i = NUM_KBJOY_KEYS - 1;

	g_Input.Joy.anKeysets[ nKeyset + NUM_KBJOY_DEVICES ][ i-- ] = (WORD)(ulKeysetReg3 & 0x000000ffL);
	ulKeysetReg3 = ulKeysetReg3 >> 8;
	g_Input.Joy.anKeysets[ nKeyset + NUM_KBJOY_DEVICES ][ i-- ] = (WORD)(ulKeysetReg3 & 0x000000ffL);

	for( ; i > -1; )
	{
		g_Input.Joy.anKeysets[ nKeyset + NUM_KBJOY_DEVICES ][ i-- ] = (WORD)(ulKeysetReg2 & 0x000000ffL);
		ulKeysetReg2 = ulKeysetReg2 >> 8;
		g_Input.Joy.anKeysets[ nKeyset + NUM_KBJOY_DEVICES ][ i-- ] = (WORD)(ulKeysetReg1 & 0x000000ffL);
		ulKeysetReg1 = ulKeysetReg1 >> 8;
	}

	if( hkInput == NULL )
		RegCloseKey( hkKey );

	return (bRegFail ? FALSE : TRUE);
}

void
WriteRegKeyset(
	HKEY hkInput,
	int  nKeyset
)
{
	ULONG ulKeysetReg1 = 0L;
	ULONG ulKeysetReg2 = 0L;
	ULONG ulKeysetReg3 = 0L;
	HKEY  hkKey        = NULL;
	int   i;

	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_WRITE,			// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	else
		hkKey = hkInput;

	/* The key values will be packed into DWORDs */
	_ASSERT(NUM_KBJOY_KEYS - 2 <= sizeof(ULONG) * 2);

	for( i = 0; i < NUM_KBJOY_KEYS - 2; )
	{
		ulKeysetReg1 = ulKeysetReg1 << 8;
		ulKeysetReg1 |= g_Input.Joy.anKeysets[ nKeyset + NUM_KBJOY_DEVICES ][ i++ ] & 0x000000ffL;
		ulKeysetReg2 = ulKeysetReg2 << 8;
		ulKeysetReg2 |= g_Input.Joy.anKeysets[ nKeyset + NUM_KBJOY_DEVICES ][ i++ ] & 0x000000ffL;
	}
	ulKeysetReg3 |= g_Input.Joy.anKeysets[ nKeyset + NUM_KBJOY_DEVICES ][ i++ ] & 0x000000ffL;
	ulKeysetReg3 = ulKeysetReg3 << 8;
	ulKeysetReg3 |= g_Input.Joy.anKeysets[ nKeyset + NUM_KBJOY_DEVICES ][ i ] & 0x000000ffL;

	if( KEYS_A_JOYSTICK == nKeyset )
	{
		WriteRegDWORD( hkKey, REG_KEYSET_A1, ulKeysetReg1 );
		WriteRegDWORD( hkKey, REG_KEYSET_A2, ulKeysetReg2 );
		WriteRegDWORD( hkKey, REG_KEYSET_A3, ulKeysetReg3 );
	}
	else
	{
		WriteRegDWORD( hkKey, REG_KEYSET_B1, ulKeysetReg1 );
		WriteRegDWORD( hkKey, REG_KEYSET_B2, ulKeysetReg2 );
		WriteRegDWORD( hkKey, REG_KEYSET_B3, ulKeysetReg3 );
	}
}

void
WriteRegDrives( HKEY hkInput )
{
	HKEY hkKey = NULL;

	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_WRITE,				// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	else
		hkKey = hkInput;

	WriteRegString( hkKey, REG_HD1, atari_h1_dir );
	WriteRegString( hkKey, REG_HD2, atari_h2_dir );
	WriteRegString( hkKey, REG_HD3, atari_h3_dir );
	WriteRegString( hkKey, REG_HD4, atari_h4_dir );

	WriteRegString( hkKey, REG_DRIVE1, sio_filename[ 0 ] );
	WriteRegString( hkKey, REG_DRIVE2, sio_filename[ 1 ] );
	WriteRegString( hkKey, REG_DRIVE3, sio_filename[ 2 ] );
	WriteRegString( hkKey, REG_DRIVE4, sio_filename[ 3 ] );
	WriteRegString( hkKey, REG_DRIVE5, sio_filename[ 4 ] );
	WriteRegString( hkKey, REG_DRIVE6, sio_filename[ 5 ] );
	WriteRegString( hkKey, REG_DRIVE7, sio_filename[ 6 ] );
	WriteRegString( hkKey, REG_DRIVE8, sio_filename[ 7 ] );

	WriteRegString( hkKey, REG_EXE_PATH, atari_exe_dir );
	WriteRegString( hkKey, REG_STATE_PATH, atari_state_dir );

	if( hkInput == NULL )
		RegCloseKey( hkKey );
}

void
WriteAtari800Registry( HKEY hkInput )
{
	HKEY hkKey = NULL;

	if( hkInput == NULL )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,	// handle of open key 
			REGNAME,			// address of name of subkey to open 
			0,					// reserved 
			KEY_ALL_ACCESS,		// security access mask 
			&hkKey 				// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	else
		hkKey = hkInput;

	WriteRegDWORD ( hkKey, REG_REFRESH_RATE,      refresh_rate         );
	WriteRegDWORD ( hkKey, REG_DEFAULT_SYSTEM,    default_system       );
	WriteRegDWORD ( hkKey, REG_DEFAULT_TV_MODE,   default_tv_mode      );
	WriteRegDWORD ( hkKey, REG_HD_READ_ONLY,      hd_read_only         );
	WriteRegDWORD ( hkKey, REG_HOLD_OPTION,       hold_option          );
	WriteRegDWORD ( hkKey, REG_ENABLE_SIO_PATCH,  enable_sio_patch     );
	WriteRegDWORD ( hkKey, REG_ENABLE_C000_RAM,   enable_c000_ram      );
	WriteRegDWORD ( hkKey, REG_ENABLE_ROM_PATCH,  enable_rom_patches   );
	WriteRegDWORD ( hkKey, REG_ENABLE_STEREO,     stereo_enabled       );
	WriteRegDWORD ( hkKey, REG_CART_TYPE,         cart_type            );
	WriteRegDWORD ( hkKey, REG_ARTIF_MODE,        global_artif_mode    );
	WriteRegString( hkKey, REG_OSA_ROM,           atari_osa_filename   );
	WriteRegString( hkKey, REG_OSB_ROM,           atari_osb_filename   );
	WriteRegString( hkKey, REG_XLXE_ROM,          atari_xlxe_filename  );
	WriteRegString( hkKey, REG_5200_ROM,          atari_5200_filename  );
	WriteRegString( hkKey, REG_BASIC_ROM,         atari_basic_filename );
	WriteRegString( hkKey, REG_PRINT_COMMAND,     print_command        );
	WriteRegString( hkKey, REG_CURRENT_ROM,       g_szCurrentRom       );
	WriteRegString( hkKey, REG_OTHER_ROM,         g_szOtherRom         );
	WriteRegString( hkKey, REG_KEY_TEMPLATE,      g_szTemplateFile     );
	WriteRegString( hkKey, REG_EXT_PALETTE,       g_szPaletteFile      );
	WriteRegDWORD ( hkKey, REG_MISC_STATES,       g_Misc.ulState       );
	WriteRegDWORD ( hkKey, REG_START_XPOS,        g_nStartX            );
	WriteRegDWORD ( hkKey, REG_START_YPOS,        g_nStartY            );
	WriteRegDWORD ( hkKey, REG_DDRAW_MODE,        g_Screen.ulMode      );
	WriteRegDWORD ( hkKey, REG_STRETCH_MODE,      g_Screen.nStretchMode      );
	WriteRegDWORD ( hkKey, REG_MEMORY_TYPE,       g_Screen.nMemoryType       );
	WriteRegDWORD ( hkKey, REG_COLOR_BLACK,       g_Screen.Pal.nBlackLevel   );
	WriteRegDWORD ( hkKey, REG_COLOR_WHITE,       g_Screen.Pal.nWhiteLevel   );
	WriteRegDWORD ( hkKey, REG_COLOR_SATURATION,  g_Screen.Pal.nSaturation   );
	WriteRegDWORD ( hkKey, REG_COLOR_SHIFT,       g_Screen.Pal.nColorShift   );
	WriteRegDWORD ( hkKey, REG_USE_VOLUME_ONLY,   g_Sound.nVolumeOnly        );
	WriteRegDWORD ( hkKey, REG_SOUND_STATE,       g_Sound.ulMode             );
	WriteRegDWORD ( hkKey, REG_SOUND_RATE,        g_Sound.nRate              );
	WriteRegDWORD ( hkKey, REG_SOUND_VOLUME,      g_Sound.nVolume            );
	WriteRegDWORD ( hkKey, REG_SOUND_UPDATE,      g_Sound.nSkipUpdate        );
	WriteRegDWORD ( hkKey, REG_JOYSTICKS,         g_Input.Joy.ulSelected     );
	WriteRegDWORD ( hkKey, REG_AUTOFIRE_MODE,     g_Input.Joy.nAutoMode      );
	WriteRegDWORD ( hkKey, REG_AUTOFIRE_STICKS,   g_Input.Joy.ulAutoSticks   );
	WriteRegDWORD ( hkKey, REG_MOUSE_MODE,        g_Input.Mouse.nMode        );
	WriteRegDWORD ( hkKey, REG_MOUSE_PORT,        g_Input.Mouse.nPort        );
	WriteRegDWORD ( hkKey, REG_MOUSE_SPEED,       g_Input.Mouse.nMouseSpeed  );
	WriteRegDWORD ( hkKey, REG_PADDLE_RANGE,      g_Input.Mouse.nPaddleRange );
	WriteRegDWORD ( hkKey, REG_JOY_INERTIA,       g_Input.Mouse.nJoyInertia  );
	WriteRegDWORD ( hkKey, REG_PEN_XOFFSET,       g_Input.Mouse.nPenOffsetX  );
	WriteRegDWORD ( hkKey, REG_PEN_YOFFSET,       g_Input.Mouse.nPenOffsetY  );
	WriteRegDWORD ( hkKey, REG_ARROWS_MODE,       g_Input.Key.nArrowsMode    );
	WriteRegDWORD ( hkKey, REG_DONT_SHOW,         g_Misc.ulDontShow          );
	WriteRegDWORD ( hkKey, REG_FILE_ASSOCIATIONS, g_Misc.ulFileAssociations  );
	WriteRegDWORD ( hkKey, REG_DOUBLE_RATE,       g_nDoubleRate        );
	WriteRegDWORD ( hkKey, REG_SPEED_PERCENT,     g_nSpeedPercent      );
	WriteRegDWORD ( hkKey, REG_CURRENT_REV,       s_nVersion           );

	WriteRegDrives( hkKey );

	WriteRegKeyset( hkKey, KEYS_A_JOYSTICK );
	WriteRegKeyset( hkKey, KEYS_B_JOYSTICK );

	if( hkInput == NULL )
		RegCloseKey( hkKey );
}

void
InitialiseRegistry(
	HKEY hkInput,
	BOOL bErasePaths
)
{
	HKEY hkKey = NULL;
	int  i;

	if( !hkInput )
	{
		DWORD disposition = REG_OPENED_EXISTING_KEY;
		
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_ALL_ACCESS,			// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			if( RegCreateKeyEx(
				HKEY_CURRENT_USER,		// handle of an open key 
				REGNAME,				// address of subkey name 
				0,						// reserved 
				PLUS_REGKEY,			// address of class string 
				REG_OPTION_NON_VOLATILE,// special options flag 
				KEY_ALL_ACCESS,			// desired security access 
				NULL,					// address of key security structure 
				&hkKey,					// address of buffer for opened handle  
				&disposition 			// address of disposition value buffer 
				) != ERROR_SUCCESS )
			{
				DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
				/* Probably should make this fault in a more elegant manner */
				exit( 1 );
			}
		}
	}
	else
		hkKey = hkInput;

	refresh_rate       = 1;
	default_system     = 3;
	default_tv_mode    = 1;
	hd_read_only       = 1;
	hold_option        = 1;
	enable_c000_ram    = 0;
	enable_sio_patch   = 1;
	enable_rom_patches = 1;
	stereo_enabled     = 0;
	cart_type          = NO_CART;
	global_artif_mode  = 0;

	g_Misc.ulState             = DEF_MISC_STATES;
	g_Misc.ulDontShow          = DEF_DONT_SHOW_FLAGS;
	g_nDoubleRate              = DEF_DOUBLE_RATE;
	g_nSpeedPercent            = DEF_SPEED_PERCENT;

	g_Sound.ulMode             = DEF_SOUND_STATE;
	g_Sound.nVolumeOnly        = DEF_USE_VOLUME_ONLY;
	g_Sound.nRate              = DEF_SOUND_RATE;
	g_Sound.nVolume            = DEF_SOUND_VOL;
	g_Sound.nSkipUpdate        = DEF_SKIP_UPDATE;

	g_Input.Joy.ulSelected     = DEF_JOY_SELECTS;
	g_Input.Joy.nAutoMode      = DEF_AUTOFIRE_MODE;
	g_Input.Joy.ulAutoSticks   = DEF_AUTOFIRE_STICKS;
	g_Input.Mouse.nMode        = DEF_MOUSE_MODE;
	g_Input.Mouse.nPort        = DEF_MOUSE_PORT;
	g_Input.Mouse.nMouseSpeed  = DEF_MOUSE_SPEED;
	g_Input.Mouse.nPaddleRange = DEF_PADDLE_RANGE;
	g_Input.Mouse.nJoyInertia  = DEF_JOY_INERTIA;
	g_Input.Mouse.nPenOffsetX  = DEF_PEN_OFFSET_X;
	g_Input.Mouse.nPenOffsetY  = DEF_PEN_OFFSET_Y;
	g_Input.Key.nArrowsMode    = DEF_ARROWS_MODE;

	g_Screen.Pal.nBlackLevel   = DEF_CLR_BLACK_LEVEL;
	g_Screen.Pal.nWhiteLevel   = DEF_CLR_WHITE_LEVEL;
	g_Screen.Pal.nSaturation   = DEF_CLR_SATURATION;
	g_Screen.Pal.nColorShift   = DEF_CLR_SHIFT;
	g_Screen.ulMode            = DEF_SCREEN_MODE;
	g_Screen.nStretchMode      = DEF_STRETCH_MODE;
	g_Screen.nMemoryType       = DEF_MEMORY_TYPE;

	/* Clear keysets A & B */
	for( i = 0; i < NUM_KBJOY_KEYS; i++ )
	{
		g_Input.Joy.anKeysets[ KEYS_A_JOYSTICK + NUM_KBJOY_DEVICES ][ i ] = 0;
		g_Input.Joy.anKeysets[ KEYS_B_JOYSTICK + NUM_KBJOY_DEVICES ][ i ] = 0;
	}
	
	if( bErasePaths || *atari_osa_filename == '\0' ) /* WinNT wants this */
		strcpy( atari_osa_filename, DEFAULT_OSA );
	if( bErasePaths || *atari_osb_filename == '\0' )
		strcpy( atari_osb_filename, DEFAULT_OSB );
	if( bErasePaths || *atari_xlxe_filename == '\0' )
		strcpy( atari_xlxe_filename, DEFAULT_OXL );
	if( bErasePaths || *atari_5200_filename == '\0' )
		strcpy( atari_5200_filename, DEFAULT_O52 );
	if( bErasePaths || *atari_basic_filename == '\0' )
		strcpy( atari_basic_filename, DEFAULT_BAS );

	if( bErasePaths || *g_szTemplateFile == '\0' )
		strcpy( g_szTemplateFile, DEFAULT_A8K );
	if( bErasePaths || *g_szPaletteFile == '\0' )
		strcpy( g_szPaletteFile, DEFAULT_ACT );
	if( bErasePaths || *atari_state_dir == '\0' )
		strcpy( atari_state_dir, DEFAULT_A8S );

	if( bErasePaths || *atari_h1_dir == '\0' )
		strcpy( atari_h1_dir, DEFAULT_HDD );
	if( bErasePaths || *atari_h2_dir == '\0' )
		strcpy( atari_h2_dir, DEFAULT_HDD );
	if( bErasePaths || *atari_h3_dir == '\0' )
		strcpy( atari_h3_dir, DEFAULT_HDD );
	if( bErasePaths || *atari_h4_dir == '\0' )
		strcpy( atari_h4_dir, DEFAULT_HDD );

	if( bErasePaths || *atari_exe_dir == '\0' )
		strcpy( atari_exe_dir, "." );

	strcpy( g_szCurrentRom, "None" );
	strcpy( g_szOtherRom,   "None" );

	strcpy( print_command, "Notepad %s" );

	for( i = 0; i < MAX_DRIVES; i++ )
	{
		strcpy( sio_filename[ i ], "Off" ) ;
		drive_status[ i ] = Off;
	}

	WriteAtari800Registry( hkKey );

	if( hkInput == NULL )
		RegCloseKey( hkKey );
}

BOOL
ReadRegPaths( HKEY hkInput )
{
	HKEY	hkKey    = NULL;
	BOOL	bRegFail = FALSE;
	char	szDirPath[ MAX_PATH ];

	GetCurrentDirectory( MAX_PATH, szDirPath );
	if( !hkInput )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_READ,				// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return FALSE;
		}
	}
	else
		hkKey = hkInput;

	if( ReadRegBinary( hkInput, REG_OSA_ROM, atari_osa_filename, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_osa_filename, DEFAULT_OSA );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_OSB_ROM, atari_osb_filename, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_osb_filename, DEFAULT_OSB );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_XLXE_ROM, atari_xlxe_filename, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_xlxe_filename, DEFAULT_OXL );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_5200_ROM, atari_5200_filename, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_5200_filename, DEFAULT_O52 );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_BASIC_ROM, atari_basic_filename, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_basic_filename, DEFAULT_BAS );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_KEY_TEMPLATE, g_szTemplateFile, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( g_szTemplateFile, DEFAULT_A8K );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_EXT_PALETTE, g_szPaletteFile, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( g_szPaletteFile, DEFAULT_ACT );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_STATE_PATH, atari_state_dir, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_state_dir, DEFAULT_A8S );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_CURRENT_ROM, g_szCurrentRom, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( g_szCurrentRom, "None" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_OTHER_ROM, g_szOtherRom, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( g_szOtherRom, "None" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_HD1, atari_h1_dir, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_h1_dir, DEFAULT_HDD );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_HD2, atari_h2_dir, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_h2_dir, DEFAULT_HDD );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_HD3, atari_h3_dir, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_h3_dir, DEFAULT_HDD );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_HD4, atari_h4_dir, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_h4_dir, DEFAULT_HDD );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkInput, REG_EXE_PATH, atari_exe_dir, MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( atari_exe_dir, "." );
		bRegFail = TRUE;
	}
	if( hkInput == NULL )
		RegCloseKey( hkKey );

	return (bRegFail ? FALSE : TRUE);
}

void
ReadRegDrives( HKEY hkInput )
{
	BOOL bRegFail = FALSE;
	HKEY hkKey    = NULL;
	int  i;

	if( !hkInput )
	{
		if( RegOpenKeyEx(
			HKEY_CURRENT_USER,		// handle of open key 
			REGNAME,				// address of name of subkey to open 
			0,						// reserved 
			KEY_READ,				// security access mask 
			&hkKey 					// address of handle of open key 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			return;
		}
	}
	else
		hkKey = hkInput;

	if( ReadRegBinary( hkKey, REG_DRIVE1, sio_filename[ 0 ], MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( sio_filename[ 0 ], "Off" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkKey, REG_DRIVE2, sio_filename[ 1 ], MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( sio_filename[ 1 ], "Off" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkKey, REG_DRIVE3, sio_filename[ 2 ], MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( sio_filename[ 2 ], "Off" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkKey, REG_DRIVE4, sio_filename[ 3 ], MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( sio_filename[ 3 ], "Off" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkKey, REG_DRIVE5, sio_filename[ 4 ], MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( sio_filename[ 4 ], "Off" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkKey, REG_DRIVE6, sio_filename[ 5 ], MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( sio_filename[ 5 ], "Off" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkKey, REG_DRIVE7, sio_filename[ 6 ], MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( sio_filename[ 6 ], "Off" );
		bRegFail = TRUE;
	}
	if( ReadRegBinary( hkKey, REG_DRIVE8, sio_filename[ 7 ], MAX_PATH, FALSE ) == READ_REG_FAIL )
	{
		strcpy( sio_filename[ 7 ], "Off" );
		bRegFail = TRUE;
	}

	for( i = 0; i < MAX_DRIVES; i++ )
	{
		if( *sio_filename[ i ] == '\0' )
			strcpy( sio_filename[ i ], "Off" );

		if( strcmp( sio_filename[ i ], "Off" ) == 0 )
			drive_status[ i ] = Off;
		else
		if( strcmp( sio_filename[ i ], "Empty" ) == 0 )
			drive_status[ i ] = NoDisk;
	}

	if( hkInput == NULL )
		RegCloseKey( hkKey );

	if( bRegFail )
		WriteAtari800Registry( NULL );
}

/*========================================================
Function : HandleRegistry
=========================================================*/
/* #FN#
   Creates the Registry entries if they don't exist and read all
   the defaults in at runtime (this is called from MainFrame) */
BOOL
/* #AS#
   TRUE if the Registry has been initialized, otherwise FALSE */
HandleRegistry( void )
{
	DWORD disposition = REG_OPENED_EXISTING_KEY;
	BOOL  bRegFail    = FALSE;
	BOOL  bInitialReg = FALSE;
	HKEY  hkInitKey;

	if( RegOpenKeyEx(
		HKEY_CURRENT_USER,		// handle of open key 
		REGNAME,				// address of name of subkey to open 
		0,						// reserved 
		KEY_ALL_ACCESS,			// security access mask 
		&hkInitKey 				// address of handle of open key 
		) != ERROR_SUCCESS )
	{
		if( RegCreateKeyEx(
			HKEY_CURRENT_USER,		// handle of an open key 
			REGNAME,				// address of subkey name 
			0,						// reserved 
			PLUS_REGKEY,			// address of class string 
			REG_OPTION_NON_VOLATILE,// special options flag 
			KEY_ALL_ACCESS,			// desired security access 
			NULL,					// address of key security structure 
			&hkInitKey,				// address of buffer for opened handle  
			&disposition 			// address of disposition value buffer 
			) != ERROR_SUCCESS )
		{
			DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			// Probably should make this fault in a more elegant manner
			exit( 1 );
		}
	}
	
	// If the key doesn't exist, fill in defaults. This is the only time these
	// will be written all at once. From here on out we trust the Registry to hold them
	// (yeah, right) and update when key dialogs are used to configure them
	if( disposition == REG_CREATED_NEW_KEY )
	{
		bInitialReg = TRUE;
		InitialiseRegistry( hkInitKey, !Misc_TestRomPaths( NULL, NULL ) );
	}
	else
	{
		// Read in the values from the Registry. Only fail and return error when it
		// is REALLY fatal (you never know what somebody might do with their registry)
		// For most of these an error will result in the value being the default run-time
		if( ReadRegDWORD( hkInitKey, REG_CURRENT_REV, (DWORD *)&s_nVersion, FALSE ) == READ_REG_FAIL )
			s_nVersion = CURRENT_REV - 1;
		
		if( s_nVersion != CURRENT_REV )
		{
			DisplayMessage( NULL, IDS_WARN_OUTDATED, 0, MB_ICONINFORMATION | MB_OK );
			s_nVersion = CURRENT_REV;

			ReadRegPaths( hkInitKey );	/* Since we already have a registry, read the paths + filenames at least */
										/* Note that this will have to change if I ever invalidate the path system (unlikely) */
			DeleteAllRegKeys( hkInitKey, REGNAME );

			RegCloseKey( hkInitKey );
			if( RegCreateKeyEx(
				HKEY_CURRENT_USER,		// handle of an open key 
				REGNAME,				// address of subkey name 
				0,						// reserved 
				PLUS_REGKEY,			// address of class string 
				REG_OPTION_NON_VOLATILE,// special options flag 
				KEY_ALL_ACCESS,			// desired security access 
				NULL,					// address of key security structure 
				&hkInitKey,				// address of buffer for opened handle  
				&disposition 			// address of disposition value buffer 
				) != ERROR_SUCCESS )
			{
				DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
				// Probably should make this fault in a more elegant manner
				exit( 1 );
			}
			InitialiseRegistry( hkInitKey, FALSE );
			bInitialReg = TRUE;
		}
		else
		{
			if( ReadRegDWORD( hkInitKey, REG_REFRESH_RATE, (DWORD *)&refresh_rate, TRUE ) == READ_REG_FAIL )
			{
				refresh_rate = 1;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_DOUBLE_RATE, (DWORD *)&g_nDoubleRate, TRUE ) == READ_REG_FAIL )
			{
				g_nDoubleRate = DEF_DOUBLE_RATE;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_ARROWS_MODE, (DWORD *)&g_Input.Key.nArrowsMode, TRUE ) == READ_REG_FAIL )
			{
				g_Input.Key.nArrowsMode = DEF_ARROWS_MODE;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_HD_READ_ONLY, (DWORD *)&hd_read_only, TRUE ) == READ_REG_FAIL )
			{
				hd_read_only = 1;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_SPEED_PERCENT, (DWORD *)&g_nSpeedPercent, TRUE ) == READ_REG_FAIL )
			{
				g_nSpeedPercent = DEF_SPEED_PERCENT;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_MOUSE_PORT, (DWORD *)&g_Input.Mouse.nPort, TRUE ) == READ_REG_FAIL )
			{
				g_Input.Mouse.nPort = DEF_MOUSE_PORT;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_MOUSE_SPEED, (DWORD *)&g_Input.Mouse.nMouseSpeed, TRUE ) == READ_REG_FAIL )
			{
				g_Input.Mouse.nMouseSpeed = DEF_MOUSE_SPEED;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_PADDLE_RANGE, (DWORD *)&g_Input.Mouse.nPaddleRange, TRUE ) == READ_REG_FAIL )
			{
				g_Input.Mouse.nPaddleRange = DEF_PADDLE_RANGE;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_JOY_INERTIA, (DWORD *)&g_Input.Mouse.nJoyInertia, TRUE ) == READ_REG_FAIL )
			{
				g_Input.Mouse.nJoyInertia = DEF_JOY_INERTIA;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_PEN_XOFFSET, (DWORD *)&g_Input.Mouse.nPenOffsetX, TRUE ) == READ_REG_FAIL )
			{
				g_Input.Mouse.nPenOffsetX = DEF_PEN_OFFSET_X;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_PEN_YOFFSET, (DWORD *)&g_Input.Mouse.nPenOffsetY, TRUE ) == READ_REG_FAIL )
			{
				g_Input.Mouse.nPenOffsetY = DEF_PEN_OFFSET_Y;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_DEFAULT_SYSTEM, (DWORD *)&default_system, TRUE ) == READ_REG_FAIL )
			{
				default_system = 3;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_DEFAULT_TV_MODE, (DWORD *)&default_tv_mode, TRUE ) == READ_REG_FAIL )
			{
				default_tv_mode = 1;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_HOLD_OPTION, (DWORD *)&hold_option, TRUE ) == READ_REG_FAIL )
			{
				hold_option = 1;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_ENABLE_C000_RAM, (DWORD *)&enable_c000_ram, TRUE ) == READ_REG_FAIL )
			{
				enable_c000_ram = 0;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_ENABLE_SIO_PATCH, (DWORD *)&enable_sio_patch, TRUE ) == READ_REG_FAIL )
			{
				enable_sio_patch = 1;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_ENABLE_ROM_PATCH, (DWORD *)&enable_rom_patches, TRUE ) == READ_REG_FAIL )
			{
				enable_rom_patches = 1;
				bRegFail = TRUE;
			}
#ifndef NO_VOL_ONLY
			if( ReadRegDWORD( hkInitKey, REG_USE_VOLUME_ONLY, (DWORD *)&g_Sound.nVolumeOnly, TRUE ) == READ_REG_FAIL )
			{
#endif /*NO_VOL_ONLY*/
				g_Sound.nVolumeOnly = DEF_USE_VOLUME_ONLY;
#ifndef NO_VOL_ONLY
				bRegFail = TRUE;
			}
#endif /*NO_VOL_ONLY*/
#ifdef STEREO
			if( ReadRegDWORD( hkInitKey, REG_ENABLE_STEREO, (DWORD *)&stereo_enabled, TRUE ) == READ_REG_FAIL )
			{
#endif /*STEREO*/
				stereo_enabled = 0;
#ifdef STEREO
				bRegFail = TRUE;
			}
#endif /*STEREO*/
			if( ReadRegDWORD( hkInitKey, REG_ARTIF_MODE, (DWORD *)&global_artif_mode, TRUE ) == READ_REG_FAIL )
			{
				global_artif_mode = 0;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_STRETCH_MODE, (DWORD *)&g_Screen.nStretchMode, TRUE ) == READ_REG_FAIL )
			{
				g_Screen.nStretchMode = DEF_STRETCH_MODE;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_MEMORY_TYPE, (DWORD *)&g_Screen.nMemoryType, TRUE ) == READ_REG_FAIL )
			{
				g_Screen.nMemoryType = DEF_MEMORY_TYPE;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_DDRAW_MODE, (DWORD *)&g_Screen.ulMode, TRUE ) == READ_REG_FAIL )
			{
				g_Screen.ulMode = DEF_SCREEN_MODE;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_FILE_ASSOCIATIONS, (DWORD *)&g_Misc.ulFileAssociations, TRUE ) == READ_REG_FAIL )
			{
				g_Misc.ulFileAssociations = DEF_FILE_ASSOCIATIONS;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_START_XPOS, (DWORD *)&g_nStartX, TRUE ) == READ_REG_FAIL )
			{
				g_nStartX = DEF_START_X;
				bRegFail = TRUE;
			}

			if( g_nStartX > GetSystemMetrics( SM_CXFULLSCREEN ) )
				g_nStartX = GetSystemMetrics( SM_CXFULLSCREEN ) - (g_Screen.ulMode & SM_WRES_DOUBLE ? ATARI_DOUBLE_VIS_WIDTH : ATARI_VIS_WIDTH);
			if( g_nStartX < 0 )
				g_nStartX = 0;
			
			if( ReadRegDWORD( hkInitKey, REG_START_YPOS, (DWORD *)&g_nStartY, TRUE ) == READ_REG_FAIL )
			{
				g_nStartY = DEF_START_Y;
				bRegFail = TRUE;
			}

			if( g_nStartY > GetSystemMetrics( SM_CYFULLSCREEN ) )
				g_nStartY = GetSystemMetrics( SM_CYFULLSCREEN ) - (g_Screen.ulMode & SM_WRES_DOUBLE ? ATARI_DOUBLE_HEIGHT : ATARI_HEIGHT);
			if( g_nStartY < 0 )
				g_nStartY = 0;

			if( ReadRegDWORD( hkInitKey, REG_SOUND_RATE, (DWORD *)&g_Sound.nRate, TRUE ) == READ_REG_FAIL )
			{
				g_Sound.nRate = DEF_SOUND_RATE;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_SOUND_STATE, (DWORD *)&g_Sound.ulMode, TRUE ) == READ_REG_FAIL )
			{
				g_Sound.ulMode = SOUND_MMSOUND;
				if( g_Sound.nRate != 22050 )
					g_Sound.ulMode |= SOUND_CUSTOM_RATE;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_SOUND_VOLUME, (DWORD *)&g_Sound.nVolume, TRUE ) == READ_REG_FAIL )
			{
				g_Sound.nVolume = DEF_SOUND_VOL;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_SOUND_UPDATE, (DWORD *)&g_Sound.nSkipUpdate, TRUE ) == READ_REG_FAIL )
			{
				g_Sound.nSkipUpdate = DEF_SKIP_UPDATE;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_MISC_STATES, (DWORD *)&g_Misc.ulState, TRUE ) == READ_REG_FAIL )
			{
				g_Misc.ulState = DEF_MISC_STATES;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_DONT_SHOW, (DWORD *)&g_Misc.ulDontShow, TRUE ) == READ_REG_FAIL )
			{
				g_Misc.ulDontShow = DEF_DONT_SHOW_FLAGS;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_AUTOFIRE_MODE, (DWORD *)&g_Input.Joy.nAutoMode, TRUE ) == READ_REG_FAIL )
			{
				g_Input.Joy.nAutoMode = DEF_AUTOFIRE_MODE;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_AUTOFIRE_STICKS, (DWORD *)&g_Input.Joy.ulAutoSticks, TRUE ) == READ_REG_FAIL )
			{
				g_Input.Joy.ulAutoSticks = DEF_AUTOFIRE_STICKS;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_JOYSTICKS, (DWORD *)&g_Input.Joy.ulSelected, TRUE ) == READ_REG_FAIL )
			{
				g_Input.Joy.ulSelected = DEF_JOY_SELECTS;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_MOUSE_MODE, (DWORD *)&g_Input.Mouse.nMode, TRUE ) == READ_REG_FAIL )
			{
				g_Input.Mouse.nMode = DEF_MOUSE_MODE;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_COLOR_BLACK, (DWORD *)&g_Screen.Pal.nBlackLevel, TRUE ) == READ_REG_FAIL )
			{
				g_Screen.Pal.nBlackLevel = DEF_CLR_BLACK_LEVEL;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_COLOR_WHITE, (DWORD *)&g_Screen.Pal.nWhiteLevel, TRUE ) == READ_REG_FAIL )
			{
				g_Screen.Pal.nWhiteLevel = DEF_CLR_WHITE_LEVEL;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_COLOR_SATURATION, (DWORD *)&g_Screen.Pal.nSaturation, TRUE ) == READ_REG_FAIL )
			{
				g_Screen.Pal.nSaturation = DEF_CLR_SATURATION;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_COLOR_SHIFT, (DWORD *)&g_Screen.Pal.nColorShift, TRUE ) == READ_REG_FAIL )
			{
				g_Screen.Pal.nColorShift = DEF_CLR_SHIFT;
				bRegFail = TRUE;
			}
			if( ReadRegDWORD( hkInitKey, REG_CART_TYPE, (DWORD *)&cart_type, TRUE ) == READ_REG_FAIL )
			{
				cart_type = NO_CART;
				bRegFail = TRUE;
			}
			if( ReadRegBinary( hkInitKey, REG_PRINT_COMMAND, print_command, 256, TRUE ) == READ_REG_FAIL )
			{
				strcpy( print_command, "Notepad %s" );
				bRegFail = TRUE;
			}

			if( !ReadRegPaths( NULL ) )
				bRegFail = TRUE;

			if( !ReadRegKeyset( hkInitKey, KEYS_A_JOYSTICK ) )
				bRegFail = TRUE;
			if( !ReadRegKeyset( hkInitKey, KEYS_B_JOYSTICK ) )
				bRegFail = TRUE;
		}
	}
	RegCloseKey( hkInitKey );

	if( bRegFail )
		WriteAtari800Registry( NULL );

	return bInitialReg;
} /* #OF# HandleRegistry */

/*========================================================
Function : GetRegKeyHandle
=========================================================*/
/* #FN#
   Opens/Creates registry key pszKeyName in HKEY_CLASSES_ROOT section */
HKEY
/* #AS#
   Opened/Created registry key handle */
GetRegKeyHandle(
	LPCSTR pszKeyName, /* #IN# Name of registry key */
	BOOL   bCreateKey  /* #IN# Create the specified key if the key does not exist in the registry */ )
{
	DWORD disposition = REG_OPENED_EXISTING_KEY;
	BOOL  bRegFail    = TRUE;
	HKEY  hkKey       = NULL;

	if( RegOpenKeyEx(
			HKEY_CLASSES_ROOT,		// handle of open key 
			pszKeyName,				// address of name of subkey to open 
			0,						// reserved 
			KEY_ALL_ACCESS,			// security access mask 
			&hkKey					// address of handle of open key 
		) != ERROR_SUCCESS )
	{
		if( bCreateKey )
			if( RegCreateKeyEx(
					HKEY_CLASSES_ROOT,			// handle of an open key 
					pszKeyName,					// address of subkey name 
					0,							// reserved 
					PLUS_REGKEY,				// address of class string 
					REG_OPTION_NON_VOLATILE,	// special options flag 
					KEY_ALL_ACCESS,				// desired security access 
					NULL,						// address of key security structure 
					&hkKey,						// address of buffer for opened handle  
					&disposition 				// address of disposition value buffer 
				) != ERROR_SUCCESS )
			{
				DisplayMessage( NULL, IDS_REG_OPEN_ERROR, 0, MB_ICONSTOP | MB_OK );
			}
			else
				bRegFail = FALSE;
	}
	else
		bRegFail = FALSE;

	return (bRegFail ? NULL : hkKey);
} /* #OF# GetRegKeyHandle */
