/****************************************************************************
File    : misc_win.c
/*
@(#) #SY# Atari800Win PLus
@(#) #IS# Miscellanous stuff implementation for Win32 platforms
@(#) #BY# Richard Lawrence, Tomasz Szymankowski
@(#) #LM# 24.06.2001
*/

#include <windows.h>
#include <shellapi.h>
#include <stdlib.h>
#include <crtdbg.h>
#include <stdio.h>
#include <io.h>
#include <fcntl.h>
#include <mmsystem.h>
#include <excpt.h>
#include "WinConfig.h"
#include "Resource.h"
#include "atari800.h"
#include "globals.h"
#include "timing.h"
#include "macros.h"
#include "crc.h"
#include "zlib.h"
#include "registry.h"
#include "display_win.h"
#include "sound_win.h"
#include "input_win.h"
#include "misc_win.h"

/* Constants definition */

#define FCT_CHECK_16STD		0x01
#define FCT_CHECK_16AGS		0x02
#define FCT_CHECK_16OSS		0x04
#define FCT_CHECK_16ROM		(FCT_CHECK_16STD | FCT_CHECK_16AGS | FCT_CHECK_16OSS)
#define FCT_CHECK_32AGS		0x08
#define FCT_CHECK_32DB		0x10
#define FCT_CHECK_32ROM		(FCT_CHECK_32AGS | FCT_CHECK_32DB)

#define	EC_BUFFER_LEN		32
#define EC_MAX_LINE_LEN		2048

/* Public objects */

struct MiscCtrl_t g_Misc =
{
	DEF_MISC_STATES,
	DEF_FILE_ASSOCIATIONS,
	DEF_DONT_SHOW_FLAGS,
	0
};

FARPROC	pgzread  = NULL;
FARPROC	pgzopen  = NULL;
FARPROC	pgzclose = NULL;
FARPROC	pgzwrite = NULL;
FARPROC	pgzerror = NULL;

struct RomTypeInfo_t g_aRomTypeInfo[] =
{
	{ RTI_OSA, 2370568491L, "Rev.A  \t| 4/800\t| PAL" },
	{ RTI_OSB, 3252116993L, "Rev.B  \t| 4/800\t| PAL" },
	{ RTI_OSB, 4051249634L, "Rev.B  \t| 4/800\t| NTSC" },
	{ RTI_XLE, 2613326695L, "Rev.1  \t| 600XL\t| 03/11/1983" },
	{ RTI_XLE, 3764596111L, "Rev.2  \t| XL/XE\t| 05/10/1983" },
	{ RTI_XLE, 3591293960L, "Rev.3  \t| 800XE\t| 03/01/1985" },
	{ RTI_XLE, 3780165629L, "Rev.4  \t| XEGS \t| 05/07/1987" },
	{ RTI_XLE, 257804588L,  "Arabic \t| 65XE \t| 07/21/1987" },
	{ RTI_XLE, 2766230028L, "Rev.3  \t| ARGS \t| ---" },
	{ RTI_XLE, 1577346053L, "Rev.2.3\t| QMEG \t| ---" },
	{ RTI_XLE, 1932228838L, "Rev.3.2\t| QMEG \t| ---" },
	{ RTI_XLE, 2265792253L, "Rev.3.8\t| QMEG \t| ---" },
	{ RTI_XLE, 2603811756L, "Rev.4.2\t| QMEG \t| ---" },
	{ RTI_XLE, 1105050917L, "Rev.4.3\t| QMEG \t| ---" },
	{ RTI_XLE, 2258206520L, "---    \t| TOMS \t| ---" },
	{ RTI_A52, 3182898204L, "---    \t| 5200 \t| ---" },
//	{ RTI_A52, 2155009960L, "FoxOS  \t| 5200 \t| ---" },
	{ RTI_BAS, 3021189661L, "Rev.A  \t| ---  \t| ---" },
	{ RTI_BAS, 266326092L,  "Rev.B  \t| ---  \t| ---" },
	{ RTI_BAS, 2190982779L, "Rev.C  \t| ---  \t| ---" }
};

const int g_nRomTypeInfoNo = sizeof(g_aRomTypeInfo)/sizeof(g_aRomTypeInfo[0]);

/* Private objects */

static BOOL   s_bCheckZlib   = TRUE;
static HANDLE s_hZlib        = NULL;
static UINT   s_unCartCheck  = FCT_CHECK_16ROM;
static BOOL   s_bBreakSearch = FALSE;


/*========================================================
Function : Misc_GetSystemInfo
=========================================================*/
/* #FN#
   Gets information about the system an emulator is run on */
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
Misc_GetSystemInfo(
	UINT *pSystemInfo
)
{
	OSVERSIONINFO osvi;
	*pSystemInfo = 0;

	/* Determine the Windows OS version */
	ZeroMemory( &osvi, sizeof(OSVERSIONINFO) );
	osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);

	if( !GetVersionEx( (OSVERSIONINFO *)&osvi ) )
		return FALSE;

	switch( osvi.dwPlatformId )
	{
		case VER_PLATFORM_WIN32_NT:
		{
			/* Test for the product */
			if( osvi.dwMajorVersion == 4 )
				*pSystemInfo = SYS_WIN_NT4;

			if( osvi.dwMajorVersion == 5 )
				*pSystemInfo = SYS_WIN_NT5;

			break;
		}

		case VER_PLATFORM_WIN32_WINDOWS:
		{
			if( osvi.dwMajorVersion  > 4 || 
			   (osvi.dwMajorVersion == 4 && osvi.dwMinorVersion > 0) )
			{
				*pSystemInfo = SYS_WIN_98;
			}
			else
				*pSystemInfo = SYS_WIN_95;

			break;
		}
	}
	/* Check if an enhanced processor instruction set is available */
	__try
	{
		int nFeatureFlags = 0;

		/* NOTE: The following sample code contains a Pentium-specific instruction.
		   It should only be run on Pentium processors. The program generates
		   unhandled exception if running on non-Pentium family processors. */
		__asm
		{
			mov eax, 1
			_emit 0x0f		; CPUID (00001111 10100010) - This is a Pentium specific
			_emit 0xa2		; instruction which gets information on the processor.
			mov nFeatureFlags, edx
		}
		/* Is IA MMX technology bit (bit 23 of edx) in feature flags set? */
		if( nFeatureFlags & 0x800000 )
			*pSystemInfo |= SYS_PRC_MMX;
	}
	__except(EXCEPTION_EXECUTE_HANDLER)
	{
		/* Do nothing */
	}
	return TRUE; 
} /* #OF# Misc_GetSystemInfo */

/*========================================================
Function : ReadDisabledROMs
=========================================================*/
/* #FN#
   Reads ROM files from disk instead from state image */
int
/* #AS#
   TRUE if succeeded, otherwise FALSE */
ReadDisabledROMs( void )
{
	int	nRomFile;

	nRomFile = _open( atari_basic_filename, O_BINARY | O_RDONLY, 0777 );
	if( nRomFile == -1 )
	{
		Aprint( "Could not open %s for reading.", atari_basic_filename );
		return FALSE;
	}

	if( _read( nRomFile, atari_basic, 8192 ) < 8192 )
	{
		Aprint( "Could not read all of atari basic from %s.", atari_basic_filename );
		return FALSE;
	}
	_close( nRomFile );

	nRomFile = _open( atari_xlxe_filename, O_BINARY | O_RDONLY, 0777 );
	if( nRomFile == -1 )
	{
		Aprint( "Could not open %s for reading.", atari_xlxe_filename );
		return FALSE;
	}

	if( _read( nRomFile, atari_xlxe_filename, 16384 ) < 16384 )
	{
		Aprint( "Could not read entire atari ROM from %s.", atari_xlxe_filename );
		return FALSE;
	}
	_close( nRomFile );

	return TRUE;
} /* #OF# ReadDisabledROMs */

/*========================================================
Function : prepend_tmpfile_path
=========================================================*/
/* #FN#
   Inserts path to a temp dir to supplied buffer */
int
/* #AS#
   A length of the path */
prepend_tmpfile_path(
	char *pszBuffer
)
{
	int	nOffset = 0;

	nOffset += GetTempPath( MAX_PATH, pszBuffer ) - 1;
	if( nOffset == -1 || pszBuffer[ nOffset ] != '\\' )
		nOffset++;

	return nOffset;
} /* #OF# prepend_tmpfile_path */

/*========================================================
Function : zlib_capable
=========================================================*/
/* #FN#
   Checks if the ZLIB library is available */
int
/* #AS#
   1 if ZLIB is available, otherwise -1 */
zlib_capable( void )
{
	if( !s_hZlib )
	{
		Aprint( "Cannot gzip/ungzip without zlib.dll loaded properly." );
		return -1;
	}
	return 1;
} /* #OF# zlib_capable */

/*========================================================
Function : Atari_Initialise
=========================================================*/
/* #FN#
   Prepares a Windows stuff for emulation; this function is invoked
   by Atari800 kernel */
void
/* #AS#
   Nothing */
Atari_Initialise(
	int  *argc,
	char *argv[]
)
{
	if( s_bCheckZlib )
	{
		s_bCheckZlib = FALSE;
		if( !s_hZlib )
		{
			s_hZlib = LoadLibrary( "ZLIB.DLL" );
			if( !s_hZlib )
				DisplayMessage( NULL, IDS_ERROR_ZLIB_LOAD, 0, MB_ICONEXCLAMATION | MB_OK );
			else
			{
				pgzread  = GetProcAddress( s_hZlib, "gzread" );
				pgzopen  = GetProcAddress( s_hZlib, "gzopen" );
				pgzclose = GetProcAddress( s_hZlib, "gzclose" );
				pgzwrite = GetProcAddress( s_hZlib, "gzwrite" );
				pgzerror = GetProcAddress( s_hZlib, "gzerror" );
				if( !pgzread || !pgzopen || !pgzclose || !pgzwrite || !pgzerror )
				{
					FreeLibrary( s_hZlib );
					s_hZlib = NULL;
					DisplayMessage( NULL, IDS_ERROR_ZLIB_USE, 0, MB_ICONEXCLAMATION | MB_OK );
				}
			}
		}
	}
	/* Check what DirectX modes are available */
	if( !g_Screen.ulModesAvail )
	{
		Screen_CheckDDrawModes();
	}
	/* Strange code here, will be removed after merging with kernel 1.1 */
	if( !g_Screen.lpbmi )
	{
		int	nRGB, i;
		/* Only initialize this the first time it's allocated, since we'll come through here
		   on changing hardware types and it will already be set up correctly at that point */
		g_Screen.lpbmi = (LPBITMAPINFO)calloc( 1, sizeof(BITMAPINFOHEADER) + sizeof(RGBQUAD) * PAL_ENTRIES_NO );

		if( !g_Screen.lpbmi )
			Atari_Exit( 1 );

		g_Screen.lpbmi->bmiHeader.biSize        = sizeof(BITMAPINFOHEADER);
		g_Screen.lpbmi->bmiHeader.biWidth       =  ATARI_VIS_WIDTH;
		g_Screen.lpbmi->bmiHeader.biHeight      = -ATARI_HEIGHT;	/* Negative because we are a top-down bitmap */
		g_Screen.lpbmi->bmiHeader.biPlanes      = 1;
		g_Screen.lpbmi->bmiHeader.biBitCount    = 8;				/* Each byte stands for a color value */
		g_Screen.lpbmi->bmiHeader.biCompression = BI_RGB;			/* Uncompressed format */
		g_Screen.lpbmi->bmiHeader.biSizeImage   = ATARI_VIS_WIDTH * ATARI_HEIGHT - (sizeof(BITMAPINFOHEADER) + sizeof(RGBQUAD) * PAL_ENTRIES_NO);
		g_Screen.lpbmi->bmiHeader.biClrUsed     = PAL_ENTRIES_NO;

		for( i = 0; i < PAL_ENTRIES_NO; i++ )
		{
			/* Get color value from kernel table */
			nRGB = colortable[ i ];
			
			g_Screen.lpbmi->bmiColors[ i ].rgbRed      = g_Screen.Pal.pe[ i ].peRed   = (nRGB & 0x00ff0000) >> 16;
			g_Screen.lpbmi->bmiColors[ i ].rgbGreen    = g_Screen.Pal.pe[ i ].peGreen = (nRGB & 0x0000ff00) >> 8;
			g_Screen.lpbmi->bmiColors[ i ].rgbBlue     = g_Screen.Pal.pe[ i ].peBlue  =  nRGB & 0x000000ff;
			g_Screen.lpbmi->bmiColors[ i ].rgbReserved = g_Screen.Pal.pe[ i ].peFlags =  0;
		}
		Screen_PrepareInterp( FALSE );
	}
	/* Clean up the Atari timer stuff */
	Timer_Reset();
	/* Clean up the input stuff */
	Input_Reset();
} /* #OF# Atari_Initialise */

/*========================================================
Function : Misc_TogglePause
=========================================================*/
/* #FN#
   Toggles between the pause on and off */
void
/* #AS#
   Nothing */
Misc_TogglePause( void )
{
	if( g_ulAtariState & ATARI_RUNNING )
	{
		if( g_ulAtariState & ATARI_PAUSED )
		{
			g_ulAtariState &= ~ATARI_PAUSED;
			Sound_Restart();

			/* Release the redrawing stuff if necessary */
			Screen_FreeRedraw();

			if( ST_DOUBLE_BUFFERS )
				Screen_Clear( FALSE, FALSE );
 		}
		else
		{
			g_ulAtariState |= ATARI_PAUSED;
			Sound_Clear( FALSE );

			if( g_hMainWnd )
				SetTimer( g_hMainWnd, TIMER_READ_JOYSTICK, 100, NULL ); 

			/* Redraw paused screen if necessary; snapshot screen
			   for redrawing */
			Screen_DrawPaused( TRUE, FALSE, TRUE, TRUE );

			Screen_ShowMousePointer( TRUE );
		}
	}
} /* #OF# Misc_TogglePause */

/*========================================================
Function : Misc_ToggleFullSpeed
=========================================================*/
/* #FN#
   Toggles between the full speed mode on and off */
void
/* #AS#
   Nothing */
Misc_ToggleFullSpeed( void )
{
	if( g_ulAtariState & ATARI_RUNNING )
	{
		if( g_Misc.ulState & MS_FULL_SPEED )
		{
			g_Misc.ulState &= ~MS_FULL_SPEED;
			Sound_Restart();
		}
		else
		{
			g_Misc.ulState |= MS_FULL_SPEED;
			Sound_Clear( FALSE );
		}
		WriteRegDWORD( NULL, REG_MISC_STATES, g_Misc.ulState );
	}
} /* #OF# Misc_ToggleFullSpeed */

/*========================================================
Function : Misc_ToggleSIOPatch
=========================================================*/
/* #FN#
   Toggles between the SIO patch mode on and off */
void
/* #AS#
   Nothing */
Misc_ToggleSIOPatch( void )
{
	if( enable_rom_patches )
	{
		if( enable_sio_patch )
		{
			enable_sio_patch = 0;
			RestoreSIO();
		}
		else
		{
			enable_sio_patch = 1;
			SetSIOEsc();
		}
		WriteRegDWORD( NULL, REG_ENABLE_SIO_PATCH, enable_sio_patch );
	}
} /* #OF# Misc_ToggleSIOPatch */

/*========================================================
Function : MonitorThreadProc
=========================================================*/
/* #FN#
   A separate thread procedure for a monitor console */
static
DWORD WINAPI
/* #AS#
   Thread return code (success or failure) */
MonitorThreadProc(
	LPVOID lpParameter
)
{
	ExitThread( monitor() );
	return 0;
} /* #OF# MonitorThreadProc */

/*========================================================
Function : Misc_LaunchMonitor
=========================================================*/
/* #FN#
   Launches the emulator monitor console */
int
/* #AS#
   An integer value: 1 to go back to emulation, 0 to exit */
Misc_LaunchMonitor( void )
{
	DWORD  dwThreadId   = 0; 
	DWORD  dwExitCode   = 0;
	HANDLE hThread      = NULL;
	ULONG  ulScreenMode = 0L;

	/* CAtari800WinView::OnKillFocus will invoke that */
//	Sound_Clear( FALSE );

	/* Unfortunately, there are some problems with automatically return to
	   (flipped) full-screen mode, go to windowed and then restore instead */
	if( g_Screen.ulMode & SM_MODE_FULL )
	{
		ulScreenMode = g_Screen.ulMode;

		g_Screen.ulMode &= ~SM_MODE_MASK;
		/* Set a windowed mode */
		g_Screen.ulMode |= SM_MODE_WIND;

		if( !Screen_ChangeMode( TRUE ) )
		{
			ulScreenMode = 0L;
		}
	}
	/* Disable the main window */
	g_ulAtariState |= ATARI_MONITOR;
	EnableWindow( g_hMainWnd, FALSE );

	/* Launch an emulated Atari monitor */
	if( (hThread = CreateThread( NULL,
								 0,
								 MonitorThreadProc,
								 (LPVOID)NULL,
								 0,
								 &dwThreadId )) )
	{
		/* This message loop will solve some problems with
           the main window repainting */
		do
		{
			MSG msg;
			if( PeekMessage( &msg, 0, 0, 0, PM_REMOVE ) )
			{
				TranslateMessage( &msg );
				DispatchMessage( &msg );
			}
			Sleep( 1 ); /* Wait a moment, please */
			GetExitCodeThread( hThread, &dwExitCode );
		}
		while( STILL_ACTIVE == dwExitCode );

		CloseHandle( hThread );
	}
	else
		/* We had few luck to create an additional thread */
		dwExitCode = monitor();

	/* Enable the main window */
	g_ulAtariState &= ~ATARI_MONITOR;
	EnableWindow( g_hMainWnd, TRUE );

	if( dwExitCode )
	{
		HWND hPopupWnd = GetLastActivePopup( g_hMainWnd );

		BringWindowToTop( g_hMainWnd );
		if( IsIconic( hPopupWnd ) )
		{
			ShowWindow( hPopupWnd, SW_RESTORE );
		}
		else
			SetForegroundWindow( hPopupWnd );

		if( ulScreenMode )
		{
			g_Screen.ulMode = ulScreenMode;

			Screen_ChangeMode( TRUE );
		}
		return 1;	/* Go back to the emulation */
	}
	else
		/* Start a quit (this will end up in Atari_Exit(0)) */
		PostMessage( g_hMainWnd, WM_CLOSE, 0, 0L );

	return 0;
} /* #OF# Misc_LaunchMonitor */

/*========================================================
Function : Atari_Exit
=========================================================*/
/* #FN#
   This function is called by Atari800 kernel when emulation is
   about exit */
int
/* #AS#
   Nonzero to go back to emulation, 0 to exit */
Atari_Exit(
	int nPanic
)
{
	if( nPanic && g_ulAtariState & ATARI_RUNNING )
		g_ulAtariState |= ATARI_CRASHED;
	
	if( ATARI_CRASHED == g_ulAtariState )
	{
		wsync_halt = 1;	/* turn off CPU */
		return 0;
	}

	if( nPanic && g_ulAtariState & ATARI_RUNNING )
	{
		BOOL bCont = FALSE;

		Screen_DrawFrozen( FALSE, TRUE, TRUE, TRUE );

		if( IDYES == DisplayMessage( NULL, IDS_WARN_ATARI_PANIC, 0, MB_ICONSTOP | MB_YESNO ) )
		{
			/* Launch the monitor */
			bCont = Misc_LaunchMonitor();
		}
		Screen_FreeRedraw();
		Screen_UseAtariPalette( FALSE );

		if( bCont )
		{
			g_ulAtariState &= ~ATARI_CRASHED; /* For the CIM service */
			return 1;
		}
		g_ulAtariState = ATARI_CRASHED;
		wsync_halt = 1;	/* Turn off CPU */
		g_nTestVal = 32767;

		InvalidateRect( g_hMainWnd, NULL, TRUE );
	}
	else
	{
		/* Reset everything DirectDraw */
		Screen_Clear( TRUE, FALSE );

		/* Reset everything MultiMedia/DirectSound */
		Sound_Clear( TRUE );

		/* Reset everything DirectInput */
		Input_Clear();

		g_ulAtariState = ATARI_UNINITIALIZED | ATARI_CLOSING;

		Remove_ROM();

		if( g_Screen.lpbmi )
			free( g_Screen.lpbmi );
		g_Screen.lpbmi = NULL;

		Clear_Temp_Files();

		if( s_hZlib )
		{
			FreeLibrary( s_hZlib );
   			pgzopen = pgzclose = pgzread = pgzwrite = pgzerror = NULL;
		}
	}
	return 0;
} /* #OF# Atari_Exit */

/*========================================================
Function : EnumWindowsProc
=========================================================*/
/* #FN#
   Receives top-level window handles */
static
BOOL CALLBACK
/* #AS#
   To continue enumeration, the callback function must return TRUE, otherwise FALSE */
EnumWindowsProc(
	HWND   hWnd,   /* #IN# Handle to parent window   */
	LPARAM lParam  /* #IN# Application-defined value */
)
{
	BOOL bResult = TRUE; /* Continue enumeration */
	char szTitle[ 32 ];

	int nLen = strlen( VERSION_INFO );

	GetWindowText( hWnd, szTitle, nLen + 1 );
	if( strncmp( VERSION_INFO, szTitle, nLen ) == 0 )
	{
		*(HWND*)lParam = hWnd;
		bResult = FALSE;
	}
	return bResult;
}

/*========================================================
Function : Misc_FindAppWindow
=========================================================*/
/* #FN#
   Tries to find an emulator window */
HWND
/* #AS#
   Handle of the window if it has been found, otherwise NULL */
Misc_FindAppWindow( void )
{
	HWND hAppWnd = NULL;

	/* Enumerate top-level windows */
	EnumWindows( EnumWindowsProc, (LPARAM)&hAppWnd );

	return hAppWnd;
} /* #OF# Misc_FindAppWindow */

/*========================================================
Function : Misc_GetHomeDirectory
=========================================================*/
/* #FN#
   Retrieves the path for the Atari800Win home directory */
BOOL
/* #AS#
   TRUE if the path was retrieved, otherwise FALSE */
Misc_GetHomeDirectory(
	LPSTR pszHomeDir
)
{
	BOOL bResult = FALSE;

	if( GetModuleFileName( NULL, pszHomeDir, MAX_PATH ) )
	{
		size_t i;
		if( i = strlen( pszHomeDir ) )
		{
			while( i && pszHomeDir[ i ] != '\\' )
				i--;
			if( i )
				pszHomeDir[ i ] = '\0';
			bResult = TRUE;
		}
	}
	return bResult;
} /* #OF# Misc_GetHomeDirectory */

/*========================================================
Function : TestAndSetPath
=========================================================*/
/* #FN#
   Searches the Atari system ROMs in home folder and its subfolders */
static
BOOL
/* #AS#
   TRUE if the ROM files has been found, otherwise FALSE */
TestAndSetPath(
	LPSTR pszFileName,
	enum  RomType rtType,
	LPSTR pszPath
)
{
	char   szStartPath[ MAX_PATH ];
	char   szImagePath[ MAX_PATH ];
	char   szPattern  [ MAX_PATH ];
	ULONG  ulCRC;
	int    nResult;

	WIN32_FIND_DATA fileData;
	HANDLE hFile;

	if( NULL == pszPath )
	{
		/* We begin searching at an Atari800Win home directory */
		Misc_GetHomeDirectory( szStartPath );
	}
	else
		strcpy( szStartPath, pszPath );

	if( szStartPath[ strlen( szStartPath ) - 1 ] != '\\' )
		strcat( szStartPath, "\\" );

	strcpy( szPattern, szStartPath );
	strcat( szPattern, "*.rom" );

	hFile = FindFirstFile( szPattern, &fileData );
	/* If any file has been found... */
	while( INVALID_HANDLE_VALUE != hFile && !s_bBreakSearch )
	{
		if( !(fileData.dwFileAttributes & (FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_SYSTEM | FILE_ATTRIBUTE_DIRECTORY)) )
		{
			strcpy( szImagePath, szStartPath );
			strcat( szImagePath, fileData.cFileName );

			ulCRC = CheckFile( szImagePath, &nResult );
			if( ulCRC != 0 )
			{
				int i;
				for( i = 0; i < g_nRomTypeInfoNo; i++ )
				{
					if( g_aRomTypeInfo[ i ].rtType == rtType &&
						g_aRomTypeInfo[ i ].ulCRC  == ulCRC )
					{
						/* Again, strncpy gives the strange effects here */
						_ASSERT(strlen( szImagePath ) < MAX_PATH);
						strcpy( pszFileName, szImagePath );

						/* The searched file has been found */
						FindClose( hFile );
						return TRUE;
					}
				}
			}
		}
		/* Try to find a next file */
		if( !FindNextFile( hFile, &fileData ) )//&& (GetLastError() == ERROR_NO_MORE_FILES) )
		{
			FindClose( hFile );
			hFile = INVALID_HANDLE_VALUE; /* Ending the loop */
		}
	}
	strcpy( szPattern, szStartPath );
	strcat( szPattern, "*.*" );

	hFile = FindFirstFile( szPattern, &fileData );
	/* If any folder has been found... */
	while( INVALID_HANDLE_VALUE != hFile && !s_bBreakSearch )
	{
		if( !(fileData.dwFileAttributes & (FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_SYSTEM)) &&
			 (fileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) && (*fileData.cFileName != '.') )
		{
			strcpy( szImagePath, szStartPath );
			strcat( szImagePath, fileData.cFileName );

			/* Search the ROM file in this subfolder */
			if( TestAndSetPath( pszFileName, rtType, szImagePath ) )
			{
				FindClose( hFile );
				return TRUE;
			}
		}
		/* Try to find a next folder */
		if( !FindNextFile( hFile, &fileData ) )//&& (GetLastError() == ERROR_NO_MORE_FILES) )
		{
			FindClose( hFile );
			hFile = INVALID_HANDLE_VALUE; /* Ending the loop */
		}
	}
	return FALSE;
} /* #OF# TestAndSetPath */

/*========================================================
Function : RomSearchingDlgProc
=========================================================*/
/* #FN#
   The "ROM Searching" dialog box procedure */
static
int CALLBACK
/* #AS#
   TRUE if the message has been handled, otherwise FALSE */
RomSearchingDlgProc(
	HWND hDialog,
	UINT uiMsg,
	UINT wParam,
	LONG lParam
)
{
	LPCSTR pszProgress[] = { "|", "/", "-", "\\" };
	static int i = -1;

	switch( uiMsg )
	{
		case WM_USER:
			if( ++i > 3 ) i = 0;
			SetWindowText( hDialog, pszProgress[ i ] );
			return TRUE;

		case WM_COMMAND:
			s_bBreakSearch = TRUE;
			DestroyWindow( hDialog );
			return TRUE;
	}
	return FALSE;
} /* #OF# RomSearchingDlgProc */

/*========================================================
Function : SearchingThreadProc
=========================================================*/
/* #FN#
   A separate thread procedure for ROM searching */
DWORD WINAPI
/* #AS#
   Thread return code (success or failure) */
SearchingThreadProc(
	LPVOID lpParameter
)
{
	LPSTR pszStartPath = (LPSTR)lpParameter;
	BOOL  bFoundRom = FALSE;

	if(	!s_bBreakSearch &&
		TestAndSetPath( atari_osa_filename,   RTI_OSA, pszStartPath ) )
		bFoundRom = TRUE;
	if(	!s_bBreakSearch &&
		TestAndSetPath( atari_osb_filename,   RTI_OSB, pszStartPath ) )
		bFoundRom = TRUE;
	if(	!s_bBreakSearch &&
		TestAndSetPath( atari_xlxe_filename,  RTI_XLE, pszStartPath ) )
		bFoundRom = TRUE;
	if(	!s_bBreakSearch &&
		TestAndSetPath( atari_5200_filename,  RTI_A52, pszStartPath ) )
		bFoundRom = TRUE;
	if(	!s_bBreakSearch &&
		TestAndSetPath( atari_basic_filename, RTI_BAS, pszStartPath ) )
		bFoundRom = TRUE;

	ExitThread( bFoundRom );
	return 0;
} /* #OF# SearchingThreadProc */

/*========================================================
Function : Misc_TestRomPaths
=========================================================*/
/* #FN#
   Routine tries to locate the system ROMs in either the home directory
   or a subdirectory "ROMs" under the home directory */
BOOL
/* #AS#
   TRUE if at least one ROM file was found, otherwise FALSE */
Misc_TestRomPaths(
	LPSTR pszStartPath,
	HWND  hWnd
)
{
	HWND   hDialog = NULL;
	HANDLE hThread = NULL;
	DWORD  dwThreadId, dwExitCode = 0;

	s_bBreakSearch = FALSE;

	/* Install the Abort Dialog handler and display the window */
	if( hWnd )
	{
		EnableWindow( hWnd, FALSE );
		hDialog = CreateDialog( g_hInstance, MAKEINTRESOURCE(IDD_ROMSEARCHING),
								hWnd, (DLGPROC)RomSearchingDlgProc );
	}
	/* The searching procedure may take a long time until it
	   ends its activity. */
	if( (hThread = CreateThread( NULL,
								 0,
								 SearchingThreadProc,
								 (LPVOID)pszStartPath,
								 0,
								 &dwThreadId )) )
	{
		int nCount = 0;
		/* This message loop will solve some problems with Abort
		   Dialog activity and the main window repainting */
		do
		{
			MSG msg;
			if( PeekMessage( &msg, 0, 0, 0, PM_REMOVE ) )
				if( !hDialog || !IsDialogMessage( hDialog, &msg ) )
				{
					TranslateMessage( &msg );
					DispatchMessage( &msg );
				}
			if( nCount++ == 20 && hDialog )
			{
				SendMessage( hDialog, WM_USER, 0, 0 );
				nCount = 0;
			}
			Sleep( 10 ); /* Wait a moment, please */
			GetExitCodeThread( hThread, &dwExitCode );
		}
		while( STILL_ACTIVE == dwExitCode );

		CloseHandle( hThread );
	}
	if(	hWnd )
	{
		EnableWindow( hWnd, TRUE );
		/* There is the Search window enabled, so we can destroy the
		   Abort box now */
		if( !s_bBreakSearch && hDialog )
			DestroyWindow( hDialog );

//		SetFocus( hWnd );
	}
	return (BOOL)dwExitCode;
} /* #OF# Misc_TestRomPaths */

/*========================================================
Function : RomTypeDlgProc
=========================================================*/
/* #FN#
   The "ROM Type" dialog box procedure. Since the one doesn't use MFC,
   it may be invoked from the pure "C" functions */
static
int CALLBACK
/* #AS#
   TRUE if the message has been handled, otherwise FALSE */
RomTypeDlgProc(
	HWND hDialog,
	UINT uiMsg,
	UINT wParam,
	LONG lParam
)
{
	UINT aunRadioIDs [ ROM_TYPES_NO ] = { IDC_ROMTYPE_16STD, IDC_ROMTYPE_16AGS, IDC_ROMTYPE_16OSS, IDC_ROMTYPE_32AGS, IDC_ROMTYPE_32DB };
	UINT aunCartTypes[ ROM_TYPES_NO ] = { NORMAL16_CART, AGS32_CART, OSS_SUPERCART, AGS32_CART, DB_SUPERCART };
	char szCartDesc  [ LOADSTRING_STRING_SIZE ];
	int  i;

	switch( uiMsg )
	{
		case WM_INITDIALOG:
			for( i = 0; i < ROM_TYPES_NO; i++ )
				EnableWindow( GetDlgItem( hDialog, aunRadioIDs[ i ] ), (BOOL)(s_unCartCheck & (1 << i)) );

			/* Check a appropriate radio button */
			CheckRadioButton( hDialog, IDC_ROMTYPE_16STD, IDC_ROMTYPE_32DB,
							  (s_unCartCheck & FCT_CHECK_16ROM ?
							  (default_system == 6 ? IDC_ROMTYPE_16AGS : IDC_ROMTYPE_16STD) :
							  (default_system == 6 ? IDC_ROMTYPE_32AGS : IDC_ROMTYPE_32DB)) );
			/* Set description of the cartridge type */
			LoadString( NULL, (s_unCartCheck & FCT_CHECK_16ROM ?
							  (default_system == 6 ? IDS_ROM_16AGS_DESC : IDS_ROM_16STD_DESC) :
							  (default_system == 6 ? IDS_ROM_32AGS_DESC : IDS_ROM_32DB_DESC)),
							  szCartDesc, LOADSTRING_STRING_SIZE );
			SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
			return TRUE;

		case WM_CLOSE:
			wParam = IDCANCEL;

		case WM_COMMAND:
			switch( wParam )
			{
				case IDC_ROMTYPE_16STD:
					LoadString( NULL, IDS_ROM_16STD_DESC, szCartDesc, LOADSTRING_STRING_SIZE );
					SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
					return TRUE;

				case IDC_ROMTYPE_16AGS:
					LoadString( NULL, IDS_ROM_16AGS_DESC, szCartDesc, LOADSTRING_STRING_SIZE );
					SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
					return TRUE;

				case IDC_ROMTYPE_16OSS:
					LoadString( NULL, IDS_ROM_16OSS_DESC, szCartDesc, LOADSTRING_STRING_SIZE );
					SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
					return TRUE;

				case IDC_ROMTYPE_32AGS:
					LoadString( NULL, IDS_ROM_32AGS_DESC, szCartDesc, LOADSTRING_STRING_SIZE );
					SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
					return TRUE;

				case IDC_ROMTYPE_32DB:
					LoadString( NULL, IDS_ROM_32DB_DESC, szCartDesc, LOADSTRING_STRING_SIZE );
					SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
					return TRUE;

				case IDOK:
					for( i = 0; i < ROM_TYPES_NO; i++ )
						if( BST_CHECKED == IsDlgButtonChecked( hDialog, aunRadioIDs[ i ] ) )
							break;
					EndDialog( hDialog, aunCartTypes[ i ] );
					return TRUE;

				case IDCANCEL:
					EndDialog( hDialog, NO_CART );
					return TRUE;
			}
			break;
	}
	return FALSE;
} /* #OF# RomTypeDlgProc */

/*========================================================
Function : Misc_FindCartType
=========================================================*/
/* #FN#
   Checks the pointed cartridge type */
int
/* #AS#
   Cartridge type */
Misc_FindCartType(
	LPCSTR pszFileName, /* #IN# Name of the file to test          */
	BOOL   bAutoDetect, /* #IN# Don't ask user for cart type      */
	BOOL   bCheckExt    /* #IN# Check if the file is an Atari one */
)
{
	UINT unFileType = IAF_CRT_IMAGE | IAF_ROM_IMAGE;
	int  nCartType  = NO_CART;

	if( !Misc_IsAtariFile( pszFileName, &unFileType ) )
	{
		if( !bCheckExt )
		{
			/* It is not a good idea to ignore the Misc_IsAtariFile()
               function result but it's the only way to load
			   a ROM file without an appropriate extension */
			unFileType = IAF_ROM_IMAGE;
		}
		else
			return NO_CART;
	}
	if( IAF_ROM_IMAGE == unFileType )
	{
		int	nFileLen = 0;
		int fd = _open( pszFileName, _O_RDONLY | _O_BINARY, 0 );
		if( -1 == fd )
			return NO_CART;

		nFileLen = _filelength( fd );
		_close( fd );

		if( nFileLen < 8193 )
		{
			nCartType = NORMAL8_CART;
		}
		else
		if( nFileLen < 16385 )
		{
			if( bAutoDetect )
				nCartType = (default_system == 6 ? AGS32_CART : NORMAL16_CART);
			else
			{
				/* Ask the user for 16K cart type */
				s_unCartCheck = FCT_CHECK_16ROM;
				nCartType = DialogBox( g_hInstance, MAKEINTRESOURCE(IDD_ROMTYPE),
									   g_hMainWnd, (DLGPROC)RomTypeDlgProc );
			}
		}
		else
		if( nFileLen < 32769 )
		{
			if( bAutoDetect )
				nCartType = (default_system == 6 ? AGS32_CART : DB_SUPERCART);
			else
			{
				/* Ask the user for 32K cart type */
				s_unCartCheck = FCT_CHECK_32ROM;
				nCartType = DialogBox( g_hInstance, MAKEINTRESOURCE(IDD_ROMTYPE),
									   g_hMainWnd, (DLGPROC)RomTypeDlgProc );
			}
		}
		else
		if( nFileLen < 40961 )
			/* BountyBob 5200 cart has got the size greater than 32K */
			nCartType = AGS32_CART;
	}
	else
	if( IAF_CRT_IMAGE == unFileType )
		nCartType = CARTRIDGE;

	return nCartType;
} /* #OF# Misc_FindCartType */

/*========================================================
Function : Misc_IsAtariFile
=========================================================*/
/* #FN#
   Checks if the pointed image is known Atari file */
BOOL
/* #AS#
   TRUE if the image might be an Atari file, otherwise FALSE */
Misc_IsAtariFile(
	LPCSTR  pszFileName, /* #IN# Name of the file to test */
	UINT   *pFileType    /* #IN/OUT# The file type */
)
{
	char szTempFile[ MAX_PATH ];
	int  fd;

	strcpy( szTempFile, pszFileName );
	strupr(	szTempFile );

	if( IAF_DSK_IMAGE & *pFileType )
	{
		if( strstr( szTempFile, ".GZ"  ) ||
			strstr( szTempFile, ".ATZ" ) ||
			strstr( szTempFile, ".XFZ" ) ||
			strstr( szTempFile, ".DCM" ) ||
			strstr( szTempFile, ".ATR" ) ||
			strstr( szTempFile, ".XFD" ) )
		{
			*pFileType = IAF_DSK_IMAGE;
			return TRUE;
		}
	}
	if( IAF_ROM_IMAGE & *pFileType )
	{
		if( strstr( szTempFile, ".ROM" ) ||
			strstr( szTempFile, ".BIN" ) )
		{
			*pFileType = IAF_ROM_IMAGE;
			return TRUE;
		}
	}
	if( IAF_CRT_IMAGE & *pFileType &&
		(fd = _open( pszFileName, _O_RDONLY | _O_BINARY, 0 )) != -1 )
	{
		char cBuffer[ 4 ];
		int  nBytesRead = 0;

		_lseek( fd, 0L, SEEK_SET );
		nBytesRead = _read( fd, cBuffer, 4 );
		_close( fd );

		if( 4 == nBytesRead &&
			'C' == cBuffer[ 0 ] &&
			'A' == cBuffer[ 1 ] &&
			'R' == cBuffer[ 2 ] &&
			'T' == cBuffer[ 3 ] )
		{
			*pFileType = IAF_CRT_IMAGE;
			return TRUE;
		}
	}
	if( IAF_A8S_IMAGE & *pFileType )
	{
		if( strstr( szTempFile, ".A8S" ) )
		{
			*pFileType = IAF_A8S_IMAGE;
			return TRUE;
		}
	}
	/* That's the last check because some carts have $FFFF header */
	if( IAF_BIN_IMAGE & *pFileType &&
		(fd = _open( pszFileName, _O_RDONLY | _O_BINARY, 0 )) != -1 )
	{
		char cBuffer[ 2 ];
		int  nBytesRead = 0;

		_lseek( fd, 0L, SEEK_SET );
		nBytesRead = _read( fd, cBuffer, 2 );
		_close( fd );

		if( 2 == nBytesRead &&
			0xff == (BYTE)cBuffer[ 0 ] &&
			0xff == (BYTE)cBuffer[ 1 ] )
		{
			*pFileType = IAF_BIN_IMAGE;
			return TRUE;
		}
	}
	*pFileType = 0;

	return FALSE;
} /* #OF# Misc_IsAtariFile */

/*========================================================
Function : Misc_IsCompressedFile
=========================================================*/
/* #FN#
   Simple routine to check if the given filename is any one of the various
   compressed types. Currently the only test for this is by extension, also
   the only way you can actually read a compressed file in, so that seems
   valid enough. */
BOOL
/* #AS#
   TRUE if the file is compressed, otherwise FALSE */
Misc_IsCompressedFile(
	LPCSTR pszFileName /* #IN# Name of the file to test */
)
{
	char szTempFile[ MAX_PATH ];

	strcpy( szTempFile, pszFileName );
	strupr(	szTempFile );

	if( strstr( szTempFile, ".GZ"  ) ||
		strstr( szTempFile, ".ATZ" ) ||
		strstr( szTempFile, ".XFZ" ) ||
		strstr( szTempFile, ".DCM" ) )
		return TRUE;

	return FALSE;
} /* #OF# Misc_IsCompressedFile */

/*========================================================
Function : Misc_RunAtariExe
=========================================================*/
/* #FN#
   Loads and executes the pointed Atari 8-bit executable file */
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
Misc_RunAtariExe(
	LPSTR pszFileName /* #IN# Name of the executable file to run */
)
{
	char szNewDir[ MAX_PATH ];
	BOOL bResult;

	strncpy( szNewDir, pszFileName, MAX_PATH );
	Misc_GetFolderPath( szNewDir, NULL );
	if( _stricmp( szNewDir, atari_exe_dir ) != 0 )
	{
		strcpy( atari_exe_dir, szNewDir );
		WriteRegString( NULL, REG_EXE_PATH, atari_exe_dir );
	}
	bResult = BIN_loader( pszFileName );

	return bResult;
} /* #OF# Misc_RunAtariExe */

/*========================================================
Function : Misc_GetFolderPath
=========================================================*/
/* #FN#
   Extracts the folder from full file path */
void
/* #AS#
   Nothing */
Misc_GetFolderPath(
	LPSTR pszPathName, /* #IN/OUT# Full file path/file folder name */
	LPSTR pszFileName  /* #OUT#    File path */
)
{
	int nPathLen = strlen( pszPathName ) - 1;
	int i;

	for( i = nPathLen; i > 0 && pszPathName[ i ] != '\\'; i-- );
	if( i > 0 || pszPathName[ i ] == '\\' )
	{
		pszPathName[ i++ ] = '\0';
	}
	if( pszFileName != NULL )
		strncpy( pszFileName, &pszPathName[ i ], MAX_PATH );

	/* Add ending backslash to drive name */
	if( strlen( pszPathName ) == 2 && pszPathName[ 1 ] == ':' )
		strcat( pszPathName, "\\" );
	
} /* #OF# Misc_GetFolderPath */

/*========================================================
Function : ConsoleHandlerRoutine
=========================================================*/
/* #FN#
   A console process uses this function to handle control signals received
   by the process */
static
BOOL WINAPI
/* #AS#
   If the function handles the control signal, it should return TRUE. If it
   returns FALSE, the next handler function in the list of handlers for this
   process is used */
ConsoleHandlerRoutine(
	DWORD dwCtrlType /* #IN# Control signal type */
)
{
	/* On Windows 95, a console application that has installed a control
	   signal handler function only gets called for the CTRL_C_EVENT and
	   CTRL_BREAK_EVENT signals; the signal handler function is never
	   called for the CTRL_SHUTDOWN_EVENT, CTRL_LOGOFF_EVENT, and
	   CTRL_CLOSE_EVENT signals. */
	switch( dwCtrlType )
	{
		case CTRL_C_EVENT:
		case CTRL_BREAK_EVENT:
		case CTRL_SHUTDOWN_EVENT:
		case CTRL_CLOSE_EVENT:
		case CTRL_LOGOFF_EVENT:
			break;

		default:
			/* Unknown type, better pass it on */
			return FALSE;
	}
	/* Handled all known events */
	return TRUE;
} /* #OF# ConsoleHandlerRoutine */

/*========================================================
Function : Misc_AllocMonitorConsole
=========================================================*/
/* #FN#
   Invokes a monitor console */
BOOL
/* #AS#
   BOOL if succeeded, otherwise FALSE */
Misc_AllocMonitorConsole(
	FILE **pOutput,
	FILE **pInput
)
{
    if( !AllocConsole() )
		return FALSE;

	SetConsoleTitle( "Atari800Win PLus Monitor" );
	*pOutput = fopen( "CON", "wt" );
	*pInput  = fopen( "CON", "rt" );
	/* Disable Ctrl+C and Ctrl+Break keystrokes */
	SetConsoleCtrlHandler( ConsoleHandlerRoutine, TRUE );

	return TRUE;
} /* #OF# Misc_AllocMonitorConsole */

/*========================================================
Function : Misc_FreeMonitorConsole
=========================================================*/
/* #FN#
   Closes a monitor console */
void
/* #AS#
   Nothing */
Misc_FreeMonitorConsole(
	FILE *pOutput,
	FILE *pInput
)
{
	/* Detach Console Routine */
	SetConsoleCtrlHandler( ConsoleHandlerRoutine, FALSE );

	if( pInput )
		fclose( pInput );
	if( pOutput )
		fclose( pOutput );

	FreeConsole();
} /* #OF# Misc_FreeMonitorConsole */

/*========================================================
Function : ExecuteCmd
=========================================================*/
/* #FN#
   Executes a system command */
static
BOOL
/* #AS#
   TRUE if succeded, otherwise FALSE */
ExecuteCmd(
	LPSTR pszCommand, /* #IN# System command to execute */
	BOOL  bGetOutput  /* #IN# Output stream to error log */
)
{	
	BOOL   bProcess   = FALSE;
	BOOL   bPipe      = FALSE;
	HANDLE hReadPipe  = NULL;
	HANDLE hWritePipe = NULL;

	PROCESS_INFORMATION pi;
	STARTUPINFO si;

	SECURITY_ATTRIBUTES sa;
	sa.nLength              = sizeof(sa);
	sa.lpSecurityDescriptor = NULL;
	sa.bInheritHandle       = TRUE;

	if( bGetOutput )
		bPipe = CreatePipe( &hReadPipe, &hWritePipe, &sa, 0 );

	/* Run a child process */
	ZeroMemory( &si, sizeof(si) );
	si.cb = sizeof(si);
	si.wShowWindow = SW_SHOWNORMAL; /* Do not display the console window */
	si.dwFlags = STARTF_USESHOWWINDOW;

	if( bPipe )
	{
		si.dwFlags   |= STARTF_USESTDHANDLES;
		si.hStdInput  = GetStdHandle( STD_INPUT_HANDLE );
		si.hStdOutput = hWritePipe;
		si.hStdError  = hWritePipe;
	}

	/* Run system command */
	bProcess =
		CreateProcess( NULL,		// no module name (use command line)
					   pszCommand,	// command line
					   NULL,		// process handle not inheritable
					   NULL,		// threat handle not inheritable
					   TRUE,		// set handle inheritance to TRUE
					   0,			// no creation flags
					   NULL,		// use parent's environment block
					   NULL,		// use parent's starting directory
					   &si,
					   &pi );

	/* We have to close the reading end of the pipe */
	if( bPipe )
		CloseHandle( hWritePipe );

	if( bProcess )
	{
		if( bPipe )
		{
			char  szBuffer[ EC_BUFFER_LEN ];	/* Buffer that receives data */
			char  szText  [ EC_MAX_LINE_LEN ];	/* Buffer for output line */
			char *pszOut = szText;
			char  c;
			DWORD dwBytesRead;					/* Number of bytes read */
			BOOL  bReadResult;

			/* Output stream to Error View buffer */
			do
			{
				/* Read the executed process output message */
				bReadResult = ReadFile( hReadPipe, (LPVOID)&szBuffer, EC_BUFFER_LEN, &dwBytesRead, NULL );
				if( bReadResult )
				{
					char *pszIn = szBuffer;
					int   i;
					for( i = 0; i < (int)dwBytesRead; i++ )
					{
						c = *(pszIn++);
						if( c >= ' ' && c <= 'z' )
							*(pszOut++) = c;
						if( c == '\0' || c == '\n' )
						{
							*pszOut = '\0';
							/* Print message line to the Error View buffer */
							Aprint( szText );
							pszOut = szText;
						}
					}
				}
			}
			while( bReadResult && dwBytesRead != 0 );
		}
		if( bPipe )
			/* Wait for ending the executed process */
			WaitForSingleObject( pi.hProcess, INFINITE );

		if( pi.hProcess && pi.hProcess != INVALID_HANDLE_VALUE )
			CloseHandle( pi.hProcess );
		if( pi.hThread && pi.hThread != INVALID_HANDLE_VALUE )
			CloseHandle( pi.hThread );
	}

	if( bPipe )
		CloseHandle( hReadPipe );

	return bProcess;
} /* #OF# ExecuteCmd */

/*========================================================
Function : Misc_ExecutePrintCmd
=========================================================*/
/* #FN#
   Executes a print command */
BOOL
/* #AS#
   TRUE if succeded, otherwise FALSE */
Misc_ExecutePrintCmd(
	LPSTR pszPrintFile /* #IN# Print command to execute */
)
{
	BOOL bResult = TRUE;

	/* Unfortunately, there are some problems with automatically return to
	   flipped full-screen mode, go to windowed instead */
	if( g_Screen.ulMode & SM_MODE_FULL /*&& g_Screen.ulMode & SM_OPTN_FLIP_BUFFERS*/ )
	{
		/* The only safe method to doing it here */
		PostMessage( g_hMainWnd, WM_COMMAND, ID_VIEW_TOGGLEMODES, 0L );
	}

	if( !(g_Misc.ulState & MS_USE_PRINT_COMMAND) )
	{
		char szPath[ MAX_PATH ];
		char szFile[ MAX_PATH ];
		int  nResult;

		strncpy( szPath, pszPrintFile, MAX_PATH );
		Misc_GetFolderPath( szPath, szFile );

		if( (nResult =
			(int)ShellExecute( g_hMainWnd,
							   "print",
							   szFile,
							   NULL,
							   szPath,
							   SW_HIDE )) < 32 )
		{
			Aprint( "Printing error (shell execution code: %d)", nResult );
			bResult = FALSE;
		}
	}
	else
	{
		char szPrintCmd[ MAX_PATH ];

		sprintf( szPrintCmd, print_command, pszPrintFile );
		if( !ExecuteCmd( szPrintCmd, FALSE ) )
			bResult = FALSE;
	}
	g_ulAtariState &= ~ATARI_PAUSED;

	return bResult;
} /* #OF# Misc_ExecutePrintCmd */

/*========================================================
Function : Misc_SetProcessPriority
=========================================================*/
/* #FN#
   Sets the process priority level */
void
/* #AS#
   Nothing */
Misc_SetProcessPriority( void )
{
	SetPriorityClass( GetCurrentProcess(),
					  g_Misc.ulState & MS_HIGH_PRIORITY ?
					  HIGH_PRIORITY_CLASS :
					  NORMAL_PRIORITY_CLASS );
} /* #OF# Misc_SetProcessPriority */
