/****************************************************************************
File    : display_win.c
/*
@(#) #SY# Atari800Win PLus
@(#) #IS# Display implementation for Win32 platforms
@(#) #BY# Richard Lawrence, Tomasz Szymankowski, Piotr Fusik
@(#) #LM# 24.06.2001
*/

#include <stdio.h>
#include <crtdbg.h>
#include <ddraw.h>
#include <mmsystem.h>
#include <windows.h>
#include <windowsx.h>
#include "WinConfig.h"
#include "Resource.h"
#include "Helpers.h"
#include "atari800.h"
#include "globals.h"
#include "macros.h"
#include "registry.h"
#include "sound_win.h"
#include "misc_win.h"
#include "input_win.h"
#include "avisave.h"
#include "display_win.h"

/* Stretching routines have been optimized by Piotr Fusik. */

#define PFUSIK_ASM

/* WARNING!: Robust interpolation routines in C seem to be faster
   on Pentium Pro systems (Celeron, Pentium II, Pentium III etc.)
*/
#define INTEL_ASM

/* Public objects */

struct ScreenCtrl_t g_Screen =
{
	DEF_SCREEN_MODE,
	0L,
	DEF_STRETCH_MODE,
	DEF_MEMORY_TYPE,
	NULL,
	DEF_SHOW_CURSOR,
	20,
	FALSE,
	NULL,
	NULL,
	/* Palette */
	{
		DEF_CLR_BLACK_LEVEL,
		DEF_CLR_WHITE_LEVEL,
		DEF_CLR_SATURATION,
		DEF_CLR_SHIFT,
		TRUE,
		{ 0 }
	}
};

void (*Atari_DisplayScreen)( UBYTE *screen );

/* Private objects */

static struct
{
	UCHAR *pSource;		/* The kernel makes his writes into this buffer  */
	UCHAR *pTarget;		/* We make the writes here by using asm routines */
	UCHAR *pMainScr;	/* Pointer to the primary buffer   */
	UCHAR *pWorkScr;	/* Pointer to the secondary buffer */
	UCHAR *pBackScr;	/* Pointer to the backup buffer    */
} s_Buffer =
{
	NULL, NULL, NULL, NULL, NULL
};

static LONG  s_lWndStyleBack = 0;
static ULONG s_ulScreenMode  = 0;
static int   s_nStretchMode  = 0;
static int   s_nMemoryType   = 0;
static ULONG s_ulLEDOffset   = 0;
static BOOL  s_bModeChanging = FALSE;
static BOOL  s_bModeChanged  = FALSE;
static BOOL  s_bIColorMap    = FALSE;
static RECT  s_rcSource;
static RECT  s_rcDestin;

static struct
{
	HDC     hDC;		/* Compatible device context */
	HBITMAP hBmp;		/* Display screenshot bitmap */
	int     nWidth;		/* Width of this bitmap      */
	int     nHeight;	/* Height of this bitmap     */
} s_Redraw =
{
	NULL, NULL, 0, 0
};

static struct
{
	LONG nPitch;
	LONG nWidth;
	LONG nHeight;
} s_Surface =
{
	0L, ATARI_WIDTH, ATARI_HEIGHT + 16
};

static struct
{
	LPDIRECTDRAW        lpDirectDraw;
	/* Do not use SURFACE3, it does not work with NT 4.0 */
	LPDIRECTDRAWSURFACE lpDDSPrimary;
	LPDIRECTDRAWSURFACE lpDDSOffscrn;
	LPDIRECTDRAWSURFACE lpDDSBackBuf;
	LPDIRECTDRAWCLIPPER lpDDClipper;
	LPDIRECTDRAWPALETTE lpDDPalette;
	/* Drawing flags and other params */
	DWORD               dwBlitFlags;
	DWORD               dwFlipFlags;
	DWORD               dwDDFxFlags[ BLIT_EFFECTS_NO ];
	DDBLTFX             fxBltEffects; /* The FX effects for blits */
	DDBLTFX            *pfxBltEffects;
	BOOL                bFlipBuffers;
} s_DDraw =
{
	NULL, NULL, NULL, NULL, NULL, NULL, DDBLT_ASYNC, 0,
	{
		DDBLTFX_MIRRORLEFTRIGHT,
		DDBLTFX_MIRRORUPDOWN,
		DDBLTFX_NOTEARING
	},
	{ 0 }, NULL, FALSE
};

static UCHAR s_aucAsciiToScreen[ 128 ] =
{
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
	0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
	0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
	0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
	0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f
};

/* Default Windows system palette (20 entries) */
static PALETTEENTRY s_apeSystemColours[ 20 ] =
{
	{ 0,    0,  0, PC_NOCOLLAPSE }, //
	{ 128,  0,  0, PC_NOCOLLAPSE },
	{ 0,  128,  0, PC_NOCOLLAPSE }, //
	{ 128,128,  0, PC_NOCOLLAPSE }, //
	{ 0,    0,128, PC_NOCOLLAPSE },
	{ 128,  0,128, PC_NOCOLLAPSE }, //
	{   0,128,128, PC_NOCOLLAPSE }, //
	{ 192,192,192, PC_NOCOLLAPSE },
	{ 192,220,192, PC_NOCOLLAPSE }, //
	{ 166,202,240, PC_NOCOLLAPSE }, //

	{ 255,251,240, PC_NOCOLLAPSE }, //
	{ 160,160,164, PC_NOCOLLAPSE },
	{ 128,128,128, PC_NOCOLLAPSE },
	{ 255,  0,  0, PC_NOCOLLAPSE }, //
	{   0,255,  0, PC_NOCOLLAPSE }, //
	{ 255,255,  0, PC_NOCOLLAPSE }, //
	{   0,  0,255, PC_NOCOLLAPSE }, //
	{ 255,  0,255, PC_NOCOLLAPSE }, //
	{   0,255,255, PC_NOCOLLAPSE }, //
	{ 255,255,255, PC_NOCOLLAPSE }
};

static UBYTE s_anIColorMap[ PAL_ENTRIES_NO ][ PAL_ENTRIES_NO ];

static BOOL GetDDErrorString( HRESULT hResult, LPSTR lpszErrorBuff, DWORD cchError );

static void Screen_DDraw                       ( UBYTE *screen );
static void Screen_DDraw_Double                ( UBYTE *screen );
static void Screen_DDraw_Double_Interpolation  ( UBYTE *screen );
static void Screen_DDraw_Double_Scanlines      ( UBYTE *screen );
static void Screen_DDraw_Full_Blt              ( UBYTE *screen );
static void Screen_DDraw_Full_Blt_Interpolation( UBYTE *screen );
static void Screen_DDraw_Full_Blt_Scanlines    ( UBYTE *screen );
static void Screen_DDraw_1024_Interpolation    ( UBYTE *screen );
static void Screen_DDraw_1024_Scanlines        ( UBYTE *screen );
static void Screen_GDI                         ( UBYTE *screen );
static void Screen_GDI_Double                  ( UBYTE *screen );
static void Screen_GDI_Double_Interpolation    ( UBYTE *screen );
static void Screen_GDI_Double_Scanlines        ( UBYTE *screen );


#ifdef _DEBUG
#define ServeDDError( nUID, hResult, bQuit ) \
		ShowDDError( nUID, hResult, bQuit, __FILE__, __LINE__ )
#else /*_DEBUG*/
#define ServeDDError( nUID, hResult, bQuit ) \
		ShowDDError( nUID, hResult, bQuit )
#endif /*_DEBUG*/


/*========================================================
Function : ShowDDError
=========================================================*/
/* #FN#
   Shows DirectDraw error */
static
void
/* #AS#
   Nothing */
ShowDDError(
	UINT    nUID,    /* #IN# */
	HRESULT hResult, /* #IN# */
	BOOL    bQuit    /* #IN# */
#ifdef _DEBUG
  , char   *pszFile,
	DWORD   dwLine
#endif /*_DEBUG*/
)
{
	char szError [ LOADSTRING_STRING_SIZE ];
	char szAction[ LOADSTRING_STRING_SIZE ];

#ifdef _DEBUG
	Aprint( "DirectDraw error: %s@%ld", pszFile, dwLine );
#endif /*_DEBUG*/

	/* Get us back to a GDI display and stop making noises */
	Screen_SetSafeDisplay( TRUE );
	Sound_Clear( FALSE );

	GetDDErrorString( hResult, szError, LOADSTRING_STRING_SIZE );
	LoadString( NULL, nUID, szAction, LOADSTRING_STRING_SIZE );
	DisplayMessage( NULL, IDS_DDERR_PROMPT, IDS_DDERR_HDR, MB_ICONSTOP | MB_OK, szAction, szError );

	/* Start a quit (this will end up in Atari_Exit()) */
	if( bQuit )
	{
		/* Make sure the atari is turned off */
		g_ulAtariState = ATARI_UNINITIALIZED;
		PostMessage( g_hMainWnd, WM_CLOSE, 0, 0L );
	}
	else
		Sound_Restart();
} /* #OF# ShowDDError */

/*========================================================
Function : AtariPlot
=========================================================*/
/* #FN#
   Routines for plotting on the Atari screen (taken out of ui.c, which
   I don't use currently */
static
void
/* #AS#
   Nothing */
AtariPlot(
	UBYTE *pScreen, /* #IN# */
	int    nFg,     /* #IN# */
	int    nBg,     /* #IN# */
	int    nCh,     /* #IN# */
	int    x,       /* #IN# */
	int    y        /* #IN# */
)
{
	int   nOffset = s_aucAsciiToScreen[ (nCh & 0x07f) ] * 8;
	int   i;
	int   j;
	char *pAtariCharset;
	char *ptr;

	if( !(g_ulAtariState & ATARI_RUNNING) )
		return;

	if( mach_xlxe != 0 )
		pAtariCharset = atarixl_os + 0x2000;
	else
		pAtariCharset = memory + 0xe000;

	ptr = &pScreen[ 24 * ATARI_WIDTH + 32 ];

	for( i = 0; i < 8; i++ )
	{
		UBYTE data = pAtariCharset[ nOffset++ ];

		for( j = 0; j < 8; j++ )
		{
			int nPixel;

			if( data & 0x80 )
				nPixel = nFg;
			else
				nPixel = nBg;

			ptr[ (y * 8 + i) * ATARI_WIDTH + (x * 8 + j) ] = nPixel;

			data = data << 1;
		}
	}
} /* #OF# AtariPlot */

/*========================================================
Function : AtariPrint
=========================================================*/
/* #FN#
   Routines for printing on the Atari screen (taken out of ui.c, which
   I don't use currently */
static
void
/* #AS#
   Nothing */
AtariPrint(
	UBYTE *pScreen,   /* #IN# */
	int    nFg,       /* #IN# */
	int    nBg,       /* #IN# */
	char  *pszString, /* #IN# */
	int    x,         /* #IN# */
	int    y          /* #IN# */
)
{
	while( *pszString )
	{
		AtariPlot( pScreen, nFg, nBg, *pszString++, x, y );
		x++;
	}
} /* #OF# AtariPrint */

/*========================================================
Function : CenterPrint
=========================================================*/
/* #FN#
   Routines for printing on the Atari screen (taken out of ui.c, which
   I don't use currently */
static
void
/* #AS#
   Nothing */
CenterPrint(
	UBYTE *pScreen,   /* #IN# */
	int    nFg,       /* #IN# */
	int    nBg,       /* #IN# */
	char  *pszString, /* #IN# */
	int    y          /* #IN# */
)
{
	AtariPrint( pScreen, nFg, nBg, pszString, (40 - strlen( pszString )) / 2, y );
} /* #OF# CenterPrint */

/*========================================================
Function : DrawPaused
=========================================================*/
/* #FN#
   Draws paused screen */
static
void
/* #AS#
   Nothing */
DrawPaused(
	UBYTE *pScreen  /* #IN# */
)
{
	int    i, j;
	UBYTE *blank;

	for( i = 0; i < ATARI_HEIGHT; i++ )
	{
		blank = pScreen + i * ATARI_WIDTH + i % 2;
		for( j = 0; j < ATARI_WIDTH; j += 2 )
		{
			*blank  = 0;
			 blank += 2;
		}
	}
	/* The 5200 console OS hasn't got the Atari fonts */
	if( 6 != default_system /* 5200 */ )
		CenterPrint( pScreen, 255, 0, "Paused: F9 to continue", 11 );

	Atari_DisplayScreen( pScreen );
	if( s_DDraw.bFlipBuffers )
		/* We have to draw the text also on a GDI surface */
		Atari_DisplayScreen( pScreen );
} /* #OF# DrawPaused */

/*========================================================
Function : RestoreSurfaces
=========================================================*/
/* #FN#
   Restores all the DirectDraw Surfaces */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
RestoreSurfaces( void )
{
	HRESULT hResult;

	/* Restore Memory Surface */
	if( s_DDraw.lpDDSOffscrn &&
		FAILED(hResult = IDirectDrawSurface_IsLost ( s_DDraw.lpDDSOffscrn )) &&
		FAILED(hResult = IDirectDrawSurface_Restore( s_DDraw.lpDDSOffscrn )) )
	{
		return FALSE;
	}
	/* Restore Primary Surface */
	if( s_DDraw.lpDDSPrimary &&
		FAILED(hResult = IDirectDrawSurface_IsLost ( s_DDraw.lpDDSPrimary )) &&
		FAILED(hResult = IDirectDrawSurface_Restore( s_DDraw.lpDDSPrimary )) )
	{
		return FALSE;
	}
	/* It is a good place to clear screen here because there may
	   be some graphics glitches after the surfaces restoring */
	Screen_Clear( FALSE, TRUE );

	return TRUE;
} /* #OF# RestoreSurfaces */

/*========================================================
Function : CreateDirectDrawObject
=========================================================*/
/* #FN#
   Creates DirectDraw object */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
CreateDirectDrawObject( void )
{
	/* Get the object if not already allocated */
	if( !s_DDraw.lpDirectDraw )
	{
		HRESULT hResult;

		hResult = DirectDrawCreate( NULL, &s_DDraw.lpDirectDraw, 0 );
		if( FAILED(hResult) )
		{
			ServeDDError( IDS_DDERR_ALLOC_OBJ, hResult, FALSE );
			return FALSE;
		}
	}
	return TRUE;
} /* #OF# CreateDirectDrawObject */

/*========================================================
Function : SetupPrimarySurface
=========================================================*/
/* #FN#
   Sets up DirectDraw Primary Surface */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
SetupPrimarySurface( void )
{
	DDSURFACEDESC ddsd;
	HRESULT hResult;

	_ASSERT(!s_DDraw.lpDDSPrimary);

	ZeroMemory( &ddsd, sizeof(DDSURFACEDESC) );
	ddsd.dwSize         = sizeof(DDSURFACEDESC);
	ddsd.dwFlags        = DDSD_CAPS;
	ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;

	if( s_DDraw.bFlipBuffers )
	{
		ddsd.dwFlags          |= DDSD_BACKBUFFERCOUNT;
		ddsd.ddsCaps.dwCaps   |= DDSCAPS_FLIP | DDSCAPS_COMPLEX;
		ddsd.dwBackBufferCount = 1; /* Only double-buffering is possible at the moment */
	}
	/* Create the primary DirectDraw surface */
	hResult = IDirectDraw_CreateSurface( s_DDraw.lpDirectDraw, &ddsd, &s_DDraw.lpDDSPrimary, NULL );
	if( FAILED(hResult) )
	{
		if( s_DDraw.lpDDSPrimary )
		{
			IDirectDrawSurface_Release( s_DDraw.lpDDSPrimary );
			s_DDraw.lpDDSPrimary = NULL;
		}
		ServeDDError( IDS_DDERR_ALLOC_PRIMARY, hResult, FALSE );
		return FALSE;
	}
	/* Get the back buffer when flipping the surfaces is active */
	if( s_DDraw.bFlipBuffers )
	{
		DDSCAPS caps;

		ZeroMemory( &caps, sizeof(DDSCAPS) );
		caps.dwCaps = DDSCAPS_BACKBUFFER;

		hResult = IDirectDrawSurface_GetAttachedSurface( s_DDraw.lpDDSPrimary, &caps, &s_DDraw.lpDDSBackBuf );
		if( FAILED(hResult) )
		{
			IDirectDrawSurface_Release( s_DDraw.lpDDSPrimary );
			s_DDraw.lpDDSPrimary = NULL;

			ServeDDError( IDS_DDERR_ALLOC_PRIMARY, hResult, FALSE );
			return FALSE;
		}
	}
	/* Create and attach the clipper object when in DirectDraw windowed */
	if( g_hViewWnd && g_Screen.ulMode & SM_MODE_WIND )
	{
		_ASSERT(!s_DDraw.lpDDClipper);

		hResult = IDirectDraw_CreateClipper( s_DDraw.lpDirectDraw, 0, &s_DDraw.lpDDClipper, NULL );
		if( SUCCEEDED(hResult) )
		{
			hResult = IDirectDrawClipper_SetHWnd( s_DDraw.lpDDClipper, 0, g_hViewWnd );
			/* Attach the clipper to a primary surface */
			if( SUCCEEDED(hResult) )
				hResult = IDirectDrawSurface_SetClipper( s_DDraw.lpDDSPrimary, s_DDraw.lpDDClipper );
		}
		if( FAILED(hResult) )
		{
			if( s_DDraw.lpDDClipper )
			{
				IDirectDrawClipper_Release( s_DDraw.lpDDClipper );
				s_DDraw.lpDDClipper = NULL;
			}
			IDirectDrawSurface_Release( s_DDraw.lpDDSPrimary );
			s_DDraw.lpDDSPrimary = NULL;

			ServeDDError( IDS_DDERR_ALLOC_PRIMARY, hResult, FALSE );
			return FALSE;
		}
	}
	return TRUE;
} /* #OF# SetupPrimarySurface */

/*========================================================
Function : SetupMemorySurface
=========================================================*/
/* #FN#
   Sets up DirectDraw Memory Surface */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
SetupMemorySurface( void )
{
	DDSURFACEDESC ddsd;
	HRESULT hResult = DD_OK;

	_ASSERT(!s_DDraw.lpDDSOffscrn);

	ZeroMemory( &ddsd, sizeof(DDSURFACEDESC) );
	ddsd.dwSize = sizeof(DDSURFACEDESC);

	if( g_Screen.ulMode & SM_MODE_FULL )
	{
		ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH | DDSD_PIXELFORMAT;
		ddsd.ddpfPixelFormat.dwSize        = sizeof(DDPIXELFORMAT);
		ddsd.ddpfPixelFormat.dwFlags       = DDPF_RGB | DDPF_PALETTEINDEXED8;
		ddsd.ddpfPixelFormat.dwRGBBitCount = 8;
	}
	else
		ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;

	ddsd.dwHeight = s_Surface.nHeight;
	ddsd.dwWidth  = s_Surface.nWidth;

	/* Try to allocate a video memory for offscreen surface */
	if( MEMORY_VIDEO == g_Screen.nMemoryType )
	{
		ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_VIDEOMEMORY | DDSCAPS_LOCALVIDMEM;

		hResult = IDirectDraw_CreateSurface( s_DDraw.lpDirectDraw, &ddsd, &s_DDraw.lpDDSOffscrn, NULL );
		if( FAILED(hResult) )
			Aprint( "Video memory allocation failed, trying AGP memory..." );
	}
	/* Try to allocate an AGP memory for offscreen surface */
	if( MEMORY_AGP == g_Screen.nMemoryType || FAILED(hResult) )
	{
		ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_VIDEOMEMORY | DDSCAPS_NONLOCALVIDMEM;

		hResult = IDirectDraw_CreateSurface( s_DDraw.lpDirectDraw, &ddsd, &s_DDraw.lpDDSOffscrn, NULL );
		if( FAILED(hResult) )
			Aprint( "AGP memory allocation failed, trying system memory..." );
	}
	/* Try to allocate a system memory for offscreen surface */
	if( MEMORY_SYSTEM == g_Screen.nMemoryType || FAILED(hResult) )
	{
		ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;

		hResult = IDirectDraw_CreateSurface( s_DDraw.lpDirectDraw, &ddsd, &s_DDraw.lpDDSOffscrn, NULL );
		if( FAILED(hResult) )
		{
			s_DDraw.lpDDSOffscrn = NULL;
			ServeDDError( IDS_DDERR_ALLOC_OFFSCR, hResult, FALSE );

			Aprint( "Could not allocate system memory surface!" );
			return FALSE;
		}
	}
	return TRUE;
} /* #OF# SetupMemorySurface */

/*========================================================
Function : SetDirectPointers
=========================================================*/
/* #FN#
   Sets up pointers directly to memory surfaces */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
SetDirectPointers( void )
{
	if( g_Screen.ulMode & SM_MODE_FULL && !(g_Screen.ulMode & SM_OPTN_SAFE_MODE) )
	{
		DDSURFACEDESC ddsd;
		HRESULT hResult;

		ZeroMemory( &ddsd, sizeof(DDSURFACEDESC) );
		ddsd.dwSize = sizeof(DDSURFACEDESC);

		hResult = IDirectDrawSurface_Lock( s_DDraw.lpDDSOffscrn, NULL, &ddsd, DDLOCK_WAIT, NULL );

		if( DDERR_SURFACELOST == hResult && RestoreSurfaces() )
			/* Try to lock the surface once again */
			hResult = IDirectDrawSurface_Lock( s_DDraw.lpDDSOffscrn, NULL, &ddsd, DDLOCK_WAIT, NULL );

		if( FAILED(hResult) )
		{
			IDirectDrawSurface_Release( s_DDraw.lpDDSOffscrn );
			s_DDraw.lpDDSOffscrn = NULL;

			ServeDDError( IDS_DDERR_ALLOC_OFFSCR, hResult, FALSE );
			return FALSE;
		}
		/* Store the pitch for the allocated surface */
		s_Surface.nPitch = ddsd.lPitch;

		/* It is VERY dangerous method, especially when using video
		   memory for off-screen surfaces (unfortunately DirectX 3.0
		   doesn't support client memory surfaces). But the method is
		   still the fastest one available here.
		*/
		if( g_Screen.ulMode & SM_ATTR_STRETCHED && g_Screen.nStretchMode != STRETCH_PIXELDOUBLING )
		{
			/* Set pointer to the offscreen surface buffer */
			s_Buffer.pTarget = (UCHAR *)ddsd.lpSurface;
		}
		else
			s_Buffer.pSource = (UCHAR *)ddsd.lpSurface;

		IDirectDrawSurface_Unlock( s_DDraw.lpDDSOffscrn, &ddsd );
	}
	return TRUE;
} /* #OF# SetDirectPointers */

/*========================================================
Function : ReleaseAllObjects
=========================================================*/
/* #FN#
   Releases all the DirectDraw Objects */
static
HRESULT
/* #AS#
   DD_OK if succeeded */
ReleaseAllObjects(
	BOOL bReleaseDD /* #IN# */
)
{
	HRESULT hResult  = DD_OK;
	HRESULT hFailure = DD_OK;

	/* Release Palette Object */
	if( s_DDraw.lpDDPalette )
	{
		if( FAILED(hResult = IDirectDrawPalette_Release( s_DDraw.lpDDPalette )) )
			hFailure = hResult;
		s_DDraw.lpDDPalette = NULL;
	}
	/* Release Clipper Object */
	if( s_DDraw.lpDDClipper )
	{
		/* It decreases only the clipper's reference count */
		if( FAILED(hResult = IDirectDrawClipper_Release( s_DDraw.lpDDClipper )) )
			hFailure = hResult;
		s_DDraw.lpDDClipper = NULL;
	}
	/* Release Memory Surface */
	if( s_DDraw.lpDDSOffscrn )
	{
		if( FAILED(hResult = IDirectDrawSurface_Release( s_DDraw.lpDDSOffscrn )) )
			hFailure = hResult;
		s_DDraw.lpDDSOffscrn = NULL;
	}
	/* Release Primary Surface */
	if( s_DDraw.lpDDSPrimary )
	{
		IDirectDrawSurface_SetClipper( s_DDraw.lpDDSPrimary, NULL );

		if( FAILED(hResult = IDirectDrawSurface_Release( s_DDraw.lpDDSPrimary )) )
			hFailure = hResult;
		s_DDraw.lpDDSPrimary = NULL;

		if( s_DDraw.bFlipBuffers )
			s_DDraw.lpDDSBackBuf = NULL;
	}
	/* Release DirectDraw object and restore display mode */
	if( bReleaseDD && s_DDraw.lpDirectDraw )
	{
		IDirectDraw_RestoreDisplayMode( s_DDraw.lpDirectDraw );
		if( FAILED(hResult = IDirectDraw_Release( s_DDraw.lpDirectDraw )) )
			hFailure = hResult;
		s_DDraw.lpDirectDraw = NULL;
	}
	return hFailure;
} /* #OF# ReleaseAllObjects */

/*========================================================
Function : Screen_FreeRedraw
=========================================================*/
/* #FN#
   Releases the redrawing stuff */
void
/* #AS#
   Nothing */
Screen_FreeRedraw( void )
{
	if( s_Redraw.hDC != NULL )
	{
		DeleteDC( s_Redraw.hDC );
		s_Redraw.hDC = NULL;
	}
	if( s_Redraw.hBmp != NULL )
	{
		DeleteObject( s_Redraw.hBmp );
		s_Redraw.hBmp = NULL;
	}
} /* #OF# Screen_FreeRedraw */

/*========================================================
Function : EnumModesCallback
=========================================================*/
/* #FN#
   Checks available display modes */
static
HRESULT WINAPI
/* #AS#
   Always DDENUMRET_OK */
EnumModesCallback(
	LPDDSURFACEDESC lpDDSurfaceDesc, /* #IN# */
	LPVOID          lpContext        /* #IN# */
)
{
	_ASSERT(lpDDSurfaceDesc);

	if( lpDDSurfaceDesc->ddpfPixelFormat.dwRGBBitCount == 8 )
	{
		if( lpDDSurfaceDesc->dwHeight == 768 && lpDDSurfaceDesc->dwWidth == 1024 )
			g_Screen.ulModesAvail |= MODE_1024_768;
		else
		if( lpDDSurfaceDesc->dwHeight == 600 && lpDDSurfaceDesc->dwWidth == 800 )
			g_Screen.ulModesAvail |= MODE_800_600;
		else
		if( lpDDSurfaceDesc->dwHeight == 480 && lpDDSurfaceDesc->dwWidth == 640 )
			g_Screen.ulModesAvail |= MODE_640_480;
		else
		if( lpDDSurfaceDesc->dwHeight == 400 && lpDDSurfaceDesc->dwWidth == 640 )
			g_Screen.ulModesAvail |= MODE_640_400;
		else
		if( lpDDSurfaceDesc->dwHeight == 384 && lpDDSurfaceDesc->dwWidth == 512 )
			g_Screen.ulModesAvail |= MODE_512_384;
		else
		if( lpDDSurfaceDesc->dwHeight == 300 && lpDDSurfaceDesc->dwWidth == 400 )
			g_Screen.ulModesAvail |= MODE_400_300;
		else
		if( lpDDSurfaceDesc->dwHeight == 400 && lpDDSurfaceDesc->dwWidth == 320 )
			g_Screen.ulModesAvail |= MODE_320_400;
		else
		if( lpDDSurfaceDesc->dwHeight == 240 && lpDDSurfaceDesc->dwWidth == 320 )
			g_Screen.ulModesAvail |= MODE_320_240;
		else
		if( lpDDSurfaceDesc->dwHeight == 200 && lpDDSurfaceDesc->dwWidth == 320 )
			g_Screen.ulModesAvail |= MODE_320_200;
	}
	return DDENUMRET_OK;
} /* #OF# EnumModesCallback */

/*========================================================
Function : GetScreenRes
=========================================================*/
/* #FN#
   Returns the width and height of current screen mode */
//static
//void
/* #AS#
   Nothing */
//GetScreenRes(
//	int *pWidth, /* #OUT# */
//	int *pHeight /* #OUT# */
//)
//{
//	DWORD anParams[ WIND_SUP_MODES_NO + FULL_SUP_MODES_NO ][ 3 ] =
//	{
//		{SM_WRES_NORMAL,  336,240}, {SM_WRES_DOUBLE,   672,480 },
//		{SM_FRES_320_200, 320,200}, {SM_FRES_320_240,  320,240 },
//		{SM_FRES_400_300, 400,300}, {SM_FRES_512_384,  512,384 },
//		{SM_FRES_640_400, 640,400}, {SM_FRES_640_480,  640,480 },
//		{SM_FRES_800_600, 800,600}, {SM_FRES_1024_768, 1024,768}
//	};
//	int i = 0, j = WIND_SUP_MODES_NO;
//
//	if( g_Screen.ulMode & SM_MODE_FULL )
//	{
//		i  = WIND_SUP_MODES_NO;
//		j += FULL_SUP_MODES_NO;
//	}
//	for( ; i < j; i++ )
//		if( g_Screen.ulMode & anParams[ i ][ 0 ] )
//		{
//			*pWidth  = anParams[ i ][ 1 ];
//			*pHeight = anParams[ i ][ 2 ];
//		}
//} /* #OF# GetScreenRes */

/*========================================================
Function : SetupPalette
=========================================================*/
/* #FN#
   Creates and attaches a palette to DirectDraw surfaces */
static
int
/* #AS#
   Nonzero if succeeded, otherwise 0 */
SetupPalette( void )
{
	if( g_Screen.ulMode & SM_MODE_FULL )
	{
		HRESULT hResult;

		if( s_DDraw.lpDirectDraw )
		{
			if( s_DDraw.lpDDPalette )
			{
				/* Release Palette Object */
				IDirectDrawPalette_Release( s_DDraw.lpDDPalette );
				s_DDraw.lpDDPalette = NULL;
			}
			hResult = IDirectDraw_CreatePalette( s_DDraw.lpDirectDraw, DDPCAPS_8BIT | DDPCAPS_ALLOW256, g_Screen.Pal.pe, &s_DDraw.lpDDPalette, NULL );
			if( FAILED(hResult) )
			{
				ServeDDError( IDS_DDERR_ALLOC_PALETTE, hResult, FALSE );
				return 0;
			}
		}
		else
			return 0; /* Something went wrong, e.g. registry are corrupted */

		/* This avoids some glitches that may appear during the palette changing */
//		IDirectDraw_WaitForVerticalBlank( s_DDraw.lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );

		if( s_DDraw.lpDDSPrimary )
		{
			IDirectDrawSurface_SetPalette( s_DDraw.lpDDSPrimary, NULL );

			hResult = IDirectDrawSurface_SetPalette( s_DDraw.lpDDSPrimary, s_DDraw.lpDDPalette );

			if( DDERR_SURFACELOST == hResult && RestoreSurfaces() )
				hResult = IDirectDrawSurface_SetPalette( s_DDraw.lpDDSPrimary, s_DDraw.lpDDPalette );

			if( FAILED(hResult) )
			{
				ServeDDError( IDS_DDERR_SETPAL_PRIMARY, hResult, FALSE );
				return 0;
			}
		}
		if( s_DDraw.lpDDSOffscrn )
		{
			IDirectDrawSurface_SetPalette( s_DDraw.lpDDSOffscrn, NULL );
			/* Don't actually want to set a palette for this if it's windowed
			   DDRAW, because that's what we use this surface for - a colorspace
			   conversion to whatever the primary surface is via StretchDIB */
			hResult = IDirectDrawSurface_SetPalette( s_DDraw.lpDDSOffscrn, s_DDraw.lpDDPalette );

			if( DDERR_SURFACELOST == hResult && RestoreSurfaces() )
				hResult = IDirectDrawSurface_SetPalette( s_DDraw.lpDDSOffscrn, s_DDraw.lpDDPalette );

			if( FAILED(hResult) )
			{
				ServeDDError( IDS_DDERR_SETPAL_MEMORY, hResult, FALSE );
				return 0;
			}
		}
		/* Using of this may seem questionable but helps under 2000 with double buffering */
		IDirectDrawPalette_SetEntries( s_DDraw.lpDDPalette, 0, 0, PAL_ENTRIES_NO, g_Screen.Pal.pe );
	}
	return 1;
} /* #OF# SetupPalette */

/*========================================================
Function : Screen_UseSystemPalette
=========================================================*/
/* #FN#
   Forces using system palette in full-screen modes */
BOOL
/* #AS#
   Nothing */
Screen_UseSystemPalette( void )
{
	if( g_Screen.ulMode & SM_MODE_FULL )
	{
		if( g_Screen.ulMode & SM_OPTN_USE_SYSPAL && !g_Screen.Pal.bUseSysCol )
		{
			/* Copy system palette entries, so menus/dialog look well in 8-bit displays */
//			memcpy( &g_Screen.Pal.pe[   0 ], &s_apeSystemColours[  0 ], sizeof(PALETTEENTRY) * 10 );
//			memcpy( &g_Screen.Pal.pe[ 246 ], &s_apeSystemColours[ 10 ], sizeof(PALETTEENTRY) * 10 );

			/* We do not need all the system colors; six of them will be enough */
			memcpy( &g_Screen.Pal.pe[   1 ], &s_apeSystemColours[  1 ], sizeof(PALETTEENTRY) );
			memcpy( &g_Screen.Pal.pe[   4 ], &s_apeSystemColours[  4 ], sizeof(PALETTEENTRY) );
			memcpy( &g_Screen.Pal.pe[   7 ], &s_apeSystemColours[  7 ], sizeof(PALETTEENTRY) );

			memcpy( &g_Screen.Pal.pe[ 247 ], &s_apeSystemColours[ 11 ], sizeof(PALETTEENTRY) * 2 );
			memcpy( &g_Screen.Pal.pe[ 255 ], &s_apeSystemColours[ 19 ], sizeof(PALETTEENTRY) );

			if( s_DDraw.lpDDPalette )
				IDirectDrawPalette_SetEntries( s_DDraw.lpDDPalette, 0, 0, PAL_ENTRIES_NO, g_Screen.Pal.pe );

			return (g_Screen.Pal.bUseSysCol = TRUE);
		}
	}
	return FALSE;
} /* #OF# Screen_UseSystemPalette */

/*========================================================
Function : Screen_UseAtariPalette
=========================================================*/
/* #FN#
   Forces using Atari800 palette in Windows environment */
BOOL
/* #AS#
   Nothing */
Screen_UseAtariPalette(
	BOOL bForceUse
)
{
	_ASSERT(g_Screen.lpbmi);

	if( bForceUse || g_Screen.Pal.bUseSysCol )
	{
		int	nRGB;
		int i;

		for( i = 0; i < PAL_ENTRIES_NO; i++ )
		{
			/* Get color value from kernel table */
			nRGB = colortable[ i ];

			g_Screen.lpbmi->bmiColors[ i ].rgbRed      = g_Screen.Pal.pe[ i ].peRed   = (nRGB & 0x00ff0000) >> 16;
			g_Screen.lpbmi->bmiColors[ i ].rgbGreen    = g_Screen.Pal.pe[ i ].peGreen = (nRGB & 0x0000ff00) >> 8;
			g_Screen.lpbmi->bmiColors[ i ].rgbBlue     = g_Screen.Pal.pe[ i ].peBlue  =  nRGB & 0x000000ff;
			g_Screen.lpbmi->bmiColors[ i ].rgbReserved = g_Screen.Pal.pe[ i ].peFlags =  0;
		}
		/* Make the changes visible for DirectDraw full-screen modes */
		if( g_Screen.ulMode & SM_MODE_FULL && s_DDraw.lpDDPalette )
			IDirectDrawPalette_SetEntries( s_DDraw.lpDDPalette, 0, 0, PAL_ENTRIES_NO, g_Screen.Pal.pe );

		return !(g_Screen.Pal.bUseSysCol = FALSE);
	}
	return FALSE;
} /* #OF# Screen_UseAtariPalette */

/*========================================================
Function : Screen_FlipToGDI
=========================================================*/
/* #FN#
   Makes the surface that GDI writes to the primary surface */
void
/* #AS#
   Nothing */
Screen_FlipToGDI(
	BOOL bUseSysPal
)
{
	if( s_DDraw.bFlipBuffers && s_DDraw.lpDirectDraw )
	{
		/* Flip to GDI surface if double-buffered mode is active */
		IDirectDraw_FlipToGDISurface( s_DDraw.lpDirectDraw );
	}
	if( bUseSysPal )
		/* Set the standard sytem colors */
		Screen_UseSystemPalette();
} /* #OF# Screen_FlipToGDI */

/*========================================================
Function : Screen_ShowMenuBar
=========================================================*/
/* #FN#
   Redraws a non-client area (menu bar, status bar and borders) */
void
/* #AS#
   Nothing */
Screen_ShowMenuBar(
	BOOL bForceShow /* #IN# Forces displaying the menu */
)
{
	if( g_hMainWnd && g_Screen.ulMode & SM_MODE_FULL &&
		(bForceShow || !ST_MENUBAR_HIDDEN) )
	{
//		DrawMenuBar( g_hMainWnd );
		/* We need menu with borders and status bar on the screen */
		RedrawWindow( g_hMainWnd,
					  NULL, NULL,
					  RDW_FRAME |
					  RDW_INVALIDATE |
					  RDW_UPDATENOW |
					  RDW_NOERASE );
	}
} /* #OF# Screen_ShowMenuBar */

/*========================================================
Function : Screen_PrepareRedraw
=========================================================*/
/* #FN#
   Prepares a GDI bitmap for repairing screen damages */
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
Screen_PrepareRedraw(
	BOOL bForcePrep /* #IN# Forces preparing a bitmap */
)
{
	BOOL bResult = FALSE;

	if( g_Screen.ulMode & SM_MODE_FULL )
	{
		if( bForcePrep || s_Redraw.hDC == NULL )
		{
			HDC hDC = NULL;	/* GDI-compatible device context for the surface */
			RECT rc;

			/* Release the redrawing stuff if necessary */
			Screen_FreeRedraw();

			GetWindowRect( g_hViewWnd, &rc );
			rc.left += CX_BORDER; rc.top += CY_BORDER;

//			GetScreenRes( &s_Redraw.nWidth, &s_Redraw.nHeight );
			s_Redraw.nWidth  = rc.right - rc.left - CX_BORDER;
			s_Redraw.nHeight = rc.bottom - rc.top - CY_BORDER;

			if( hDC = GetDC( g_hViewWnd ) )
			{
				if( (s_Redraw.hBmp = CreateCompatibleBitmap( hDC, s_Redraw.nWidth, s_Redraw.nHeight )) != NULL )
				{
					/* The bitmap is empty, so let's copy the contents of the surface to it.
					   For that we need to select it into a device context. We create one. */
					if( (s_Redraw.hDC = CreateCompatibleDC( hDC )) != NULL )
					{
						/* Select OffscrBmp into OffscrDC */
						HBITMAP hOldBmp = (HBITMAP)SelectObject( s_Redraw.hDC, s_Redraw.hBmp );
						/* Now we can copy the contents of the surface to the offscreen bitmap */
						BitBlt( s_Redraw.hDC, 0, 0, s_Redraw.nWidth, s_Redraw.nHeight, hDC, 0, 0 /*rc.left, rc.top*/, SRCCOPY );
						/* De-select OffscrBmp */
						SelectObject( s_Redraw.hDC, hOldBmp );

						bResult = TRUE;
					}
					else
						Screen_FreeRedraw();
				}
				/* We do not need hDC anymore. Free it */
				ReleaseDC( g_hViewWnd, hDC );
			}
		}
	}
	return bResult;
} /* #OF# Screen_PrepareRedraw */

/*========================================================
Function : Screen_DrawPaused
=========================================================*/
/* #FN#
   Paints paused screen after changing a display mode */
void
/* #AS#
   Nothing */
Screen_DrawPaused(
	BOOL bForceMenu, /* #IN# Forces displaying the menu */
	BOOL bUseSysPal, /* #IN# Forces setting the system palette */
	BOOL bForcePrep,
	BOOL bRedraw
)
{
	if( g_ulAtariState & ATARI_PAUSED && s_bModeChanged )
	{
		if( bRedraw )
			DrawPaused( (UBYTE *)Screen_GetBuffer() );

		Screen_FlipToGDI( bUseSysPal );
		Screen_PrepareRedraw( bForcePrep );

		if( bForceMenu || ST_MENUBAR_HIDDEN )
			Screen_ShowMenuBar( TRUE );
	}
} /* #OF# Screen_DrawPaused */

/*========================================================
Function : Screen_DrawFrozen
=========================================================*/
/* #FN#
   Paints lastly built screen */
void
/* #AS#
   Nothing */
Screen_DrawFrozen(
	BOOL bForceMenu, /* #IN# Forces displaying the menu */
	BOOL bUseSysPal, /* #IN# Forces setting the system palette */
	BOOL bForcePrep,
	BOOL bRedraw
)
{
	if( bRedraw && !(g_ulAtariState & ATARI_UNINITIALIZED) )
	{
		Atari_DisplayScreen( (UBYTE *)Screen_GetBuffer() );
		if( s_DDraw.bFlipBuffers )
			/* We have to redraw the screen also on a GDI surface */
			Atari_DisplayScreen( (UBYTE *)Screen_GetBuffer() );
	}
	Screen_FlipToGDI( bUseSysPal );
	Screen_PrepareRedraw( bForcePrep );

	if( bForceMenu || ST_MENUBAR_HIDDEN )
		Screen_ShowMenuBar( TRUE );
} /* #OF# Screen_DrawFrozen */

/*========================================================
Function : Screen_Redraw
=========================================================*/
/* #FN#
   Redraws an emulated Atari screen */
void
/* #AS#
   Nothing */
Screen_Redraw(
	HDC hDC /* #IN# Handle of the device context to be used for rendering an image */
)
{
	/* For windowed modes we may redraw the screen in the very simple way */
	if( g_Screen.ulMode & SM_MODE_WIND )
	{
		HDC hBackDC = g_Screen.hDC;
		g_Screen.hDC = hDC;

		Atari_DisplayScreen( (UBYTE *)Screen_GetBuffer() );

		g_Screen.hDC = hBackDC;
	}
	/* For DirectDraw fullscreen modes we have to use a prepared bitmap */
	else if( NULL != s_Redraw.hBmp )
	{
		/* Select OffscrBmp into OffscrDC */
		HBITMAP hOldBmp = (HBITMAP)SelectObject( s_Redraw.hDC, s_Redraw.hBmp );
		/* Now we can copy the contents of the surface to the offscreen bitmap */
		BitBlt( hDC, 0, 0, s_Redraw.nWidth, s_Redraw.nHeight, s_Redraw.hDC, 0, 0, SRCCOPY );
		/* De-select OffscrBmp */
		SelectObject( s_Redraw.hDC, hOldBmp );
	}
} /* #OF# Screen_Redraw */

/*========================================================
Function : Screen_SetWindowSize
=========================================================*/
/* #FN#
   Sets main window size and position */
void
/* #AS#
   TRUE if succeeded, otherwise FALSE */
Screen_SetWindowSize(
	HWND hView,
	UINT nFlags
)
{
	if( g_hMainWnd )
	{
		int  nMenuHeight = GetSystemMetrics( SM_CYMENU );
		int  nHeight = ATARI_HEIGHT;
		int  nWidth  = ATARI_VIS_WIDTH;
		RECT rc;

		if( g_Screen.ulMode & SM_WRES_DOUBLE )
		{
			nHeight = ATARI_DOUBLE_HEIGHT;
			nWidth  = ATARI_DOUBLE_VIS_WIDTH;
		}
		if( !hView )
			hView = g_hViewWnd;

		/* Set main window size and position */
		for( ;; )
		{
			SetWindowPos( g_hMainWnd, HWND_NOTOPMOST, g_nStartX, g_nStartY,
				nWidth +
					GetSystemMetrics( SM_CXDLGFRAME ) * 2 +
					GetSystemMetrics( SM_CXEDGE ) * 2,
				nHeight + g_Screen.nStatusSize +
					nMenuHeight +
					GetSystemMetrics( SM_CYCAPTION ) +
					GetSystemMetrics( SM_CYDLGFRAME ) * 2 +
					GetSystemMetrics( SM_CYEDGE ) * 2,
				nFlags );

			if( hView )
			{
				/* Add extra space to the window's height when a menu bar wraps
				   to two or more rows; GetMenuBarInfo and GetMenuInfo require
				   Windows 98 or "better"... */
				GetClientRect( hView, &rc );

				if( rc.bottom != nHeight )
					nMenuHeight += nHeight - rc.bottom;
				else
					break;
			}
			else
				break;
		}
	}
} /* #OF# Screen_SetWindowSize */

/*========================================================
Function : SetupWindowedDisplay
=========================================================*/
/* #FN#
   Sets up windowed mode display */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
SetupWindowedDisplay( void )
{
	_ASSERT(g_Screen.ulMode & SM_MODE_WIND);

	/* If we are recovering from a DirectDraw mode, the window frame will be
	   whatever size that mode was instead of what we want it to be, so set
	   window size here */
	Screen_SetWindowSize( g_hViewWnd, SWP_SHOWWINDOW | SWP_NOSENDCHANGING );

	s_rcSource.left   = 0;
	s_rcSource.top    = 0;
	s_rcSource.right  = ATARI_VIS_WIDTH;
	s_rcSource.bottom = ATARI_HEIGHT;

	g_Screen.lpbmi->bmiHeader.biWidth  =  ATARI_VIS_WIDTH;
	g_Screen.lpbmi->bmiHeader.biHeight = -ATARI_HEIGHT;	/* Negative because we are a top-down bitmap */

	if( g_Screen.ulMode & SM_OPTN_USE_GDI )
	{
		/* Specifics for non-DirectDraw windowed modes */
		if( g_Screen.ulMode & SM_WRES_DOUBLE )
		{
			g_Screen.ulMode |= SM_ATTR_STRETCHED;

			g_Screen.lpbmi->bmiHeader.biWidth  =  ATARI_DOUBLE_VIS_WIDTH;
			g_Screen.lpbmi->bmiHeader.biHeight = -ATARI_DOUBLE_HEIGHT;

			if( STRETCH_PIXELDOUBLING != g_Screen.nStretchMode )
			{
				if( STRETCH_INTERPOLATION == g_Screen.nStretchMode )
				{
					Atari_DisplayScreen = Screen_GDI_Double_Interpolation;
				}
				else
					Atari_DisplayScreen = Screen_GDI_Double_Scanlines;
			}
			else
				Atari_DisplayScreen = Screen_GDI_Double;
		}
		else
			Atari_DisplayScreen = Screen_GDI;
	}
	else
	{
		/* Specifics for DirectDraw windowed modes */
		if( g_Screen.ulMode & SM_WRES_DOUBLE )
		{
			g_Screen.ulMode |= SM_ATTR_STRETCHED;

			if( STRETCH_PIXELDOUBLING != g_Screen.nStretchMode )
			{
				g_Screen.lpbmi->bmiHeader.biHeight = -ATARI_DOUBLE_HEIGHT;
				s_rcSource.bottom = ATARI_DOUBLE_HEIGHT;
				s_Surface.nHeight *= 2;

				if( STRETCH_INTERPOLATION == g_Screen.nStretchMode )
				{
					g_Screen.lpbmi->bmiHeader.biWidth = ATARI_DOUBLE_VIS_WIDTH;
					s_rcSource.right = ATARI_DOUBLE_VIS_WIDTH;
					s_Surface.nWidth = ATARI_DOUBLE_VIS_WIDTH;

					Atari_DisplayScreen = Screen_DDraw_Double_Interpolation;
				}
				else
					Atari_DisplayScreen = Screen_DDraw_Double_Scanlines;
			}
			else
				Atari_DisplayScreen = Screen_DDraw_Double;
		}
		else
			Atari_DisplayScreen = Screen_DDraw;

		Screen_ComputeClipArea( g_hViewWnd );
	}
	/* Set the disk drive activity LED offset */
	s_ulLEDOffset = (ATARI_HEIGHT - DISKLED_FONT_HEIGHT) * ATARI_WIDTH + ATARI_WIDTH - ATARI_HORZ_CLIP - DISKLED_FONT_WIDTH;

	return TRUE;
} /* #OF# SetupWindowedDisplay */

/*========================================================
Function : SetupFullScreenDisplay
=========================================================*/
/* #FN#
   Sets up DirectDraw full screen mode display */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
SetupFullScreenDisplay( void )
{
	HRESULT hResult;

	_ASSERT(g_Screen.ulMode & SM_MODE_FULL);

	Atari_DisplayScreen = Screen_DDraw_Full_Blt;

	if( g_Screen.ulMode & SM_FRES_320_200 )
	{
		hResult = IDirectDraw_SetDisplayMode( s_DDraw.lpDirectDraw, 320, 200, 8 );
		if( FAILED(hResult) )
		{
			ServeDDError( IDS_DDERR_INIT_RES, hResult, FALSE );
			return FALSE;
		}
		s_rcDestin.left   = 0;
		s_rcDestin.top    = 0;
		s_rcDestin.right  = 320;
		s_rcDestin.bottom = 200;
		s_rcSource.left   = ATARI_HORZ_CLIP + ((ATARI_VIS_WIDTH - 320) >> 1);
		s_rcSource.top    = (ATARI_HEIGHT - 200) >> 1;
		s_rcSource.right  = ATARI_WIDTH - s_rcSource.left;
		s_rcSource.bottom = ATARI_HEIGHT - s_rcSource.top;

		g_Screen.ulMode |= SM_ATTR_SMALL_DLG | SM_ATTR_NO_MENU;
		/* Set the disk drive activity LED offset */
		s_ulLEDOffset = (s_rcSource.bottom - DISKLED_FONT_HEIGHT) * ATARI_WIDTH + s_rcSource.right - DISKLED_FONT_WIDTH;
	}
	else
	if( g_Screen.ulMode & SM_FRES_320_240 )
	{
		hResult = IDirectDraw_SetDisplayMode( s_DDraw.lpDirectDraw, 320, 240, 8 );
		if( FAILED(hResult) )
		{
			ServeDDError( IDS_DDERR_INIT_RES, hResult, FALSE );
			return FALSE;
		}
		s_rcDestin.left   = 0;
		s_rcDestin.top    = 0;
		s_rcDestin.right  = 320;
		s_rcDestin.bottom = 240;
		s_rcSource.left   = ATARI_HORZ_CLIP + ((ATARI_VIS_WIDTH - 320) >> 1);
		s_rcSource.top    = 0;
		s_rcSource.right  = ATARI_WIDTH - s_rcSource.left;
		s_rcSource.bottom = ATARI_HEIGHT;

		g_Screen.ulMode |= SM_ATTR_SMALL_DLG | SM_ATTR_NO_MENU;
		/* Set the disk drive activity LED offset */
		s_ulLEDOffset = (s_rcSource.bottom - DISKLED_FONT_HEIGHT) * ATARI_WIDTH + s_rcSource.right - DISKLED_FONT_WIDTH;
	}
	else
	if( g_Screen.ulMode & SM_FRES_320_400 )
	{
		hResult = IDirectDraw_SetDisplayMode( s_DDraw.lpDirectDraw, 320, 400, 8 );
		if( FAILED(hResult) )
		{
			ServeDDError( IDS_DDERR_INIT_RES, hResult, FALSE );
			return FALSE;
		}
		s_rcDestin.left   = 0;
		s_rcDestin.top    = 0;
		s_rcDestin.right  = 320;
		s_rcDestin.bottom = 400;
		s_rcSource.left   = ATARI_HORZ_CLIP + ((ATARI_VIS_WIDTH - 320) >> 1);
		s_rcSource.top    = (ATARI_HEIGHT - 200) >> 1;
		s_rcSource.right  = ATARI_WIDTH - s_rcSource.left;
		s_rcSource.bottom = ATARI_HEIGHT - s_rcSource.top;

		g_Screen.ulMode |= SM_ATTR_SMALL_DLG | SM_ATTR_STRETCHED | SM_ATTR_NO_MENU;
		/* Set the disk drive activity LED offset */
		s_ulLEDOffset = (s_rcSource.bottom - DISKLED_FONT_HEIGHT) * ATARI_WIDTH + s_rcSource.right - DISKLED_FONT_WIDTH;

		if( STRETCH_PIXELDOUBLING != g_Screen.nStretchMode )
		{
			s_rcSource.top     = (ATARI_DOUBLE_HEIGHT - 400) >> 1;
			s_rcSource.bottom  = ATARI_DOUBLE_HEIGHT - s_rcSource.top;
			s_Surface.nHeight *= 2;

			if( STRETCH_INTERPOLATION == g_Screen.nStretchMode )
			{
				s_rcSource.left  = (ATARI_VIS_WIDTH - 320) >> 1;
				s_rcSource.right = ATARI_VIS_WIDTH - s_rcSource.left;
				s_Surface.nWidth = ATARI_VIS_WIDTH;

				Atari_DisplayScreen = Screen_DDraw_Full_Blt_Interpolation;
			}
			else
				Atari_DisplayScreen = Screen_DDraw_Full_Blt_Scanlines;
		}
	}
	else
	if( g_Screen.ulMode & SM_FRES_400_300 )
	{
		hResult = IDirectDraw_SetDisplayMode( s_DDraw.lpDirectDraw, 400, 300, 8 );
		if( FAILED(hResult) )
		{
			ServeDDError( IDS_DDERR_INIT_RES, hResult, FALSE );
			return FALSE;
		}
		s_rcDestin.left   = (400 - ATARI_VIS_WIDTH) >> 1;
		s_rcDestin.top    = (300 - ATARI_HEIGHT) >> 1;
		s_rcDestin.right  = s_rcDestin.left + ATARI_VIS_WIDTH;
		s_rcDestin.bottom = s_rcDestin.top + ATARI_HEIGHT;
		s_rcSource.left   = ATARI_HORZ_CLIP;
		s_rcSource.top    = 0;
		s_rcSource.right  = ATARI_WIDTH - ATARI_HORZ_CLIP;
		s_rcSource.bottom = ATARI_HEIGHT;

		g_Screen.ulMode |= SM_ATTR_SMALL_DLG;
		/* Set the disk drive activity LED offset */
		s_ulLEDOffset = (s_rcSource.bottom - DISKLED_FONT_HEIGHT) * ATARI_WIDTH + s_rcSource.right - DISKLED_FONT_WIDTH;
	}
	else
	if( g_Screen.ulMode & SM_FRES_512_384 )
	{
		hResult = IDirectDraw_SetDisplayMode( s_DDraw.lpDirectDraw, 512, 384, 8 );
		if( FAILED(hResult) )
		{
			ServeDDError( IDS_DDERR_INIT_RES, hResult, FALSE );
			return FALSE;
		}
		s_rcDestin.left   = (512 - ATARI_VIS_WIDTH) >> 1;
		s_rcDestin.top    = (384 - ATARI_HEIGHT) >> 1;
		s_rcDestin.right  = s_rcDestin.left + ATARI_VIS_WIDTH;
		s_rcDestin.bottom = s_rcDestin.top + ATARI_HEIGHT;
		s_rcSource.left   = ATARI_HORZ_CLIP;
		s_rcSource.top    = 0;
		s_rcSource.right  = ATARI_WIDTH - ATARI_HORZ_CLIP;
		s_rcSource.bottom = ATARI_HEIGHT;

		/* Set the disk drive activity LED offset */
		s_ulLEDOffset = (s_rcSource.bottom - DISKLED_FONT_HEIGHT) * ATARI_WIDTH + s_rcSource.right - DISKLED_FONT_WIDTH;
	}
	else
	if( g_Screen.ulMode & SM_FRES_640_400 )
	{
		hResult = IDirectDraw_SetDisplayMode( s_DDraw.lpDirectDraw, 640, 400, 8 );
		if( FAILED(hResult) )
		{
			ServeDDError( IDS_DDERR_INIT_RES, hResult, FALSE );
			return FALSE;
		}
		s_rcDestin.left   = 0;
		s_rcDestin.top    = 0;
		s_rcDestin.right  = 640;
		s_rcDestin.bottom = 400;
		s_rcSource.left   = ATARI_HORZ_CLIP + ((ATARI_VIS_WIDTH - 320) >> 1);
		s_rcSource.top    = (ATARI_HEIGHT - 200) >> 1;
		s_rcSource.right  = ATARI_WIDTH - s_rcSource.left;
		s_rcSource.bottom = ATARI_HEIGHT - s_rcSource.top;

		g_Screen.ulMode |= SM_ATTR_STRETCHED | SM_ATTR_NO_MENU;
		/* Set the disk drive activity LED offset */
		s_ulLEDOffset = (s_rcSource.bottom - DISKLED_FONT_HEIGHT) * ATARI_WIDTH + s_rcSource.right - DISKLED_FONT_WIDTH;

		if( STRETCH_PIXELDOUBLING != g_Screen.nStretchMode )
		{
			s_rcSource.top     = (ATARI_DOUBLE_HEIGHT - 400) >> 1;
			s_rcSource.bottom  = ATARI_DOUBLE_HEIGHT - s_rcSource.top;
			s_Surface.nHeight *= 2;

			if( STRETCH_INTERPOLATION == g_Screen.nStretchMode )
			{
				s_rcSource.left  = (ATARI_DOUBLE_VIS_WIDTH - 640) >> 1;
				s_rcSource.right = ATARI_DOUBLE_VIS_WIDTH - s_rcSource.left;
				s_Surface.nWidth = ATARI_DOUBLE_VIS_WIDTH;

				Atari_DisplayScreen = Screen_DDraw_Full_Blt_Interpolation;
			}
			else
				Atari_DisplayScreen = Screen_DDraw_Full_Blt_Scanlines;
		}
	}
	else
	if( g_Screen.ulMode & SM_FRES_640_480 )
	{
		hResult = IDirectDraw_SetDisplayMode( s_DDraw.lpDirectDraw, 640, 480, 8 );
		if( FAILED(hResult) )
		{
			ServeDDError( IDS_DDERR_INIT_RES, hResult, FALSE );
			return FALSE;
		}
		s_rcDestin.left   = 0;
		s_rcDestin.top    = 0;
		s_rcDestin.right  = 640;
		s_rcDestin.bottom = 480;
		s_rcSource.left   = ATARI_HORZ_CLIP + ((ATARI_VIS_WIDTH - 320) >> 1);
		s_rcSource.top    = 0;
		s_rcSource.right  = ATARI_WIDTH - s_rcSource.left;
		s_rcSource.bottom = ATARI_HEIGHT;

		g_Screen.ulMode |= SM_ATTR_STRETCHED | SM_ATTR_NO_MENU;
		/* Set the disk drive activity LED offset */
		s_ulLEDOffset = (s_rcSource.bottom - DISKLED_FONT_HEIGHT) * ATARI_WIDTH + s_rcSource.right - DISKLED_FONT_WIDTH;

		if( STRETCH_PIXELDOUBLING != g_Screen.nStretchMode )
		{
			s_rcSource.bottom  = ATARI_DOUBLE_HEIGHT;
			s_Surface.nHeight *= 2;

			if( STRETCH_INTERPOLATION == g_Screen.nStretchMode )
			{
				s_rcSource.left  = (ATARI_DOUBLE_VIS_WIDTH - 640) >> 1;
				s_rcSource.right = ATARI_DOUBLE_VIS_WIDTH - s_rcSource.left;
				s_Surface.nWidth = ATARI_DOUBLE_VIS_WIDTH;

				Atari_DisplayScreen = Screen_DDraw_Full_Blt_Interpolation;
			}
			else
				Atari_DisplayScreen = Screen_DDraw_Full_Blt_Scanlines;
		}
	}
	else
	if( g_Screen.ulMode & SM_FRES_800_600 )
	{
		hResult = IDirectDraw_SetDisplayMode( s_DDraw.lpDirectDraw, 800, 600, 8 );
		if( FAILED(hResult) )
		{
			ServeDDError( IDS_DDERR_INIT_RES, hResult, FALSE );
			return FALSE;
		}
		s_rcDestin.left   = (800 - ATARI_DOUBLE_VIS_WIDTH) >> 1;
		s_rcDestin.top    = (600 - ATARI_DOUBLE_HEIGHT) >> 1;
		s_rcDestin.right  = s_rcDestin.left + ATARI_DOUBLE_VIS_WIDTH;
		s_rcDestin.bottom = s_rcDestin.top + ATARI_DOUBLE_HEIGHT;
		s_rcSource.left   = ATARI_HORZ_CLIP;
		s_rcSource.top    = 0;
		s_rcSource.right  = ATARI_WIDTH - ATARI_HORZ_CLIP;
		s_rcSource.bottom = ATARI_HEIGHT;

		g_Screen.ulMode |= SM_ATTR_STRETCHED;
		/* Set the disk drive activity LED offset */
		s_ulLEDOffset = (s_rcSource.bottom - DISKLED_FONT_HEIGHT) * ATARI_WIDTH + s_rcSource.right - DISKLED_FONT_WIDTH;

		if( STRETCH_PIXELDOUBLING != g_Screen.nStretchMode )
		{
			s_rcSource.bottom  = ATARI_DOUBLE_HEIGHT;
			s_Surface.nHeight *= 2;

			if( STRETCH_INTERPOLATION == g_Screen.nStretchMode )
			{
				s_rcSource.left  = 0;
				s_rcSource.right = ATARI_DOUBLE_VIS_WIDTH;
				s_Surface.nWidth = ATARI_DOUBLE_VIS_WIDTH;

				Atari_DisplayScreen = Screen_DDraw_Full_Blt_Interpolation;
			}
			else
				Atari_DisplayScreen = Screen_DDraw_Full_Blt_Scanlines;
		}
	}
	else
	if( g_Screen.ulMode & SM_FRES_1024_768 )
	{
		hResult = IDirectDraw_SetDisplayMode( s_DDraw.lpDirectDraw, 1024, 768, 8 );
		if( FAILED(hResult) )
		{
			ServeDDError( IDS_DDERR_INIT_RES, hResult, FALSE );
			return FALSE;
		}
		s_rcDestin.left   = (1024 - ATARI_TRIPLE_VIS_WIDTH) >> 1;
		s_rcDestin.top    = (768 - ATARI_TRIPLE_HEIGHT) >> 1;
		s_rcDestin.right  = s_rcDestin.left + ATARI_TRIPLE_VIS_WIDTH;
		s_rcDestin.bottom = s_rcDestin.top + ATARI_TRIPLE_HEIGHT;
		s_rcSource.left   = ATARI_HORZ_CLIP;
		s_rcSource.top    = 0;
		s_rcSource.right  = ATARI_WIDTH - ATARI_HORZ_CLIP;
		s_rcSource.bottom = ATARI_HEIGHT;

		g_Screen.ulMode |= SM_ATTR_STRETCHED;
		/* Set the disk drive activity LED offset */
		s_ulLEDOffset = (s_rcSource.bottom - DISKLED_FONT_HEIGHT) * ATARI_WIDTH + s_rcSource.right - DISKLED_FONT_WIDTH;

		if( STRETCH_PIXELDOUBLING != g_Screen.nStretchMode )
		{
			s_rcSource.bottom  = ATARI_TRIPLE_HEIGHT;
			s_Surface.nHeight *= 3;

			if( STRETCH_INTERPOLATION == g_Screen.nStretchMode )
			{
				s_rcSource.left  = 0;
				s_rcSource.right = ATARI_TRIPLE_VIS_WIDTH;
				s_Surface.nWidth = ATARI_TRIPLE_VIS_WIDTH;

				Atari_DisplayScreen = Screen_DDraw_1024_Interpolation;
			}
			else
				Atari_DisplayScreen = Screen_DDraw_1024_Scanlines;
		}
	}
	return TRUE;
} /* #OF# SetupFullScreenDisplay */

/*========================================================
Function : Screen_ToggleWindowed
=========================================================*/
/* #FN#
   Toggles between a single and double windowed mode */
int
/* #AS#
   Screen_Initialise() return value */
Screen_ToggleWindowed( void )
{
	if( g_Screen.ulMode & SM_WRES_DOUBLE )
	{
		g_Screen.ulMode &= ~SM_WRES_MASK;
		g_Screen.ulMode |= SM_WRES_NORMAL;
	}
	else
	{
		g_Screen.ulMode &= ~SM_WRES_MASK;
		g_Screen.ulMode |= SM_WRES_DOUBLE;
	}
	WriteRegDWORD( NULL, REG_DDRAW_MODE, g_Screen.ulMode );
	/* See CMainFrame::OnSize method */
//	g_nTestVal = (ST_DOUBLE_REFRESH ? g_nDoubleRate : refresh_rate) - 1;

	return Screen_ChangeMode( FALSE );
} /* #OF# Screen_ToggleWindowed */

/*========================================================
Function : Screen_ToggleModes
=========================================================*/
/* #FN#
   Toggles between a windowed and full screen modes */
int
/* #AS#
   Screen_Initialise() return value */
Screen_ToggleModes( void )
{
	if( g_Screen.ulMode & SM_MODE_FULL )
	{
		Screen_ShowMousePointer( TRUE );

		g_Screen.ulMode &= ~SM_MODE_FULL;
		g_Screen.ulMode |= SM_MODE_WIND;
	}
	else
	{
		_ASSERT(g_Screen.ulMode & SM_MODE_WIND);

		g_Screen.ulMode &= ~SM_MODE_WIND;
		g_Screen.ulMode |= SM_MODE_FULL;
	}
	WriteRegDWORD( NULL, REG_DDRAW_MODE, g_Screen.ulMode );
	/* See CMainFrame::OnSize method */
//	g_nTestVal = (ST_DOUBLE_REFRESH ? g_nDoubleRate : refresh_rate) - 1;

	return Screen_ChangeMode( FALSE );
} /* #OF# Screen_ToggleModes */

/*========================================================
Function : Screen_Initialise
=========================================================*/
/* #FN#
   Initializes DirectDraw/GDI display */
int
/* #AS#
   1 if succeeded, otherwise 0 */
Screen_Initialise(
	BOOL bForceInit /* #IN# Forces display initialization */
)
{
//	ULONG ulScreenMode = g_Screen.ulMode;
	HRESULT	hResult;

	if( !s_Buffer.pMainScr )
		s_Buffer.pMainScr = malloc( ATARI_WIDTH * (ATARI_HEIGHT + 16) );
	if( !s_Buffer.pWorkScr )
		s_Buffer.pWorkScr = malloc( ATARI_TRIPLE_VIS_SCREEN_SIZE + 1 );
	if( !s_Buffer.pBackScr )
		s_Buffer.pBackScr = malloc( ATARI_WIDTH * (ATARI_HEIGHT + 16) );

	if( !s_Buffer.pMainScr || !s_Buffer.pWorkScr )
		return 0;

	if( !bForceInit && s_DDraw.lpDirectDraw &&
		s_ulScreenMode == g_Screen.ulMode && s_nStretchMode == g_Screen.nStretchMode && s_nMemoryType == g_Screen.nMemoryType )
		return 1;

	s_bModeChanged  = FALSE;
	s_bModeChanging = TRUE;

	g_Screen.ulMode &= ~SM_ATTR_MASK;

	/* Close Video Output file */
	if( g_Screen.pfOutput )
	{
		/* Description should be changed */
		Video_CloseOutput();
		/* The tray bar has to be redrawn */
		UpdateIndicator( ID_INDICATOR_AVI );
//		RedrawWindow( g_hMainWnd, NULL, NULL, RDW_INVALIDATE | RDW_NOERASE | RDW_FRAME );
	}
	if( s_Buffer.pBackScr && s_Buffer.pSource && !(g_Screen.ulMode & SM_OPTN_SAFE_MODE) )
 		/* Save the generated atari screen */
		CopyMemory( s_Buffer.pBackScr, s_Buffer.pSource, ATARI_WIDTH * (ATARI_HEIGHT + 16) );

	/* There is the same schema of using buffers for all the windowed modes;
	   when in fullscreen the offscreen surface is used as source or target */
	s_Buffer.pSource = s_Buffer.pMainScr;
	s_Buffer.pTarget = s_Buffer.pWorkScr;

	/* Set DirectDraw control params */
	s_DDraw.pfxBltEffects = NULL;
	s_DDraw.bFlipBuffers  = FALSE;
	s_DDraw.dwBlitFlags   = 0;
	s_DDraw.dwFlipFlags   = 0;

	if( g_Screen.ulMode & SM_OPTN_DDBLT_WAIT )
	{
		s_DDraw.dwBlitFlags |= DDBLT_WAIT;
		s_DDraw.dwFlipFlags |= DDFLIP_WAIT;
	}
	/* Generic size of the offscreen surface for the DirectDraw */
	s_Surface.nHeight = ATARI_HEIGHT + 16;
	s_Surface.nWidth  = ATARI_WIDTH;
	s_Surface.nPitch  = 0L;

	if( !s_lWndStyleBack )
		s_lWndStyleBack = GetWindowLong( g_hMainWnd, GWL_STYLE );

	/* Release also DirectDraw object if windowed non-DirectDraw mode is chosen */
	ReleaseAllObjects( g_Screen.ulMode & SM_MODE_WIND && g_Screen.ulMode & SM_OPTN_USE_GDI );

	/* Handle windowed DirectDraw or non-DirectDraw modes */
	if( g_Screen.ulMode & SM_MODE_WIND )
	{
		/* Restore oryginal window style */
		if( g_hMainWnd && s_lWndStyleBack )
			SetWindowLong( g_hMainWnd, GWL_STYLE, s_lWndStyleBack );

		/* Restore display mode */
		if( s_DDraw.lpDirectDraw )
			IDirectDraw_RestoreDisplayMode( s_DDraw.lpDirectDraw );

		/* Specifics for DirectDraw windowed modes */
		if( !(g_Screen.ulMode & SM_OPTN_USE_GDI) )
		{
			if( !CreateDirectDrawObject() )
			{
				g_ulAtariState  = ATARI_UNINITIALIZED;
				s_bModeChanging = FALSE;
				return 0;
			}
			_ASSERT(s_DDraw.lpDDSPrimary == NULL && s_DDraw.lpDDSOffscrn == NULL);
			_ASSERT(s_DDraw.lpDirectDraw != NULL);

			hResult = IDirectDraw_SetCooperativeLevel( s_DDraw.lpDirectDraw, g_hMainWnd, DDSCL_NORMAL | DDSCL_NOWINDOWCHANGES );
			if( FAILED(hResult) )
			{
				ServeDDError( IDS_DDERR_SET_MODE, hResult, FALSE );
				s_bModeChanging = FALSE;
				return 0;
			}
		}
		if( !SetupWindowedDisplay() )
		{
			s_bModeChanging = FALSE;
			return 0;
		}
	}
	else /* Handle fullscreen (exclusive screen) DirectDraw modes */
	{
		_ASSERT(g_Screen.ulMode & SM_MODE_FULL);

		if( !CreateDirectDrawObject() )
		{
			g_ulAtariState = ATARI_UNINITIALIZED;
			s_bModeChanging = FALSE;
			return 0;
		}
		if( g_hMainWnd )
			SetWindowLong( g_hMainWnd, GWL_STYLE, WS_BORDER );

		/* Set DirectDraw cooperative level */
		hResult = IDirectDraw_SetCooperativeLevel( s_DDraw.lpDirectDraw, g_hMainWnd, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN /*| DDSCL_ALLOWREBOOT*/ );
		if( FAILED(hResult) )
		{
			ServeDDError( IDS_DDERR_SET_MODE, hResult, FALSE );
			s_bModeChanging = FALSE;
			return 0;
		}
		if( !SetupFullScreenDisplay() )
		{
			s_bModeChanging = FALSE;
			return 0;
		}
		/* Flipped buffers are available for full screen without menu only */
		s_DDraw.bFlipBuffers = g_Screen.ulMode & SM_OPTN_FLIP_BUFFERS;
	}
	/* Continue handling DirectDraw modes (windowed and fullscreen) */
	if( g_Screen.ulMode & SM_MODE_FULL || !(g_Screen.ulMode & SM_OPTN_USE_GDI) )
	{
		/* Primary surface for DirectDraw modes */
		if( !SetupPrimarySurface() )
		{
			s_bModeChanging = FALSE;
			return 0;
		}
		/* Memory surface is used for a full screen and both the windowed
		   modes now */
		if( !SetupMemorySurface() )
		{
			s_bModeChanging = FALSE;
			return 0;
		}
		/* Set direct pointers to the memory surface */
		if( !SetDirectPointers() )
		{
			s_bModeChanging = FALSE;
			return 0;
		}
		/* Create and set a palette to the created DirectDraw surfaces */
		if( !SetupPalette() )
		{
			s_bModeChanging = FALSE;
			return 0;
		}
		/* DirectDraw special effects have been enabled */
		if( g_Screen.ulMode & SM_DDFX_MASK )
		{
			ULONG ulIndex = (g_Screen.ulMode >> 28) & 0x0f;
			int   i;

			ZeroMemory( &s_DDraw.fxBltEffects, sizeof(DDBLTFX) );
			s_DDraw.fxBltEffects.dwSize = sizeof(DDBLTFX);
			for( i = 0; i < BLIT_EFFECTS_NO; i++ )
				if( ulIndex & (1 << i) )
				{
					s_DDraw.fxBltEffects.dwDDFX = s_DDraw.dwDDFxFlags[ i ];
					break;
				}
			s_DDraw.pfxBltEffects = &s_DDraw.fxBltEffects;
			s_DDraw.dwBlitFlags   |= DDBLT_DDFX;
		}
	}
	/* Set kernel pointer to our screen buffer */
	atari_screen = (ULONG *)s_Buffer.pSource;

	if( g_Misc.ulState & MS_CAPTURE_CTRLESC &&
		g_Screen.ulMode & SM_MODE_WIND && s_ulScreenMode & SM_MODE_FULL )
		/* Re-enable a Ctrl+Esc keystroke capture for a windowed mode we are switched to */
		Input_EnableEscCapture( TRUE );

	if( ST_DOUBLE_BUFFERS && !(g_ulAtariState & ATARI_PAUSED) )
		/* Needed for double-buffering modes to clean the menus */
		Screen_ShowMenuBar( TRUE );

	/* Clean up the screen */
	Screen_Clear( FALSE, FALSE );

	if( s_Buffer.pBackScr && s_Buffer.pSource && !(g_Screen.ulMode & SM_OPTN_SAFE_MODE) )
		/* Restore the generated atari screen */
		CopyMemory( s_Buffer.pSource, s_Buffer.pBackScr, ATARI_WIDTH * (ATARI_HEIGHT + 16) );

	s_ulScreenMode  = g_Screen.ulMode;//ulScreenMode;
	s_nStretchMode  = g_Screen.nStretchMode;
	s_nMemoryType   = g_Screen.nMemoryType;
	s_bModeChanging = FALSE;
	s_bModeChanged  = TRUE;

	return 1;
} /* #OF# Screen_Initialise */

/*========================================================
Function : Screen_ComputeClipArea
=========================================================*/
/* #FN#
   Computes destination area size and position */
void
/* #AS#
   Nothing */
Screen_ComputeClipArea(
	HWND hViewWnd
)
{
	if( NULL == hViewWnd || g_Screen.ulMode & SM_OPTN_USE_GDI || g_Screen.ulMode & SM_MODE_FULL )
		return;	/* Don't need to worry about this for GDI screens */

	GetWindowRect( hViewWnd, &s_rcDestin );

	s_rcDestin.left += GetSystemMetrics( SM_CXEDGE );
	s_rcDestin.top  += GetSystemMetrics( SM_CYEDGE );

	s_rcDestin.right  = s_rcDestin.left + (g_Screen.ulMode & SM_WRES_DOUBLE ? ATARI_DOUBLE_VIS_WIDTH : ATARI_VIS_WIDTH);
	s_rcDestin.bottom = s_rcDestin.top  + (g_Screen.ulMode & SM_WRES_DOUBLE ? ATARI_DOUBLE_HEIGHT : ATARI_HEIGHT);
} /* #OF# Screen_ComputeClipArea */

/*========================================================
Function : Screen_SetSafeDisplay
=========================================================*/
/* #FN#
   Sets safe display screen for compatibility */
int
/* #AS#
   The value returned by Screen_Initialise function */
Screen_SetSafeDisplay(
	BOOL bForceGDI /* #IN# Use GDI flag */
)
{
	int nResult = 0;

	Screen_ShowMousePointer( TRUE );

	g_Screen.ulMode &= ~SM_MODE_MASK;
	/* Set windowed mode */
	g_Screen.ulMode |= SM_MODE_WIND;
	if( bForceGDI )
		g_Screen.ulMode |= SM_OPTN_USE_GDI;

	nResult = Screen_Initialise( TRUE );
	/* If there still are some problems */
	if( !nResult )
	{
		g_Screen.ulMode &= ~SM_WRES_MASK;
		g_Screen.ulMode |= (SM_WRES_NORMAL | SM_OPTN_USE_GDI);

		if( !Screen_Initialise( TRUE ) )
			/* Set normal windowed size */
			Screen_SetWindowSize( g_hViewWnd, SWP_SHOWWINDOW | SWP_NOSENDCHANGING );
	}
	WriteRegDWORD( NULL, REG_DDRAW_MODE, g_Screen.ulMode );
	/* Force screen refreshing */
	g_nTestVal = (ST_DOUBLE_REFRESH ? g_nDoubleRate : refresh_rate) - 1;

	return nResult;
} /* #OF# Screen_SetSafeDisplay */

/*========================================================
Function : Screen_Clear
=========================================================*/
/* #FN#
   Clears screen buffers */
void
/* #AS#
   Nothing */
Screen_Clear(
	BOOL bPermanent, /* #IN# */
	BOOL bDeepClear
)
{
	if( bDeepClear )
	{
		if( s_Buffer.pMainScr && !(g_ulAtariState & ATARI_PAUSED) )
			ZeroMemory( s_Buffer.pMainScr, ATARI_WIDTH * (ATARI_HEIGHT + 16) );
	}
	if( s_Buffer.pWorkScr )
		ZeroMemory( s_Buffer.pWorkScr, ATARI_TRIPLE_VIS_SCREEN_SIZE );

	/* Force screen refreshing */
	g_nTestVal = (ST_DOUBLE_REFRESH ? g_nDoubleRate : refresh_rate) - 1;

	/* Restore the Atari palette if necessary */
	Screen_UseAtariPalette( FALSE );

	if( bDeepClear || (!ST_ATARI_FAILED &&
		/* If paused clear only when switching between modes */
		(!(g_ulAtariState & ATARI_PAUSED) || s_bModeChanging)) )
	{
		DDSURFACEDESC ddsd;
		DDBLTFX ddbltfx;

		ZeroMemory( &ddsd, sizeof(DDSURFACEDESC) );
		ddsd.dwSize = sizeof(DDSURFACEDESC);

		ZeroMemory( &ddbltfx, sizeof(DDBLTFX) );
		ddbltfx.dwSize      = sizeof(DDBLTFX);
		ddbltfx.dwFillColor = 0;

		if( bDeepClear )
		{
			if( s_DDraw.lpDDSOffscrn )
				IDirectDrawSurface_Blt( s_DDraw.lpDDSOffscrn, NULL, NULL, NULL, DDBLT_COLORFILL , &ddbltfx );
		}
		if( g_Screen.ulMode & SM_MODE_FULL )
		{
			if( s_DDraw.lpDDSPrimary )
			{
				IDirectDraw_WaitForVerticalBlank( s_DDraw.lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );

				/* Clear the primary buffer */
				IDirectDrawSurface_Blt( s_DDraw.lpDDSPrimary, NULL, NULL, NULL, DDBLT_COLORFILL , &ddbltfx );

				if( s_DDraw.bFlipBuffers && s_DDraw.lpDDSBackBuf )
				{
					/* Clear the secondary buffer if double-buffered mode is used */
					IDirectDrawSurface_Blt( s_DDraw.lpDDSBackBuf, NULL, NULL, NULL, DDBLT_COLORFILL , &ddbltfx );
				}
				else if( !(g_ulAtariState & ATARI_PAUSED && s_bModeChanging) )
				{
					/* WARNING: There is a non-client area (a menu bar, status bar and borders)
					   to redraw after cleaning when in a full-screen mode with visible menus
					*/
					Screen_ShowMenuBar( FALSE );
				}
			}
		}
	}
	if( bPermanent )
	{
		/* Release DirectDraw Objects */
		if( FAILED(
			ReleaseAllObjects( TRUE )) )
		{
			DisplayMessage( NULL, IDS_DDERR_RELEASE_ALL, 0, MB_ICONEXCLAMATION | MB_OK );
		}
		/* Release allocated buffers */
		if( s_Buffer.pMainScr )
		{
			free( s_Buffer.pMainScr );
			s_Buffer.pMainScr = s_Buffer.pSource = NULL;
		}
		if( s_Buffer.pWorkScr )
		{
			free( s_Buffer.pWorkScr );
			s_Buffer.pWorkScr = s_Buffer.pTarget = NULL;
		}
		if( s_Buffer.pBackScr )
		{
			free( s_Buffer.pBackScr );
			s_Buffer.pBackScr = NULL;
		}
	}
	/* Clear redrawing stuff if necessary */
	if( !(g_ulAtariState & ATARI_PAUSED) || bPermanent )
	{
		/* Release redraw bitmap after using */
		Screen_FreeRedraw();
	}
} /* #OF# Screen_Clear */

/*========================================================
Function : Screen_ChangeMode
=========================================================*/
/* #FN#
   Changes display mode */
int
/* #AS#
   1 if succeeded, otherwise 0 */
Screen_ChangeMode(
	BOOL bForceInit /* #IN# Forces display reinitialization */
)
{
	int nResult = Screen_Initialise( bForceInit );

	if( !nResult )
	{
		Screen_SetSafeDisplay( TRUE );
	}
	/* Refresh paused screen if necessary */
	Screen_DrawPaused( g_Screen.ulMode & SM_MODE_FULL, FALSE, TRUE, TRUE );

	return nResult;
} /* #OF# Screen_ChangeMode */

/*========================================================
Function : Screen_CheckDDrawModes
=========================================================*/
/* #FN#
   Checks DirectDraw modes */
void
/* #AS#
   Nothing */
Screen_CheckDDrawModes( void )
{
	if( !s_DDraw.lpDirectDraw )
	{
		if( SUCCEEDED(DirectDrawCreate( NULL, &s_DDraw.lpDirectDraw, 0 )) )
		{
			IDirectDraw_EnumDisplayModes( s_DDraw.lpDirectDraw, 0, NULL, NULL, EnumModesCallback );
			IDirectDraw_Release( s_DDraw.lpDirectDraw );
		}
		s_DDraw.lpDirectDraw = NULL;
	}
	else
		IDirectDraw_EnumDisplayModes( s_DDraw.lpDirectDraw, 0, NULL, NULL, EnumModesCallback );
} /* #OF# Screen_CheckDDrawModes */

/*========================================================
Function : Screen_ShowMousePointer
=========================================================*/
/* #FN#
   Makes the mouse pointer visible or hidden */
void
/* #AS#
   Nothing */
Screen_ShowMousePointer(
	BOOL bShow /* #IN# Show mouse pointer flag */
)
{
	if( g_Screen.ulMode & SM_MODE_FULL && g_Screen.ulMode & SM_OPTN_HIDE_CURSOR ||
		g_Misc.ulState & MS_MOUSE_CAPTURED )
	{
		if( bShow )
		{
			if( g_Screen.nShowCursor <= 0 )
				ShowCursor( TRUE );

			if( g_Misc.ulState & MS_MOUSE_CAPTURED )
			{
				ClipCursor( NULL );
				g_Screen.nShowCursor = 10;
			}
			else
				g_Screen.nShowCursor = DEF_SHOW_CURSOR;
		}
		else
		{
			ShowCursor( FALSE ); /* Hide the mouse cursor */
			/* Re-capture the cursor */
			Input_RefreshBounds( g_hViewWnd, TRUE );
		}
	}
} /* #OF# Screen_ShowMousePointer */

/*========================================================
Function : Update_PEN
=========================================================*/
/* #FN#
   Draws light pen/light gun pointer on the Atari screen */
_inline
static
void
/* #AS#
   Nothing */
Update_PEN(
	UBYTE *screen /* #IN# Pointer to Atari screen */
)
{
	if( g_Misc.ulState & MS_MOUSE_CAPTURED &&
		/* Draw the light pen cursor if the right mouse button is
		   pressed or the user has checked the appropriate option */
		MOUSE_PEN == g_Input.Mouse.nMode &&
		(g_Input.Mouse.nButtons & 2 || g_Misc.ulState & MS_SHOW_PEN_POINTER) )
	{
		UWORD *ptr = &((UWORD *)screen)[ 12 + g_Input.Mouse.nX + ATARI_WIDTH * g_Input.Mouse.nY ];
		*ptr ^= 0xffff;
		ptr[ ATARI_WIDTH / 2 ] ^= 0xffff;
	}
} /* #OF# Update_PEN */

/*========================================================
Function : Screen_GetBuffer
=========================================================*/
/* #FN#
   Returns pointer to the atari buffer */
UCHAR*
Screen_GetBuffer( void )
{
	return s_Buffer.pSource; /* Screen buffer */
} /* #OF# Screen_GetBuffer */

/*========================================================
Function : Screen_FreeInterp
=========================================================*/
/* #FN#
   Invalidates a interpolation table with palette indexes */
void
/* #AS#
   Nothing */
Screen_FreeInterp( void )
{
	s_bIColorMap = FALSE;
} /* #OF# Screen_FreeInterp */

//#pragma optimize("at", on)

/*========================================================
Function : Screen_PrepareInterp
=========================================================*/
/* #FN#
   Prepares a interpolation table with palette indexes */
void
/* #AS#
   Nothing */
Screen_PrepareInterp(
	BOOL bForcePrep
)
{
	if( bForcePrep ||
		STRETCH_INTERPOLATION == g_Screen.nStretchMode && !s_bIColorMap )
	{
		/* Build a new interpolation map */
		UBYTE pixel;
		int r1, g1, b1, r2, g2, b2;
		int	nRGB;
		int i, j;

		HANDLE       hLogPal  = NULL;  /* Handle to a logical palette */
		LPLOGPALETTE lpPal    = NULL;  /* Pointer to a logical palette */
		HPALETTE     hPalette = NULL;

		hLogPal = GlobalAlloc( GHND/*LPTR*/, sizeof(LOGPALETTE) + sizeof(PALETTEENTRY) * PAL_ENTRIES_NO );
		/* If not enough memory, clean up and return NULL */
		if( !hLogPal )
			return;

		lpPal = (LPLOGPALETTE)GlobalLock( (HGLOBAL)hLogPal );
		/* Init palette header info */
		lpPal->palVersion    = 0x300;
		lpPal->palNumEntries = (WORD)PAL_ENTRIES_NO;

		/* Copy entries */
		for( i = 0; i < PAL_ENTRIES_NO; i++ )
		{
			/* Get color value from kernel table */
			nRGB = colortable[ i ];

			lpPal->palPalEntry[ i ].peRed   = (nRGB & 0x00ff0000) >> 16;
			lpPal->palPalEntry[ i ].peGreen = (nRGB & 0x0000ff00) >> 8;
			lpPal->palPalEntry[ i ].peBlue  =  nRGB & 0x000000ff;
			lpPal->palPalEntry[ i ].peFlags =  0;
		}
		/* Create palette */
		hPalette = CreatePalette( lpPal );

		GlobalUnlock( (HGLOBAL) hLogPal );
		GlobalFree( (HGLOBAL) hLogPal );

		if( hPalette )
		{
			for( i = 0; i < PAL_ENTRIES_NO; i++ )
			{
				r1 = g_Screen.Pal.pe[ i ].peRed;
				g1 = g_Screen.Pal.pe[ i ].peGreen;
				b1 = g_Screen.Pal.pe[ i ].peBlue;

				for( j = i; j < PAL_ENTRIES_NO; j++ )
				{
					if( i == j )
					{
						s_anIColorMap[ i ][ j ] = i;
					}
					else
					{
						r2 = g_Screen.Pal.pe[ j ].peRed;
						g2 = g_Screen.Pal.pe[ j ].peGreen;
						b2 = g_Screen.Pal.pe[ j ].peBlue;

						pixel = GetNearestPaletteIndex( hPalette, RGB((r1 + r2) / 2, (g1 + g2) / 2, (b1 + b2) / 2) );

						s_anIColorMap[ i ][ j ] = pixel;
						s_anIColorMap[ j ][ i ] = pixel;
					}
				}
			}
			DeleteObject( hPalette );
			s_bIColorMap = TRUE;
		}
	}
} /* #OF# Screen_PrepareInterp */

#define	Inter1a( pDst, pSrc1, pSrc2 ) \
{\
	*pDst++ = *pSrc1;\
	/* Fill in the gap with the average of the pixels either side */\
	*pDst++ = s_anIColorMap[ *pSrc1++ ][ *pSrc2++ ];\
}
#define	Inter4a( pDst, pSrc1, pSrc2 ) \
{\
	Inter1a( pDst, pSrc1, pSrc2 );\
	Inter1a( pDst, pSrc1, pSrc2 );\
	Inter1a( pDst, pSrc1, pSrc2 );\
	Inter1a( pDst, pSrc1, pSrc2 );\
}
#define	Inter16a( pDst, pSrc1, pSrc2 ) \
{\
	Inter4a( pDst, pSrc1, pSrc2 );\
	Inter4a( pDst, pSrc1, pSrc2 );\
	Inter4a( pDst, pSrc1, pSrc2 );\
	Inter4a( pDst, pSrc1, pSrc2 );\
}
#define	Inter64a( pDst, pSrc1, pSrc2 ) \
{\
	Inter16a( pDst, pSrc1, pSrc2 );\
	Inter16a( pDst, pSrc1, pSrc2 );\
	Inter16a( pDst, pSrc1, pSrc2 );\
	Inter16a( pDst, pSrc1, pSrc2 );\
}
#define	Inter2Line( pDst, pSrc1, pSrc2 ) \
{\
	Inter64a( pDst, pSrc1, pSrc2 );\
	Inter64a( pDst, pSrc1, pSrc2 );\
	Inter64a( pDst, pSrc1, pSrc2 );\
	Inter64a( pDst, pSrc1, pSrc2 );\
	Inter64a( pDst, pSrc1, pSrc2 );\
	Inter4a( pDst, pSrc1, pSrc2 );\
	Inter4a( pDst, pSrc1, pSrc2 );\
	Inter4a( pDst, pSrc1, pSrc2 );\
	Inter1a( pDst, pSrc1, pSrc2 );\
	Inter1a( pDst, pSrc1, pSrc2 );\
	Inter1a( pDst, pSrc1, pSrc2 );\
	/* The last pixel is a simple copy of the one before it */\
	*pDst = *pSrc1;\
}

/*========================================================
Function : Interpolate2
=========================================================*/
/* #FN#
   Doubles the Atari screen using interpolation */
_inline
static
void
/* #AS#
   Nothing */
Interpolate2(
	int    xs, /* #IN# x pos of the source left-top corner */
	int    ws, /* #IN# Source width       */
	int    hs, /* #IN# Source height      */
	int    wd, /* #IN# Destination width  */
	int    hd, /* #IN# Destination height */
	UBYTE *pSource,
	UBYTE *pTarget,
	int    nPitch,
	BOOL   bVertFlip
)
{
	UBYTE *pSourceLine = (UBYTE*)(pSource + xs);
	UBYTE *pTargetLine = (UBYTE*)pTarget;

	/* Get the scan line widths of each DIB */

	const int nIncS = ws + xs + xs;

	int nIncD1 = nPitch ? nPitch : wd;
	int nIncD2 = nIncD1 + nIncD1;
	int l;

#ifndef PFUSIK_ASM
	UBYTE *pd, *ps1, *ps2;
	int i, j;
#endif

	if( bVertFlip )
	{
		/* Get pointers to the start points in the source and destination
		   DIBs. Note that this will be the bottom left corner of the DIB
		   as the scan lines are reversed in memory */
		pTargetLine = (UBYTE*)(pTarget + (hd - 1) * nIncD1);
		nIncD1 = -nIncD1;
		nIncD2 = -nIncD2;
	}

#ifdef PFUSIK_ASM
	if( wd > ws )
	{
		/* First line is interpolated horizontally:
		   Interpolate ws bytes starting from pSourceLine
		   to 2 * ws bytes starting from pTargetLine */
		_asm
		{
			mov		esi, dword ptr [pSourceLine]		;; [---eax---] [---edx---]
			mov		edi, dword ptr [pTargetLine]		;; al ah -- -- dl dh -- -- bl
			mov		ecx, dword ptr [ws]
			shr		ecx, 1
			dec		ecx
			xor		edx, edx							;; .. .. .. .. 00 00 00 00 ..
			mov		bl, byte ptr [esi]					;; .. .. .. .. 00 00 00 00 AA
			inc		esi
		int_horz:
		;; [esi] BB CC
		;; [edi] AA AB BB BC
			mov		dx, word ptr [esi]					;; .. .. .. .. BB CC 00 00 AA
			mov		al, dl								;; BB .. .. .. BB CC 00 00 AA
			mov		ah, byte ptr [edx + s_anIColorMap]	;; BB BC .. .. BB CC 00 00 AA
			add		esi, 2
			shl		eax, 16								;; 00 00 BB BC BB CC 00 00 AA
			mov		al, bl								;; AA 00 BB BC BB CC 00 00 AA
			mov		bl, dh								;; AA 00 BB BC BB CC 00 00 CC
			mov		dh, al								;; AA 00 BB BC BB AA 00 00 CC
			mov		ah, byte ptr [edx + s_anIColorMap]	;; AA AB BB BC BB AA 00 00 CC
			mov		dword ptr [edi], eax
			add		edi, 4
			dec		ecx
			jnz		int_horz
		;; [esi] BB
		;; [edi] AA AB BB BB
			mov		dl, byte ptr [esi]					;; .. .. .. .. BB .. 00 00 AA
			mov		dh, bl								;; .. .. .. .. BB AA 00 00 AA
			mov		al, dl								;; BB .. .. .. BB AA 00 00 AA
			mov		ah, dl								;; BB BB .. .. BB AA 00 00 AA
			shl		eax, 16								;; 00 00 BB BB BB AA 00 00 AA
			mov		ah, byte ptr [edx + s_anIColorMap]	;; 00 AB BB BB BB AA 00 00 AA
			mov		al, bl								;; AA AB BB BB BB AA 00 00 AA
			mov		dword ptr [edi], eax
		}
		/* Interpolate horizontally and vertically */
		if( g_Screen.ulMode & SM_OPTN_OPTIMIZE_PRO )
		{
			for( l = 1; l < hs; l++ )
			{
				pSourceLine += nIncS;
				_asm
				{
					mov		esi, dword ptr [pSourceLine]	;; The source pointer
					mov		edi, dword ptr [pTargetLine]	;; The dest pointer
					mov		eax, dword ptr [nIncD1]			;; Offset between stretched lines
					mov		ecx, dword ptr [ws]
					push	ebp
					mov		ebp, eax
					shr		ecx, 1
					dec		ecx
					push	ecx
				;; [esi]            AA BB CC
				;; [edi]            PP   PQ   QQ   QR
				;; [edi + ebp]     PPAA PQAB QQBB QRBC
				;; [edi + 2 * ebp]  AA   AB   BB   BC
				int_both_pro:
																		;;           [---eax---] [-------edx-------]
																		;; [bx] [cl] al ah [] [] [dl] [dh] [--] [--]
					movzx	ebx, byte ptr [esi + 1]						;;  bb
					movzx	ecx, byte ptr [esi + 2]						;;       CC
					shl		ebx, 8										;;  BB
					movzx	eax, byte ptr [ebx + ecx + s_anIColorMap]	;;           BC 00 00 00
					shl		eax, 8										;;           00 BC 00 00
					movzx	ecx, byte ptr [edi + 3]						;;       QR
					movzx	edx, byte ptr [eax + ecx + s_anIColorMap]	;;                       QRBC  00   00   00
					shl		edx, 8										;;                        00  QRBC  00   00
					movzx	ecx, byte ptr [edi + 2]						;;       QQ
					mov		dl, byte ptr [ebx + ecx + s_anIColorMap]	;;                       QQBB
					mov		al, bh										;;           BB
					movzx	ecx, byte ptr [esi]							;;       AA
					shl		eax, 8										;;           00 BB BC 00
					mov		al, byte ptr [ebx + ecx + s_anIColorMap]	;;           AB
					shl		eax, 8										;;           00 AB BB BC
					movzx	ebx, byte ptr [edi]							;;  pp
					add		eax, ecx									;;           AA
					shl		ebx, 8										;;  PP
					mov		dword ptr [edi + 2 * ebp], eax				;; Store AA AB BB BC
					movzx	ecx, byte ptr [ebx + ecx + s_anIColorMap]	;;      PPAA
					movzx	ebx, dl										;; qqbb
					shl		edx, 8										;;                        00  QQBB QRBC  00
					shl		ebx, 8										;; QQBB
					add		esi, 2										;; Increment source pointer
					mov		dl, byte ptr [ebx + ecx + s_anIColorMap]	;;                       PQAB
					shl		edx, 8										;;                        00  PQAB QQBB QRBC
					add		edx, ecx									;;                       PPAA
					mov		dword ptr [edi + ebp], edx					;; Store PPAA PQAB QQBB QRBC
					add		edi, 4										;; Increment dest pointer
					dec		[esp]										;; Decrement loop counter
					jnz		int_both_pro

					pop		eax											;; Pop decrement loop counter
				;; [esi]            AA BB
				;; [edi]            PP   PQ   QQ   QQ
				;; [edi + ebp]     PPAA PQAB QQBB QQBB
				;; [edi + 2 * ebp]  AA   AB   BB   BB
					movzx	ebx, byte ptr [esi + 1]						;;  bb
					movzx	ecx, byte ptr [edi + 2]						;;       QQ
					mov		eax, ebx									;;           BB 00 00 00
					shl		ebx, 8										;;  BB
					add		eax, ebx									;;              BB
					movzx	edx, byte ptr [ebx + ecx + s_anIColorMap]	;;                       QQBB  00   00   00
					mov		dh, dl										;;                            QQBB
					movzx	ecx, byte ptr [esi]							;;       AA
					shl		eax, 8										;;           00 BB BB 00
					mov		al, byte ptr [ebx + ecx + s_anIColorMap]	;;           AB
					movzx	ebx, byte ptr [edi]							;;  pp
					shl		eax, 8										;;           00 AB BB BB
					shl		ebx, 8										;;  PP
					add		eax, ecx									;;           AA
					mov		dword ptr [edi + 2 * ebp], eax				;; Store AA AB BB BB
					movzx	ecx, byte ptr [ebx + ecx + s_anIColorMap]	;;      PPAA
					mov		ebx, 0ff00h
					and		ebx, edx									;; QQBB
					shl		edx, 8										;;                        00  QQBB QQBB
					mov		dl, byte ptr [ebx + ecx + s_anIColorMap]	;;                       PQAB QQBB QQBB
					shl		edx, 8										;;                        00  PQAB QQBB QQBB
					add		edx, ecx									;;                       PPAA PQAB QQBB QQBB
					mov		dword ptr [edi + ebp], edx	;; Store PPAA PQAB QQBB QQBB

					pop		ebp
				}
				pTargetLine += nIncD2;
			}
		}
		else
		{
			for( l = 1; l < hs; l++ )
			{
				pSourceLine += nIncS;
				/* Interpolate ws bytes starting from pSourceLine
				   to 2 * ws bytes starting from pTargetLine + 2 * nIncD1
				   Then interpolate these 2 * ws bytes starting from pTargetLine + 2 * nIncD1
				   with 2 * ws bytes starting from pTargetLine
				   to 2 * ws bytes starting from pTargetLine + nIncD1 */
				_asm
				{
					mov		esi, dword ptr [pSourceLine]	;; The source pointer
					mov		edi, dword ptr [pTargetLine]	;; The dest pointer
				;; Here's a little trick:
				;; s_anIColorMap[y][x] is accessed by [ebx + ecx]
				;; cl = x, bh = y
				;; both ecx and ebx contain parts of s_anIColorMap offset,
				;; ecx contains loop count in high word
				;; i.e.
				;; ecx = (s_anIColorMap & 0x0000ff00) + (loopCount << 16) + x;
				;; ebx = (s_anIColorMap & 0xffff00ff) - (loopCount << 16) + (y << 8);
					mov		ecx, dword ptr [ws]			;; Loop count = number of words - 2
					shr		ecx, 1						;; (loop count is tested for < 0, not == 0
					dec		ecx							;; and last word is processed after the loop)
					dec		ecx
					shl		ecx, 16						;; Loop count in high word of ecx
					mov		ebx, offset s_anIColorMap	;; Interpolation lookup table
					mov		ch, bh
					sub		ebx, ecx
					mov		eax, dword ptr [nIncD1]		;; Offset between stretched lines
					push	ebp
					mov		ebp, eax
				;; [esi]            AA BB CC
				;; [edi]            PP   PQ   QQ   QR
				;; [edi + ebp]     PPAA PQAB QQBB QRBC
				;; [edi + 2 * ebp]  AA   AB   BB   BC
				int_both:
														;;           [---eax---] [-------edx-------]
														;; [bh] [cl] al ah [] [] [dl] [dh] [--] [--]
					mov		eax, ebx					;; Copy ebx to eax, so [eax + ecx] can be used
					mov		bh, byte ptr [esi + 1]		;;  BB
					mov		cl, byte ptr [esi + 2]		;;       CC
					mov		ah, byte ptr [ebx + ecx]	;;              BC
					mov		cl, byte ptr [edi + 3]		;;       QR
					mov		dh, byte ptr [eax + ecx]	;;                            QRBC
					mov		cl, byte ptr [edi + 2]		;;       QQ
					mov		dl, byte ptr [ebx + ecx]	;;                       QQBB
					mov		al, bh						;;           BB
					mov		cl, byte ptr [esi]			;;       AA
					shl		eax, 16						;;           00 00 BB BC
					mov		ah, byte ptr [ebx + ecx]	;;              AB
					mov		bh, byte ptr [edi]			;;  PP
					mov		al, cl						;;           AA
					mov		dword ptr [edi + 2 * ebp], eax	;; Store AA AB BB BC
					mov		cl, byte ptr [ebx + ecx]	;;      PPAA
					mov		bh, dl						;; QQBB
					shl		edx, 16						;;                        00   00  QQBB QRBC
					add		esi, 2						;; Increment source pointer
					mov		dh, byte ptr [ebx + ecx]	;;                            PQAB
					add		ebx, 10000h
					mov		dl, cl						;;                       PPAA
					mov		dword ptr [edi + ebp], edx	;; Store PPAA PQAB QQBB QRBC
					add		edi, 4						;; Increment dest pointer
					sub		ecx, 10000h
					jae		int_both
	
				;; [esi]            AA BB
				;; [edi]            PP   PQ   QQ   QQ
				;; [edi + ebp]     PPAA PQAB QQBB QQBB
				;; [edi + 2 * ebp]  AA   AB   BB   BB
					mov		bh, byte ptr [esi + 1]		;;  BB
					mov		cl, byte ptr [edi + 2]		;;       QQ
					mov		ah, bh						;;              BB
					mov		dh, byte ptr [ebx + ecx]	;;                            QQBB
					mov		al, bh						;;           BB
					mov		dl, dh						;;                       QQBB
					mov		cl, byte ptr [esi]			;;       AA
					shl		eax, 16						;;           00 00 BB BB
					mov		ah, byte ptr [ebx + ecx]	;;              AB
					mov		bh, byte ptr [edi]			;;  PP
					mov		al, cl						;;           AA
					mov		dword ptr [edi + 2 * ebp], eax	;; Store AA AB BB BB
					mov		cl, byte ptr [ebx + ecx]	;;      PPAA
					mov		bh, dl						;; QQBB
					shl		edx, 16						;;                        00   00  QQBB QQBB
					mov		dh, byte ptr [ebx + ecx]	;;                            PQAB
					mov		dl, cl						;;                       PPAA
					mov		dword ptr [edi + ebp], edx	;; Store PPAA PQAB QQBB QQBB
					pop		ebp
				}
				pTargetLine += nIncD2;
			}
		}
	}
	else
	{
		/* Interpolate vertically only */
		memcpy( pTargetLine, pSourceLine, wd);

		if( g_Screen.ulMode & SM_OPTN_OPTIMIZE_PRO )
		{
			/* Optimized for Pentium Pro, Pentium II etc. */
			for( l = 1; l < hs; l++ )
			{
				pSourceLine += nIncS;
				memcpy( pTargetLine + nIncD2, pSourceLine, wd);
				_asm
				{
					mov		edi, dword ptr [pTargetLine]
					mov		ebx, dword ptr [nIncD1]
					mov		ecx, dword ptr [wd]
					shr		ecx, 1
				int_vert_pro:
					movzx	esi, byte ptr [edi]
					movzx	edx, byte ptr [edi + 2 * ebx]
					shl		esi, 8
					mov		al, byte ptr [esi + edx + s_anIColorMap]
					mov		byte ptr [edi + ebx], al
					inc		edi
					movzx	esi, byte ptr [edi]
					movzx	edx, byte ptr [edi + 2 * ebx]
					shl		esi, 8
					mov		al, byte ptr [esi + edx + s_anIColorMap]
					mov		byte ptr [edi + ebx], al
					inc		edi
					dec		ecx
					jnz		int_vert_pro
				}
				pTargetLine += nIncD2;
			}
		}
		else
		{
			/* Optimized for Pentium MMX, Pentium, 486, etc. */
			for( l = 1; l < hs; l++ )
			{
				pSourceLine += nIncS;
				/* Move wd bytes starting from pSourceLine
				   to wd bytes starting from pTargetLine + 2 * nIncD1
				   Then interpolate these wd bytes starting from pTargetLine + 2 * nIncD1
				   with wd bytes starting from pTargetLine
				   to wd bytes starting from pTargetLine + nIncD1 */
				_asm
				{
					mov		esi, dword ptr [pSourceLine]		;; [---eax---] [---edx---]
					mov		edi, dword ptr [pTargetLine]		;; al ah -- -- dl dh -- --
					mov		ebx, dword ptr [nIncD1]
					mov		ecx, dword ptr [wd]
					shr		ecx, 1
					xor		eax, eax							;; 00 00 00 00 .. .. .. ..
					xor		edx, edx							;; 00 00 00 00 00 00 00 00
				int_vert:
				;; [esi]           RR SS
				;; [edi]           PP QQ
				;; [edi + ebx]     PR QS
				;; [edi + 2 * ebx] RR SS
					mov		ax, word ptr [esi]					;; RR SS 00 00 .. .. 00 00
					mov		dx, word ptr [edi]					;; RR SS 00 00 PP QQ 00 00
					mov		word ptr [edi + 2 * ebx], ax
					xchg	ah, dl								;; RR PP 00 00 SS QQ 00 00
					add		esi, 2
					mov		al, byte ptr [eax + s_anIColorMap]	;; PR PP 00 00 SS QQ 00 00
					mov		ah, byte ptr [edx + s_anIColorMap]	;; PR QS 00 00 SS QQ 00 00
					mov		word ptr [edi + ebx], ax
					add		edi, 2
					dec		ecx
					jnz		int_vert
				}
				pTargetLine += nIncD2;
			}
		}
	}
	memcpy( pTargetLine + nIncD1, pTargetLine, wd );
#else /* PFUSIK_ASM */
	if( wd > ws )
	{
		/* Process each source line */
		for( l = 0; l < hs; l++ )
		{
			/* Copy the source pixels to every other destination pixel */
			ps1 = pSourceLine;
			ps2 = pSourceLine + 1;
			pd  = pTargetLine;

			Inter2Line( pd, ps1, ps2 );

			pTargetLine += nIncD2;
			pSourceLine += nIncS;
		}
	}
	else
	{
		/* Process each source line */
		for( l = 0; l < hs; l++ )
		{
			/* Copy the source lines */
			memcpy( pTargetLine, pSourceLine, wd );

			pTargetLine += nIncD2;
			pSourceLine += nIncS;
		}
	}
	/* Process the missing lines in the destination */
	pSourceLine = (UBYTE*)(bVertFlip ? (pTarget + (hd - 1) * -nIncD1) : pTarget);

	for( l = 0, j = wd >> 4; l < hs - 1; l++ )
	{
		ps1 = pSourceLine;
		ps2 = pSourceLine + nIncD2;
		pd  = pSourceLine + nIncD1;

		for( i = 0; i < j; i++ )
		{
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
		}
		pSourceLine += nIncD2;
	}
	/* The last line is a simple copy of the one before it */
	memcpy( pSourceLine + (bVertFlip ? -nIncD1 : nIncD1), pSourceLine, wd );
#endif /* PFUSIK_ASM */
} /* #OF# Interpolate2 */

#define	Inter1b( pDst, pSrc1, pSrc2 ) \
{\
	*pDst++ = *pSrc1;\
	/* Fill in the gap with the average of the pixels either side */\
	*pDst++ = s_anIColorMap[ *pSrc1++ ][ *pSrc2++ ];\
	*pDst++ = *(pDst - 1);\
}
#define	Inter4b( pDst, pSrc1, pSrc2 ) \
{\
	Inter1b( pDst, pSrc1, pSrc2 );\
	Inter1b( pDst, pSrc1, pSrc2 );\
	Inter1b( pDst, pSrc1, pSrc2 );\
	Inter1b( pDst, pSrc1, pSrc2 );\
}
#define	Inter16b( pDst, pSrc1, pSrc2 ) \
{\
	Inter4b( pDst, pSrc1, pSrc2 );\
	Inter4b( pDst, pSrc1, pSrc2 );\
	Inter4b( pDst, pSrc1, pSrc2 );\
	Inter4b( pDst, pSrc1, pSrc2 );\
}
#define	Inter64b( pDst, pSrc1, pSrc2 ) \
{\
	Inter16b( pDst, pSrc1, pSrc2 );\
	Inter16b( pDst, pSrc1, pSrc2 );\
	Inter16b( pDst, pSrc1, pSrc2 );\
	Inter16b( pDst, pSrc1, pSrc2 );\
}
#define	Inter3Line( pDst, pSrc1, pSrc2 ) \
{\
	Inter64b( pDst, pSrc1, pSrc2 );\
	Inter64b( pDst, pSrc1, pSrc2 );\
	Inter64b( pDst, pSrc1, pSrc2 );\
	Inter64b( pDst, pSrc1, pSrc2 );\
	Inter64b( pDst, pSrc1, pSrc2 );\
	Inter4b( pDst, pSrc1, pSrc2 );\
	Inter4b( pDst, pSrc1, pSrc2 );\
	Inter4b( pDst, pSrc1, pSrc2 );\
	Inter1b( pDst, pSrc1, pSrc2 );\
	Inter1b( pDst, pSrc1, pSrc2 );\
	Inter1b( pDst, pSrc1, pSrc2 );\
	/* The last pixel is a simple copy of the one before it */\
	*pDst++ = *pSrc1;\
	*pDst   = *pSrc1;\
}

/*========================================================
Function : Interpolate3
=========================================================*/
/* #FN#
   Triples the Atari screen using interpolation */
_inline
static
void
/* #AS#
   Nothing */
Interpolate3(
	int    xs, /* #IN# x pos of the source left-top corner */
	int    ws, /* #IN# Source width       */
	int    hs, /* #IN# Source height      */
	int    wd, /* #IN# Destination width  */
	int    hd, /* #IN# Destination height */
	UBYTE *pSource,
	UBYTE *pTarget,
	int    nPitch
)
{
	UBYTE *pSourceLine = (UBYTE*)(pSource + xs);
	UBYTE *pTargetLine = (UBYTE*)pTarget;

	UBYTE *pd, *ps1, *ps2;

	/* Get the scan line widths of each DIB */
	const int nIncS  = ws + xs + xs;
	const int nIncD1 = nPitch ? nPitch : wd;
	const int nIncD2 = nIncD1 + nIncD1;
	const int nIncD3 = nIncD2 + nIncD1;

	int i, j, l;

	/* Process each source line */
	for( l = 0; l < hs; l++ )
	{
		/* Copy the source pixels to every other destination pixel */
		ps1 = pSourceLine;
		ps2 = pSourceLine + 1;
		pd  = pTargetLine;

		Inter3Line( pd, ps1, ps2 );

		pTargetLine += nIncD3;
		pSourceLine += nIncS;
	}
	/* Process the missing lines in the destination */
	pSourceLine = (UBYTE*)pTarget;

	for( l = 0, j = wd >> 4; l < hs - 1; l++ )
	{
		ps1 = pSourceLine;
		ps2 = pSourceLine + nIncD3;
		pd  = pSourceLine + nIncD2;

		memcpy( pSourceLine + nIncD1, ps1, wd );

		for( i = 0; i < j; i++ )
		{
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
			*pd++ = s_anIColorMap[ *ps1++ ][ *ps2++ ];
		}
		pSourceLine += nIncD3;
	}
	/* The two last lines are a simple copy of the one before them */
	memcpy( pSourceLine + nIncD1, pSourceLine, wd );
	memcpy( pSourceLine + nIncD2, pSourceLine, wd );
} /* #OF# Interpolate3 */

//#pragma optimize("", on)

/*========================================================
Function : LockMemorySurface
=========================================================*/
/* #FN#
   Locks the DirectDraw memory surface for direct accessing to be safe */
static
_inline
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
LockMemorySurface(
	BOOL bLock
)
{
	static DDSURFACEDESC ddsd;
	if( bLock )
	{
		HRESULT hResult;

		ZeroMemory( &ddsd, sizeof(DDSURFACEDESC) );
		ddsd.dwSize = sizeof(DDSURFACEDESC);

		hResult = IDirectDrawSurface_Lock( s_DDraw.lpDDSOffscrn, NULL, &ddsd, DDLOCK_WAIT, NULL );

		if( DDERR_SURFACELOST == hResult && RestoreSurfaces() )
			/* Try to lock the surface once again */
			hResult = IDirectDrawSurface_Lock( s_DDraw.lpDDSOffscrn, NULL, &ddsd, DDLOCK_WAIT, NULL );

		if( FAILED(hResult) )
			return FALSE;

		s_Buffer.pTarget = ddsd.lpSurface;
		s_Surface.nPitch = ddsd.lPitch;
	}
	else
		IDirectDrawSurface_Unlock( s_DDraw.lpDDSOffscrn, &ddsd );

	return TRUE;
} /* #OF# LockMemorySurface */

/*========================================================
Function : Screen_DDraw
=========================================================*/
/* #FN#
   Draws Atari screen in DirectDraw windowed mode x 1 */
static
void
/* #AS#
   Nothing */
Screen_DDraw(
	UBYTE *screen /* #IN# Pointer to Atari screen */
)
{
	HRESULT hResult;
	HDC hdc;

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	if( g_Misc.ulState & MS_SHOW_DRIVE_LED )
		Update_LED( screen + s_ulLEDOffset );
#endif
	Update_PEN( screen );

	/* We have to copy the full size atari screen to another buffer to chop
	   off the left and right overflow (reason: allowing StretchDIB to cut the
	   rectangle for us is VERY slow). It so happens everything involved is
	   DWORD aligned, and the offsets and such are constants, so I'm sticking
	   this asm routine in here... */
	_asm
	{
		mov		esi, dword ptr [s_Buffer.pSource]	;; The source pointer
		add		esi, ATARI_HORZ_CLIP
		mov		edi, dword ptr [s_Buffer.pTarget]	;; The dest pointer
		mov		eax, ATARI_HEIGHT					;; Number of lines
	scan_line:
		mov		ecx, 054h		;; Our count of DWORDs to copy (ATARI_VIS_WIDTH/4)
		rep		movsd			;; Move that string
		add		esi, 030h		;; Add in the offset to the next scan line for src

		dec		eax
		jnz		scan_line
	}
	/* Direct using of a primary buffer spares 10-15% of emulation speed but
	   unfortunately we can't use clipper objects/special effects in this case */
//	if( DD_OK == (hResult = IDirectDrawSurface_GetDC( s_DDraw.lpDDSPrimary, &hdc )) )
//	{
//		StretchDIBits( hdc,
//					   s_rcDestin.left, s_rcDestin.top, s_rcDestin.right - s_rcDestin.left, s_rcDestin.bottom - s_rcDestin.top,
//					   s_rcSource.left, s_rcSource.top, s_rcSource.right - s_rcSource.left, s_rcSource.bottom - s_rcSource.top,
//					   s_Buffer.pTarget, g_Screen.lpbmi, DIB_RGB_COLORS, SRCCOPY);
//
//		IDirectDrawSurface_ReleaseDC( s_DDraw.lpDDSPrimary, hdc );
//	}
	if( DD_OK == (hResult = IDirectDrawSurface_GetDC( s_DDraw.lpDDSOffscrn, &hdc )) )
	{
		/* This StretchDIB basically does only the color space conversion to the memory surface */
		StretchDIBits( hdc,
					   0, 0, ATARI_VIS_WIDTH, ATARI_HEIGHT,
					   0, 0, ATARI_VIS_WIDTH, ATARI_HEIGHT,
					   s_Buffer.pTarget, g_Screen.lpbmi, DIB_RGB_COLORS, SRCCOPY );

		IDirectDrawSurface_ReleaseDC( s_DDraw.lpDDSOffscrn, hdc );
	}
	else
	{
		ServeDDError( IDS_DDERR_SURFACE_LOCK, hResult, FALSE );
		return;
	}
	if( g_Screen.ulMode & SM_OPTN_DDVBL_WAIT )
		IDirectDraw_WaitForVerticalBlank( s_DDraw.lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );
	/* Blit to the primary surface from a memory one */
	if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_DDraw.lpDDSPrimary, &s_rcDestin, s_DDraw.lpDDSOffscrn, &s_rcSource, s_DDraw.dwBlitFlags, s_DDraw.pfxBltEffects )) )
		return;

	if( hResult == DDERR_WASSTILLDRAWING )
		return;

	if( hResult == DDERR_SURFACELOST )
		RestoreSurfaces();
	else
		ServeDDError( IDS_DDERR_SURFACE_BLT, hResult, FALSE );
} /* #OF# Screen_DDraw */

/*========================================================
Function : Screen_DDraw_Double
=========================================================*/
/* #FN#
   Draws Atari screen in DirectDraw windowed mode x 2 */
static
void
/* #AS#
   Nothing */
Screen_DDraw_Double(
	UBYTE *screen /* #IN# Pointer to Atari screen */
)
{
	HRESULT	hResult;
	HDC hdc;

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	if( g_Misc.ulState & MS_SHOW_DRIVE_LED )
		Update_LED( screen + s_ulLEDOffset );
#endif
	Update_PEN( screen );

	_asm
	{
		mov		esi, dword ptr [s_Buffer.pSource]	;; The source pointer
		add		esi, ATARI_HORZ_CLIP
		mov		edi, dword ptr [s_Buffer.pTarget]	;; The dest pointer
		mov		eax, ATARI_HEIGHT					;; Number of lines
	scan_line:
		mov		ecx, 054h		;; Our count of DWORDs to copy (ATARI_VIS_WIDTH/4)
		rep		movsd			;; Move that string
		add		esi, 030h		;; Add in the offset to the next scan line for src

		dec		eax
		jnz		scan_line
	}
	_ASSERT(s_DDraw.lpDDSOffscrn != NULL);

	if( DD_OK == (hResult = IDirectDrawSurface_GetDC( s_DDraw.lpDDSOffscrn, &hdc )) )
	{
		/* This StretchDIB basically does only the color space conversion to the memory surface */
		StretchDIBits( hdc,
					   0, 0, ATARI_VIS_WIDTH, ATARI_HEIGHT,
					   0, 0, ATARI_VIS_WIDTH, ATARI_HEIGHT,
					   s_Buffer.pTarget, g_Screen.lpbmi, DIB_RGB_COLORS, SRCCOPY );

		IDirectDrawSurface_ReleaseDC( s_DDraw.lpDDSOffscrn, hdc );
	}
	else
	{
		ServeDDError( IDS_DDERR_SURFACE_LOCK, hResult, FALSE );
		return;
	}
	if( g_Screen.ulMode & SM_OPTN_DDVBL_WAIT )
		IDirectDraw_WaitForVerticalBlank( s_DDraw.lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );
	/* Blit to the primary surface from a memory one */
	if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_DDraw.lpDDSPrimary, &s_rcDestin, s_DDraw.lpDDSOffscrn, &s_rcSource, s_DDraw.dwBlitFlags, s_DDraw.pfxBltEffects )) )
		return;

	if( hResult == DDERR_WASSTILLDRAWING )
		return;

	if( hResult == DDERR_SURFACELOST )
		RestoreSurfaces();
	else
		ServeDDError( IDS_DDERR_SURFACE_BLT, hResult, FALSE );
} /* #OF# Screen_DDraw_Double */

/*========================================================
Function : Screen_DDraw_Double_Interpolation
=========================================================*/
/* #FN#
   Draws Atari screen in windowed mode x 2 with interpolation */
static
void
/* #AS#
   Nothing */
Screen_DDraw_Double_Interpolation(
	UBYTE *screen /* #IN# Pointer to Atari screen */
)
{
	HRESULT	hResult;
	HDC	hdc;

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	if( g_Misc.ulState & MS_SHOW_DRIVE_LED )
		Update_LED( screen + s_ulLEDOffset );
#endif
	Update_PEN( screen );

	Interpolate2( ATARI_HORZ_CLIP, ATARI_VIS_WIDTH, ATARI_HEIGHT,
				  ATARI_DOUBLE_VIS_WIDTH, ATARI_DOUBLE_HEIGHT,
				  s_Buffer.pSource, s_Buffer.pTarget, 0, FALSE );

	_ASSERT(s_DDraw.lpDDSOffscrn != NULL);

	if( DD_OK == (hResult = IDirectDrawSurface_GetDC( s_DDraw.lpDDSOffscrn, &hdc )) )
	{
		/* This StretchDIB basically does only the color space conversion to the memory surface */
		StretchDIBits( hdc,
					   0, 0, ATARI_DOUBLE_VIS_WIDTH, ATARI_DOUBLE_HEIGHT,
					   0, 0, ATARI_DOUBLE_VIS_WIDTH, ATARI_DOUBLE_HEIGHT,
					   s_Buffer.pTarget, g_Screen.lpbmi, DIB_RGB_COLORS, SRCCOPY );

		IDirectDrawSurface_ReleaseDC( s_DDraw.lpDDSOffscrn, hdc );
	}
	else
	{
		ServeDDError( IDS_DDERR_SURFACE_LOCK, hResult, FALSE );
		return;
	}
	if( g_Screen.ulMode & SM_OPTN_DDVBL_WAIT )
		IDirectDraw_WaitForVerticalBlank( s_DDraw.lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );
	/* Blit to the primary surface from a memory one */
	if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_DDraw.lpDDSPrimary, &s_rcDestin, s_DDraw.lpDDSOffscrn, &s_rcSource, s_DDraw.dwBlitFlags, s_DDraw.pfxBltEffects )) )
		return;

	if( hResult == DDERR_WASSTILLDRAWING )
		return;

	if( hResult == DDERR_SURFACELOST )
		RestoreSurfaces();
	else
		ServeDDError( IDS_DDERR_SURFACE_BLT, hResult, FALSE );
} /* #OF# Screen_DDraw_Double_Interpolation */

/*========================================================
Function : Screen_DDraw_Double_Scanlines
=========================================================*/
/* #FN#
   Draws Atari screen in windowed mode x 2 with scan lines */
static
void
/* #AS#
   Nothing */
Screen_DDraw_Double_Scanlines(
	UBYTE *screen /* #IN# Pointer to Atari screen */
)
{
	HRESULT	hResult;
	HDC	hdc;

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	if( g_Misc.ulState & MS_SHOW_DRIVE_LED )
		Update_LED( screen + s_ulLEDOffset );
#endif
	Update_PEN( screen );

#ifdef PFUSIK_ASM
	if( g_Screen.ulMode & SM_OPTN_USE_MMX )
	{
		static ULONG colMask[2] = {0xf0f0f0f0, 0xf0f0f0f0};
		static ULONG lumMask[2] = {0x07070707, 0x07070707};
		_asm
		{
			mov		esi, dword ptr [s_Buffer.pSource]	;; The source pointer
			add		esi, ATARI_HORZ_CLIP
			mov		edi, dword ptr [s_Buffer.pTarget]	;; The dest pointer
			mov		ebx, ATARI_HEIGHT		;; Number of line pairs (ATARI_HEIGHT)
			mov		edx, ATARI_VIS_WIDTH	;; Offset from normal line to dark line
			movq	mm2, qword ptr [colMask]
			movq	mm3, qword ptr [lumMask]
		mmx_line:
			mov		ecx, 0x15				;; Count of OWORDs (ATARI_VIS_WIDTH/16)
		mmx_oword:
			movq	mm0, qword ptr [esi]	;; Load original pixels
			movq	mm4, qword ptr [esi + 8]
			add		esi, 16					;; Increment source pointer
			movq	mm1, mm0				;; Copy original pixels
			movq	mm5, mm4
			movq	qword ptr [edi], mm0	;; Store original pixels
			movq	qword ptr [edi + 8], mm4
			psrlq	mm1, 1					;; Darken brightness
			psrlq	mm5, 1
			pand	mm0, mm2				;; Extract colors
			pand	mm4, mm2
			pand	mm1, mm3				;; Extract darkened brightness
			pand	mm5, mm3
			por		mm0, mm1				;; Combine color with brightness
			por		mm4, mm5
			movq	qword ptr [edi + edx], mm0	;; Store darkened pixels
			movq	qword ptr [edi + edx + 8], mm4
			add		edi, 16					;; Increment dest pointer
			dec		ecx						;; Decrement QWORD counter
			jnz		mmx_oword				;; Back to start of QWORD operation

			add		esi, 0x30				;; Add in the offset to the next scan line for src
			add		edi, edx				;; Skip over the dark pixels
			dec		ebx
			jnz		mmx_line

			emms							;; Empty MMX state
		}
	}
	else
	{
		_asm
		{
			mov		esi, dword ptr [s_Buffer.pSource]	;; The source pointer
			add		esi, ATARI_HORZ_CLIP
			mov		edi, dword ptr [s_Buffer.pTarget]	;; The dest pointer
			mov		ebx, ATARI_HEIGHT		;; Number of line pairs (ATARI_HEIGHT)
		new_line:
			mov		ecx, 0x54				;; Our count of DWORDs to copy (ATARI_VIS_WIDTH/4)
			rep		movsd					;; Move that string
	
			sub		esi, ATARI_VIS_WIDTH	;; Move back to origin of original scan line
			mov		ecx, 0x54				;; Count of DWORDs (ATARI_VIS_WIDTH/4)
		dark_line:
			mov		eax, dword ptr [esi]	;; Make copy of original pixels
			mov		edx, eax				;; Another copy of original pixels
			and		eax, 0xf0f0f0f0			;; Extract high nybbles - colors
			shr		edx, 1					;; Darken brightness
			and		edx, 0x07070707			;; Extract darkened brightness
			add		eax, edx				;; Combine color with brightness
			add		esi, 4					;; Increment source pointer
			mov		dword ptr [edi], eax	;; Store darkened pixels
			add		edi, 4					;; Increment dest pointer
			dec		ecx						;; Decrement our DWORD counter
			jnz		dark_line				;; Back to start of DWORD operation
	
			add		esi, 0x30				;; Add in the offset to the next scan line for src
			dec		ebx
			jnz		new_line
		}
	}
#else
	_asm
	{
		mov		esi, dword ptr [s_Buffer.pSource]	;; The source pointer
		add		esi, ATARI_HORZ_CLIP
		mov		edi, dword ptr [s_Buffer.pTarget]	;; The dest pointer
		mov		eax, ATARI_HEIGHT					;; Number of line pairs (ATARI_HEIGHT)
	new_line:
		mov		ecx, 054h				;; Our count of DWORDs to copy (ATARI_VIS_WIDTH/4)
		rep		movsd					;; Move that string

		sub		esi, ATARI_VIS_WIDTH	;; Move back to origin of original scan line
		mov		edx, ATARI_VIS_WIDTH	;; The pixel counter
	dark_line:
		mov		bl, byte ptr [esi]		;; Make copy of original pixel
		mov		cl, bl					;; Another copy of original pixel
		and		bl, 0x0f				;; Bottom nybble of this pixel
		shr		bl, 1					;; nybble / 2
		and		cl, 0xf0				;; Top nybble of this pixel
		add		cl, bl					;; Make darkened pixel by adding nybbles
		inc		esi						;; Move to next source pixel
		mov		byte ptr [edi], cl		;; Copy darkened pixel into destination point
		inc		edi
		dec		edx     				;; Decrement our pixel counter
		jnz		dark_line				;; Back to start of pixel operation

		add		esi, 030h				;; Add in the offset to the next scan line for src
		dec		eax
		jnz		new_line
	}
#endif
	_ASSERT(s_DDraw.lpDDSOffscrn != NULL);

	if( DD_OK == (hResult = IDirectDrawSurface_GetDC( s_DDraw.lpDDSOffscrn, &hdc )) )
	{
		/* This StretchDIB basically does only the color space conversion to the memory surface */
		StretchDIBits( hdc,
					   0, 0, ATARI_VIS_WIDTH, ATARI_DOUBLE_HEIGHT,
					   0, 0, ATARI_VIS_WIDTH, ATARI_DOUBLE_HEIGHT,
					   s_Buffer.pTarget, g_Screen.lpbmi, DIB_RGB_COLORS, SRCCOPY );

		IDirectDrawSurface_ReleaseDC( s_DDraw.lpDDSOffscrn, hdc );
	}
	else
	{
		ServeDDError( IDS_DDERR_SURFACE_LOCK, hResult, FALSE );
		return;
	}
	if( g_Screen.ulMode & SM_OPTN_DDVBL_WAIT )
		IDirectDraw_WaitForVerticalBlank( s_DDraw.lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );
	/* Blit to the primary surface from a memory one */
	if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_DDraw.lpDDSPrimary, &s_rcDestin, s_DDraw.lpDDSOffscrn, &s_rcSource, s_DDraw.dwBlitFlags, s_DDraw.pfxBltEffects )) )
		return;

	if( hResult == DDERR_WASSTILLDRAWING )
		return;

	if( hResult == DDERR_SURFACELOST )
		RestoreSurfaces();
	else
		ServeDDError( IDS_DDERR_SURFACE_BLT, hResult, FALSE );
} /* #OF# Screen_DDraw_Double_Scanlines */

/*========================================================
Function : Screen_DDraw_Full_Blt
=========================================================*/
/* #FN#
   Draws Atari screen in full screen mode */
static
void
/* #AS#
   Nothing */
Screen_DDraw_Full_Blt(
	UBYTE *screen /* #IN# Pointer to Atari screen */
)
{
	HRESULT	hResult;

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	if( g_Misc.ulState & MS_SHOW_DRIVE_LED )
		Update_LED( screen + s_ulLEDOffset );
#endif
	Update_PEN( screen );

	if( g_Screen.ulMode & SM_OPTN_SAFE_MODE )
	{
		if( !LockMemorySurface( TRUE ) )
			return;

		if( s_Surface.nPitch != s_Surface.nWidth )
		{
			UCHAR *pSource = s_Buffer.pSource;
			UCHAR *pTarget = s_Buffer.pTarget;
			int i;

			for( i = 0; i < ATARI_HEIGHT; i++ )
			{
				CopyMemory( pTarget, pSource, ATARI_WIDTH );
				pTarget += s_Surface.nPitch;
				pSource += ATARI_WIDTH;
			}
		}
		else
			CopyMemory( s_Buffer.pTarget, s_Buffer.pSource, ATARI_SCREEN_SIZE );

		LockMemorySurface( FALSE );
	}

	if( s_DDraw.bFlipBuffers )
	{
		if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_DDraw.lpDDSBackBuf, &s_rcDestin, s_DDraw.lpDDSOffscrn, &s_rcSource, s_DDraw.dwBlitFlags, s_DDraw.pfxBltEffects )) )
		{
			/* Flip the DirectDraw surfaces */
			if( DD_OK == (hResult = IDirectDrawSurface_Flip( s_DDraw.lpDDSPrimary, NULL, s_DDraw.dwFlipFlags )) )
				return;
		}
	}
	else
	{
		if( g_Screen.ulMode & SM_OPTN_DDVBL_WAIT )
			IDirectDraw_WaitForVerticalBlank( s_DDraw.lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );
		/* Blit to the primary surface from a memory one */
		if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_DDraw.lpDDSPrimary, &s_rcDestin, s_DDraw.lpDDSOffscrn, &s_rcSource, s_DDraw.dwBlitFlags, s_DDraw.pfxBltEffects )) )
			return;
	}
	if( hResult == DDERR_WASSTILLDRAWING )
		return;

	if( hResult == DDERR_SURFACELOST )
		RestoreSurfaces();
	else
		ServeDDError( IDS_DDERR_SURFACE_BLT, hResult, FALSE );
} /* #OF# Screen_DDraw_Full_Blt */

/*========================================================
Function : Screen_DDraw_Full_Blt_Interpolation
=========================================================*/
/* #FN#
   Draws Atari screen in full mode with interpolation */
static
void
/* #AS#
   Nothing */
Screen_DDraw_Full_Blt_Interpolation(
	UBYTE *screen /* #IN# Pointer to Atari screen */
)
{
	HRESULT	hResult;

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	if( g_Misc.ulState & MS_SHOW_DRIVE_LED )
		Update_LED( screen + s_ulLEDOffset );
#endif
	Update_PEN( screen );

	if( MEMORY_SYSTEM != g_Screen.nMemoryType )
	{
		Interpolate2( ATARI_HORZ_CLIP, ATARI_VIS_WIDTH, ATARI_HEIGHT,
					  s_Surface.nWidth, ATARI_DOUBLE_HEIGHT,
					  s_Buffer.pSource, s_Buffer.pWorkScr, 0, FALSE );

		if( g_Screen.ulMode & SM_OPTN_SAFE_MODE )
		{
			if( !LockMemorySurface( TRUE ) )
				return;
		}
		if( s_Surface.nPitch != s_Surface.nWidth )
		{
			UCHAR *pSource = s_Buffer.pWorkScr;
			UCHAR *pTarget = s_Buffer.pTarget;
			int i;

			for( i = 0; i < ATARI_DOUBLE_HEIGHT; i++ )
			{
				CopyMemory( pTarget, pSource, s_Surface.nWidth );
				pTarget += s_Surface.nPitch;
				pSource += s_Surface.nWidth;
			}
		}
		else
			CopyMemory( s_Buffer.pTarget, s_Buffer.pWorkScr, s_Surface.nWidth * ATARI_DOUBLE_HEIGHT );
	}
	else
	{
		if( g_Screen.ulMode & SM_OPTN_SAFE_MODE )
		{
			if( !LockMemorySurface( TRUE ) )
				return;
		}
		Interpolate2( ATARI_HORZ_CLIP, ATARI_VIS_WIDTH, ATARI_HEIGHT,
					  s_Surface.nWidth, ATARI_DOUBLE_HEIGHT,
					  s_Buffer.pSource, s_Buffer.pTarget, s_Surface.nPitch, FALSE );
	}
	if( g_Screen.ulMode & SM_OPTN_SAFE_MODE )
		LockMemorySurface( FALSE );

	if( s_DDraw.bFlipBuffers )
	{
		if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_DDraw.lpDDSBackBuf, &s_rcDestin, s_DDraw.lpDDSOffscrn, &s_rcSource, s_DDraw.dwBlitFlags, s_DDraw.pfxBltEffects )) )
		{
			/* Flip the DirectDraw surfaces */
			if( DD_OK == (hResult = IDirectDrawSurface_Flip( s_DDraw.lpDDSPrimary, NULL, s_DDraw.dwFlipFlags )) )
				return;
		}
	}
	else
	{
		if( g_Screen.ulMode & SM_OPTN_DDVBL_WAIT )
			IDirectDraw_WaitForVerticalBlank( s_DDraw.lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );
		/* Blit to the primary surface from a memory one */
		if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_DDraw.lpDDSPrimary, &s_rcDestin, s_DDraw.lpDDSOffscrn, &s_rcSource, s_DDraw.dwBlitFlags, s_DDraw.pfxBltEffects )) )
			return;
	}
	if( hResult == DDERR_WASSTILLDRAWING )
		return;

	if( hResult == DDERR_SURFACELOST )
		RestoreSurfaces();
	else
		ServeDDError( IDS_DDERR_SURFACE_BLT, hResult, FALSE );
} /* #OF# Screen_DDraw_Full_Blt_Interpolation */

/*========================================================
Function : Screen_DDraw_Full_Blt_Scanlines
=========================================================*/
/* #FN#
   Draws Atari screen in full mode with scan lines */
static
void
/* #AS#
   Nothing */
Screen_DDraw_Full_Blt_Scanlines(
	UBYTE *screen /* #IN# Pointer to Atari screen */
)
{
	HRESULT	hResult;
	int nCache;

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	if( g_Misc.ulState & MS_SHOW_DRIVE_LED )
		Update_LED( screen + s_ulLEDOffset );
#endif
	Update_PEN( screen );

	if( g_Screen.ulMode & SM_OPTN_SAFE_MODE )
	{
		if( !LockMemorySurface( TRUE ) )
			return;
	}
	/* Calculate cache size of the memory surface */
	nCache = s_Surface.nPitch - s_Surface.nWidth;

#ifdef PFUSIK_ASM
	_asm
	{
		mov		esi, dword ptr [s_Buffer.pSource]	;; The source pointer
		add		esi, ATARI_HORZ_CLIP
		mov		edi, dword ptr [s_Buffer.pTarget]	;; The dest pointer
		add		edi, ATARI_HORZ_CLIP
		add		nCache, 0x30			;; Skip margins in dest
		mov		ebx, ATARI_HEIGHT		;; Number of line pairs (ATARI_HEIGHT)
	new_line:
		mov		ecx, 0x54				;; Our count of DWORDs to copy (ATARI_VIS_WIDTH/4)
		rep		movsd					;; Move that string

		sub		esi, ATARI_VIS_WIDTH	;; Move back to origin of original scan line
		add		edi, 0x30				;; Skip margins in dest
		mov		ecx, 0x54				;; Count of DWORDs (ATARI_VIS_WIDTH/4)
	dark_line:
		mov		eax, dword ptr [esi]	;; Make copy of original pixels
		mov		edx, eax				;; Another copy of original pixels
		and		eax, 0xf0f0f0f0			;; Extract high nybbles - colors
		shr		edx, 1					;; Darken brightness
		and		edx, 0x07070707			;; Extract darkened brightness
		add		eax, edx				;; Combine color with brightness
		add		esi, 4					;; Increment source pointer
		mov		dword ptr [edi], eax	;; Store darkened pixels
		add		edi, 4					;; Increment dest pointer
		dec		ecx						;; Decrement our DWORD counter
		jnz		dark_line				;; Back to start of DWORD operation

		add		esi, 0x30				;; Add in the offset to the next scan line for src
		add		edi, nCache
		dec		ebx
		jnz		new_line
	}
#else
	_asm
	{
		mov		esi, dword ptr [s_Buffer.pSource]	;; The source pointer
		mov		edi, dword ptr [s_Buffer.pTarget]	;; The dest pointer
		mov		eax, ATARI_HEIGHT					;; Number of line pairs (ATARI_HEIGHT)
	new_line:
		mov		ecx, 060h				;; Our count of DWORDs to copy (ATARI_WIDTH/4)
		rep		movsd					;; Move that string

		sub		esi, ATARI_WIDTH		;; Move back to origin of original scan line
		mov		edx, ATARI_WIDTH		;; The pixel counter
	dark_line:
		mov		bl, byte ptr [esi]		;; Make copy of original pixel
		mov		cl, bl					;; Another copy of original pixel
		and		bl, 0x0f				;; Bottom nybble of this pixel
		shr		bl, 1					;; nybble / 2
		and		cl, 0xf0				;; Top nybble of this pixel
		add		cl, bl					;; Make darkened pixel by adding nybbles
		inc		esi						;; Move to next source pixel
		mov		byte ptr [edi], cl		;; Copy darkened pixel into destination point
		add		edi, nCache
		dec		edx						;; Decrement our pixel counter
		jnz		dark_line				;; Back to start of pixel operation

		dec		eax
		jnz		new_line
	}
#endif
	if( g_Screen.ulMode & SM_OPTN_SAFE_MODE )
		LockMemorySurface( FALSE );

	if( s_DDraw.bFlipBuffers )
	{
		if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_DDraw.lpDDSBackBuf, &s_rcDestin, s_DDraw.lpDDSOffscrn, &s_rcSource, s_DDraw.dwBlitFlags, s_DDraw.pfxBltEffects )) )
		{
			/* Flip the DirectDraw surfaces */
			if( DD_OK == (hResult = IDirectDrawSurface_Flip( s_DDraw.lpDDSPrimary, NULL, s_DDraw.dwFlipFlags )) )
				return;
		}
	}
	else
	{
		if( g_Screen.ulMode & SM_OPTN_DDVBL_WAIT )
			IDirectDraw_WaitForVerticalBlank( s_DDraw.lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );
		/* Blit to the primary surface from a memory one */
		if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_DDraw.lpDDSPrimary, &s_rcDestin, s_DDraw.lpDDSOffscrn, &s_rcSource, s_DDraw.dwBlitFlags, s_DDraw.pfxBltEffects )) )
			return;
	}
	if( hResult == DDERR_WASSTILLDRAWING )
		return;

	if( hResult == DDERR_SURFACELOST )
		RestoreSurfaces();
	else
		ServeDDError( IDS_DDERR_SURFACE_BLT, hResult, FALSE );
} /* #OF# Screen_DDraw_Full_Blt_Scanlines */

/*========================================================
Function : Screen_DDraw_1024_Interpolation
=========================================================*/
/* #FN#
   Draws Atari screen in full 1024 mode with interpolation */
static
void
/* #AS#
   Nothing */
Screen_DDraw_1024_Interpolation(
	UBYTE *screen /* #IN# Pointer to Atari screen */
)
{
	HRESULT	hResult;

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	if( g_Misc.ulState & MS_SHOW_DRIVE_LED )
		Update_LED( screen + s_ulLEDOffset );
#endif
	Update_PEN( screen );

	if( MEMORY_SYSTEM != g_Screen.nMemoryType )
	{
		Interpolate3( ATARI_HORZ_CLIP, ATARI_VIS_WIDTH, ATARI_HEIGHT,
					  ATARI_TRIPLE_VIS_WIDTH, ATARI_TRIPLE_HEIGHT,
					  s_Buffer.pSource, s_Buffer.pWorkScr, 0 );

		if( g_Screen.ulMode & SM_OPTN_SAFE_MODE )
		{
			if( !LockMemorySurface( TRUE ) )
				return;
		}
		if( s_Surface.nPitch != s_Surface.nWidth )
		{
			UCHAR *pSource = s_Buffer.pWorkScr;
			UCHAR *pTarget = s_Buffer.pTarget;
			int i;

			for( i = 0; i < ATARI_TRIPLE_HEIGHT; i++ )
			{
				CopyMemory( pTarget, pSource, ATARI_TRIPLE_VIS_WIDTH );
				pTarget += s_Surface.nPitch;
				pSource += ATARI_TRIPLE_VIS_WIDTH;
			}
		}
		else
			CopyMemory( s_Buffer.pTarget, s_Buffer.pWorkScr, ATARI_TRIPLE_VIS_SCREEN_SIZE );
	}
	else
	{
		if( g_Screen.ulMode & SM_OPTN_SAFE_MODE )
		{
			if( !LockMemorySurface( TRUE ) )
				return;
		}
		Interpolate3( ATARI_HORZ_CLIP, ATARI_VIS_WIDTH, ATARI_HEIGHT,
					  ATARI_TRIPLE_VIS_WIDTH, ATARI_TRIPLE_HEIGHT,
					  s_Buffer.pSource, s_Buffer.pTarget, s_Surface.nPitch );
	}
	if( g_Screen.ulMode & SM_OPTN_SAFE_MODE )
		LockMemorySurface( FALSE );

	if( s_DDraw.bFlipBuffers )
	{
		if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_DDraw.lpDDSBackBuf, &s_rcDestin, s_DDraw.lpDDSOffscrn, &s_rcSource, s_DDraw.dwBlitFlags, s_DDraw.pfxBltEffects )) )
		{
			/* Flip the DirectDraw surfaces */
			if( DD_OK == (hResult = IDirectDrawSurface_Flip( s_DDraw.lpDDSPrimary, NULL, s_DDraw.dwFlipFlags )) )
				return;
		}
	}
	else
	{
		if( g_Screen.ulMode & SM_OPTN_DDVBL_WAIT )
			IDirectDraw_WaitForVerticalBlank( s_DDraw.lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );
		/* Blit to the primary surface from a memory one */
		if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_DDraw.lpDDSPrimary, &s_rcDestin, s_DDraw.lpDDSOffscrn, &s_rcSource, s_DDraw.dwBlitFlags, s_DDraw.pfxBltEffects )) )
			return;
	}
	if( hResult == DDERR_WASSTILLDRAWING )
		return;

	if( hResult == DDERR_SURFACELOST )
		RestoreSurfaces();
	else
		ServeDDError( IDS_DDERR_SURFACE_BLT, hResult, FALSE );
} /* #OF# Screen_DDraw_1024_Interpolation */

/*========================================================
Function : Screen_DDraw_1024_Scanlines
=========================================================*/
/* #FN#
   Draws Atari screen in full 1024 mode with scan lines */
static
void
/* #AS#
   Nothing */
Screen_DDraw_1024_Scanlines(
	UBYTE *screen /* #IN# Pointer to Atari screen */
)
{
	HRESULT	hResult;
	int nCache;

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	if( g_Misc.ulState & MS_SHOW_DRIVE_LED )
		Update_LED( screen + s_ulLEDOffset );
#endif
	Update_PEN( screen );

	if( g_Screen.ulMode & SM_OPTN_SAFE_MODE )
	{
		if( !LockMemorySurface( TRUE ) )
			return;
	}
	/* Calculate cache size of the memory surface */
	nCache = s_Surface.nPitch - s_Surface.nWidth;

#ifdef PFUSIK_ASM
	_asm
	{
		mov		esi, dword ptr [s_Buffer.pSource]	;; The source pointer
		add		esi, ATARI_HORZ_CLIP
		mov		edi, dword ptr [s_Buffer.pTarget]	;; The dest pointer
		add		edi, ATARI_HORZ_CLIP
		add		nCache, 0x30			;; Skip margins in dest
		mov		ebx, ATARI_HEIGHT		;; Number of line pairs (ATARI_HEIGHT)
	new_line:
		mov		ecx, 0x54				;; Our count of DWORDs to copy (ATARI_VIS_WIDTH/4)
		rep		movsd					;; Move that string

		sub		esi, ATARI_VIS_WIDTH	;; Move back to origin of original scan line
		add		edi, 0x30				;; Skip margins in dest
		mov		ecx, 0x54				;; Our count of DWORDs to copy (ATARI_VIS_WIDTH/4)
		rep		movsd					;; Move that string

		sub		esi, ATARI_VIS_WIDTH	;; Move back to origin of original scan line
		add		edi, 0x30				;; Skip margins in dest
		mov		ecx, 0x54				;; Count of DWORDs (ATARI_VIS_WIDTH/4)
	dark_line:
		mov		eax, dword ptr [esi]	;; Make copy of original pixels
		mov		edx, eax				;; Another copy of original pixels
		and		eax, 0xf0f0f0f0			;; Extract high nybbles - colors
		shr		edx, 1					;; Darken brightness
		and		edx, 0x07070707			;; Extract darkened brightness
		add		eax, edx				;; Combine color with brightness
		add		esi, 4					;; Increment source pointer
		mov		dword ptr [edi], eax	;; Store darkened pixels
		add		edi, 4					;; Increment dest pointer
		dec		ecx						;; Decrement our DWORD counter
		jnz		dark_line				;; Back to start of DWORD operation

		add		esi, 0x30				;; Add in the offset to the next scan line for src
		add		edi, nCache
		dec		ebx
		jnz		new_line
	}
#else
	_asm
	{
		;; Order in this routine has been manipulated to maximize pairing on a Pentium processor
		mov		esi, dword ptr [s_Buffer.pSource]	;; the source pointer
		mov		edi, dword ptr [s_Buffer.pTarget]	;; the dest pointer
		mov		eax, ATARI_HEIGHT					;; Number of line pairs (ATARI_HEIGHT)
	new_line:
		mov		ecx, 060h			;; Our count of DWORDs to copy (ATARI_WIDTH/4)
		rep		movsd				;; Move that string
		sub		esi, ATARI_WIDTH	;; Move back to origin of this scan line
		mov		ecx, 060h			;; Going to move another set of words...
		rep		movsd				;; Make another copy of scanline

		sub		esi, ATARI_WIDTH	;; Move back to origin of original scan line
		mov		edx, ATARI_WIDTH	;; The pixel counter
	dark_line:
		mov		bl, byte ptr [esi]	;; Make copy of original pixel
		mov		cl, bl				;; Another copy of original pixel
		and		bl, 0x0f			;; Bottom nybble of this pixel
		shr		bl, 1				;; nybble / 2
		and		cl, 0xf0			;; Top nybble of this pixel
		add		cl, bl				;; Make darkened pixel by adding nybbles
		inc		esi					;; move to next source pixel
		mov		byte ptr [edi], cl	;; Copy darkened pixel into destination point
		add		edi, nCache
		dec		edx					;; Decrement our pixel counter
		jnz		dark_line			;; Back to start of pixel operation

		dec		eax					;; Decrement our line counter
		jnz		new_line
	}
#endif
	if( g_Screen.ulMode & SM_OPTN_SAFE_MODE )
		LockMemorySurface( FALSE );

	if( s_DDraw.bFlipBuffers )
	{
		if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_DDraw.lpDDSBackBuf, &s_rcDestin, s_DDraw.lpDDSOffscrn, &s_rcSource, s_DDraw.dwBlitFlags, s_DDraw.pfxBltEffects )) )
		{
			/* Flip the DirectDraw surfaces */
			if( DD_OK == (hResult = IDirectDrawSurface_Flip( s_DDraw.lpDDSPrimary, NULL, s_DDraw.dwFlipFlags )) )
				return;
		}
	}
	else
	{
		if( g_Screen.ulMode & SM_OPTN_DDVBL_WAIT )
			IDirectDraw_WaitForVerticalBlank( s_DDraw.lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );
		/* Blit to the primary surface from a memory one */
		if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_DDraw.lpDDSPrimary, &s_rcDestin, s_DDraw.lpDDSOffscrn, &s_rcSource, s_DDraw.dwBlitFlags, s_DDraw.pfxBltEffects )) )
			return;
	}
	if( hResult == DDERR_WASSTILLDRAWING )
		return;

	if( hResult == DDERR_SURFACELOST )
		RestoreSurfaces();
	else
		ServeDDError( IDS_DDERR_SURFACE_BLT, hResult, FALSE );
} /* #OF# Screen_DDraw_Full_Blt_Scanlines */

/*========================================================
Function : Screen_GDI_Double
=========================================================*/
/* #FN#
   Draws Atari screen in GDI windowed mode x 2 */
static
void
/* #AS#
   Nothing */
Screen_GDI_Double(
	UBYTE *screen /* #IN# Pointer to Atari screen */
)
{
	UBYTE *pSourceLine = (UBYTE *)s_Buffer.pSource + ATARI_HORZ_CLIP;
	UBYTE *pTargetLine = (UBYTE *)s_Buffer.pTarget;
#ifdef PFUSIK_ASM
	int nIncD = ATARI_DOUBLE_VIS_WIDTH;
#else
	int nIncD = 0;
#endif

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	if( g_Misc.ulState & MS_SHOW_DRIVE_LED )
		Update_LED( screen + s_ulLEDOffset );
#endif
	Update_PEN( screen );

	if( g_Screen.pfOutput )
	{
		/* We have to flip the top-down bitmap verticaly, because
		   video compressors do not support bitmap mirroring
		*/
		pTargetLine += ATARI_DOUBLE_VIS_SCREEN_SIZE - 2 * ATARI_DOUBLE_VIS_WIDTH;
#ifdef PFUSIK_ASM
		nIncD = -3 * ATARI_DOUBLE_VIS_WIDTH;
#else
		nIncD = -4 * ATARI_DOUBLE_VIS_WIDTH;
#endif
	}

#ifdef PFUSIK_ASM
	if( g_Screen.ulMode & SM_OPTN_USE_MMX )
	{
#ifdef INTEL_ASM
		int nWidth = ATARI_VIS_WIDTH;
		int nSrcWidth = ATARI_WIDTH;
		int nDestWidth = ATARI_DOUBLE_VIS_WIDTH;
		int nHeight = ATARI_HEIGHT;
		/* Size: 2x2, Mode: DUPLICATE, Cpu: MMX */
		_ASSERT(nWidth % 16 == 0);
		_asm
		{
			mov		ebx, dword ptr [pSourceLine]
			mov		eax, dword ptr [pTargetLine]
			sub		ebx, 32
			sub		eax, 64
			mov		ecx, dword ptr [nWidth]
			mov		edi, dword ptr [nDestWidth]
			and		ecx, 0ffffffe0h
			mov		esi, dword ptr [nHeight]
			lea		ecx, [eax + 2 * ecx]
		pixel_2x2_duplicate_mmx:
			movq	mm0, qword ptr [ebx + 32]
			movq	mm1, mm0
			mov		edx, dword ptr [eax + 64]	; Cache read -- better performance
			add		ebx, 32
			mov		edx, dword ptr [eax + 96]	; Cache read -- better performance
			add		eax, 64
			punpcklbw	mm0, mm0
			movq	mm2, qword ptr [ebx + 8]	; 2nd read
			punpckhbw	mm1, mm1
			movq	qword ptr [eax + edi], mm0	; 2a write -- row no. 2
			movq	qword ptr [eax + edi + 8], mm1	; 2b write -- row no. 2
			movq	mm3, mm2
			movq	qword ptr [eax], mm0		; 1a write -- row no. 1
			punpcklbw	mm2, mm2
			movq	qword ptr [eax + 8], mm1	; 1b write
			movq	qword ptr [eax + 16], mm2	; 1c write
			punpckhbw	mm3, mm3
			movq	mm4, qword ptr [ebx + 16]	; 3rd read
			movq	qword ptr [eax + 24], mm3	; 1d write
			movq	mm5, mm4
			movq	qword ptr [eax + edi + 16], mm2	; 2c write
			punpcklbw	mm4, mm4
			movq	qword ptr [eax + edi + 24], mm3	; 2d write
			punpckhbw	mm5, mm5
			movq	qword ptr [eax + 32], mm4	; 1e write
			movq	mm6, qword ptr [ebx + 24]	; 4th read
			movq	qword ptr [eax + 40], mm5	; 1f write
			movq	mm7, mm6
			movq	qword ptr [eax + edi + 32], mm4	; 2e write
			punpcklbw	mm6, mm6
			movq	qword ptr [eax + edi + 40], mm5	; 2f write
			punpckhbw	mm7, mm7
			movq	qword ptr [eax + 48], mm6	; 1g write
			movq	qword ptr [eax + 56], mm7	; 1h write
			movq	qword ptr [eax + edi + 48], mm6	; 2g write
			movq	qword ptr [eax + edi + 56], mm7	; 2h write
			cmp		ecx, eax
			jne		pixel_2x2_duplicate_mmx

			test	dword ptr [nWidth], 16
			jz		cont_2x2_duplicate_mmx
			movq	mm0, qword ptr [ebx + 32]
			movq	mm1, mm0
			mov		edx, dword ptr [eax + 64]	; Cache read -- better performance
			add		ebx, 16
			mov		edx, dword ptr [eax + 96]	; Cache read -- better performance
			add		eax, 32
			punpcklbw	mm0, mm0
			movq	mm2, qword ptr [ebx + 24]	; 2nd read
			punpckhbw	mm1, mm1
			movq	qword ptr [eax + edi + 32], mm0	; 2a write -- row no. 2
			movq	qword ptr [eax + edi + 40], mm1	; 2b write -- row no. 2
			movq	mm3, mm2
			movq	qword ptr [eax + 32], mm0		; 1a write -- row no. 1
			punpcklbw	mm2, mm2
			movq	qword ptr [eax + 40], mm1	; 1b write
			movq	qword ptr [eax + 48], mm2	; 1c write
			punpckhbw	mm3, mm3
			movq	qword ptr [eax + 56], mm3	; 1d write
			movq	qword ptr [eax + edi + 48], mm2	; 2c write
			movq	qword ptr [eax + edi + 56], mm3	; 2d write

		cont_2x2_duplicate_mmx:
			mov		edx, dword ptr [nWidth]
			lea		ecx, [ecx + 2 * edi]
			sub		eax, edx
			sub		ebx, edx
			sub		eax, edx
			add		ebx, dword ptr [nSrcWidth]
			lea		eax, [eax + 2 * edi]
			dec		esi
			jnz		pixel_2x2_duplicate_mmx

			emms
		}
#else /*INTEL_ASM*/
		_asm
		{
			mov		esi, dword ptr [pSourceLine]	;; The source pointer
			mov		edi, dword ptr [pTargetLine]	;; The dest pointer
			mov		ebx, ATARI_HEIGHT				;; Number of line pairs (ATARI_HEIGHT)
		mmx_line:
			mov		ecx, 0x15						;; Count of OWORDs (ATARI_VIS_WIDTH/16)
		mmx_oword:
			movq	mm0, qword ptr [esi]			;; mm0=HGFEDCBA
			movq	mm2, qword ptr [esi + 8]		;; mm2=PONMLKJI
			movq	mm1, mm0						;; mm1=HGFEDCBA
			movq	mm3, mm2						;; mm3=PONMLKJI
			punpcklbw	mm0, mm0					;; mm0=DDCCBBAA
			punpckhbw	mm1, mm1					;; mm1=HHGGFFEE
			punpcklbw	mm2, mm2					;; mm2=LLKKJJII
			punpckhbw	mm3, mm3					;; mm3=PPOONNMM
			movq	qword ptr [edi], mm0			;; Store upper copies
			movq	qword ptr [edi + 8], mm1
			movq	qword ptr [edi + 16], mm2
			movq	qword ptr [edi + 24], mm3
			movq	qword ptr [edi + ATARI_DOUBLE_VIS_WIDTH], mm0	;; Store lower copies
			movq	qword ptr [edi + ATARI_DOUBLE_VIS_WIDTH + 8], mm1
			movq	qword ptr [edi + ATARI_DOUBLE_VIS_WIDTH + 16], mm2
			movq	qword ptr [edi + ATARI_DOUBLE_VIS_WIDTH + 24], mm3
			add		esi, 16							;; Increment source pointer
			add		edi, 32							;; Increment dest pointer
			dec		ecx								;; Decrement OWORD counter
			jnz		mmx_oword						;; Back to start of OWORD->HWORD operation

			add		esi, 0x30						;; Add in the offset to the next scan line for src
			add		edi, dword ptr [nIncD]			;; Top-down: ATARI_DOUBLE_VIS WIDTH, Bottom-up: -3 * ATARI_DOUBLE_VIS_WIDTH
			dec		ebx
			jnz		mmx_line

			emms									;; Empty MMX state
		}
#endif /*INTEL_ASM*/
	}
	else
	{
		int lineCount;
		_asm
		{
			mov		esi, dword ptr [pSourceLine]	;; The source pointer
			mov		edi, dword ptr [pTargetLine]	;; The dest pointer
			mov		dword ptr [lineCount], ATARI_HEIGHT
		new_line_2:
			mov		ecx, 0x54
		copy_scanline_2:
			mov		eax, dword ptr [esi]
			add		esi, 4
			add		edi, 8
			mov		edx, eax
			mov		ebx, eax
			bswap	edx
			and		eax, 00000ffffh
			and		edx, 0ffff0000h
			add		edx, eax
			mov		eax, ebx
			bswap	eax
			rol		edx, 8
			mov		dword ptr [edi - 8], edx
			and		ebx, 0ffff0000h
			and		eax, 00000ffffh
			mov		dword ptr [edi - 8 + ATARI_DOUBLE_VIS_WIDTH], edx
			add		ebx, eax
			ror		ebx, 8
			dec		ecx
			mov		[edi - 4], ebx
			mov		[edi - 4 + ATARI_DOUBLE_VIS_WIDTH], ebx
			jnz		copy_scanline_2
			add		esi, 0x30						;; Add in the offset to the next scan line for src
			add		edi, dword ptr [nIncD]			;; Top-down: ATARI_DOUBLE_VIS WIDTH, Bottom-up: -3 * ATARI_DOUBLE_VIS_WIDTH
			dec		dword ptr [lineCount]
			jnz		new_line_2
		}
	}
#else
	_asm
	{
		;; Order in this routine has been manipulated to maximize pairing on a Pentium processor
		mov		ebx, dword ptr [pSourceLine]	;; The source pointer
		mov		edi, dword ptr [pTargetLine]	;; The dest pointer
		mov		eax, 0f00000h		;; Number of scan lines in the source (shifted 16 bits)
	new_line_1:
		mov		ecx, 0150h			;; We're going to copy ATARI_VIS_WIDTH pixel pairs
	copy_bytes_1:
		mov		dl, byte ptr [ebx]	;; Get this pixel for doubling
		mov		byte ptr [edi], dl	;; Store copy 1
		inc		edi
		inc		ebx					;; Next source
		mov		byte ptr [edi], dl	;; Store copy 2
		inc		edi					;; Next destination

		dec		ecx
		jnz		copy_bytes_1		;; Do this until ATARI_VIS_WIDTH pairs are done

		mov		esi, edi			;; Current destination pointer into source index
		mov		ecx, 0a8h			;; Our count of DWORDs to copy (ATARI_VIS_WIDTH * 2 / 4)
		sub		esi, 02a0h			;; Move back ATARI_VIS_WIDTH * 2 (beginning of pixel-doubled scan line)
		add		ebx, 030h			;; Add in the offset to the next scan line for src
		rep		movsd				;; Move that string
		add		edi, dword ptr [nIncD]	;; Top-down: 0, Bottom-up: -4 * ATARI_DOUBLE_VIS_WIDTH

		sub		eax, 10000h			;; Decrement high word of eax
		jnz		new_line_1			;; Jump if we're not done
	}
#endif
	if( g_Screen.pfOutput )
		/* Save the DIB to video output stream */
		Video_SaveFrame( s_Buffer.pTarget, ATARI_DOUBLE_VIS_SCREEN_SIZE );

	/* Make the prepared bitmap visible on the screen */
	StretchDIBits( g_Screen.hDC,
				   0, 0, ATARI_DOUBLE_VIS_WIDTH, ATARI_DOUBLE_HEIGHT,
				   0, 0, ATARI_DOUBLE_VIS_WIDTH, ATARI_DOUBLE_HEIGHT,
				   s_Buffer.pTarget, g_Screen.lpbmi, DIB_RGB_COLORS, SRCCOPY );
} /* #OF# Screen_GDI_Double */

/*========================================================
Function : Screen_GDI_Double_Interpolation
=========================================================*/
/* #FN#
   Draws Atari screen in GDI windowed mode x 2 with interpolation */
static
void
/* #AS#
   Nothing */
Screen_GDI_Double_Interpolation(
	UBYTE *screen /* #IN# Pointer to Atari screen */
)
{
#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	if( g_Misc.ulState & MS_SHOW_DRIVE_LED )
		Update_LED( screen + s_ulLEDOffset );
#endif
	Update_PEN( screen );

	if( g_Screen.pfOutput )
	{
		/* We have to flip the top-down bitmap verticaly, because
		   video compressors do not support bitmap mirroring
		*/
		Interpolate2( ATARI_HORZ_CLIP, ATARI_VIS_WIDTH, ATARI_HEIGHT,
					  ATARI_DOUBLE_VIS_WIDTH, ATARI_DOUBLE_HEIGHT,
					  s_Buffer.pSource, s_Buffer.pTarget, 0, TRUE );
		/* Save the DIB to video output stream */
		Video_SaveFrame( s_Buffer.pTarget, ATARI_DOUBLE_VIS_SCREEN_SIZE );
	}
	else
	{
		Interpolate2( ATARI_HORZ_CLIP, ATARI_VIS_WIDTH, ATARI_HEIGHT,
					  ATARI_DOUBLE_VIS_WIDTH, ATARI_DOUBLE_HEIGHT,
					  s_Buffer.pSource, s_Buffer.pTarget, 0, FALSE );
	}
	/* Make the prepared bitmap visible on the screen */
	StretchDIBits( g_Screen.hDC,
				   0, 0, ATARI_DOUBLE_VIS_WIDTH, ATARI_DOUBLE_HEIGHT,
				   0, 0, ATARI_DOUBLE_VIS_WIDTH, ATARI_DOUBLE_HEIGHT,
				   s_Buffer.pTarget, g_Screen.lpbmi, DIB_RGB_COLORS, SRCCOPY );
} /* #OF# Screen_GDI_Double_Interpolation */

/*========================================================
Function : Screen_GDI_Double_Scanlines
=========================================================*/
/* #FN#
   Draws Atari screen in GDI windowed mode x 2 with scan lines */
static
void
/* #AS#
   Nothing */
Screen_GDI_Double_Scanlines(
	UBYTE *screen /* #IN# Pointer to Atari screen */
)
{
	UBYTE *pSourceLine = (UBYTE *)s_Buffer.pSource + ATARI_HORZ_CLIP;
	UBYTE *pTargetLine = (UBYTE *)s_Buffer.pTarget;
	int nIncD = ATARI_DOUBLE_VIS_WIDTH;

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	if( g_Misc.ulState & MS_SHOW_DRIVE_LED )
		Update_LED( screen + s_ulLEDOffset );
#endif
	Update_PEN( screen );

	if( g_Screen.pfOutput )
	{
		/* We have to flip the top-down bitmap verticaly, because
		   video compressors do not support bitmap mirroring
		*/
		pTargetLine += ATARI_DOUBLE_VIS_SCREEN_SIZE - 2 * ATARI_DOUBLE_VIS_WIDTH;
		nIncD = -3 * ATARI_DOUBLE_VIS_WIDTH;
	}

#ifdef PFUSIK_ASM
	if( g_Screen.ulMode & SM_OPTN_USE_MMX )
	{
		static ULONG colMask[2] = {0xf0f0f0f0, 0xf0f0f0f0};
		static ULONG lumMask[2] = {0x07070707, 0x07070707};
		_asm
		{
			mov		esi, dword ptr [pSourceLine]	;; The source pointer
			mov		edi, dword ptr [pTargetLine]	;; The dest pointer
			mov		ebx, ATARI_HEIGHT				;; Number of line pairs (ATARI_HEIGHT)
			movq	mm2, qword ptr [colMask]
			movq	mm3, qword ptr [lumMask]
		mmx_line:
			mov		ecx, 0x2a				;; Count of QWORDs (ATARI_VIS_WIDTH/8)
		mmx_qword:
			movq	mm0, qword ptr [esi]	;; Load 8 source pixels
			movq	mm4, mm0
			punpcklbw	mm0, mm0			;; Double the pixels
			punpckhbw	mm4, mm4
			movq	mm1, mm0				;; Copy original pixels
			movq	mm5, mm4
			movq	qword ptr [edi], mm0	;; Store normal pixels
			movq	qword ptr [edi + 8], mm4
			pand	mm0, mm2				;; Extract colors
			pand	mm4, mm2
			psrlq	mm1, 1					;; Darken brightness
			psrlq	mm5, 1
			add		esi, 8					;; Increment source pointer
			pand	mm1, mm3				;; Extract darkened brightness
			pand	mm5, mm3
			por		mm0, mm1				;; Combine color with brightness
			por		mm4, mm5
			movq	qword ptr [edi + ATARI_DOUBLE_VIS_WIDTH], mm0	;; Store lower copies
			movq	qword ptr [edi + ATARI_DOUBLE_VIS_WIDTH + 8], mm4
			add		edi, 16					;; Increment dest pointer
			dec		ecx						;; Decrement DWORD counter
			jnz		mmx_qword				;; Back to start of QWORD->OWORD operation

			add		esi, 0x30				;; Add in the offset to the next scan line for src
			add		edi, dword ptr [nIncD]	;; Top-down: ATARI_DOUBLE_VIS WIDTH, Bottom-up: -3 * ATARI_DOUBLE_VIS_WIDTH
			dec		ebx
			jnz		mmx_line

			emms							;; Empty MMX state
		}
	}
	else
	{
		_asm
		{
			mov		esi, dword ptr [pSourceLine]	;; The source pointer
			mov		edi, dword ptr [pTargetLine]	;; The dest pointer
			mov		ebx, ATARI_HEIGHT				;; Number of line pairs (ATARI_HEIGHT)
		new_line_1:
			mov		ecx, 0xa8						;; Count of WORDs to copy (ATARI_VIS_WIDTH/2)
		copy_scanline_1:
			xor		edx, edx						;; edx=0000
			mov		ax, word ptr [esi]				;; eax=..HL
			mov		dl, al							;; edx=000L
			mov		al, ah							;; eax=..HH
			mov		dh, dl							;; edx=00LL
			shl		eax, 16							;; eax=HH00
			or		eax, edx						;; eax=HHLL
			mov		dword ptr [edi], eax			;; Store normal pixels
			mov		edx, eax						;; Copy of normal pixels
			and		eax, 0xf0f0f0f0					;; Extract high nybbles - colors
			shr		edx, 1							;; Darken brightness
			add		esi, 2							;; Increment source pointer
			and		edx, 0x07070707					;; Extract darkened brightness
			add		eax, edx						;; Combine color with brightness
			mov		dword ptr [edi + ATARI_DOUBLE_VIS_WIDTH], eax	;; Store darkened pixels
			add		edi, 4							;; Increment dest pointer
			dec		ecx								;; Decrement our WORD counter
			jnz		copy_scanline_1					;; Back to start of WORD operation

			add		esi, 0x30						;; Add in the offset to the next scan line for src
			add		edi, dword ptr [nIncD]			;; Top-down: ATARI_DOUBLE_VIS WIDTH, Bottom-up: -3 * ATARI_DOUBLE_VIS_WIDTH
			dec		ebx
			jnz		new_line_1
		}
	}
#else
	_asm
	{
		;; Order in this routine has been manipulated to maximize pairing on a Pentium processor
		mov		esi, dword ptr [pSourceLine]	;; The source pointer
		mov		edi, dword ptr [pTargetLine]	;; The dest pointer
		dec		edi
		mov		eax, 0f00000h			;; Number of scan lines in the source (shifted 16 bits)
	new_line_1:
		mov		ax, ATARI_VIS_WIDTH		;; We're going to copy ATARI_VIS_WIDTH pixel pairs
	copy_scanline_1:
		mov		dl, byte ptr [esi]								;; Get this pixel
		inc		edi												;; Next destination
		mov		cl, dl											;; Make a copy of original pixel
		mov		bl, dl											;; Another copy of original pixel
		and		bl, 0x0f										;; Bottom nybble of this pixel
		mov		byte ptr [edi], dl								;; Store copy 1 of regular pixel
		shr		bl, 1											;; bits / 2
		and		cl, 0xf0										;; Top nybble of this pixel
		add		bl, cl											;; (pixel val / 16) * 16 + (val % 16) / 2
		mov		byte ptr [edi + ATARI_DOUBLE_VIS_WIDTH], bl	;; Store darkline pixel 1
		inc		esi												;; Next source pixel
		inc		edi												;; Next destination
		mov		byte ptr [edi], dl								;; Store copy 2 of regular pixel
		mov		byte ptr [edi + ATARI_DOUBLE_VIS_WIDTH], bl	;; Store darkline pixel 2

		dec		ax					;; Decrement counter
		jnz		copy_scanline_1		;; Do this until ATARI_VIS_WIDTH pairs are done

		add		edi, dword ptr [nIncD]			;; Top-down: ATARI_DOUBLE_VIS WIDTH, Bottom-up: -3 * ATARI_DOUBLE_VIS_WIDTH
		add		esi, 030h			;; Add in the offset to the next scan line for src

		sub		eax, 10000h			;; Decrement high word of eax
		jnz		new_line_1			;; Jump if we're not done
	}
#endif
	if( g_Screen.pfOutput )
		/* Save the DIB to video output stream */
		Video_SaveFrame( s_Buffer.pTarget, ATARI_DOUBLE_VIS_SCREEN_SIZE );

	/* Make the prepared bitmap visible on the screen */
	StretchDIBits( g_Screen.hDC,
				   0, 0, ATARI_DOUBLE_VIS_WIDTH, ATARI_DOUBLE_HEIGHT,
				   0, 0, ATARI_DOUBLE_VIS_WIDTH, ATARI_DOUBLE_HEIGHT,
				   s_Buffer.pTarget, g_Screen.lpbmi, DIB_RGB_COLORS, SRCCOPY );
} /* #OF# Screen_GDI_Double_Scanlines */

/*========================================================
Function : Screen_GDI
=========================================================*/
/* #FN#
   Draws Atari screen in GDI windowed mode x 1 */
static
void
/* #AS#
   Nothing */
Screen_GDI(
	UBYTE *screen /* #IN# Pointer to Atari screen */
)
{
#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	if( g_Misc.ulState & MS_SHOW_DRIVE_LED )
		Update_LED( screen + s_ulLEDOffset );
#endif
	Update_PEN( screen );

	if( g_Screen.pfOutput )
	{
		/* We have to flip the top-down bitmap verticaly, because
		   video compressors do not support bitmap mirroring
		*/
		_asm
		{
			mov		esi, dword ptr [s_Buffer.pSource]	;; The source pointer
			add		esi, ATARI_HORZ_CLIP
			mov		edi, dword ptr [s_Buffer.pTarget]	;; The dest pointer
			add		edi, 0139b0h						;; ATARI_VIS_SCREEN_SIZE - ATARI_VIS_WIDTH
			mov		eax, ATARI_HEIGHT					;; Number of lines
		scan_line_1:
			mov		ecx, 054h		;; Our count of DWORDs to copy (ATARI_VIS_WIDTH/4)
			rep		movsd			;; Move that string
			add		esi, 030h		;; Add in the offset to the next scan line for source
			sub		edi, 02a0h		;; 2 * ATARI_VIS_WIDTH

			dec		eax
			jnz		scan_line_1
		}
		/* Save the DIB to video output stream */
		Video_SaveFrame( s_Buffer.pTarget, ATARI_VIS_SCREEN_SIZE );
	}
	else
	{
		/* The top-down bitmap will be mirrored by StretchDIBits */
		_asm
		{
			mov		esi, dword ptr [s_Buffer.pSource]	;; the source pointer
			add		esi, ATARI_HORZ_CLIP
			mov		edi, dword ptr [s_Buffer.pTarget]	;; the dest pointer
			mov		eax, ATARI_HEIGHT					;; number of lines
		scan_line_2:
			mov		ecx, 054h		;; Our count of DWORDs to copy (ATARI_VIS_WIDTH/4)
			rep		movsd			;; Move that string
			add		esi, 030h		;; add in the offset to the next scan line for src

			dec		eax
			jnz		scan_line_2
		}
	}
	/* Make the prepared bitmap visible on the screen */
	StretchDIBits( g_Screen.hDC,
				   0, 0, ATARI_VIS_WIDTH, ATARI_HEIGHT,
				   0, 0, ATARI_VIS_WIDTH, ATARI_HEIGHT,
				   s_Buffer.pTarget, g_Screen.lpbmi, DIB_RGB_COLORS, SRCCOPY );
} /* #OF# Screen_GDI */


/* TO DO:
   The methods below should be used EVERY time when the screen
   freezing for GUI is needed. Maybe in the next release...
*/

#define AF_BITMAP	0x01
#define AF_COLORS	0x02
#define AF_LOCKED	0x04

/*========================================================
Function : AllocUserScreen
=========================================================*/
/* #FN#
   Prepares the screen for using dialogs safely */
static
UINT
/* #AS#
   Flags to use as the input of the FreeUserScreen method */
AllocUserScreen( void )
{
	UINT uiResult = 0;

	if( g_Screen.ulMode & SM_MODE_FULL )
	{
		if( ST_ATARI_STOPPED )
		{
			/* It may be a good idea to handle the case here,
			   for example when the DisplayMessage is called
			   after cleaning the screen using 'deep' flag */
			if( !g_Screen.bLocked )
			{
				Screen_FlipToGDI( FALSE );

				if( ST_MENUBAR_HIDDEN )
					Screen_ShowMenuBar( TRUE );
			}
		}
		/* There is a valid bitmap to prepare */
		else if( NULL == s_Redraw.hBmp )
		{
			/* Prepare a screen bitmap for redrawing */
			Screen_DrawFrozen( FALSE, FALSE, FALSE, FALSE );
			uiResult |= AF_BITMAP;
		}
		/* Set a system palette if possible and necessary */
		if( !g_Screen.Pal.bUseSysCol && Screen_UseSystemPalette() )
			uiResult |= AF_COLORS;

		/* Set the 'menu-active' semafore */
		if( !g_Screen.bLocked )
		{
			g_Screen.bLocked = TRUE;
			uiResult |= AF_LOCKED;
		}
	}
	return uiResult;
} /* #OF# AllocUserScreen */

/*========================================================
Function : FreeUserScreen
=========================================================*/
/* #FN#
   Frees the screen allocated by AllocUserScreen */
static
void
/* #AS#
   Nothing */
FreeUserScreen(
	UINT uiAlloc
)
{
	if( uiAlloc & AF_BITMAP )
	{
		Screen_FreeRedraw();

		/* TO DO:
		   I'm not sure if the screen cleaning should be placed here;
		   Used temporarily for 2.8
		*/
		if( ST_DOUBLE_BUFFERS && !(g_Screen.ulMode & SM_ATTR_NO_MENU) )
			/* Needed for double-buffering modes to clean the menus */
			Screen_ShowMenuBar( TRUE );

		/* Clean up the screen */
		Screen_Clear( FALSE, FALSE );
	}
	if( uiAlloc & AF_COLORS )
		Screen_UseAtariPalette( FALSE );

	if( uiAlloc & AF_LOCKED )
		g_Screen.bLocked = FALSE;
} /* #OF# FreeUserScreen */

/*========================================================
Function : Screen_DisplayMessage
=========================================================*/
/* #FN#
   Creates, displays, and operates a message box */
int
/* #AS#
   The MessageBox API function return value */
Screen_DisplayMessage(
	HWND hWnd,	    /* #IN# Parent window     */
	UINT uFormat,	/* #IN# Text resource id  */
	UINT uCaption,	/* #IN# Title resource id */
	UINT uType,		/* #IN# Message box style */
	...
)
{
	va_list argList;

	char szFormat [ LOADSTRING_STRING_SIZE ];
	char szCaption[ LOADSTRING_STRING_SIZE ];
	char szMessage[ LOADSTRING_STRING_SIZE * 3 ];
	int  nResult;

	/* Prepare the emulator display for using the GUI */
	UINT uiAlloc = AllocUserScreen();

	if( !(uFormat &&
		LoadString( NULL, uFormat, szFormat, LOADSTRING_STRING_SIZE )) )
	{
		strcpy( szFormat, "An internal error has occured (resource load)!" );
	}
	/* Format the message string */
	va_start( argList, uType );
	vsprintf( szMessage, szFormat, argList );

	if( !(uCaption &&
		LoadString( NULL, uCaption, szCaption, LOADSTRING_STRING_SIZE )) )
	{
		strcpy( szCaption, "Atari800Win PLus" );
	}
	if( !uType )
		uType = MB_ICONSTOP | MB_OK;

	Screen_ShowMousePointer( TRUE );

	/* Display the message box */
	nResult = MessageBox( (hWnd ? hWnd : g_hMainWnd), szMessage, szCaption, uType );

	/* Free the allocated user screen */
	FreeUserScreen( uiAlloc );

	va_end( argList );

	return nResult;
} /* #OF# Screen_DisplayMessage */

/*========================================================
Function : GetDDErrorString
=========================================================*/
/* #FN#
   Outputs a debug string to debugger */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
GetDDErrorString(
	HRESULT hResult,       /* #IN# */
	LPSTR   lpszErrorBuff, /* #IN# */
	DWORD   cchError       /* #IN# */
)
{
    DWORD cLen;
    LPSTR lpszError;
    TCHAR szMsg[ 256 ];

    /* Check parameters */
    if( !lpszErrorBuff || !cchError )
    {
        /* Error, invalid parameters */
        return FALSE;
    }

    switch( hResult )
    {
		/* The request completed successfully */
		case DD_OK:
			lpszError = TEXT("DD_OK");
			break;

#ifdef WIN_TRANSLATE_ERRORS

		/* This object is already initialized */
		case DDERR_ALREADYINITIALIZED:
			lpszError = TEXT("DDERR_ALREADYINITIALIZED");
			break;

		/* This surface can not be attached to the requested surface */
		case DDERR_CANNOTATTACHSURFACE:
			lpszError = TEXT("DDERR_CANNOTATTACHSURFACE");
			break;

		/* This surface can not be detached from the requested surface */
		case DDERR_CANNOTDETACHSURFACE:
			lpszError = TEXT("DDERR_CANNOTDETACHSURFACE");
			break;

		/* Support is currently not available */
		case DDERR_CURRENTLYNOTAVAIL:
			lpszError = TEXT("DDERR_CURRENTLYNOTAVAIL");
			break;

		/* An exception was encountered while performing the requested operation */
		case DDERR_EXCEPTION:
			lpszError = TEXT("DDERR_EXCEPTION");
			break;

		/* Generic failure */
		case DDERR_GENERIC:
			lpszError = TEXT("DDERR_GENERIC");
			break;

		/* Height of rectangle provided is not a multiple of reqd alignment */
		case DDERR_HEIGHTALIGN:
			lpszError = TEXT("DDERR_HEIGHTALIGN");
			break;

		/* Unable to match primary surface creation request with existing
		   primary surface */
		case DDERR_INCOMPATIBLEPRIMARY:
			lpszError = TEXT("DDERR_INCOMPATIBLEPRIMARY");
			break;

		/* One or more of the caps bits passed to the callback are incorrect */
		case DDERR_INVALIDCAPS:
			lpszError = TEXT("DDERR_INVALIDCAPS");
			break;

		/* DirectDraw does not support provided Cliplist */
		case DDERR_INVALIDCLIPLIST:
			lpszError = TEXT("DDERR_INVALIDCLIPLIST");
			break;

		/* DirectDraw does not support the requested mode */
		case DDERR_INVALIDMODE:
			lpszError = TEXT("DDERR_INVALIDMODE");
			break;

		/* DirectDraw received a pointer that was an invalid DIRECTDRAW object */
		case DDERR_INVALIDOBJECT:
			lpszError = TEXT("DDERR_INVALIDOBJECT");
			break;

		/* One or more of the parameters passed to the callback function are
		   incorrect */
		case DDERR_INVALIDPARAMS:
			lpszError = TEXT("DDERR_INVALIDPARAMS");
			break;

		/* Pixel format was invalid as specified */
		case DDERR_INVALIDPIXELFORMAT:
			lpszError = TEXT("DDERR_INVALIDPIXELFORMAT");
			break;

		/* Rectangle provided was invalid */
		case DDERR_INVALIDRECT:
			lpszError = TEXT("DDERR_INVALIDRECT");
			break;

		/* Operation could not be carried out because one or more surfaces are locked */
		case DDERR_LOCKEDSURFACES:
			lpszError = TEXT("DDERR_LOCKEDSURFACES");
			break;

		/* There is no 3D present. */
		case DDERR_NO3D:
			lpszError = TEXT("DDERR_NO3D");
			break;

		/* Operation could not be carried out because there is no alpha accleration
		   hardware present or available */
		case DDERR_NOALPHAHW:
			lpszError = TEXT("DDERR_NOALPHAHW");
			break;

		/* no clip list available */
		case DDERR_NOCLIPLIST:
			lpszError = TEXT("DDERR_NOCLIPLIST");
			break;

		/* Operation could not be carried out because there is no color conversion
		   hardware present or available */
		case DDERR_NOCOLORCONVHW:
			lpszError = TEXT("DDERR_NOCOLORCONVHW");
			break;

		/* Create function called without DirectDraw object method SetCooperativeLevel
		   being called */
		case DDERR_NOCOOPERATIVELEVELSET:
			lpszError = TEXT("DDERR_NOCOOPERATIVELEVELSET");
			break;

		/* Surface doesn't currently have a color key */
		case DDERR_NOCOLORKEY:
			lpszError = TEXT("DDERR_NOCOLORKEY");
			break;

		/* Operation could not be carried out because there is no hardware support
		   of the dest color key */
		case DDERR_NOCOLORKEYHW:
			lpszError = TEXT("DDERR_NOCOLORKEYHW");
			break;

		/* No DirectDraw support possible with current display driver */
		case DDERR_NODIRECTDRAWSUPPORT:
			lpszError = TEXT("DDERR_NODIRECTDRAWSUPPORT");
			break;

		/* Operation requires the application to have exclusive mode but the
		   application does not have exclusive mode */
		case DDERR_NOEXCLUSIVEMODE:
			lpszError = TEXT("DDERR_NOEXCLUSIVEMODE");
			break;

		/* Flipping visible surfaces is not supported */
		case DDERR_NOFLIPHW:
			lpszError = TEXT("DDERR_NOFLIPHW");
			break;

		/* There is no GDI present */
		case DDERR_NOGDI:
			lpszError = TEXT("DDERR_NOGDI");
			break;

		/* Operation could not be carried out because there is no hardware present
		   or available */
		case DDERR_NOMIRRORHW:
			lpszError = TEXT("DDERR_NOMIRRORHW");
			break;

		/* Requested item was not found */
		case DDERR_NOTFOUND:
			lpszError = TEXT("DDERR_NOTFOUND");
			break;

		/* Operation could not be carried out because there is no overlay hardware
		   present or available */
		case DDERR_NOOVERLAYHW:
			lpszError = TEXT("DDERR_NOOVERLAYHW");
			break;

		/* Operation could not be carried out because there is no appropriate raster
		   op hardware present or available */
		case DDERR_NORASTEROPHW:
			lpszError = TEXT("DDERR_NORASTEROPHW");
			break;

		/* Operation could not be carried out because there is no rotation hardware
		   present or available */
		case DDERR_NOROTATIONHW:
			lpszError = TEXT("DDERR_NOROTATIONHW");
			break;

		/* Operation could not be carried out because there is no hardware support
		   for stretching */
		case DDERR_NOSTRETCHHW:
			lpszError = TEXT("DDERR_NOSTRETCHHW");
			break;

		/* DirectDrawSurface is not in 4 bit color palette and the requested operation
		   requires 4 bit color palette */
		case DDERR_NOT4BITCOLOR:
			lpszError = TEXT("DDERR_NOT4BITCOLOR");
			break;

		/* DirectDrawSurface is not in 4 bit color index palette and the requested
		   operation requires 4 bit color index palette */
		case DDERR_NOT4BITCOLORINDEX:
			lpszError = TEXT("DDERR_NOT4BITCOLORINDEX");
			break;

		/* DirectDraw Surface is not in 8 bit color mode and the requested operation
		   requires 8 bit color */
		case DDERR_NOT8BITCOLOR:
			lpszError = TEXT("DDERR_NOT8BITCOLOR");
			break;

		/* Operation could not be carried out because there is no texture mapping
		   hardware present or available */
		case DDERR_NOTEXTUREHW:
			lpszError = TEXT("DDERR_NOTEXTUREHW");
			break;

		/* Operation could not be carried out because there is no hardware support
		   for vertical blank synchronized operations */
		case DDERR_NOVSYNCHW:
			lpszError = TEXT("DDERR_NOVSYNCHW");
			break;

		/* Operation could not be carried out because there is no hardware support
		   for zbuffer blting */
		case DDERR_NOZBUFFERHW:
			lpszError = TEXT("DDERR_NOZBUFFERHW");
			break;

		/* Overlay surfaces could not be z layered based on their BltOrder because
		   the hardware does not support z layering of overlays */
		case DDERR_NOZOVERLAYHW:
			lpszError = TEXT("DDERR_NOZOVERLAYHW");
			break;

		/* The hardware needed for the requested operation has already been
		   allocated */
		case DDERR_OUTOFCAPS:
			lpszError = TEXT("DDERR_OUTOFCAPS");
			break;

		/* DirectDraw does not have enough memory to perform the operation */
		case DDERR_OUTOFMEMORY:
			lpszError = TEXT("DDERR_OUTOFMEMORY");
			break;

		/* DirectDraw does not have enough memory to perform the operation */
		case DDERR_OUTOFVIDEOMEMORY:
			lpszError = TEXT("DDERR_OUTOFVIDEOMEMORY");
			break;

		/* Hardware does not support clipped overlays */
		case DDERR_OVERLAYCANTCLIP:
			lpszError = TEXT("DDERR_OVERLAYCANTCLIP");
			break;

		/* Can only have ony color key active at one time for overlays */
		case DDERR_OVERLAYCOLORKEYONLYONEACTIVE:
			lpszError = TEXT("DDERR_OVERLAYCOLORKEYONLYONEACTIVE");
			break;

		/* Access to this palette is being refused because the palette is already
		   locked by another thread */
		case DDERR_PALETTEBUSY:
			lpszError = TEXT("DDERR_PALETTEBUSY");
			break;

		/* No src color key specified for this operation */
		case DDERR_COLORKEYNOTSET:
			lpszError = TEXT("DDERR_COLORKEYNOTSET");
			break;

		/* This surface is already attached to the surface it is being attached to */
		case DDERR_SURFACEALREADYATTACHED:
			lpszError = TEXT("DDERR_SURFACEALREADYATTACHED");
			break;

		/* This surface is already a dependency of the surface it is being made a
		   dependency of */
		case DDERR_SURFACEALREADYDEPENDENT:
			lpszError = TEXT("DDERR_SURFACEALREADYDEPENDENT");
			break;

		/* Access to this surface is being refused because the surface is already
		   locked by another thread */
		case DDERR_SURFACEBUSY:
			lpszError = TEXT("DDERR_SURFACEBUSY");
			break;

		/* Access to this surface is being refused because no driver exists
		   which can supply a pointer to the surface.
		   This is most likely to happen when attempting to lock the primary
		   surface when no DCI provider is present.
		   Will also happen on attempts to lock an optimized surface. */
		case DDERR_CANTLOCKSURFACE:
			lpszError = TEXT("DDERR_CANTLOCKSURFACE");
			break;

		/* Access to Surface refused because Surface is obscured */
		case DDERR_SURFACEISOBSCURED:
			lpszError = TEXT("DDERR_SURFACEISOBSCURED");
			break;

		/* Access to this surface is being refused because the surface is gone.
		   The DIRECTDRAWSURFACE object representing this surface should
		   have Restore called on it. */
		case DDERR_SURFACELOST:
			lpszError = TEXT("DDERR_SURFACELOST");
			break;

		/* The requested surface is not attached */
		case DDERR_SURFACENOTATTACHED:
			lpszError = TEXT("DDERR_SURFACENOTATTACHED");
			break;

		/* Height requested by DirectDraw is too large */
		case DDERR_TOOBIGHEIGHT:
			lpszError = TEXT("DDERR_TOOBIGHEIGHT");
			break;

		/* Size requested by DirectDraw is too large. The individual height and
		   width are OK */
		case DDERR_TOOBIGSIZE:
			lpszError = TEXT("DDERR_TOOBIGSIZE");
			break;

		/* Width requested by DirectDraw is too large */
		case DDERR_TOOBIGWIDTH:
			lpszError = TEXT("DDERR_TOOBIGWIDTH");
			break;

		/* Action not supported */
		case DDERR_UNSUPPORTED:
			lpszError = TEXT("DDERR_UNSUPPORTED");
			break;

		/* FOURCC format requested is unsupported by DirectDraw */
		case DDERR_UNSUPPORTEDFORMAT:
			lpszError = TEXT("DDERR_UNSUPPORTEDFORMAT");
			break;

		/* Bitmask in the pixel format requested is unsupported by DirectDraw */
		case DDERR_UNSUPPORTEDMASK:
			lpszError = TEXT("DDERR_UNSUPPORTEDMASK");
			break;

		/* Vertical blank is in progress */
		case DDERR_VERTICALBLANKINPROGRESS:
			lpszError = TEXT("DDERR_VERTICALBLANKINPROGRESS");
			break;

		/* Informs DirectDraw that the previous Blt which is transfering information
		   to or from this Surface is incomplete */
		case DDERR_WASSTILLDRAWING:
			lpszError = TEXT("DDERR_WASSTILLDRAWING");
			break;

		/* Rectangle provided was not horizontally aligned on reqd. boundary */
		case DDERR_XALIGN:
			lpszError = TEXT("DDERR_XALIGN");
			break;

		/* The GUID passed to DirectDrawCreate is not a valid DirectDraw driver
		   identifier */
		case DDERR_INVALIDDIRECTDRAWGUID:
			lpszError = TEXT("DDERR_INVALIDDIRECTDRAWGUID");
			break;

		/* A DirectDraw object representing this driver has already been created
		   for this process */
		case DDERR_DIRECTDRAWALREADYCREATED:
			lpszError = TEXT("DDERR_DIRECTDRAWALREADYCREATED");
			break;

		/* A hardware only DirectDraw object creation was attempted but the driver
		   did not support any hardware */
		case DDERR_NODIRECTDRAWHW:
			lpszError = TEXT("DDERR_NODIRECTDRAWHW");
			break;

		/* this process already has created a primary surface */
		case DDERR_PRIMARYSURFACEALREADYEXISTS:
			lpszError = TEXT("DDERR_PRIMARYSURFACEALREADYEXISTS");
			break;

		/* software emulation not available */
		case DDERR_NOEMULATION:
			lpszError = TEXT("DDERR_NOEMULATION");
			break;

		/* Region passed to Clipper::GetClipList is too small */
		case DDERR_REGIONTOOSMALL:
			lpszError = TEXT("DDERR_REGIONTOOSMALL");
			break;

		/* An attempt was made to set a clip list for a clipper objec that
		   is already monitoring an hwnd */
		case DDERR_CLIPPERISUSINGHWND:
			lpszError = TEXT("DDERR_CLIPPERISUSINGHWND");
			break;

		/* No clipper object attached to surface object */
		case DDERR_NOCLIPPERATTACHED:
			lpszError = TEXT("DDERR_NOCLIPPERATTACHED");
			break;

		/* Clipper notification requires an HWND or
		   no HWND has previously been set as the CooperativeLevel HWND */
		case DDERR_NOHWND:
			lpszError = TEXT("DDERR_NOHWND");
			break;

		/* HWND used by DirectDraw CooperativeLevel has been subclassed,
		   this prevents DirectDraw from restoring state */
		case DDERR_HWNDSUBCLASSED:
			lpszError = TEXT("DDERR_HWNDSUBCLASSED");
			break;

		/* The CooperativeLevel HWND has already been set.
		   It can not be reset while the process has surfaces or palettes created */
		case DDERR_HWNDALREADYSET:
			lpszError = TEXT("DDERR_HWNDALREADYSET");
			break;

		/* No palette object attached to this surface */
		case DDERR_NOPALETTEATTACHED:
			lpszError = TEXT("DDERR_NOPALETTEATTACHED");
			break;

		/* No hardware support for 16 or 256 color palettes */
		case DDERR_NOPALETTEHW:
			lpszError = TEXT("DDERR_NOPALETTEHW");
			break;

		/* If a clipper object is attached to the source surface passed into a
		   BltFast call */
		case DDERR_BLTFASTCANTCLIP:
			lpszError = TEXT("DDERR_BLTFASTCANTCLIP");
			break;

		/* No blter */
		case DDERR_NOBLTHW:
			lpszError = TEXT("DDERR_NOBLTHW");
			break;

		/* No DirectDraw ROP hardware */
		case DDERR_NODDROPSHW:
			lpszError = TEXT("DDERR_NODDROPSHW");
			break;

		/* Returned when GetOverlayPosition is called on a hidden overlay */
		case DDERR_OVERLAYNOTVISIBLE:
			lpszError = TEXT("DDERR_OVERLAYNOTVISIBLE");
			break;

		/* Returned when GetOverlayPosition is called on a overlay that UpdateOverlay
		   has never been called on to establish a destionation */
		case DDERR_NOOVERLAYDEST:
			lpszError = TEXT("DDERR_NOOVERLAYDEST");
			break;

		/* returned when the position of the overlay on the destionation is no longer
		   legal for that destionation */
		case DDERR_INVALIDPOSITION:
			lpszError = TEXT("DDERR_INVALIDPOSITION");
			break;

		/* Returned when an overlay member is called for a non-overlay surface */
		case DDERR_NOTAOVERLAYSURFACE:
			lpszError = TEXT("DDERR_NOTAOVERLAYSURFACE");
			break;

		/* An attempt was made to set the cooperative level when it was already
		   set to exclusive */
		case DDERR_EXCLUSIVEMODEALREADYSET:
			lpszError = TEXT("DDERR_EXCLUSIVEMODEALREADYSET");
			break;

		/* An attempt has been made to flip a surface that is not flippable */
		case DDERR_NOTFLIPPABLE:
			lpszError = TEXT("DDERR_NOTFLIPPABLE");
			break;

		/* Can't duplicate primary & 3D surfaces, or surfaces that are implicitly
		   created */
		case DDERR_CANTDUPLICATE:
			lpszError = TEXT("DDERR_CANTDUPLICATE");
			break;

		/* Surface was not locked.  An attempt to unlock a surface that was not
		   locked at all, or by this process, has been attempted */
		case DDERR_NOTLOCKED:
			lpszError = TEXT("DDERR_NOTLOCKED");
			break;

		/* Windows can not create any more DCs */
		case DDERR_CANTCREATEDC:
			lpszError = TEXT("DDERR_CANTCREATEDC");
			break;

		/* No DC was ever created for this surface */
		case DDERR_NODC:
			lpszError = TEXT("DDERR_NODC");
			break;

		/* This surface can not be restored because it was created in a different
		   mode */
		case DDERR_WRONGMODE:
			lpszError = TEXT("DDERR_WRONGMODE");
			break;

		/* This surface can not be restored because it is an implicitly created
		   surface */
		case DDERR_IMPLICITLYCREATED:
			lpszError = TEXT("DDERR_IMPLICITLYCREATED");
			break;

		/* The surface being used is not a palette-based surface */
		case DDERR_NOTPALETTIZED:
			lpszError = TEXT("DDERR_NOTPALETTIZED");
			break;

		/* The display is currently in an unsupported mode */
		case DDERR_UNSUPPORTEDMODE:
			lpszError = TEXT("DDERR_UNSUPPORTEDMODE");
			break;

		/* Operation could not be carried out because there is no mip-map
		   texture mapping hardware present or available */
		case DDERR_NOMIPMAPHW:
			lpszError = TEXT("DDERR_NOMIPMAPHW");
			break;

		/* The requested action could not be performed because the surface was of
		   the wrong type */
		case DDERR_INVALIDSURFACETYPE:
			lpszError = TEXT("DDERR_INVALIDSURFACETYPE");
			break;

		/* Device does not support optimized surfaces, therefore no video memory optimized surfaces */
		case DDERR_NOOPTIMIZEHW:
			lpszError = TEXT("DDERR_NOOPTIMIZEHW");
			break;

		/* Surface is an optimized surface, but has not yet been allocated any memory */
		case DDERR_NOTLOADED:
			lpszError = TEXT("DDERR_NOTLOADED");
			break;

		/* A DC has already been returned for this surface. Only one DC can be
		   retrieved per surface */
		case DDERR_DCALREADYCREATED:
			lpszError = TEXT("DDERR_DCALREADYCREATED");
			break;

		/* An attempt was made to allocate non-local video memory from a device
		   that does not support non-local video memory */
		case DDERR_NONONLOCALVIDMEM:
			lpszError = TEXT("DDERR_NONONLOCALVIDMEM");
			break;

		/* The attempt to page lock a surface failed */
		case DDERR_CANTPAGELOCK:
			lpszError = TEXT("DDERR_CANTPAGELOCK");
			break;

		/* The attempt to page unlock a surface failed */
		case DDERR_CANTPAGEUNLOCK:
			lpszError = TEXT("DDERR_CANTPAGEUNLOCK");
			break;

		/* An attempt was made to page unlock a surface with no outstanding page locks */
		case DDERR_NOTPAGELOCKED:
			lpszError = TEXT("DDERR_NOTPAGELOCKED");
			break;

		/* There is more data available than the specified buffer size could hold */
		case DDERR_MOREDATA:
			lpszError = TEXT("DDERR_MOREDATA");
			break;

		/* The video port is not active */
		case DDERR_VIDEONOTACTIVE:
			lpszError = TEXT("DDERR_VIDEONOTACTIVE");
			break;

		/* Surfaces created by one direct draw device cannot be used directly by
		   another direct draw device */
		case DDERR_DEVICEDOESNTOWNSURFACE:
			lpszError = TEXT("DDERR_DEVICEDOESNTOWNSURFACE");
			break;

		/* An attempt was made to invoke an interface member of a DirectDraw object
		   created by CoCreateInstance() before it was initialized */
		case DDERR_NOTINITIALIZED:
			lpszError = TEXT("DDERR_NOTINITIALIZED");
			break;

#endif /*WIN_TRANSLATE_ERRORS*/

			/* Unknown DD/App Error */
		default:
			wsprintf( szMsg, "Error #%ld", (DWORD)hResult );
			lpszError = szMsg;
			break;
    }

    /* Copy DD Error string to buff */
    cLen = strlen( lpszError );
    if( cLen >= cchError )
    {
        cLen = cchError - 1;
    }
    if( cLen )
    {
        strncpy( lpszErrorBuff, lpszError, cLen );
        lpszErrorBuff[ cLen ] = 0;
    }
	return TRUE;
} /* #OF# GetDDErrorString */
