/****************************************************************************
File    : avisave.c
/*
@(#) #SY# Atari800Win PLus
@(#) #IS# AVI API implementation for Win32 platforms
@(#) #BY# Tomasz Szymankowski
@(#) #LM# 29.03.2001
*/

/**************************************************************************
 *
 *  THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
 *  KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
 *  PURPOSE.
 *
 *  Copyright (C) 1992 - 1997 Microsoft Corporation.  All Rights Reserved.
 *
 **************************************************************************/
/*
	This file has been modified for Atari800Win PLus purpose
*/

#define AVIIF_KEYFRAME  0x00000010L /* This frame is a key frame */

#include <stdio.h>
#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>
#include <vfw.h>
#include "WinConfig.h"
#include "Resource.h"
#include "atari800.h"
#include "globals.h"
#include "macros.h"
#include "display_win.h"
#include "timing.h"
#include "misc_win.h"
#include "avisave.h"


/* Private objects */

static PAVISTREAM s_ps           = NULL; 
static PAVISTREAM s_psCompressed = NULL; 

static DWORD s_dwFrame = 0;


/*========================================================
Function : AVI_Init
=========================================================*/
/* #FN#
   Initializes Video for Windows environment */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
AVI_Init( void )
{
	/* First let's make sure we are running on 1.1 */
	WORD wVer = HIWORD( VideoForWindowsVersion() );
	if( wVer < 0x010a )
	{
		/* Oops, we are too old, blow out of here */
		return FALSE;
	}
	AVIFileInit();

	return TRUE;
} /* #OF# AVI_Init */

/*========================================================
Function : AVI_FileOpenWrite
=========================================================*/
/* #FN#
   Opens an AVI file */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
AVI_FileOpenWrite(
	PAVIFILE *pFile,      /* #IN# */
    char     *pszFileName /* #IN# */
)
{
	if( (AVIERR_OK !=
		AVIFileOpen( pFile,					/* Returned file pointer  */
					 pszFileName,			/* File name              */
					 OF_WRITE | OF_CREATE,	/* Mode to open file with */
					 NULL ))				/* Use handler determined from file extension... */
	  )
		return FALSE;

	return TRUE;
} /* #OF# AVI_FileOpenWrite */

/*========================================================
Function : AVI_CreateStream
=========================================================*/
/* #FN#
   Opens a video stream for AVI file */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
AVI_CreateStream(
	PAVIFILE    pFile,       /* #IN# */
    PAVISTREAM *ps,          /* #IN# */
    int         nRate,		 /* #IN# Sample per second */
    ULONG       nBufferSize, /* #IN# */
    int         nRectWidth,  /* #IN# */
    int         nRectHeight  /* #IN# */
)
{
	AVISTREAMINFO strhdr;

	ZeroMemory( &strhdr, sizeof(strhdr) );
	/* Fill in the stream header for the video stream... */
	strhdr.fccType               = streamtypeVIDEO;
	strhdr.fccHandler            = 0;
	strhdr.dwScale               = 1;
	strhdr.dwRate                = nRate;
	strhdr.dwSuggestedBufferSize = nBufferSize;

	SetRect( &strhdr.rcFrame, 0, 0, nRectWidth, nRectHeight );

	/* Create the stream */
	if( (AVIERR_OK !=
		AVIFileCreateStream( pFile,		/* File pointer            */
							 ps,		/* Returned stream pointer */
							 &strhdr ))	/* Stream header           */
	  )
		return FALSE;

	return TRUE;
} /* #OF# AVI_CreateStream */

/*========================================================
Function : AVI_SetOptions
=========================================================*/
/* #FN#
   Sets options for a video stream */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
AVI_SetOptions(
	PAVISTREAM *ps,           /* #IN# */
	PAVISTREAM *psCompressed, /* #IN# */
	HWND  hWnd,               /* #IN# */
	BOOL *pError              /* #OUT# */
)
{
	AVICOMPRESSOPTIONS opts;
	AVICOMPRESSOPTIONS FAR* aopts[ 1 ] = { &opts };
	*pError = FALSE;

	ZeroMemory( &opts, sizeof(opts) );

	if( !AVISaveOptions( hWnd, 0, 1, ps, (LPAVICOMPRESSOPTIONS FAR *)&aopts ) )
		return FALSE;

	if( (AVIERR_OK !=
		AVIMakeCompressedStream( psCompressed, *ps, &opts, NULL ))
	  )
	{
		*pError = TRUE;
		return FALSE;
	}
	return TRUE;
} /* #OF# AVI_SetOptions */

/*========================================================
Function : AVI_SetFormat
=========================================================*/
/* #FN#
   Sets frame format for a video stream */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
AVI_SetFormat(
	PAVISTREAM *psCompressed, /* #IN# */
	LPBITMAPINFOHEADER lpbi   /* #IN# */
)
{
	if( (AVIERR_OK !=
		AVIStreamSetFormat( *psCompressed, 0,
							 lpbi,	/* Stream format */
							 lpbi->biSize +
							 lpbi->biClrUsed * sizeof(RGBQUAD) ))
	  )
		return FALSE;

	return TRUE;
} /* #OF# AVI_SetFormat */

/*========================================================
Function : AVI_AddFrame
=========================================================*/
/* #FN#
   Adds a video frame to the opened stream */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
AVI_AddFrame(
	PAVISTREAM psCompressed, /* #IN# */
	int        nTime,        /* #IN# */
	UCHAR     *pBitmapBits,  /* #IN# */
	LONG       nBufferLen    /* #IN# */
)
{
	if( (AVIERR_OK !=
		AVIStreamWrite( psCompressed,			/* Stream pointer     */
						nTime,					/* Time of this frame */
						1,						/* Number to write    */
						(LPBYTE)pBitmapBits,	/* Pointer to data    */
						nBufferLen,				/* Size of this frame */
						AVIIF_KEYFRAME,			/* Flags....          */
						NULL, NULL ))
	  )
		return FALSE;

	return TRUE;
} /* #OF# AVI_AddFrame */

/*========================================================
Function : AVI_SaveInfo
=========================================================*/
/* #FN#
   Saves simple information to an AVI file */
static
BOOL
/* #AS#
   Always TRUE */
AVI_SaveInfo(
	PAVIFILE pFile,  /* #IN# */
	char    *pszInfo /* #IN# */
)
{
	int nInfoLen = strlen( pszInfo ) + 1;
	int nHeadLen = 3 * sizeof(DWORD);

	LPSTR psz = (LPSTR)calloc( nHeadLen + nInfoLen, 1 );
	if( psz )
	{
		PDWORD pdw = (PDWORD)psz;
		pdw[ 0 ] = mmioStringToFOURCC( "INFO", 0 );
		pdw[ 1 ] = mmioStringToFOURCC( "ISBJ", 0 ); /* Subject */
		pdw[ 2 ] = nInfoLen;
		strncpy( psz + nHeadLen, pszInfo, nInfoLen );

		if( !AVIFileWriteData( pFile, mmioStringToFOURCC( "LIST", 0 ), psz, nHeadLen + nInfoLen ) )
		{
			free( psz );
			return TRUE;
		}
		free( psz );
	}
	return FALSE;
} /* #OF# AVI_SaveInfo */

/*========================================================
Function : AVI_CloseStream
=========================================================*/
/* #FN#
   Closes a video stream */
static
BOOL
/* #AS#
   Always TRUE */
AVI_CloseStream(
	PAVISTREAM *ps,          /* #IN# */
	PAVISTREAM *psCompressed /* #IN# */
)
{
	if( *ps )
	{
		AVIStreamClose( *ps );
		*ps = NULL;
	}
	if( *psCompressed )
	{
		AVIStreamClose( *psCompressed );
		*psCompressed = NULL;
	}
	return TRUE;
} /* #OF# AVI_CloseStream */

/*========================================================
Function : AVI_CloseFile
=========================================================*/
/* #FN#
   Closes an AVI file */
static
BOOL
/* #AS#
   Always TRUE */
AVI_CloseFile(
	PAVIFILE pFile /* #IN# */
)
{
	if( pFile )
		AVIFileRelease( pFile );

	return TRUE;
} /* #OF# AVI_CloseFile */

/*========================================================
Function : AVI_Exit
=========================================================*/
/* #FN#
   Closes a video stream */
static
BOOL
/* #AS#
   Always TRUE */
AVI_Exit( void )
{
	AVIFileExit();

	return TRUE;
} /* #OF# AVI_Exit */

/*========================================================
Function : Video_OpenOutput
=========================================================*/
/* #FN#
   Opens a video file for writing */
void
/* #AS#
   Nothing */
Video_OpenOutput(
	char *pszOutFileName /* #IN# */
)
{
	if( g_Screen.pfOutput )
		/* Close AVI Output file */
		Video_CloseOutput();

	/* Initialise AVI stuff */
	if( AVI_Init() )
	{
		PAVIFILE pfVideoOutput = NULL;

		if( AVI_FileOpenWrite( &pfVideoOutput, pszOutFileName ) )
		{
			/* Calculate the frame rate */
			int  nFrameRate = (TV_PAL == tv_mode ? g_Timer.nPalFreq : g_Timer.nNtscFreq) / (ST_DOUBLE_REFRESH ? g_nDoubleRate : refresh_rate);
			BOOL bShowMsg   = TRUE;

			if( AVI_CreateStream( pfVideoOutput, &s_ps, nFrameRate,
								  g_Screen.lpbmi->bmiHeader.biSizeImage,
								  g_Screen.lpbmi->bmiHeader.biWidth,
								 -g_Screen.lpbmi->bmiHeader.biHeight ) )
			{
				if( AVI_SetOptions( &s_ps, &s_psCompressed, g_hMainWnd, &bShowMsg ) )
				{
					bShowMsg = TRUE;

					/* Video compressors do not support bitmap mirroring */
					g_Screen.lpbmi->bmiHeader.biHeight *= -1;

					if( AVI_SetFormat( &s_psCompressed, &g_Screen.lpbmi->bmiHeader ) )
					{
						s_dwFrame = 0;
						/* Unleash the video recording */
						g_Screen.pfOutput = pfVideoOutput;

						/* Initialization succeeded, exit */
						return;
					}
					g_Screen.lpbmi->bmiHeader.biHeight *= -1;
				}
				AVI_CloseStream( &s_ps, &s_psCompressed );
			}
			AVI_CloseFile( pfVideoOutput );

			if( bShowMsg )
				DisplayMessage( NULL, IDS_ERROR_VFW_STREAM, 0, MB_ICONEXCLAMATION | MB_OK );
		}
		else
			DisplayMessage( NULL, IDS_ERROR_FILE_OPEN, 0, MB_ICONEXCLAMATION | MB_OK, pszOutFileName );

		AVI_Exit();
	}
	else
		DisplayMessage( NULL, IDS_ERROR_VFW_INIT, 0, MB_ICONEXCLAMATION | MB_OK );
} /* #OF# Video_OpenOutput */

/*========================================================
Function : Video_SaveFrame
=========================================================*/
/* #FN#
   Saves one frame to the video stream */
void
/* #AS#
   Nothing */
Video_SaveFrame(
	UCHAR *pBitmapBits, /* #IN# */
	LONG   nBufferLen   /* #IN# */
)
{
	if( g_Screen.pfOutput )
	{
		/* Add a next frame to the stream */
		AVI_AddFrame( s_psCompressed, ++s_dwFrame, pBitmapBits, nBufferLen );
	}
} /* #OF# Video_SaveFrame */

/*========================================================
Function : Video_CloseOutput
=========================================================*/
/* #FN#
   Closes the video file */
void
/* #AS#
   Nothing */
Video_CloseOutput( void )
{
	if( g_Screen.pfOutput )
	{
		AVI_CloseStream( &s_ps, &s_psCompressed );
		AVI_SaveInfo( g_Screen.pfOutput, "Created by Atari800Win PLus" );
		AVI_CloseFile( g_Screen.pfOutput );
		AVI_Exit();

		DisplayMessage( NULL, IDS_GFX_FILE_CLOSED, 0, MB_ICONINFORMATION | MB_OK );

		/* Restore mirroring of the top-down bitmaps */
		g_Screen.lpbmi->bmiHeader.biHeight *= -1;
		g_Screen.pfOutput = NULL;
	}
} /* #OF# Video_CloseOutput */
