/****************************************************************************
File    : MainFrame.cpp
/*
@(#) #SY# Atari800Win
@(#) #IS# CMainFrame implementation file
@(#) #BY# Richard Lawrence, Tomasz Szymankowski
@(#) #LM# 22.12.2000
*/

/*
Copyright (c) 1998 Richard Lawrence

This program is free software; you can redistribute it and/or modify it under the terms 
of the GNU General Public License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later version. This program is 
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details. You should have received a copy of the GNU
General Public License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "StdAfx.h"
#include "Atari800Win.h"
#include "Atari800WinDoc.h"
#include "Atari800WinView.h"
#include "Helpers.h"
#include "DriveDlg.h"
#include "SettingsDlg.h"
#include "FileAssociationsDlg.h"
#include "MiscellanousDlg.h"
#include "RomImagesDlg.h"
#include "GraphicsDlg.h"
#include "SoundDlg.h"
#include "HarddiskDlg.h"
#include "KeyboardDlg.h"
#include "JoystickDlg.h"
#include "MouseDlg.h"
#include "ErrorLogDlg.h"
#include "PaletteDlg.h"
#include "ConvertTypeDlg.h"
#include "WarningDlg.h"
#include "AboutDlg.h"
#include "MainFrame.h"

//#include "core.h"			// AtariWin core

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#define ID_ATARI_MACHINETYPE_BASE		ID_ATARI_MACHINETYPE_800OSA
#define ID_ATARI_VIDEOSYSTEM_BASE		ID_ATARI_VIDEOSYSTEM_PAL
#define ID_ATARI_AUTOFIRE_BASE			ID_ATARI_AUTOFIRE_OFF
#define ID_GRAPHICS_ARTIFACTING_BASE	ID_GRAPHICS_ARTIFACTING_NONE

#define SetClearView( val ) \
		if( m_pMainView ) m_pMainView->m_bClearScreen = val


WORD s_aShowMenuBarCmd[] =
{
	ID_FILE_LOADEXECUTABLE,
	ID_FILE_ATTACHDISKIMAGE_DRIVE1,
	ID_FILE_ATTACHDISKIMAGE_DRIVE2,
	ID_FILE_ATTACHDISKIMAGE_DRIVE3,
	ID_FILE_ATTACHDISKIMAGE_DRIVE4,
	ID_FILE_ATTACHDISKIMAGE_DRIVE5,
	ID_FILE_ATTACHDISKIMAGE_DRIVE6,
	ID_FILE_ATTACHDISKIMAGE_DRIVE7,
	ID_FILE_ATTACHDISKIMAGE_DRIVE8,
	ID_FILE_DISKDRIVES,
	ID_FILE_ATTACHCARTRIDGEIMAGE,
	ID_FILE_ROMIMAGES,
	ID_FILE_HARDDISKS,
	ID_FILE_READSTATE,
	ID_FILE_SAVESTATE_NORMAL,
	ID_FILE_SAVESTATE_VERBOSE,
	ID_ATARI_SETTINGS,
	ID_ATARI_KEYBOARD,
	ID_ATARI_JOYSTICKS,
	ID_ATARI_MOUSE,
	ID_GRAPHICS_GRAPHICSOPTIONS,
	ID_GRAPHICS_PALETTE,
	ID_GRAPHICS_SAVESCREENSHOT_NORMAL,
	ID_GRAPHICS_SAVESCREENSHOT_INTERLACED,
	ID_GRAPHICS_PERFORMANCETEST,
	ID_SOUND_SOUNDOPTIONS,
	ID_SOUND_PERFORMANCETEST,
	ID_SOUND_SAVESOUND,
	ID_MISC_EXTRAOPTIONS,
	ID_MISC_FILEASSOCIATIONS,
	ID_MISC_CONVERT_DCMTOATR,
	ID_MISC_CONVERT_ROMTOCART,
	ID_MISC_CLEARALLSETTINGS,
	ID_MISC_VIEWLOGFILE,
	ID_FILE_PRINT,
	ID_FILE_PRINT_DIRECT,
	ID_FILE_PRINT_PREVIEW,
	ID_FILE_PRINT_SETUP,
	ID_APP_ABOUT,
	ID_ATARI_PAUSE
};

const int s_nShowMenuBarCmdNo = sizeof(s_aShowMenuBarCmd)/sizeof(s_aShowMenuBarCmd[0]);


/////////////////////////////////////////////////////////////////////////////
// CMainFrame window

IMPLEMENT_DYNCREATE(CMainFrame, CFrameWnd)

BEGIN_MESSAGE_MAP(CMainFrame, CFrameWnd)
	//{{AFX_MSG_MAP(CMainFrame)
	ON_WM_CREATE()
	ON_COMMAND(ID_FILE_LOADEXECUTABLE, OnFileLoadExecutable)
	ON_COMMAND(ID_FILE_ATTACHDISKIMAGE_DRIVE1, OnFileAttachDiskImageDrive1)
	ON_COMMAND(ID_FILE_ATTACHDISKIMAGE_DRIVE2, OnFileAttachDiskImageDrive2)
	ON_COMMAND(ID_FILE_ATTACHDISKIMAGE_DRIVE3, OnFileAttachDiskImageDrive3)
	ON_COMMAND(ID_FILE_ATTACHDISKIMAGE_DRIVE4, OnFileAttachDiskImageDrive4)
	ON_COMMAND(ID_FILE_ATTACHDISKIMAGE_DRIVE5, OnFileAttachDiskImageDrive5)
	ON_COMMAND(ID_FILE_ATTACHDISKIMAGE_DRIVE6, OnFileAttachDiskImageDrive6)
	ON_COMMAND(ID_FILE_ATTACHDISKIMAGE_DRIVE7, OnFileAttachDiskImageDrive7)
	ON_COMMAND(ID_FILE_ATTACHDISKIMAGE_DRIVE8, OnFileAttachDiskImageDrive8)
	ON_COMMAND(ID_FILE_ATTACHCARTRIDGEIMAGE, OnFileAttachCartridgeImage)
	ON_COMMAND(ID_FILE_TURNDISK, OnFileTurnDisk)
	ON_COMMAND(ID_FILE_DISKDRIVES, OnFileDiskDrives)
	ON_COMMAND(ID_FILE_ROMIMAGES, OnFileRomImages)
	ON_COMMAND(ID_FILE_HARDDISKS, OnFileHardDisks)
	ON_COMMAND(ID_FILE_READSTATE, OnFileReadState)
	ON_COMMAND(ID_FILE_SAVESTATE_NORMAL, OnFileSaveStateNormal)
	ON_COMMAND(ID_FILE_SAVESTATE_VERBOSE, OnFileSaveStateVerbose)
	ON_COMMAND(ID_ATARI_MACHINETYPE, OnAtariMachineType)
	ON_COMMAND(ID_ATARI_MACHINETYPE_800OSA, OnAtariMachineType800osa)
	ON_COMMAND(ID_ATARI_MACHINETYPE_800OSB, OnAtariMachineType800osb)
	ON_COMMAND(ID_ATARI_MACHINETYPE_800XL, OnAtariMachineType800xl)
	ON_COMMAND(ID_ATARI_MACHINETYPE_130XE, OnAtariMachineType130xe)
	ON_COMMAND(ID_ATARI_MACHINETYPE_320XECOMPY, OnAtariMachineType320xeCompy)
	ON_COMMAND(ID_ATARI_MACHINETYPE_320XERAMBO, OnAtariMachineType320xeRambo)
	ON_COMMAND(ID_ATARI_MACHINETYPE_5200, OnAtariMachineType5200)
	ON_COMMAND(ID_ATARI_VIDEOSYSTEM, OnAtariVideoSystem)
	ON_COMMAND(ID_ATARI_VIDEOSYSTEM_NTSC, OnAtariVideoSystemNtsc)
	ON_COMMAND(ID_ATARI_VIDEOSYSTEM_PAL, OnAtariVideoSystemPal)
	ON_COMMAND(ID_ATARI_SETTINGS, OnAtariSettings)
	ON_COMMAND(ID_ATARI_SIOPATCH, OnAtariSioPatch)
	ON_COMMAND(ID_ATARI_FULLSPEED, OnAtariFullSpeed)
	ON_COMMAND(ID_ATARI_PAUSE, OnAtariPause)
	ON_COMMAND(ID_ATARI_KEYBOARD, OnAtariKeyboard)
	ON_COMMAND(ID_ATARI_JOYSTICKS, OnAtariJoysticks)
	ON_COMMAND(ID_ATARI_MOUSE, OnAtariMouse)
	ON_COMMAND(ID_ATARI_WARMSTART, OnAtariWarmstart)
	ON_COMMAND(ID_ATARI_COLDSTART, OnAtariColdstart)
	ON_COMMAND(ID_GRAPHICS_GRAPHICSOPTIONS, OnGraphicsOptions)
	ON_COMMAND(ID_GRAPHICS_TOGGLEMODES, OnGraphicsToggleModes)
	ON_COMMAND(ID_GRAPHICS_GDIFORWINDOWS, OnGraphicsGdiForWindows)
	ON_COMMAND(ID_GRAPHICS_SCANLINES, OnGraphicsScanlines)
	ON_COMMAND(ID_GRAPHICS_ARTIFACTING, OnGraphicsArtifacting)
	ON_COMMAND(ID_GRAPHICS_ARTIFACTING_NONE, OnGraphicsArtifactingNone)
	ON_COMMAND(ID_GRAPHICS_ARTIFACTING_BLUEBROWN1, OnGraphicsArtifactingBluebrown1)
	ON_COMMAND(ID_GRAPHICS_ARTIFACTING_BLUEBROWN2, OnGraphicsArtifactingBluebrown2)
	ON_COMMAND(ID_GRAPHICS_ARTIFACTING_GTIA, OnGraphicsArtifactingGtia)
	ON_COMMAND(ID_GRAPHICS_ARTIFACTING_CTIA, OnGraphicsArtifactingCtia)
	ON_COMMAND(ID_GRAPHICS_PALETTE, OnGraphicsPalette)
	ON_COMMAND(ID_GRAPHICS_SAVESCREENSHOT_NORMAL, OnGraphicsSaveScreenshotNormal)
	ON_COMMAND(ID_GRAPHICS_SAVESCREENSHOT_INTERLACED, OnGraphicsSaveScreenshotInterlaced)
	ON_COMMAND(ID_GRAPHICS_PERFORMANCETEST, OnGraphicsPerformanceTest)
	ON_COMMAND(ID_SOUND_SOUNDOPTIONS, OnSoundOptions)
	ON_COMMAND(ID_SOUND_DIGITIZEDEFFECTS, OnSoundDigitizedEffects)
	ON_COMMAND(ID_SOUND_STEREO, OnSoundStereo)
	ON_COMMAND(ID_SOUND_MUTE, OnSoundMute)
	ON_COMMAND(ID_SOUND_SAVESOUND, OnSoundSaveSound)
	ON_COMMAND(ID_SOUND_PERFORMANCETEST, OnSoundPerformanceTest)
	ON_COMMAND(ID_MISC_MOUSECAPTURE, OnMiscMouseCapture)
	ON_COMMAND(ID_MISC_EXTRAOPTIONS, OnMiscExtraOptions)
	ON_COMMAND(ID_MISC_FILEASSOCIATIONS, OnMiscFileAssociations)
	ON_COMMAND(ID_MISC_CONVERT_DCMTOATR, OnMiscConvertDcmToAtr)
	ON_COMMAND(ID_MISC_CONVERT_ROMTOCART, OnMiscConvertRomToCart)
	ON_COMMAND(ID_MISC_CONVERT_ASCIITOATASCII, OnMiscConvertAsciiToAtascii)
	ON_COMMAND(ID_MISC_CONVERT_ATASCIITOASCII, OnMiscConvertAtasciiToAscii)
	ON_COMMAND(ID_MISC_CLEARALLSETTINGS, OnMiscClearAllSettings)
	ON_COMMAND(ID_MISC_RESTARTEMULATION, OnMiscRestartEmulation)
	ON_COMMAND(ID_MISC_VIEWLOGFILE, OnMiscViewLogFile)
	ON_COMMAND(ID_MISC_MONITOR, OnMiscMonitor)
	ON_COMMAND(ID_HELP_KEYBOARD_LAYOUT, OnHelpKeyboardLayout)
	ON_COMMAND(ID_HELP_KEYBOARD_SHORTCUTS, OnHelpKeyboardShortcuts)
	ON_COMMAND(ID_HELP_WEB_PAGE, OnHelpWebPage)
	ON_COMMAND(ID_APP_ABOUT, OnHelpAbout)
	ON_UPDATE_COMMAND_UI(ID_FILE_TURNDISK, OnUpdateFileTurnDisk)
	ON_UPDATE_COMMAND_UI(ID_ATARI_MACHINETYPE_800OSA, OnUpdateAtariMachineType)
	ON_UPDATE_COMMAND_UI(ID_ATARI_VIDEOSYSTEM_NTSC, OnUpdateAtariVideoSystem)
	ON_UPDATE_COMMAND_UI(ID_ATARI_SIOPATCH, OnUpdateAtariSioPatch)
	ON_UPDATE_COMMAND_UI(ID_ATARI_FULLSPEED, OnUpdateAtariFullSpeed)
	ON_UPDATE_COMMAND_UI(ID_ATARI_PAUSE, OnUpdateAtariPause)
	ON_UPDATE_COMMAND_UI(ID_GRAPHICS_TOGGLEMODES, OnUpdateGraphicsToggleModes)
	ON_UPDATE_COMMAND_UI(ID_GRAPHICS_GDIFORWINDOWS, OnUpdateGraphicsGdiForWindows)
	ON_UPDATE_COMMAND_UI(ID_GRAPHICS_SCANLINES, OnUpdateGraphicsScanlines)
	ON_UPDATE_COMMAND_UI(ID_GRAPHICS_ARTIFACTING_NONE, OnUpdateGraphicsArtifacting)
	ON_UPDATE_COMMAND_UI(ID_SOUND_DIGITIZEDEFFECTS, OnUpdateSoundDigitizedEffects)
	ON_UPDATE_COMMAND_UI(ID_SOUND_STEREO, OnUpdateSoundStereo)
	ON_UPDATE_COMMAND_UI(ID_SOUND_MUTE, OnUpdateSoundMute)
	ON_UPDATE_COMMAND_UI(ID_SOUND_SAVESOUND, OnUpdateSoundSaveSound)
	ON_UPDATE_COMMAND_UI(ID_MISC_MOUSECAPTURE, OnUpdateMiscMouseCapture)
	ON_WM_MENUSELECT()
	ON_WM_NCLBUTTONDOWN()
	ON_WM_NCLBUTTONDBLCLK()
	ON_WM_WINDOWPOSCHANGING()
	ON_WM_SIZE()
	ON_WM_TIMER()
	ON_WM_CLOSE()
	ON_WM_DROPFILES()
	ON_WM_NCMOUSEMOVE()
	ON_WM_MOVE()
	ON_UPDATE_COMMAND_UI(ID_ATARI_AUTOFIRE_OFF, OnUpdateAtariAutofire)
	ON_COMMAND(ID_ATARI_AUTOFIRE_OFF, OnAtariAutofireOff)
	ON_COMMAND(ID_ATARI_AUTOFIRE_FIREDEPENDENT, OnAtariAutofireFireDependent)
	ON_COMMAND(ID_ATARI_AUTOFIRE_ALLTIME, OnAtariAutofireAllTime)
	ON_COMMAND(ID_ATARI_AUTOFIRE, OnAtariAutofire)
	ON_UPDATE_COMMAND_UI(ID_ATARI_MACHINETYPE_800OSB, OnUpdateAtariMachineType)
	ON_UPDATE_COMMAND_UI(ID_ATARI_MACHINETYPE_800XL, OnUpdateAtariMachineType)
	ON_UPDATE_COMMAND_UI(ID_ATARI_MACHINETYPE_130XE, OnUpdateAtariMachineType)
	ON_UPDATE_COMMAND_UI(ID_ATARI_MACHINETYPE_320XECOMPY, OnUpdateAtariMachineType)
	ON_UPDATE_COMMAND_UI(ID_ATARI_MACHINETYPE_320XERAMBO, OnUpdateAtariMachineType)
	ON_UPDATE_COMMAND_UI(ID_ATARI_MACHINETYPE_5200, OnUpdateAtariMachineType)
	ON_UPDATE_COMMAND_UI(ID_ATARI_VIDEOSYSTEM_PAL, OnUpdateAtariVideoSystem)
	ON_UPDATE_COMMAND_UI(ID_GRAPHICS_ARTIFACTING_BLUEBROWN1, OnUpdateGraphicsArtifacting)
	ON_UPDATE_COMMAND_UI(ID_GRAPHICS_ARTIFACTING_BLUEBROWN2, OnUpdateGraphicsArtifacting)
	ON_UPDATE_COMMAND_UI(ID_GRAPHICS_ARTIFACTING_GTIA, OnUpdateGraphicsArtifacting)
	ON_UPDATE_COMMAND_UI(ID_GRAPHICS_ARTIFACTING_CTIA, OnUpdateGraphicsArtifacting)
	ON_UPDATE_COMMAND_UI(ID_ATARI_AUTOFIRE_FIREDEPENDENT, OnUpdateAtariAutofire)
	ON_UPDATE_COMMAND_UI(ID_ATARI_AUTOFIRE_ALLTIME, OnUpdateAtariAutofire)
	//}}AFX_MSG_MAP
	/* This is not supported by ClassWizard */
	ON_WM_EXITMENULOOP()
	/* User defined commands */
	ON_MESSAGE(WM_USER, OnMessageCmdLine)
	/* Global help commands */
	ON_COMMAND(ID_HELP_FINDER, OnHelpFinder)
	ON_COMMAND(ID_HELP, OnHelp)
	ON_COMMAND(ID_DEFAULT_HELP, OnHelpFinder)
	ON_COMMAND(ID_CONTEXT_HELP, OnContextHelp)
//	ON_COMMAND(ID_HELP_INDEX, CFrameWnd::OnHelpIndex)
END_MESSAGE_MAP()

static UINT indicators[] =
{
	ID_SEPARATOR /* status line indicator */
};


/*========================================================
Method   : CMainFrame::CMainFrame
=========================================================*/
/* #FN#
   Standard constructor */
CMainFrame::
CMainFrame()
{
	ULONG ulMiscStates = g_ulMiscStates;

	m_bPosChanging = TRUE;
	m_bCleanUpMenu = FALSE;
	m_bMenuShowed  = FALSE;

	/* Get system information the emulator is running on */
	GetSystemVersion( &g_unSystemInfo );

	/* These variables were not written in the Registry */
	m_strSoundName    = "atarisnd.wav";
	m_strSnapshotName = "atari000.pcx";

	g_nPalFreq  = (50 * g_nSpeedPercent) / 100;
	g_nNtscFreq = (60 * g_nSpeedPercent) / 100;

	/* The last session was crashed */
	if( g_ulMiscStates & MS_LAST_BOOT_FAILED &&
		!FindAppWindow() )
	{
		if( IDYES == DisplayMessage( NULL, IDS_WARN_LOCKUP, 0, MB_ICONQUESTION | MB_YESNO ) )
			InitialiseRegistry( NULL, TRUE );
	}
	/* Assign functions to arrow keys */
	SetArrowKeys( g_nArrowsMode );

	/* Load keys template and external palette */
	if( g_ulMiscStates & MS_USE_KEY_TEMPLATE && 
		!ReadKeyTemplate( g_szTemplateFile ) )
	{
		g_ulMiscStates &= ~MS_USE_KEY_TEMPLATE;
		DisplayMessage( NULL, IDS_ERROR_NO_KEYTEMP, 0, MB_ICONEXCLAMATION | MB_OK, g_szTemplateFile );
	}
	if( g_ulMiscStates & MS_USE_EXT_PALETTE && 
		!read_palette( g_szPaletteFile ) )
	{
		g_ulMiscStates &= ~MS_USE_EXT_PALETTE;
		DisplayMessage( NULL, IDS_ERROR_NO_PALETTE, 0, MB_ICONEXCLAMATION | MB_OK, g_szPaletteFile );
	}
	if( g_ulMiscStates != ulMiscStates )
		WriteRegDWORD( NULL, REG_MISC_STATES, g_ulMiscStates );

	if( g_ulMiscStates & MS_CAPTURE_CTRLESC )
		/* Disable task switching */
		EnableEscCapture( TRUE );

	/* Register extensions automatically if the user wants it */
	if( g_ulMiscStates & MS_AUTO_REG_FILES )
	{
		char szHomeDir[ MAX_PATH ];

		if( !GetHomeDirectory( szHomeDir ) )
			GetCurrentDirectory( MAX_PATH, szHomeDir );

		WriteRegFileExt( g_ulFileAssociations, szHomeDir, g_szCmdLine );
	}
	/* Call the main function of the Atari800 emulator */
	main( g_argc, g_argv );

} /* #OF# CMainFrame::CMainFrame */

/*========================================================
Method   : CMainFrame::~CMainFrame
=========================================================*/
/* #FN#
   Destructor */
CMainFrame::
~CMainFrame()
{
	/* In case desktop isn't getting refreshed when exiting directly from
	   DirectDraw full screen */
	::InvalidateRect( NULL, NULL, FALSE );
} /* #OF# CMainFrame::~CMainFrame */


/////////////////////////////////////////////////////////////////////////////
// CMainFrame implementation

/*========================================================
Method   : CMainFrame::PreCreateWindow
=========================================================*/
/* #FN#
   Called by the framework before the creation of the Windows window */
BOOL
/* #AS#
   Nonzero if the window creation should continue, 0 to indicate creation
   failure */
CMainFrame::
PreCreateWindow(
	CREATESTRUCT& cs
)
{
	cs.x = g_nStartX;
	cs.y = g_nStartY;

	cs.style |= CS_OWNDC;
	cs.style &= ~WS_THICKFRAME;

	return CFrameWnd::PreCreateWindow( cs );
} /* #OF# CMainFrame::PreCreateWindow */

/*========================================================
Method   : CMainFrame::ActivateFrame
=========================================================*/
/* #FN#
   Activates and restores the frame window */
void
/* #AS#
   Nothing */
CMainFrame::
ActivateFrame(
	int nCmdShow /* #IN# Parameter to pass to CWnd::ShowWindow */
)
{
	/* Get a real size of the status bar */
	CRect rect;
	m_wndStatusBar.GetWindowRect( rect );
	g_nStatusSize = rect.Height();

	CFrameWnd::ActivateFrame( nCmdShow );
} /* #OF# CMainFrame::ActivateFrame */

/*========================================================
Method   : CMainFrame::DriveInsert
=========================================================*/
/* #FN#
   Inserts disk image to pointed virtual drive */
void
/* #AS#
   Nothing */
CMainFrame::
DriveInsert(
	int nDriveNum
)
{
	CString strDiskName = sio_filename[ nDriveNum - 1 ];
	char    szPrompt[ 64 ];

	sprintf( szPrompt, "Select disk to insert in drive %d", nDriveNum );

	if( PickFileName( TRUE, strDiskName, szPrompt,
					  PF_DSK_FILTER, NULL, PF_LOAD_FLAGS ) &&
		!strDiskName.IsEmpty() )
	{
		SIO_Dismount( nDriveNum );
		SIO_Mount( nDriveNum, (LPSTR)(LPCSTR)strDiskName, FALSE );
		WriteRegDrives( NULL );
	}
	CleanScreen();
} /* #OF# CMainFrame::DriveInsert */

/*========================================================
Method   : CMainFrame::SaveState
=========================================================*/
/* #FN#
   Saves state of the Atari to a file */
void
/* #AS#
   Nothing */
CMainFrame::
SaveState(
	int nSaveRom
)
{
	if( zlib_capable() == -1 )
	{
		DisplayMessage( NULL, IDS_ERROR_NO_ZLIB, 0, MB_ICONEXCLAMATION | MB_OK );
	}
	else
	if( !nSaveRom || (nSaveRom &&
		IDYES == DisplayMessage( NULL, IDS_WARN_VERBOSE, 0, MB_ICONQUESTION | MB_YESNO )) )
	{
		char szStateName[ MAX_PATH ];

		/* Get the most recently used file name to use in a file dialog */
		strncpy( szStateName, atari_state_dir, MAX_PATH );

		if( PickFileName( FALSE, szStateName, "Save Atari saved-state file",
						  PF_A8S_FILTER, "a8s", PF_SAVE_FLAGS, TRUE, DEFAULT_A8S ) &&
			*szStateName != '\0' )
		{
			if( _stricmp( szStateName, atari_state_dir ) != 0 )
			{
				strcpy( atari_state_dir, szStateName );
				WriteRegString( NULL, REG_STATE_PATH, atari_state_dir );
			}
			if( !SaveAtariState( atari_state_dir, "wb", nSaveRom ) )
				DisplayMessage( NULL, IDS_ERROR_A8S_SAVE, 0, MB_ICONEXCLAMATION | MB_OK );
		}
		DescribeAtariSystem();
	}
	CleanScreen();
} /* #OF# CMainFrame::SaveState */

/*========================================================
Method   : CMainFrame::SaveSnapshot
=========================================================*/
/* #FN#
   Saves snapshot of the Atari screen to .PCX file */
void
/* #AS#
   Nothing */
CMainFrame::
SaveSnapshot(
	int nInterlace
)
{
	static BOOL bChanged = FALSE;
	CString strSnapshotName, strPrompt;

	if( !bChanged )
		m_strSnapshotName = Find_PCX_name();

	strSnapshotName = m_strSnapshotName;
	strPrompt.Format( "Save Atari %s screenshot file", nInterlace ? "interlaced" : "normal" );

	if( PickFileName( FALSE, m_strSnapshotName, strPrompt,
					  PF_PCX_FILTER, "pcx", PF_SAVE_FLAGS, FALSE ) &&
		!m_strSnapshotName.IsEmpty() )
	{
		if( !Save_PCX_file( nInterlace, (LPSTR)(LPCSTR)m_strSnapshotName ) )
			DisplayMessage( NULL, IDS_ERROR_PCX_SAVE, 0, MB_ICONEXCLAMATION | MB_OK );

		if( !bChanged &&
			strSnapshotName.CompareNoCase( m_strSnapshotName.Right( m_strSnapshotName.GetLength() - m_strSnapshotName.ReverseFind( '\\' ) - 1 ) ) != 0 )
			bChanged = TRUE;
	}
	CleanScreen();
} /* #OF# CMainFrame::SaveSnapshot */

/*========================================================
Method   : CMainFrame::PreTranslateMessage
=========================================================*/
/* #FN#
   Translates window messages before they are dispatched to the
   TranslateMessage and DispatchMessage Windows functions */
BOOL
/* #AS#
   Nonzero if the message was translated and should not be dispatched;
   0 if the message was not translated and should be dispatched */
CMainFrame::
PreTranslateMessage(
	MSG *pMsg
)
{
	WPARAM wp = pMsg->wParam;

	switch( pMsg->message )
	{
		/* This needs to be handled here instead of a OnXXX message
		   because we want to intercept control keys before they are
		   parsed as accelerators */
		case WM_KEYDOWN:
		{
			if( wp == (WPARAM)g_nCurrentVK )
				return TRUE;

			if( g_bKBJoystick || g_anToggleKeys[ wp ] )
			{
				if( ToggleKeyDown( wp, pMsg->lParam ) )
					return TRUE;
			}
			if( g_ulMiscStates & MS_USE_KEY_TEMPLATE &&
				g_aucKeysCovered[ wp ] )
			{
				g_nNewKey = g_nKBTable[ wp ];
				
				if( g_nNewKey != AKEY_NONE )
				{
					if( SHIFT_KEY )
						g_nNewKey |= AKEY_SHFT;
					if( CTRL_KEY )
						g_nNewKey |= AKEY_CTRL;

					g_nNewVK = wp;
				}
				else
					g_nNewVK = 0;
				return TRUE;
			}
			if( SHIFT_KEY && CTRL_KEY )
			{
				g_nNewKey = g_anShftCtrlKeys[ wp ];
				g_nNewVK  = (g_nNewKey != AKEY_NONE ? wp : 0);
				return TRUE;
			}
			if( CTRL_KEY )
			{
				g_nNewKey = g_anCtrlKeys[ wp ];
				g_nNewVK  = (g_nNewKey != AKEY_NONE ? wp : 0);
				return TRUE;
			}
			if( SHIFT_KEY )
			{
				g_nNewKey = g_anShftKeys[ wp ];
				g_nNewVK  = (g_nNewKey != AKEY_NONE ? wp : 0);
				/* Do standard translate for function keys */
				if( wp < VK_F1 || wp > VK_F12 )
					return TRUE;
			}
			if( g_anNormKeys[ wp ] != AKEY_NONE )
			{
				g_nNewKey = g_anNormKeys[ wp ];
				g_nNewVK  = (g_nNewKey != AKEY_NONE ? wp : 0);
				/* The CapsLock indicator should be used */
//				if( g_nNewVK != VK_CAPITAL ) /* CapsLock */
				return TRUE;
			}
			break;
		}

		case WM_KEYUP:
		{
			if( g_bKBJoystick || g_anToggleKeys[ wp ] )
			{
				if( ToggleKeyUp( wp, pMsg->lParam ) )
					return TRUE;
			}
			if( wp == (WPARAM)g_nCurrentVK )
			{
				g_nCurrentKey = AKEY_NONE;
				g_nCurrentVK  = 0;
				KEYPRESSED    = FALSE;
			}
			if( wp == (WPARAM)g_nNewVK )
			{
				if( !(g_nNewKey & SPECIAL_HANDLER_MASK) )
				{
					g_nNewKey = AKEY_NONE;
					g_nNewVK  = 0;
				}
			}
			break;
		}

		case WM_SYSKEYUP:
		{
			/* Special handling for Ctrl+Esc key sequence */
			if( VK_ESCAPE == wp && CTRL_KEY )
			{
				/* The Atari_Keyboard function will "release"
				   this key automatically; I must look around
				   if there is a better way for doing that */
				g_nNewKey = g_anCtrlKeys[ wp ];
				g_nNewVK  = (g_nNewKey != AKEY_NONE ? wp : 0);
				return TRUE;
			}
			break;
		}
	}
	return CFrameWnd::PreTranslateMessage( pMsg );
} /* #OF# CMainFrame::PreTranslateMessage */


/////////////////////////////////////////////////////////////////////////////
// CMainFrame diagnostics

#ifdef _DEBUG
void
CMainFrame::
AssertValid() const
{
	CFrameWnd::AssertValid();
}

void
CMainFrame::
Dump( CDumpContext &dc ) const
{
	CFrameWnd::Dump( dc );
}

#endif //_DEBUG


/////////////////////////////////////////////////////////////////////////////
// CMainFrame message handlers

/*========================================================
Method   : CMainFrame::OnCreate
=========================================================*/
/* #FN#
   Called by the framework when an application requests that the Windows
   window be created */
int
/* #AS#
   0 to continue the creation, 1 to destroy the Windows window */
CMainFrame::
OnCreate(
	LPCREATESTRUCT lpCreateStruct /* #IN# Pointer to the creation info structure */
)
{
	LARGE_INTEGER lnTimerRes;

	QueryPerformanceFrequency( &lnTimerRes ); 
	if( lnTimerRes.LowPart == 0 )
	{
		DisplayMessage( NULL, IDS_ERROR_NO_HITIMER, 0, MB_ICONSTOP | MB_OK );
		return FALSE;
	}

	if( CFrameWnd::OnCreate( lpCreateStruct ) == -1 )
		return -1;
	
	if( !m_wndStatusBar.Create( this ) ||
		!m_wndStatusBar.SetIndicators( indicators,
		  sizeof(indicators)/sizeof(UINT) ) )
	{
		TRACE0("Failed to create status bar\n");
		return -1; // fail to create
	}
	/* The status bar should be accesible by app class */
	m_pMainApp = (CAtari800WinApp*)AfxGetApp();
	ASSERT(m_pMainApp);
	m_pMainApp->m_pStatusBar = &m_wndStatusBar;

	/* Remove the "Maximize" option of the sys menu */
	CMenu *pSysMenu = GetSystemMenu( FALSE );
	if( pSysMenu != NULL )
	{
		pSysMenu->RemoveMenu( SC_MAXIMIZE, MF_BYCOMMAND );
		pSysMenu->RemoveMenu( SC_SIZE, MF_BYCOMMAND );
	}
	DragAcceptFiles( TRUE );

	/* Get global handles */
	g_hMainWnd  = GetSafeHwnd(); /* "C" modules need this */
	g_hInstance = AfxGetInstanceHandle();

	return 0;
} /* #OF# CMainFrame::OnCreate */

/*========================================================
Method   : CMainFrame::OnFileLoadExecutable
=========================================================*/
/* #FN#
   Loads an Atari executable file directly */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileLoadExecutable()
{
	UINT    unFileType  = IAF_BIN_IMAGE;
	CString	strFileName = atari_exe_dir;

	if( PickFileName( TRUE, strFileName, "Select Atari executable file",
					  PF_BIN_FILTER, "xex", PF_LOAD_FLAGS ) &&
		!strFileName.IsEmpty() )
	{
		if( IsAtariFile( strFileName, &unFileType ) )
		{
			RestartEmulation();

			if( !RunAtariExe( (LPSTR)(LPCSTR)strFileName ) )
				DisplayMessage( NULL, IDS_ERROR_BIN_RUN, 0, MB_ICONEXCLAMATION | MB_OK, strFileName );
		}
		else
			DisplayMessage( NULL, IDS_ERROR_BIN_LOAD, 0, MB_ICONEXCLAMATION | MB_OK, strFileName );
	}
	DescribeAtariSystem();
	/* Clean up the screen */
	CleanScreen();
} /* #OF# CMainFrame::OnFileLoadExecutable */

/*========================================================
Method   : CMainFrame::OnFileAttachDiskImageDrive1
=========================================================*/
/* #FN#
   Attaches a disk image to an emulated drive 1 */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileAttachDiskImageDrive1() 
{
	DriveInsert( 1 );
} /* #OF# CMainFrame::OnFileAttachDiskImageDrive1 */

/*========================================================
Method   : CMainFrame::OnFileAttachDiskImageDrive2
=========================================================*/
/* #FN#
   Attaches a disk image to an emulated drive 2 */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileAttachDiskImageDrive2() 
{
	DriveInsert( 2 );
} /* #OF# CMainFrame::OnFileAttachDiskImageDrive2 */

/*========================================================
Method   : CMainFrame::OnFileAttachDiskImageDrive3
=========================================================*/
/* #FN#
   Attaches a disk image to an emulated drive 3 */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileAttachDiskImageDrive3() 
{
	DriveInsert( 3 );
} /* #OF# CMainFrame::OnFileAttachDiskImageDrive3 */

/*========================================================
Method   : CMainFrame::OnFileAttachDiskImageDrive4
=========================================================*/
/* #FN#
   Attaches a disk image to an emulated drive 4 */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileAttachDiskImageDrive4() 
{
	DriveInsert( 4 );
} /* #OF# CMainFrame::OnFileAttachDiskImageDrive4 */

/*========================================================
Method   : CMainFrame::OnFileAttachDiskImageDrive5
=========================================================*/
/* #FN#
   Attaches a disk image to an emulated drive 5 */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileAttachDiskImageDrive5() 
{
	DriveInsert( 5 );
} /* #OF# CMainFrame::OnFileAttachDiskImageDrive5 */

/*========================================================
Method   : CMainFrame::OnFileAttachDiskImageDrive6
=========================================================*/
/* #FN#
   Attaches a disk image to an emulated drive 6 */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileAttachDiskImageDrive6() 
{
	DriveInsert( 6 );
} /* #OF# CMainFrame::OnFileAttachDiskImageDrive6 */

/*========================================================
Method   : CMainFrame::OnFileAttachDiskImageDrive7
=========================================================*/
/* #FN#
   Attaches a disk image to an emulated drive 7 */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileAttachDiskImageDrive7() 
{
	DriveInsert( 7 );
} /* #OF# CMainFrame::OnFileAttachDiskImageDrive7 */

/*========================================================
Method   : CMainFrame::OnFileAttachDiskImageDrive8
=========================================================*/
/* #FN#
   Attaches a disk image to an emulated drive 8 */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileAttachDiskImageDrive8() 
{
	DriveInsert( 8 );
} /* #OF# CMainFrame::OnFileAttachDiskImageDrive8 */

/*========================================================
Method   : CMainFrame::OnFileTurnDisk
=========================================================*/
/* #FN#
   Turns disk images in an emulated drive 1 */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileTurnDisk()
{
	char szDiskPath[ MAX_PATH ];

//	ReadRegBinary( NULL, g_apszDriveRegID[ 0 ], szDiskPath, MAX_PATH, FALSE );
	strncpy( szDiskPath, sio_filename[ 0 ], MAX_PATH );
	if( strcmp( szDiskPath, "Empty" ) != 0 &&
		strcmp( szDiskPath, "Off" ) != 0 )
	{
		char  szDrive [ _MAX_DRIVE ];
		char  szDir   [ _MAX_DIR ];
		char  szFile  [ _MAX_FNAME ];
		char  szExt   [ _MAX_EXT ];
		char  szCurDir[ MAX_PATH ];
		char  szNewDir[ MAX_PATH ];

		const char acSides[ 3 ][ 2 ] = { {'1', '9'}, {'A', 'Z'}, {'a', 'z'} };
		char  cDiskSide, cNewDiskSide;
		BOOL  bTurnDisk = FALSE;
		int   i; /* Loops index */

		CFile cfFile;

		GetCurrentDirectory( MAX_PATH, szCurDir );
		strncpy( szNewDir, szDiskPath, MAX_PATH );

		for( i = strlen( szNewDir ); i > 0 && szNewDir[ i ] != '\\'; i-- );
		if( i > 0 )
			szNewDir[ i ] = '\0';

		SetCurrentDirectory( szNewDir );

		/* Split disk full path */
		_tsplitpath( szDiskPath, szDrive, szDir, szFile, szExt );

		cNewDiskSide = cDiskSide = szFile[ strlen( szFile ) - 1 ];

		for( i = 0; i < 3 && !bTurnDisk; i++ )
		{
			if( cNewDiskSide >= acSides[ i ][ 0 ] && cNewDiskSide <= acSides[ i ][ 1 ] )
			{
				do
				{
					if( ++cNewDiskSide > acSides[ i ][ 1 ] )
						cNewDiskSide = acSides[ i ][ 0 ];

					szFile[ strlen( szFile ) - 1 ] = cNewDiskSide;
					_makepath( szDiskPath, szDrive, szDir, szFile, szExt );
					if( cfFile.Open( szDiskPath, CFile::modeRead | CFile::typeBinary, NULL ) )
					{
						bTurnDisk = TRUE;
						cfFile.Close();
						break;
					}
				}
				while( cNewDiskSide != cDiskSide );
			}
		}
		if( bTurnDisk )
		{
			SIO_Dismount( 1 );
			SIO_Mount( 1, szDiskPath, FALSE );
			WriteRegDrives( NULL );
		}
		/* Restore directory */
		SetCurrentDirectory( szCurDir );
	}
} /* #OF# CMainFrame::OnFileTurnDisk */

/*========================================================
Method   : CMainFrame::OnFileDiskDrives
=========================================================*/
/* #FN#
   Displays Drive Selections dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileDiskDrives()
{
	CDriveDlg dlgDrive( this );

	dlgDrive.DoModal();
	CleanScreen();
} /* #OF# CMainFrame::OnFileDiskDrives */

/*========================================================
Method   : CMainFrame::OnFileAttachCartridgeImage
=========================================================*/
/* #FN#
   Displays Attach Cartridge file dialog */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileAttachCartridgeImage()
{
	char szNewCart[ MAX_PATH ];

	strncpy( szNewCart, g_szOtherRom, MAX_PATH );

	if( PickCartridge( szNewCart ) )
	{
		int nCartType = FindCartType( szNewCart, FALSE, FALSE );

		if( NO_CART != nCartType && /* CARE: read errors will be ignored! */
			(_stricmp( szNewCart, g_szOtherRom ) != 0 ||
			cart_type != nCartType || !rom_inserted) )
		{
			AttachCartridge( szNewCart, nCartType );
			strcpy( g_szOtherRom, g_szCurrentRom );

			WriteRegString( NULL, REG_CURRENT_ROM, g_szCurrentRom );
			WriteRegString( NULL, REG_OTHER_ROM,   g_szOtherRom );
			WriteRegDWORD ( NULL, REG_CART_TYPE,   cart_type );

			RestartEmulation();
		}
	}
	CleanScreen();
} /* #OF# CMainFrame::OnFileAttachCartridgeImage */

/*========================================================
Method   : CMainFrame::OnFileRomImages
=========================================================*/
/* #FN#
   Displays ROM Images dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileRomImages() 
{
	CRomImagesDlg dlgRomImages( this );

	dlgRomImages.DoModal();
	CleanScreen();
} /* #OF# CMainFrame::OnFileRomImages */

/*========================================================
Method   : CMainFrame::OnFileHardDisks
=========================================================*/
/* #FN#
   Displays Pick Virtual Harddisk Directories dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileHardDisks() 
{
	CHarddiskDlg dlgHardDisk( this );

	dlgHardDisk.DoModal();
	CleanScreen();
} /* #OF# CMainFrame::OnFileHardDisks */

/*========================================================
Method   : CMainFrame::OnFileReadState
=========================================================*/
/* #FN#
   Restores saved state file */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileReadState()
{
	if( zlib_capable() == -1 )
	{
		DisplayMessage( NULL, IDS_ERROR_NO_ZLIB, 0, MB_ICONEXCLAMATION | MB_OK );
	}
	else
	{
		char szStateName[ MAX_PATH ];

		/* Get the most recently used file name to use in a file dialog */
		strncpy( szStateName, atari_state_dir, MAX_PATH );

		if( PickFileName( TRUE, szStateName, "Load Atari saved-state file",
						  PF_A8S_FILTER, "a8s", PF_LOAD_FLAGS, TRUE, DEFAULT_A8S ) &&
			*szStateName != '\0' )
		{
			if( _stricmp( szStateName, atari_state_dir ) != 0 )
			{
				strcpy( atari_state_dir, szStateName );
				WriteRegString( NULL, REG_STATE_PATH, atari_state_dir );
			}
			if( ReadAtariState( atari_state_dir, "rb" ) )
			{
				if( !(g_ulAtariState & ATARI_PAUSED) )
					OnAtariPause();

				DescribeAtariSystem();
				return;
			}
			else
				DisplayMessage( NULL, IDS_ERROR_A8S_READ, 0, MB_ICONEXCLAMATION | MB_OK );
		}
		else
			g_ulAtariState &= ~ATARI_PAUSED;
	}
	CleanScreen();
} /* #OF# CMainFrame::OnFileReadState */

/*========================================================
Method   : CMainFrame::OnFileSaveStateNormal
=========================================================*/
/* #FN#
   Dumps the exact state of the Atari to a file */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileSaveStateNormal() 
{
	SaveState( 0 );
} /* #OF# CMainFrame::OnFileSaveStateNormal */

/*========================================================
Method   : CMainFrame::OnFileSaveStateVerbose
=========================================================*/
/* #FN#
   Saves a memory image with the ROMs */
void
/* #AS#
   Nothing */
CMainFrame::
OnFileSaveStateVerbose() 
{
	SaveState( 1 );
} /* #OF# CMainFrame::OnFileSaveStateVerbose */

/*========================================================
Method   : CMainFrame::OnAtariMachineType
=========================================================*/
/* #FN#
   Flips emulated machines */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariMachineType()
{
	int nDefaultSystem = default_system;

	if( 5 == nDefaultSystem && 2 == Ram256 )
	{
		Ram256 = 1; /* Rambo */
		g_ulMiscStates |= MS_RAMBO_MODE;
		WriteRegDWORD( NULL, REG_MISC_STATES, g_ulMiscStates );
	}
	else if( ++nDefaultSystem > 6 )
	{
		nDefaultSystem = 1;
	}
	else if( 5 == nDefaultSystem )
	{
		Ram256 = 2; /* Compy Shop */
		g_ulMiscStates &= ~MS_RAMBO_MODE;
		WriteRegDWORD( NULL, REG_MISC_STATES, g_ulMiscStates );
	}
	MachineTypeChanged( nDefaultSystem );
} /* #OF# CMainFrame::OnAtariMachineType */

/*========================================================
Method   : CMainFrame::MachineTypeChanged
=========================================================*/
/* #FN#
   Makes an emulated machine changing visible for the emulator */
void
/* #AS#
   Nothing */
CMainFrame::
MachineTypeChanged(
	int nDefaultSystem
)
{
	if( nDefaultSystem != default_system &&
		(6 == default_system ||
		 6 == nDefaultSystem) )
	{
		if( rom_inserted )
		{
			Remove_ROM();
			strcpy( g_szCurrentRom, "None" );

			WriteRegString( NULL, REG_CURRENT_ROM, g_szCurrentRom );
			WriteRegDWORD ( NULL, REG_CART_TYPE,   cart_type );
		}
	}
	default_system = nDefaultSystem;
	DescribeAtariSystem();

	WriteRegDWORD( NULL, REG_DEFAULT_SYSTEM, default_system );
	RestartEmulation();
} /* #OF# CMainFrame::MachineTypeChanged */

/*========================================================
Method   : CMainFrame::OnAtariMachineType800osa
=========================================================*/
/* #FN#
   Switches the emulator to an Atari OS-A emulation */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariMachineType800osa()
{
	MachineTypeChanged( 1 /* OS-A */ );
} /* #OF# CMainFrame::OnAtariMachineType800osa */

/*========================================================
Method   : CMainFrame::OnAtariMachineType800osb
=========================================================*/
/* #FN#
   Switches the emulator to an Atari OS-B emulation */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariMachineType800osb() 
{
	MachineTypeChanged( 2 /* OS-B */ );
} /* #OF# CMainFrame::OnAtariMachineType800osb */

/*========================================================
Method   : CMainFrame::OnAtariMachineType800xl
=========================================================*/
/* #FN#
   Switches the emulator to an Atari XL emulation */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariMachineType800xl() 
{
	MachineTypeChanged( 3 /* 800 XL */ );
} /* #OF# CMainFrame::OnAtariMachineType800xl */

/*========================================================
Method   : CMainFrame::OnAtariMachineType130xe
=========================================================*/
/* #FN#
   Switches the emulator to an Atari 130 XE emulation */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariMachineType130xe() 
{
	MachineTypeChanged( 4 /* 130 XE */ );
} /* #OF# CMainFrame::OnAtariMachineType130xe */

/*========================================================
Method   : CMainFrame::OnAtariMachineType320xeCompy
=========================================================*/
/* #FN#
   Switches the emulator to an Atari 320 XE Compy Shop emulation */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariMachineType320xeCompy() 
{
	Ram256 = 2;

	g_ulMiscStates &= ~MS_RAMBO_MODE;
	WriteRegDWORD( NULL, REG_MISC_STATES, g_ulMiscStates );

	MachineTypeChanged( 5 /* 320 Compy Shop */ );
} /* #OF# CMainFrame::OnAtariMachineType320xeCompy */

/*========================================================
Method   : CMainFrame::OnAtariMachineType320xeRambo
=========================================================*/
/* #FN#
   Switches the emulator to an Atari 320 XE Rambo emulation */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariMachineType320xeRambo() 
{
	Ram256 = 1;

	g_ulMiscStates |= MS_RAMBO_MODE;
	WriteRegDWORD( NULL, REG_MISC_STATES, g_ulMiscStates );

	MachineTypeChanged( 5 /* 320 Rambo */ );
} /* #OF# CMainFrame::OnAtariMachineType320xeRambo */

/*========================================================
Method   : CMainFrame::OnAtariMachineType5200
=========================================================*/
/* #FN#
   Switches the emulator to an Atari 5200 emulation */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariMachineType5200() 
{
	MachineTypeChanged( 6 /* 5200 */ );
} /* #OF# CMainFrame::OnAtariMachineType5200 */

/*========================================================
Method   : CMainFrame::OnAtariVideoSystem
=========================================================*/
/* #FN#
   Flips emulated video standards */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariVideoSystem()
{
	if( ++default_tv_mode > 2 )
		default_tv_mode = 1;

	WriteRegDWORD( NULL, REG_DEFAULT_TV_MODE, default_tv_mode );
	RestartEmulation();
} /* #OF# CMainFrame::OnAtariVideoSystem */

/*========================================================
Method   : CMainFrame::OnAtariVideoSystemNtsc
=========================================================*/
/* #FN#
	Sets a state of the object regarding to NTSC menu */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariVideoSystemNtsc() 
{
	if( 2 != default_tv_mode )
	{
		default_tv_mode = 2;

		WriteRegDWORD( NULL, REG_DEFAULT_TV_MODE, default_tv_mode );
		RestartEmulation();
	}
} /* #OF# CMainFrame::OnAtariVideoSystemNtsc */

/*========================================================
Method   : CMainFrame::OnAtariVideoSystemPal
=========================================================*/
/* #FN#
	Sets a state of the object regarding to PAL menu */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariVideoSystemPal() 
{
	if( 1 != default_tv_mode )
	{
		default_tv_mode = 1;

		WriteRegDWORD( NULL, REG_DEFAULT_TV_MODE, default_tv_mode );
		RestartEmulation();
	}
} /* #OF# CMainFrame::OnAtariVideoSystemPal */
 
/*========================================================
Method   : CMainFrame::OnAtariSettings
=========================================================*/
/* #FN#
   Displays Atari Settings dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariSettings()
{
	CSettingsDlg dlgSettings( this );

	dlgSettings.DoModal();
	CleanScreen();
} /* #OF# CMainFrame::OnAtariSettings */

/*========================================================
Method   : CMainFrame::OnAtariSioPatch
=========================================================*/
/* #FN#
   Toggles a SIO patch between on and off */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariSioPatch()
{
	ToggleSIOPatch();

	DescribeAtariSystem();
} /* #OF# CMainFrame::OnAtariSioPatch */

/*========================================================
Method   : CMainFrame::OnAtariFullSpeed
=========================================================*/
/* #FN#
   Toggles a full speed mode between on and off */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariFullSpeed()
{
	ToggleFullSpeed();

	DescribeAtariSystem();
} /* #OF# CMainFrame::OnAtariFullSpeed */

/*========================================================
Method   : CMainFrame::OnAtariPause
=========================================================*/
/* #FN#
   Toggles a pause mode between on and off */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariPause()
{
	TogglePause();

	DescribeAtariSystem();
} /* #OF# CMainFrame::OnAtariPause */

/*========================================================
Method   : CMainFrame::OnAtariKeyboard
=========================================================*/
/* #FN#
   Displays Keyboard Templates dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariKeyboard()
{
	CKeyboardDlg dlgKeyboard( this );

	dlgKeyboard.DoModal();
	CleanScreen();
} /* #OF# CMainFrame::OnAtariKeyboard */

/*========================================================
Method   : CMainFrame::OnAtariJoysticks
=========================================================*/
/* #FN#
   Displays Joystick Options dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariJoysticks() 
{
	CJoystickDlg dlgJoystick( this );

	dlgJoystick.DoModal();
	CleanScreen();
} /* #OF# CMainFrame::OnAtariJoysticks */

/*========================================================
Method   : CMainFrame::OnAtariMouse
=========================================================*/
/* #FN#
   Displays Mouse Options dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariMouse()
{
	CMouseDlg dlgMouse( this );

	dlgMouse.DoModal();
	CleanScreen();
} /* #OF# CMainFrame::OnAtariMouse */

/*========================================================
Method   : CMainFrame::OnAtariWarmstart
=========================================================*/
/* #FN#
   Forces a warm start of an emulated machine */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariWarmstart() 
{
	if( Atari5200 == machine ) /* 5200 has no warmstart */
	{
		OnAtariColdstart();
		return;
	}

	if( g_ulAtariState & ATARI_UNINITIALIZED )
	{
		/* Non Hercules contra plures... */
		RestartEmulation();
	}
	else
	{
		if( g_ulAtariState & ATARI_CRASHED )
		{
			g_ulAtariState = ATARI_RUNNING;
			wsync_halt = 0;	/* Turn on CPU */
			g_nTestVal = 0;
			/* CAtari800WinView::OnSetFocus won't invoke that */
			RestartSound();
			CleanScreen( FALSE );
		}
		/* Do warm reset */
		Warmstart();
	}
	DescribeAtariSystem();
} /* #OF# CMainFrame::OnAtariWarmstart */

/*========================================================
Method   : CMainFrame::OnAtariColdstart
=========================================================*/
/* #FN#
   Forces a cold start of an emulated machine */
void
/* #AS#
   Nothing */
CMainFrame::
OnAtariColdstart()
{
	if( g_ulAtariState & ATARI_UNINITIALIZED )
	{
		/* Non Hercules contra plures... */
		RestartEmulation();
	}
	else
	{
		if( g_ulAtariState & ATARI_CRASHED )
		{
			g_ulAtariState = ATARI_RUNNING;
			wsync_halt = 0;	/* Turn on CPU */
			g_nTestVal = 0;
			/* CAtari800WinView::OnSetFocus won't invoke that */
			RestartSound();
			CleanScreen( FALSE );
		}
		/* Do cold reset */
		Coldstart();
		/* This looks strange but will be necessary when PORTB_handler@PIA.c
		   is called indirectly by Coldstart (vide line 799, it looks like
		   an error in Atari800 kernel, doesn't it?) */
		(enable_rom_patches && enable_sio_patch) ? SetSIOEsc() : RestoreSIO();
	}
	DescribeAtariSystem();
} /* #OF# CMainFrame::OnAtariColdstart */

/*========================================================
Method   : CMainFrame::OnGraphicsOptions
=========================================================*/
/* #FN#
   Displays Graphics Options dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsOptions() 
{
	CGraphicsDlg dlgGraphics( this );

	/* Show the window because if we changed DDraw modes the frame will
	   be missing until forced to redraw here */
	if( IDOK == dlgGraphics.DoModal() )
		ShowWindow( SW_SHOWNORMAL );

	CleanScreen();
} /* #OF# CMainFrame::OnGraphicsOptions */

/*========================================================
Method   : CMainFrame::OnGraphicsToggleModes
=========================================================*/
/* #FN#
   Toggles between windowed and full-screen modes */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsToggleModes()
{
	ToggleModes();

	/* There should be an info about pause mode on the status bar
	   in a full-screen mode and none about it in a windowed mode */
	DescribeAtariSystem( g_ulAtariState & ATARI_PAUSED );
} /* #OF# CMainFrame::OnGraphicsToggleModes */

/*========================================================
Method   : CMainFrame::OnGraphicsGdiForWindows
=========================================================*/
/* #FN#
   Switches between GDI and DirectDraw for windowed modes */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsGdiForWindows()
{
	ULONG ulSavedMode = g_ulScreenMode;

	if( g_ulScreenMode & SM_OPTN_USE_GDI )
		g_ulScreenMode &= ~SM_OPTN_USE_GDI;
	else
		g_ulScreenMode |= SM_OPTN_USE_GDI;

	if( !InitialiseScreen( FALSE ) )
	{
		g_ulScreenMode = ulSavedMode;
		InitialiseScreen( FALSE );
	}
	WriteRegDWORD( NULL, REG_DDRAW_MODE, g_ulScreenMode );
} /* #OF# CMainFrame::OnGraphicsGdiForWindows */

/*========================================================
Method   : CMainFrame::OnGraphicsScanlines
=========================================================*/
/* #FN#
   Toggles displaying scanlines between on and off */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsScanlines()
{
	ULONG ulSavedMode = g_ulScreenMode;

	if( g_ulScreenMode & SM_OPTN_SHOW_SCANLINES )
		g_ulScreenMode &= ~SM_OPTN_SHOW_SCANLINES;
	else
		g_ulScreenMode |= SM_OPTN_SHOW_SCANLINES;

	if( !InitialiseScreen( FALSE ) )
	{
		g_ulScreenMode = ulSavedMode;
		InitialiseScreen( FALSE );
	}
	WriteRegDWORD( NULL, REG_DDRAW_MODE, g_ulScreenMode );
} /* #OF# CMainFrame::OnGraphicsScanlines */

/*========================================================
Method   : CMainFrame::OnGraphicsArtifacting
=========================================================*/
/* #FN#
   Flips artifacting modes */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsArtifacting()
{
	if( ++global_artif_mode > 4 )
		global_artif_mode = 0;
	artif_init();

	WriteRegDWORD( NULL, REG_ARTIF_MODE, global_artif_mode );
} /* #OF# CMainFrame::OnGraphicsArtifacting */

/*========================================================
Method   : CMainFrame::OnGraphicsArtifactingNone
=========================================================*/
/* #FN#
   Turns off artifacting */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsArtifactingNone()
{
	global_artif_mode = 0;

	artif_init();
	WriteRegDWORD( NULL, REG_ARTIF_MODE, global_artif_mode );
} /* #OF# CMainFrame::OnGraphicsArtifactingNone */

/*========================================================
Method   : CMainFrame::OnGraphicsArtifactingBluebrown1
=========================================================*/
/* #FN#
   Turns on the blue-brown 1 artifacting */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsArtifactingBluebrown1()
{
	global_artif_mode = 1;

	artif_init();
	WriteRegDWORD( NULL, REG_ARTIF_MODE, global_artif_mode );
} /* #OF# CMainFrame::OnGraphicsArtifactingBluebrown1 */

/*========================================================
Method   : CMainFrame::OnGraphicsArtifactingBluebrown2
=========================================================*/
/* #FN#
   Turns on the blue-brown 2 artifacting */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsArtifactingBluebrown2()
{
	global_artif_mode = 2;

	artif_init();
	WriteRegDWORD( NULL, REG_ARTIF_MODE, global_artif_mode );
} /* #OF# CMainFrame::OnGraphicsArtifactingBluebrown2 */

/*========================================================
Method   : CMainFrame::OnGraphicsArtifactingGtia
=========================================================*/
/* #FN#
   Turns on the GTIA artifacting */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsArtifactingGtia()
{
	global_artif_mode = 3;

	artif_init();
	WriteRegDWORD( NULL, REG_ARTIF_MODE, global_artif_mode );
} /* #OF# CMainFrame::OnGraphicsArtifactingGtia */

/*========================================================
Method   : CMainFrame::OnGraphicsArtifactingCtia
=========================================================*/
/* #FN#
   Turns on the CTIA artifacting */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsArtifactingCtia()
{
	global_artif_mode = 4;

	artif_init();
	WriteRegDWORD( NULL, REG_ARTIF_MODE, global_artif_mode );
} /* #OF# CMainFrame::OnGraphicsArtifactingCtia */

/*========================================================
Method   : CMainFrame::OnGraphicsPalette
=========================================================*/
/* #FN#
   Displays the "Palette Options" dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsPalette()
{
	CPaletteDlg dlgPalette( this );

	dlgPalette.DoModal();
	CleanScreen();
} /* #OF# CMainFrame::OnGraphicsPalette */

/*========================================================
Method   : CMainFrame::OnGraphicsSaveScreenshotNormal
=========================================================*/
/* #FN#
   Saves normal screen snapshot */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsSaveScreenshotNormal()
{
	SaveSnapshot( 0 );
} /* #OF# CMainFrame::OnGraphicsSaveScreenshotNormal */

/*========================================================
Method   : CMainFrame::OnGraphicsSaveScreenshotInterlaced
=========================================================*/
/* #FN#
   Saves interlaced screen snapshot */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsSaveScreenshotInterlaced()
{
	SaveSnapshot( 1 );
} /* #OF# CMainFrame::OnGraphicsSaveScreenshotInterlaced */

/*========================================================
Method   : CMainFrame::OnGraphicsPerformanceTest
=========================================================*/
/* #FN#
   Renders speed of the current graphics mode */
void
/* #AS#
   Nothing */
CMainFrame::
OnGraphicsPerformanceTest()
{
	char   szReport[ LOADSTRING_STRING_SIZE ] = { " " };
	ULONG  ulStartTime;
	ULONG  ulTotalTime;
	double nResult;

	DisplayMessage( NULL, IDS_GFX_TEST_BEGIN, 0, MB_ICONINFORMATION | MB_OK );
	ulStartTime = timeGetTime();

	for( int i = 0; i < 30; i++ )
		Atari_DisplayScreen( (UBYTE *)atari_screen );

	ulTotalTime = timeGetTime() - ulStartTime;
	nResult     = (double)(ulTotalTime / 30.0f);

	if( nResult > 20.0f )
		LoadString( NULL, IDS_WARN_SPEED, szReport, LOADSTRING_STRING_SIZE );

	DisplayMessage( NULL, IDS_GFX_TEST_RESULT, IDS_BENCHMARK_RESULTS, MB_ICONINFORMATION | MB_OK, ulTotalTime, nResult, szReport );
	CleanScreen();
} /* #OF# CMainFrame::OnGraphicsPerformanceTest */

/*========================================================
Method   : CMainFrame::OnSoundOptions
=========================================================*/
/* #FN#
   Displays Sound Options dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnSoundOptions() 
{
	CSoundDlg dlgSound( this );

	dlgSound.DoModal();
	CleanScreen();
} /* #OF# CMainFrame::OnSoundOptions */

/*========================================================
Method   : CMainFrame::OnSoundDigitizedEffects
=========================================================*/
/* #FN#
   Toggles digitized effect between on and off */
void
/* #AS#
   Nothing */
CMainFrame::
OnSoundDigitizedEffects()
{
	g_nUseVolumeOnly = g_nUseVolumeOnly ? 0 : 1;

	WriteRegDWORD( NULL, REG_USE_VOLUME_ONLY, g_nUseVolumeOnly );
	ClearSound( TRUE );

	InitialiseSound();
} /* #OF# CMainFrame::OnSoundDigitizedEffects */

/*========================================================
Method   : CMainFrame::OnSoundStereo
=========================================================*/
/* #FN#
   Toggles a stereo support between on and off */
void
/* #AS#
   Nothing */
CMainFrame::
OnSoundStereo()
{
#ifndef STEREO
	stereo_enabled = 0;
#else
	stereo_enabled = stereo_enabled ? 0 : 1;
#endif
	WriteRegDWORD( NULL, REG_ENABLE_STEREO, stereo_enabled );
	ClearSound( TRUE );

	InitialiseSound();
} /* #OF# CMainFrame::OnSoundStereo */

/*========================================================
Method   : CMainFrame::OnSoundStereo
=========================================================*/
/* #FN#
   Toggles a sound playback between on and off */
void
/* #AS#
   Nothing */
CMainFrame::
OnSoundMute() 
{
	if( g_ulSoundState & SOUND_NOSOUND )
	{
		g_ulSoundState &= ~SOUND_NOSOUND;
		/* Registry is corrupted? */
		if( !(g_ulSoundState & (SOUND_MMSOUND | SOUND_DIRECTSOUND)) )
			g_ulSoundState |= SOUND_MMSOUND;
	}
	else
		g_ulSoundState |= SOUND_NOSOUND;

	WriteRegDWORD( NULL, REG_SOUND_STATE, g_ulSoundState );
	ClearSound( TRUE );

	InitialiseSound();
} /* #OF# CMainFrame::OnSoundMute */

/*========================================================
Method   : CMainFrame::OnSoundSaveSound
=========================================================*/
/* #FN#
   Saves the output of the Pokey processor to a file */
void
/* #AS#
   Nothing */
CMainFrame::
OnSoundSaveSound()
{
	int nResult = IDOK;

	if( g_pfSndOutput )
		/* Close Sound Output file */
		CloseSndOutput(); /* The sound is turned off now */
	else
	{
		if( g_ulSoundState & SOUND_NOSOUND )
			DisplayMessage( NULL, IDS_SFX_RECORD, 0, MB_ICONINFORMATION | MB_OK );

		if( !(g_ulDontShowFlags & DONT_SHOW_SOUNDFILE_WARN) )
		{
			CWarningDlg	dlgWarning( this );
			char szMessage[ LOADSTRING_STRING_SIZE ];

			LoadString( NULL, IDS_WARN_SOUNDFILE, szMessage, LOADSTRING_STRING_SIZE );
			dlgWarning.m_strWarnText = szMessage;
			dlgWarning.m_ulWarnBit   = DONT_SHOW_SOUNDFILE_WARN;
			nResult = dlgWarning.DoModal();
		}
		if( nResult == IDOK )
		{
			if( PickFileName( FALSE, m_strSoundName, "Name of WAV file to save",
							  PF_WAV_FILTER, "wav", PF_SAVE_FLAGS, FALSE ) &&
				!m_strSoundName.IsEmpty() )
			{
				/* Open Sound Output file */
				OpenSndOutput( (LPSTR)(LPCSTR)m_strSoundName );
			}
		}
	}
	CleanScreen();
} /* #OF# CMainFrame::OnSoundSaveSound */

/*========================================================
Method   : CMainFrame::OnSoundPerformanceTest
=========================================================*/
/* #FN#
   Tests the Pokey processor overhead with current settings */
void
/* #AS#
   Nothing */
CMainFrame::
OnSoundPerformanceTest()
{
	if( !(g_ulAtariState & ATARI_RUNNING) || g_ulSoundState & SOUND_NOSOUND )
	{
		DisplayMessage( NULL, IDS_SFX_TEST_WARNING, 0, MB_ICONINFORMATION | MB_OK );
	}
	else
	{
		int   nSampleSize = g_nSoundRate / (default_tv_mode == 1 ? g_nPalFreq : g_nNtscFreq);
		char *pszBuffer   = NULL;

		if( pszBuffer = (char*)calloc( 1, nSampleSize + 1 ) )
		{
			ULONG ulStartTime;
			ULONG ulTotalTime;

			DisplayMessage( NULL, IDS_SFX_TEST_BEGIN, 0, MB_ICONINFORMATION | MB_OK );

			ulStartTime = timeGetTime();
			/* Begin performance test */
			for( int i = 0; i < 60; i++ )
				Pokey_process( (UCHAR*)pszBuffer, nSampleSize );

			ulTotalTime = timeGetTime() - ulStartTime;
			free( pszBuffer );

			DisplayMessage( NULL, IDS_SFX_TEST_RESULT, IDS_BENCHMARK_RESULTS, MB_ICONINFORMATION | MB_OK, nSampleSize, ulTotalTime, (float)(ulTotalTime / 60.0f) );
		}
	}
	CleanScreen();
} /* #OF# CMainFrame::OnSoundPerformanceTest */

/*========================================================
Method   : CMainFrame::OnMiscMouseCapture
=========================================================*/
/* #FN#
   Toggles the mouse pointer capturing between on and off */
void
/* #AS#
   Nothing */
CMainFrame::
OnMiscMouseCapture()
{
	ToggleMouseCapture();
} /* #OF# CMainFrame::OnMiscMouseCapture */

/*========================================================
Method   : CMainFrame::OnMiscExtraOptions
=========================================================*/
/* #FN#
   Displays the "Extra Options" dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnMiscExtraOptions()
{
	CMiscellanousDlg dlgMiscellanous( this );

	dlgMiscellanous.DoModal();
	CleanScreen();
} /* #OF# CMainFrame::OnMiscExtraOptions */

/*========================================================
Method   : CMainFrame::OnMiscFileAssociations
=========================================================*/
/* #FN#
   Displays the "File Associations" dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnMiscFileAssociations()
{
	CFileAssociationsDlg dlgFileAssociations( this );

	dlgFileAssociations.DoModal();
	CleanScreen();
} /* #OF# CMainFrame::OnMiscFileAssociations */

/*========================================================
Method   : CMainFrame::OnMiscConvertDcmToAtr
=========================================================*/
/* #FN#
   Converts .DCM files to .ATR files */
void
/* #AS#
   Nothing */
CMainFrame::
OnMiscConvertDcmToAtr()
{
	char  szLocalFileBuffer[ 8192 ] = { '\0' };
	DWORD dwFlags = OFN_PATHMUSTEXIST | OFN_EXPLORER | OFN_FILEMUSTEXIST | OFN_ALLOWMULTISELECT | OFN_HIDEREADONLY;

	if( g_ulScreenMode & SM_MODE_FULL )
		dwFlags |= OFN_ENABLEHOOK;

	CFileDialog	dlgDiskImage( TRUE, NULL, NULL, dwFlags, PF_DCM_FILTER, this );

	dlgDiskImage.m_ofn.lpstrTitle = "Select DCM file(s) to convert";
	dlgDiskImage.m_ofn.lpstrFile  = szLocalFileBuffer;
	dlgDiskImage.m_ofn.nMaxFile   = 8192;
	if( g_ulScreenMode & SM_MODE_FULL )
		dlgDiskImage.m_ofn.lpfnHook = FileDialogHookProc;

	if( IDOK == dlgDiskImage.DoModal() )
	{
		char     szDestDir[ MAX_PATH ];
		CString  strInput;
		FILE    *fpInput, *fpOutput;
		int      i;

		strcpy( szDestDir, dlgDiskImage.GetPathName() );
		for( i = strlen( szDestDir ); i > 0 && szDestDir[ i ] != '\\' && szDestDir[i] != '.'; i-- );
		if( i == 0 || szDestDir[ i ] != '.' )
			i = strlen( szDestDir );
		strcpy( &szDestDir[ i ], ".atr" );

		dwFlags = OFN_PATHMUSTEXIST | OFN_EXPLORER | OFN_HIDEREADONLY;

		if( g_ulScreenMode & SM_MODE_FULL )
			dwFlags |= OFN_ENABLEHOOK;

		CFileDialog	dlgDestDir( FALSE, NULL, szDestDir, dwFlags, NULL, this );

		dlgDestDir.m_ofn.lpstrTitle = "Select destination directory to write file(s)";
		if( g_ulScreenMode & SM_MODE_FULL )
			dlgDestDir.m_ofn.lpfnHook = FileDialogHookProc;

		if( IDOK == dlgDestDir.DoModal()  )
		{
			strcpy( szDestDir, dlgDestDir.GetPathName() );

			for( i = strlen( szDestDir ); i > 0 && szDestDir[ i ] != '\\'; i-- );
			szDestDir[ i ] = '\0';

			if( *szDestDir != '\0' )
			{
				POSITION pos = dlgDiskImage.GetStartPosition();
				ASSERT(pos);
				if( !pos )
				{
					CleanScreen();
					return;
				}
				strInput = dlgDiskImage.GetNextPathName( pos );
				while( strInput != "" )
				{
					char szFullDestName[ MAX_PATH ];
					char szThisFileName[ MAX_PATH ];

					strcpy( szThisFileName, strInput );
					fpInput = fopen( szThisFileName, "rb" );
					if( !fpInput )
					{
						DisplayMessage( NULL, IDS_ERROR_FILE_OPEN, 0, MB_ICONEXCLAMATION | MB_OK, szThisFileName );
						CleanScreen();
						return;
					}
					for( i = strlen( szThisFileName ); i > 0 && szThisFileName[ i ]!= '\\'; i-- );
					if( szThisFileName[ i ] == '\\' )
						i++;

					strcpy( szFullDestName, szDestDir );
					strcat( szFullDestName, "\\" );
					strcat( szFullDestName, &szThisFileName[ i ] );

					for( i = strlen( szFullDestName ); i > 0 && szFullDestName[ i ] != '\\' && szFullDestName[ i ] != '.'; i-- );
					if( i == 0 || szFullDestName[ i ] != '.' )
						i = strlen( szFullDestName );
					strcpy( &szFullDestName[ i ], ".atr" );

					fpOutput = fopen( szFullDestName, "rb" );
					if( fpOutput )
					{
						if( IDNO == DisplayMessage( NULL, IDS_WARN_FILE_EXISTS, 0, MB_ICONQUESTION | MB_YESNO, szFullDestName ) )
						{
							fclose( fpInput );
							CleanScreen();
							return;
						}
						fclose( fpOutput );
					}
					fpOutput = fopen( szFullDestName, "wb" );
					if( !fpOutput )
					{
						DisplayMessage( NULL, IDS_DCM2ATR_MSG3, 0, MB_ICONEXCLAMATION | MB_OK, szFullDestName );
						CleanScreen();
						return;
					}
					if( !dcmtoatr( fpInput, fpOutput, szThisFileName, szFullDestName ) )
					{
						int nResult;
						nResult = DisplayMessage( NULL, IDS_DCM2ATR_MSG1, 0, MB_ICONQUESTION | MB_YESNO );
						fclose( fpInput );
						fclose( fpOutput );
						if( nResult == IDYES )
						{
							CErrorLogDlg dlgErrorLog( this );
							dlgErrorLog.DoModal();
						}
						CleanScreen();
						return;
					}
					fclose( fpOutput );
					fclose( fpInput );
					strInput = (pos ? dlgDiskImage.GetNextPathName( pos ) : "");
				}
			}
			DisplayMessage( NULL, IDS_DCM2ATR_MSG2, 0, MB_ICONINFORMATION | MB_OK );
		}
	}
	CleanScreen();
} /* #OF# CMainFrame::OnMiscConvertDcmToAtr */

/*========================================================
Method   : CMainFrame::OnMiscConvertRomToCart
=========================================================*/
/* #FN#
   Converts .ROM/.BIN file to .CRT one */
void
/* #AS#
   Nothing */
CMainFrame::
OnMiscConvertRomToCart()
{
	char szFileName[ MAX_PATH ];

	strncpy( szFileName, g_szOtherRom, MAX_PATH );
	GetFolderPath( szFileName, NULL );

	if( PickFileName( TRUE, szFileName, "Select ROM image to convert",
					  PF_ROM_FILTER, "rom", PF_LOAD_FLAGS ) &&
		*szFileName != '\0' )
	{
		UBYTE ubImage[ 32769 ];
		int   nCartType, nFileLen;
		int   fd, i;

		if( -1 == (fd = open( szFileName, O_RDONLY | O_BINARY, 0777 )) )
		{
			DisplayMessage( NULL, IDS_ERROR_R2C_READ, 0, MB_ICONEXCLAMATION | MB_OK );
			CleanScreen();
			return;
		}
		nFileLen = read( fd, ubImage, sizeof(ubImage) );
		close( fd );

		if( nFileLen < 8193 )
			nCartType = NORMAL8_CART;
		else
		if( nFileLen < 16385 )
			nCartType = FindCartType( szFileName, FALSE, FALSE );
		else
		/* BountyBob 5200 cart has got the size greater than 32K 
		   but we are not prepared for this here... */
		if( nFileLen < 32769 )
			nCartType = AGS32_CART;

		for( i = strlen( szFileName ) - 1; i > 0 && szFileName[ i ] != '.' && szFileName[ i ] != '\\'; i-- );
		if( i > 0 )
		{
			if( szFileName[ i ] == '.' )
				szFileName[ i ] = '\0';
			strcat( szFileName, ".crt" );
		}

		if( nCartType != NO_CART &&
			PickFileName( FALSE, szFileName, "Name of CART file to save",
						  PF_CAR_FILTER, "crt", PF_SAVE_FLAGS, FALSE ) &&
			*szFileName != '\0' )
		{
			Header_t header;

			int anMapCartType[ 4 ] = { NORMAL8_CART, NORMAL16_CART, OSS_SUPERCART, AGS32_CART };
			int nCheckSum = 0;

			for( i = 0; i < nFileLen; i++ )
				nCheckSum += ubImage[ i ];

			for( i = 0; i < 4 && nCartType != anMapCartType[ i ]; i++ );
			nCartType = i + 1;

			header.ubId       [ 0 ] = 'C';
			header.ubId       [ 1 ] = 'A';
			header.ubId       [ 2 ] = 'R';
			header.ubId       [ 3 ] = 'T';
			header.ubType     [ 0 ] = (nCartType >> 24) & 0xff;
			header.ubType     [ 1 ] = (nCartType >> 16) & 0xff;
			header.ubType     [ 2 ] = (nCartType >> 8)  & 0xff;
			header.ubType     [ 3 ] = nCartType & 0xff;
			header.ubCheckSum [ 0 ] = (nCheckSum >> 24) & 0xff;
			header.ubCheckSum [ 1 ] = (nCheckSum >> 16) & 0xff;
			header.ubCheckSum [ 2 ] = (nCheckSum >> 8)  & 0xff;
			header.ubCheckSum [ 3 ] = nCheckSum & 0xff;
			header.ubGash     [ 0 ] = '\0';
			header.ubGash     [ 1 ] = '\0';
			header.ubGash     [ 2 ] = '\0';
			header.ubGash     [ 3 ] = '\0';

			if( -1 == (fd = open( szFileName, O_CREAT | O_TRUNC | O_WRONLY | O_BINARY, 0777 )) )
			{
				DisplayMessage( NULL, IDS_ERROR_R2C_SAVE, 0, MB_ICONEXCLAMATION | MB_OK );
				CleanScreen();
				return;
			}
			write( fd, &header, sizeof(header) );
			write( fd, ubImage, nFileLen );
			close( fd );
		}
	}
	CleanScreen();
} /* #OF# CMainFrame::OnMiscConvertRomToCart */

/*========================================================
Method   : CMainFrame::OnMiscClearAllSettings
=========================================================*/
/* #FN#
   Clears all saved settings/drives/ROMs */
void
/* #AS#
   Nothing */
CMainFrame::
OnMiscClearAllSettings()
{
	ULONG ulMiscStates = g_ulMiscStates;
	int   nResult;

	nResult = DisplayMessage( NULL, IDS_WARN_CLEAR, 0, MB_ICONQUESTION | MB_YESNO );
	if( nResult == IDYES )
	{
		DeleteAllRegKeys( NULL, REGNAME );
		HandleRegistry();

		/* These are not stored in the Registry, but should be re-initialized */
		g_ulAtariState = ATARI_UNINITIALIZED;

		if( g_ulMiscStates & MS_USE_KEY_TEMPLATE && 
			!ReadKeyTemplate( g_szTemplateFile ) )
		{
			g_ulMiscStates &= ~MS_USE_KEY_TEMPLATE;
			DisplayMessage( NULL, IDS_ERROR_NO_KEYTEMP, 0, MB_ICONEXCLAMATION | MB_OK, g_szTemplateFile );
		}
		if( g_ulMiscStates & MS_USE_EXT_PALETTE && 
			!read_palette( g_szPaletteFile ) )
		{
			g_ulMiscStates &= ~MS_USE_EXT_PALETTE;
			DisplayMessage( NULL, IDS_ERROR_NO_PALETTE, 0, MB_ICONEXCLAMATION | MB_OK, g_szPaletteFile );
		}
		if( g_ulMiscStates != ulMiscStates )
			WriteRegDWORD( NULL, REG_MISC_STATES, g_ulMiscStates );

		/* Restart the emulation now */
		RestartEmulation();
	}
} /* #OF# CMainFrame::OnMiscClearAllSettings */

/*========================================================
Method   : CMainFrame::OnMiscRestartEmulation
=========================================================*/
/* #FN#
   Reinitializes the Windows stuff and emulated Atari */
void
/* #AS#
   Nothing */
CMainFrame::
OnMiscRestartEmulation()
{
	/* We are going to detach ROM. Should this be optional? */
	Remove_ROM();
	strcpy( g_szCurrentRom, "None" );

	WriteRegString( NULL, REG_CURRENT_ROM, g_szCurrentRom );
	WriteRegDWORD ( NULL, REG_CART_TYPE,   cart_type );

	/* Force Windows stuff initialization */
	RestartEmulation();
} /* #OF# CMainFrame::OnMiscRestartEmulation */

/*========================================================
Method   : CMainFrame::OnMiscViewLogFile
=========================================================*/
/* #FN#
   Displays the "Error Log" dialog box */
void
/* #AS#
   Nothing */
CMainFrame::
OnMiscViewLogFile() 
{
	CErrorLogDlg dlgErrorLog( this );

	dlgErrorLog.DoModal();
	CleanScreen();
} /* #OF# CMainFrame::OnMiscViewLogFile */

/*========================================================
Method   : CMainFrame::OnMiscMonitor
=========================================================*/
/* #FN#
   Launches a monitor console */
void
/* #AS#
   Nothing */
CMainFrame::
OnMiscMonitor() 
{
	LaunchMonitor();
	CleanScreen( FALSE );
} /* #OF# CMainFrame::OnMiscMonitor */

/*========================================================
Method   : CMainFrame::OnContextHelp
=========================================================*/
/* #FN#
   Enables context-sensitive help mode */
void
/* #AS#
   Nothing */
CMainFrame::
OnContextHelp()
{
	if( g_ulScreenMode & SM_MODE_FULL /*&& g_ulScreenMode & SM_OPTN_FLIP_BUFFERS*/ )
		return;

	/* Go to context help mode (show the "question mark" mouse pointer) */
	CFrameWnd::OnContextHelp();
} /* #OF# CMainFrame::OnContextHelp */

/*========================================================
Method   : CMainFrame::OnHelpFinder
=========================================================*/
/* #FN#
   Displays the "Index" help page */
void
/* #AS#
   Nothing */
CMainFrame::
OnHelpFinder()
{
	/* Unfortunately, there are some problems with automatically return to
	   flipped full-screen mode, go to windowed instead */
	if( g_ulScreenMode & SM_MODE_FULL /*&& g_ulScreenMode & SM_OPTN_FLIP_BUFFERS*/ )
		SetSafeDisplay( FALSE );

	/* Invoke the help contents page */
	CFrameWnd::OnHelpFinder();
} /* #OF# CMainFrame::OnHelpFinder */

/*========================================================
Method   : CMainFrame::OnHelp
=========================================================*/
/* #FN#
   Displays a file menu help page */
void
/* #AS#
   Nothing */
CMainFrame::
OnHelp()
{
	/* Unfortunately, there are some problems with automatically return to
	   flipped full-screen mode, go to windowed instead */
	if( g_ulScreenMode & SM_MODE_FULL /*&& g_ulScreenMode & SM_OPTN_FLIP_BUFFERS*/ )
		SetSafeDisplay( FALSE );

	/* Invoke the help contents page */
	CFrameWnd::OnHelp();
} /* #OF# CMainFrame::OnHelp */

/*========================================================
Method   : CMainFrame::OnHelpKeyboardLayout
=========================================================*/
/* #FN#
   Displays the "Keyboard Layout" help page */
void
/* #AS#
   Nothing */
CMainFrame::
OnHelpKeyboardLayout()
{
	/* Unfortunately, there are some problems with automatically return to
	   flipped full-screen mode, go to windowed instead */
	if( g_ulScreenMode & SM_MODE_FULL /*&& g_ulScreenMode & SM_OPTN_FLIP_BUFFERS*/ )
		SetSafeDisplay( FALSE );

	/* Invoke the default keyboard layout help topic */
	AfxGetApp()->WinHelp( HID_BASE_COMMAND + ID_HELP_KEYBOARD_LAYOUT, HELP_CONTEXT );
} /* #OF# CMainFrame::OnHelpKeyboardLayout */

/*========================================================
Method   : CMainFrame::OnHelpKeyboardShortcuts
=========================================================*/
/* #FN#
   Displays the "Keyboard Shortcuts" help page */
void
/* #AS#
   Nothing */
CMainFrame::
OnHelpKeyboardShortcuts()
{
	/* Unfortunately, there are some problems with automatically return to
	   flipped full-screen mode, go to windowed instead */
	if( g_ulScreenMode & SM_MODE_FULL /*&& g_ulScreenMode & SM_OPTN_FLIP_BUFFERS*/ )
		SetSafeDisplay( FALSE );

	/* Invoke the keyboard shortcuts help topic */
	AfxGetApp()->WinHelp( HID_BASE_COMMAND + ID_HELP_KEYBOARD_SHORTCUTS, HELP_CONTEXT );
} /* #OF# CMainFrame::OnHelpKeyboardShortcuts */

/*========================================================
Method   : CMainFrame::OnHelpWebPage
=========================================================*/
/* #FN#
   Opens the project home page */
void
/* #AS#
   Nothing */
CMainFrame::
OnHelpWebPage()
{
	char szWebPage[ LOADSTRING_STRING_SIZE ];
	
	if( LoadString( NULL, IDS_WEB_PAGE, szWebPage, LOADSTRING_STRING_SIZE ) )
	{
		/* Unfortunately, there are some problems with automatically return to
		   flipped full-screen mode, go to windowed instead */
		if( g_ulScreenMode & SM_MODE_FULL /*&& g_ulScreenMode & SM_OPTN_FLIP_BUFFERS*/ )
			SetSafeDisplay( FALSE );

		/* Open Windows Explorer */
		ShellExecute( g_hMainWnd, "open", szWebPage, NULL, NULL, SW_SHOWNORMAL );
	}
} /* #OF# CMainFrame::OnHelpWebPage */

/*========================================================
Method   : CMainFrame::OnHelpAbout
=========================================================*/
/* #FN#
   Displays the "About" window */
void
/* #AS#
   Nothing */
CMainFrame::
OnHelpAbout()
{
	CAboutDlg dlgAbout;

	dlgAbout.DoModal();
	CleanScreen();
} /* #OF# CMainFrame::OnHelpAbout */

/*========================================================
Method   : CMainFrame::OnTimer
=========================================================*/
/* #FN#
   The framework calls this member function after each interval specified
   in the SetTimer member function used to install a timer */
void
/* #AS#
   Nothing */
CMainFrame::
OnTimer(
	UINT nIDEvent /* #IN# Specifies the identifier of the timer */
)
{
	if( TIMER_READ_JOYSTICK == nIDEvent )
	{
		ReadJoystickInput( 0, 0 );
		ReadJoystickInput( 1, 0 );
		ReadJoystickInput( 2, 0 );
		ReadJoystickInput( 3, 0 );

		if( !(g_ulAtariState & ATARI_PAUSED) )
			KillTimer( TIMER_READ_JOYSTICK );
	}
	CFrameWnd::OnTimer( nIDEvent );
} /* #OF# CMainFrame::OnTimer */

/*========================================================
Method   : CMainFrame::OnDropFiles
=========================================================*/
/* #FN#
   The framework calls this member function when the user releases
   the left mouse button over a window that has registered itself
   as the recipient of dropped files */
void
/* #AS#
   Nothing */
CMainFrame::
OnDropFiles(
	HDROP hDropInfo
)
{
	UINT    unFileType  = IAF_ATARI_FILE;
	CString strFileName = "";

	if( ::DragQueryFile( hDropInfo, (UINT)-1, NULL, 0 ) )
	{
		if( ::DragQueryFile( hDropInfo, 0, NULL, 0 ) < MAX_PATH )
		{
			::DragQueryFile( hDropInfo, 0, strFileName.GetBuffer( MAX_PATH ), MAX_PATH );
			strFileName.ReleaseBuffer();
		}
	}
	::DragFinish( hDropInfo );

	if( IsAtariFile( strFileName, &unFileType ) )
	{
		/* Is it an Atari executable? */
		if( IAF_BIN_IMAGE == unFileType )
		{
			/* This is not a good idea to restart the emulation here
			   by calling RestartEmulation( TRUE ), because the main
			   window has lost the focus */

			if( !RunAtariExe( (LPSTR)(LPCSTR)strFileName ) )
				DisplayMessage( NULL, IDS_ERROR_BIN_RUN, 0, MB_ICONEXCLAMATION | MB_OK, strFileName );
		}
		else
		/* Is it an Atari disk image? */
		if( IAF_DSK_IMAGE == unFileType )
		{
			SIO_Dismount( 1 );
			SIO_Mount( 1, (LPSTR)(LPCSTR)strFileName, FALSE );
			WriteRegDrives( NULL );

			Coldstart();
			/* This looks strange but will be necessary when PORTB_handler@PIA.c
			   is called indirectly by Coldstart (vide line 799, it looks like
			   an error in Atari800 kernel, doesn't it?) */
			(enable_rom_patches && enable_sio_patch) ? SetSIOEsc() : RestoreSIO();
		}
		/* Is it an Atari state file? */
		if( IAF_A8S_IMAGE == unFileType )
		{
			if( zlib_capable() == -1 )
			{
				DisplayMessage( NULL, IDS_ERROR_NO_ZLIB, 0, MB_ICONEXCLAMATION | MB_OK );
				return;
			}
			if( _stricmp( strFileName, atari_state_dir ) != 0 )
			{
				strcpy( atari_state_dir, strFileName );
				WriteRegString( NULL, REG_STATE_PATH, atari_state_dir );
			}
			if( !ReadAtariState( atari_state_dir, "rb" ) )
				DisplayMessage( NULL, IDS_ERROR_A8S_READ, 0, MB_ICONEXCLAMATION | MB_OK );
		}
		else
		/* Is it an Atari cartridge image? */
		if( IAF_ROM_IMAGE == unFileType || IAF_CRT_IMAGE == unFileType )
		{
			AttachCartridge( (LPSTR)(LPCSTR)strFileName, NO_CART );
			strcpy( g_szOtherRom, g_szCurrentRom );

			WriteRegString( NULL, REG_CURRENT_ROM, g_szCurrentRom );
			WriteRegString( NULL, REG_OTHER_ROM,   g_szCurrentRom );
			WriteRegDWORD ( NULL, REG_CART_TYPE,   cart_type );

			/* This is not a good idea to restart the emulation here
			   by calling RestartEmulation( TRUE ), because the main
			   window has lost the focus */
			RestartEmulation( FALSE ); /* Don't reinitialise Windows stuff */
		}
	}
	/* The main window has lost the focus, we have to take it back */
//	SetActiveWindow(); /* Doesn't work at the moment */

	DescribeAtariSystem();
//	CFrameWnd::OnDropFiles( hDropInfo );
}

/*========================================================
Method   : CMainFrame::OnMove
=========================================================*/
/* #FN#
   The framework calls this function after the CWnd object has been moved */
void
/* #AS#
   Nothing */
CMainFrame::
OnMove(
	int x, /* #IN# New x-coordinate location of the upper-left corner */
	int y  /* #IN# New y-coordinate location of the upper-left corner */
)
{
	CFrameWnd::OnMove( x, y );

	if( g_hViewWnd )
	{
		/* Compute area for DirectDraw display */
		ComputeClipArea();
		/* Cache boundaries of the view window */
		RefreshViewBounds( FALSE );
		/* Force screen refreshing */
		g_nTestVal = (ST_DOUBLE_REFRESH ? g_nStretchedRate : refresh_rate) - 1;
	}
} /* #OF# CMainFrame::OnMove */

/*========================================================
Method   : CMainFrame::OnSize
=========================================================*/
/* #FN#
   The framework calls this member function after the windows size has
   changed */
void
/* #AS#
   Nothing */
CMainFrame::
OnSize(
	UINT nType, /* #IN# Type of resizing requested    */
	int cx,     /* #IN# New width of the client area  */
	int cy      /* #IN# New height of the client area */
)
{
	CFrameWnd::OnSize( nType, cx, cy );

	if( g_hViewWnd )
	{
		/* Compute area for DirectDraw display */
		ComputeClipArea();
		/* Cache boundaries of the view window */
		RefreshViewBounds( FALSE );
		/* Force screen refreshing */
		g_nTestVal = (ST_DOUBLE_REFRESH ? g_nStretchedRate : refresh_rate) - 1;
	}
} /* #OF# CMainFrame::OnSize */

/*========================================================
Method   : CMainFrame::OnClose
=========================================================*/
/* #FN#
   The framework calls this function as a signal that the CWnd or an
   application is to terminate */
void
/* #AS#
   Nothing */
CMainFrame::
OnClose() 
{
	RECT rc;

	if( g_pfSndOutput )
		/* Close Sound Output file */
		CloseSndOutput();

	GetWindowRect( &rc );

	if( (rc.left != 0) && (g_nStartX != rc.left) )
		WriteRegDWORD( NULL, REG_START_XPOS, (DWORD)rc.left );

	if( (rc.top != 0) && (g_nStartY != rc.top) )
		WriteRegDWORD( NULL, REG_START_YPOS, (DWORD)rc.top );
	
	CFrameWnd::OnClose();
} /* #OF# CMainFrame::OnClose */

/*========================================================
Method   : CMainFrame::OnUpdateFileTurnDisk
=========================================================*/
/* #FN#
   Updates a state of the menu option */
void
/* #AS#
   Nothing */
CMainFrame::
OnUpdateFileTurnDisk(
	CCmdUI *pCmdUI /* #IN# The CCmdUI object that handles the update */
)
{
	pCmdUI->Enable( 
		strcmp( sio_filename[ 0 ], "Off" )   != 0 &&
		strcmp( sio_filename[ 0 ], "Empty" ) != 0 );
} /* #OF# CMainFrame::OnUpdateFileTurnDisk */

/*========================================================
Method   : CMainFrame::OnUpdateAtariMachineType
=========================================================*/
/* #FN#
   Updates a state of the menu option */
void
/* #AS#
   Nothing */
CMainFrame::
OnUpdateAtariMachineType(
	CCmdUI *pCmdUI /* #IN# The CCmdUI object that handles the update */
)
{
	int nDefaultSystem = default_system;
	if( (5 == nDefaultSystem && 1 == Ram256) || nDefaultSystem > 5 )
		nDefaultSystem++;

	pCmdUI->SetRadio( (pCmdUI->m_nID - ID_ATARI_MACHINETYPE_BASE) == DWORD(nDefaultSystem - 1) );
} /* #OF# CMainFrame::OnUpdateAtariMachineType */

/*========================================================
Method   : CMainFrame::OnUpdateAtariVideoSystem
=========================================================*/
/* #FN#
   Updates a state of the menu option */
void
/* #AS#
   Nothing */
CMainFrame::
OnUpdateAtariVideoSystem(
	CCmdUI *pCmdUI /* #IN# The CCmdUI object that handles the update */
)
{
	pCmdUI->SetRadio( (pCmdUI->m_nID - ID_ATARI_VIDEOSYSTEM_BASE) == DWORD(default_tv_mode - 1) );
} /* #OF# CMainFrame::OnUpdateAtariVideoSystem */

/*========================================================
Method   : CMainFrame::OnUpdateAtariSioPatch
=========================================================*/
/* #FN#
   Updates a state of the menu option */
void
/* #AS#
   Nothing */
CMainFrame::
OnUpdateAtariSioPatch(
	CCmdUI *pCmdUI /* #IN# The CCmdUI object that handles the update */
)
{
	pCmdUI->SetCheck( enable_sio_patch );
	pCmdUI->Enable( enable_rom_patches );
} /* #OF# CMainFrame::OnUpdateAtariSioPatch */

/*========================================================
Method   : CMainFrame::OnUpdateAtariFullSpeed
=========================================================*/
/* #FN#
   Updates a state of the menu option */
void
/* #AS#
   Nothing */
CMainFrame::
OnUpdateAtariFullSpeed(
	CCmdUI *pCmdUI /* #IN# The CCmdUI object that handles the update */
)
{
	pCmdUI->SetCheck( (g_ulMiscStates & MS_FULL_SPEED) != 0 );
} /* #OF# CMainFrame::OnUpdateAtariFullSpeed */

/*========================================================
Method   : CMainFrame::OnUpdateAtariPause
=========================================================*/
/* #FN#
   Updates a state of the menu option */
void
/* #AS#
   Nothing */
CMainFrame::
OnUpdateAtariPause(
	CCmdUI *pCmdUI /* #IN# The CCmdUI object that handles the update */
)
{
	pCmdUI->SetCheck( (g_ulAtariState & ATARI_PAUSED) != 0 );
} /* #OF# CMainFrame::OnUpdateAtariPause */

/*========================================================
Method   : CMainFrame::OnUpdateAtariAutofire
=========================================================*/
/* #FN#
   Updates a state of the menu option */
void
/* #AS#
   Nothing */
CMainFrame::
OnUpdateAtariAutofire(
	CCmdUI *pCmdUI /* #IN# The CCmdUI object that handles the update */
)
{
	pCmdUI->SetRadio( (pCmdUI->m_nID - ID_ATARI_AUTOFIRE_BASE) == DWORD(g_nAutofireMode) );
} /* #OF# CMainFrame::OnUpdateAtariAutofire */

/*========================================================
Method   : CMainFrame::OnUpdateGraphicsToggleModes
=========================================================*/
/* #FN#
   Updates a state of the menu option */
void
/* #AS#
   Nothing */
CMainFrame::
OnUpdateGraphicsToggleModes(
	CCmdUI *pCmdUI /* #IN# The CCmdUI object that handles the update */
)
{
	pCmdUI->SetCheck( (g_ulScreenMode & SM_MODE_FULL) != 0 );
} /* #OF# CMainFrame::OnUpdateGraphicsToggleModes */

/*========================================================
Method   : CMainFrame::OnUpdateGraphicsGdiForWindows
=========================================================*/
/* #FN#
   Updates a state of the menu option */
void
/* #AS#
   Nothing */
CMainFrame::
OnUpdateGraphicsGdiForWindows(
	CCmdUI *pCmdUI /* #IN# The CCmdUI object that handles the update */
)
{
	pCmdUI->SetCheck( (g_ulScreenMode & SM_OPTN_USE_GDI) != 0 );
} /* #OF# CMainFrame::OnUpdateGraphicsGdiForWindows */

/*========================================================
Method   : CMainFrame::OnUpdateGraphicsScanlines
=========================================================*/
/* #FN#
   Updates a state of the menu option */
void
/* #AS#
   Nothing */
CMainFrame::
OnUpdateGraphicsScanlines(
	CCmdUI *pCmdUI /* #IN# The CCmdUI object that handles the update */
)
{
	pCmdUI->SetCheck( (g_ulScreenMode & SM_OPTN_SHOW_SCANLINES) != 0 );
} /* #OF# CMainFrame::OnUpdateGraphicsScanlines */

/*========================================================
Method   : CMainFrame::OnUpdateGraphicsArtifacting
=========================================================*/
/* #FN#
   Updates a state of the menu option */
void
/* #AS#
   Nothing */
CMainFrame::
OnUpdateGraphicsArtifacting(
	CCmdUI *pCmdUI /* #IN# The CCmdUI object that handles the update */
)
{
	pCmdUI->SetRadio( (pCmdUI->m_nID - ID_GRAPHICS_ARTIFACTING_BASE) == (DWORD)global_artif_mode );
} /* #OF# CMainFrame::OnUpdateGraphicsArtifacting */

/*========================================================
Method   : CMainFrame::OnUpdateSoundDigitizedEffects
=========================================================*/
/* #FN#
   Updates a state of the menu option */
void
/* #AS#
   Nothing */
CMainFrame::
OnUpdateSoundDigitizedEffects(
	CCmdUI *pCmdUI /* #IN# The CCmdUI object that handles the update */
)
{
	pCmdUI->SetCheck( g_nUseVolumeOnly );
} /* #OF# CMainFrame::OnUpdateSoundDigitizedEffects */

/*========================================================
Method   : CMainFrame::OnUpdateSoundStereo
=========================================================*/
/* #FN#
   Updates a state of the menu option */
void
/* #AS#
   Nothing */
CMainFrame::
OnUpdateSoundStereo(
	CCmdUI *pCmdUI /* #IN# The CCmdUI object that handles the update */
)
{
	pCmdUI->SetCheck( stereo_enabled );
} /* #OF# CMainFrame::OnUpdateSoundStereo */

/*========================================================
Method   : CMainFrame::OnUpdateSoundMute
=========================================================*/
/* #FN#
   Updates a state of the menu option */
void
/* #AS#
   Nothing */
CMainFrame::
OnUpdateSoundMute(
	CCmdUI *pCmdUI /* #IN# The CCmdUI object that handles the update */
)
{
	pCmdUI->SetCheck( (g_ulSoundState & SOUND_NOSOUND) != 0 );
} /* #OF# CMainFrame::OnUpdateSoundMute */

/*========================================================
Method   : CMainFrame::OnUpdateSoundSaveSound
=========================================================*/
/* #FN#
   Updates a state of the menu option */
void
/* #AS#
   Nothing */
CMainFrame::
OnUpdateSoundSaveSound(
	CCmdUI *pCmdUI /* #IN# The CCmdUI object that handles the update */
)
{
	pCmdUI->SetCheck( (BOOL)(g_pfSndOutput != NULL) );
} /* #OF# CMainFrame::OnUpdateSoundSaveSound */

/*========================================================
Method   : CMainFrame::OnUpdateMiscMouseCapture
=========================================================*/
/* #FN#
   Updates a state of the menu option */
void
/* #AS#
   Nothing */
CMainFrame::
OnUpdateMiscMouseCapture(
	CCmdUI *pCmdUI /* #IN# The CCmdUI object that handles the update */
)
{
	pCmdUI->SetCheck( (BOOL)(g_ulMiscStates & MS_MOUSE_CAPTURED) );
} /* #OF# CMainFrame::OnUpdateMiscMouseCapture */

/*========================================================
Method   : CMainFrame::OnNcLButtonDown
=========================================================*/
/* #FN#
   The framework calls this member function when the user
   presses the left mouse button while the cursor is within
   a nonclient area of the CWnd object */
void
/* #AS#
   Nothing */
CMainFrame::
OnNcLButtonDown(
	UINT   nHitTest, /* #IN# The hit-test code */
	CPoint point     /* #IN# The x and y screen coordinates of the cursor position */
)
{
	if( HTMAXBUTTON == nHitTest && g_ulScreenMode & SM_MODE_WIND )
	{
		CRect rect;
		POINT pt;

		m_bPosChanging = FALSE;
		Default(); /* Draw a pushed 'maximize' button */
		m_bPosChanging = TRUE;

		GetCursorPos( &pt );

		GetWindowRect( rect );
		rect.top    += GetSystemMetrics( SM_CYDLGFRAME );
		rect.bottom  = rect.top + GetSystemMetrics( SM_CYCAPTION );
		rect.right  -= GetSystemMetrics( SM_CXDLGFRAME ) + GetSystemMetrics( SM_CXSIZE );
		rect.left    = rect.right - GetSystemMetrics( SM_CXSIZE );
		rect.InflateRect( 1, 1, 1, 0 );

		if( rect.PtInRect( pt ) )
			ToggleWindowedModes();

		return;
	}
	CFrameWnd::OnNcLButtonDown( nHitTest, point );
} /* #OF# CMainFrame::OnNcLButtonDown */

/*========================================================
Method   : CMainFrame::OnNcLButtonDblClk
=========================================================*/
/* #FN#
   The framework calls this member function when the user
   double-clicks the left mouse button while the cursor is
   within a nonclient area of CWnd */
void
/* #AS#
   Nothing */
CMainFrame::
OnNcLButtonDblClk(
	UINT   nHitTest, /* #IN# The hit-test code */
	CPoint point     /* #IN# The x and y screen coordinates of the cursor position */
)
{
	if( HTCAPTION == nHitTest && g_ulScreenMode & SM_MODE_WIND )
	{
		ToggleWindowedModes();
		return;
	}
	CFrameWnd::OnNcLButtonDblClk( nHitTest, point );
} /* #OF# CMainFrame::OnNcLButtonDblClk */

/*========================================================
Method   : CMainFrame::OnNcMouseMove
=========================================================*/
/* #FN#
   The framework calls this member function when the cursor
   is moved within a nonclient area */
void
/* #AS#
   Nothing */
CMainFrame::
OnNcMouseMove(
	UINT   nHitTest, /* #IN# The hit-test code */
	CPoint point     /* #IN# The x and y screen coordinates of the cursor position */
)
{
	if( !(g_ulMiscStates & MS_MOUSE_CAPTURED) )
		ShowMousePointer( TRUE );
	
	CFrameWnd::OnNcMouseMove( nHitTest, point );
} /* #OF# CMainFrame::OnNcMouseMove */

/*========================================================
Method   : CMainFrame::OnWindowPosChanging
=========================================================*/
/* #FN#
   The framework calls this member function when the size,
   position, or Z-order is about to change */
void
/* #AS#
   Nothing */
CMainFrame::
OnWindowPosChanging(
	WINDOWPOS FAR *lpWndPos /* #IN# The windows new size and position */
)
{
	CFrameWnd::OnWindowPosChanging( lpWndPos );

	if( !m_bPosChanging )
		lpWndPos->flags |= SWP_NOMOVE | SWP_NOSIZE;
} /* #OF# CMainFrame::OnWindowPosChanging */

/*========================================================
Method   : CMainFrame::OnExitMenuLoop
=========================================================*/
/* #FN#
   The framework calls this member function when a menu modal loop
   has been exited */
void
/* #AS#
   Nothing */
CMainFrame::
OnExitMenuLoop(
	BOOL bIsTrackPopupMenu /* #IN# Specifies whether the menu involved is a pop-up menu */
)
{
	if( g_ulScreenMode & SM_MODE_FULL && /* Take an action only for flipped full-screen */
		g_ulScreenMode & SM_OPTN_FLIP_BUFFERS )
	{
		if( m_bCleanUpMenu )
		{
			/* Clean up the whole screen */
			CleanScreen( FALSE );
		}
		else if( m_bMenuShowed )
			ShowEmulatorSpeed( TRUE );
	}
	else
		ShowEmulatorSpeed( FALSE );

	m_bCleanUpMenu = FALSE;
	m_bMenuShowed  = FALSE;
} /* #OF# CMainFrame::OnExitMenuLoop */

/*========================================================
Method   : CMainFrame::OnMenuSelect
=========================================================*/
/* #FN#
   This method is called by the framework when the user selects a menu item */
void
/* #AS#
   Nothing */
CMainFrame::
OnMenuSelect(
	UINT  nItemID,
	UINT  nFlags,
	HMENU hSysMenu
)
{
	if( 0xffff != nFlags && hSysMenu )
	{
		m_bCleanUpMenu = TRUE;

		if( !m_bMenuShowed && nFlags & MF_POPUP &&
			hSysMenu == ::GetMenu( GetSafeHwnd() ) )
		{
			SetClearView( FALSE );

			ShowMousePointer( TRUE );
			/* Get surface for GDI if necessary */
			FlipToGDI();
			ShowMenuBar( TRUE );
			ShowEmulatorSpeed( TRUE );

			SetClearView( TRUE );
			m_bMenuShowed = TRUE;
		}
		else
		if( !(nFlags & MF_POPUP) )
		{
			for( int i = 0; i < s_nShowMenuBarCmdNo; i++ )
			{
				if( nItemID == s_aShowMenuBarCmd[ i ] )
				{
					m_bCleanUpMenu = FALSE;
					break;
				}
			}
		}
	}
	CFrameWnd::OnMenuSelect( nItemID, nFlags, hSysMenu );
} /* #OF# CMainFrame::OnMenuSelect */

/*========================================================
Method   : CMainFrame::OnCommand
=========================================================*/
/* #FN#
   The framework calls this member function when the user selects
   an item from a menu, when a child control sends a notification
   message, or when an accelerator keystroke is translated */
BOOL
/* #AS#
   Nonzero if this message was processed; otherwise 0 */
CMainFrame::
OnCommand(
	WPARAM wParam,
	LPARAM lParam
)
{
	if( 1 == HIWORD(wParam) ) /* There was an accelerator keystroke */
	{
		WORD wAccID = LOWORD(wParam);
		for( int i = 0; i < s_nShowMenuBarCmdNo - 1 /* Avoid showing the menu when PAUSE (F9) */; i++ )
		{
			if( wAccID == s_aShowMenuBarCmd[ i ] )
			{
				SetClearView( FALSE );

				ShowMousePointer( TRUE );
				/* Get surface for GDI if necessary */
				FlipToGDI();
				ShowMenuBar( TRUE );
				ShowEmulatorSpeed( TRUE );

				SetClearView( TRUE );
				break;
			}
		}
	}
	return CFrameWnd::OnCommand( wParam, lParam );
} /* #OF# CMainFrame::OnCommand */

/*========================================================
Method   : CMainFrame::OnMessageCmdLine
=========================================================*/
/* #FN#
   The WM_USER message handler (a part of a single instance support) */
LRESULT
/* #AS#
   Nonzero if the message was handled; otherwise 0 */
CMainFrame::
OnMessageCmdLine(
	WPARAM wParam,
	LPARAM lParam
)
{
	HANDLE hFileMap = OpenFileMapping( FILE_MAP_READ, FALSE, "Atari800CmdLine" );
	LPSTR  pszMsg   = (LPSTR)MapViewOfFile( hFileMap, FILE_MAP_READ, 0, 0, 0 );

	if( pszMsg )
	{
		int nCmdIndex = 0;

		CopyMemory( g_szCmdLine, pszMsg, (int)lParam );
		UnmapViewOfFile( (LPCVOID)pszMsg );

		g_argc = (int)wParam;
		for( int i = 0; i < g_argc; i++ )
		{
			g_argv[ i ] = &g_szCmdLine[ nCmdIndex ];
			nCmdIndex += strlen( &g_szCmdLine[ nCmdIndex ] ) + 1;
		}
		/* Use g_argv and g_argc while rebooting */
		RestartEmulation( TRUE, FALSE );
	}
	return TRUE;
} /* #OF# CMainFrame::OnMessageCmdLine */


void
CMainFrame::
OnAtariAutofire()
{
	int nAutofireMode = g_nAutofireMode;

	if( ++nAutofireMode > AUTOFIRE_CONT )
		nAutofireMode = AUTOFIRE_OFF;

	AutofireChanged( nAutofireMode );
}

void
CMainFrame::
AutofireChanged(
	int nAutofireMode
)
{
	for( int i = 0; i < 4; i++ )
	{
		if( g_ulAutofireSticks & (1 << i) )
			TRIG_auto[ i ] = nAutofireMode;
	}
	if( g_nAutofireMode != nAutofireMode )
	{
		g_nAutofireMode = nAutofireMode;
		WriteRegDWORD( NULL, REG_AUTOFIRE_MODE, g_nAutofireMode );
	}
}

void
CMainFrame::
OnAtariAutofireOff()
{
	AutofireChanged( AUTOFIRE_OFF );
}

void
CMainFrame::
OnAtariAutofireFireDependent()
{
	AutofireChanged( AUTOFIRE_FIRE );
}

void
CMainFrame::
OnAtariAutofireAllTime()
{
	AutofireChanged( AUTOFIRE_CONT );
}

void
CMainFrame::
OnMiscConvertAsciiToAtascii()
{
	ConvertAtasciiToAscii( FALSE );
	CleanScreen();
}

void
CMainFrame::
OnMiscConvertAtasciiToAscii()
{
	ConvertAtasciiToAscii( TRUE );
	CleanScreen();
}

void
CMainFrame::
ConvertAtasciiToAscii(
	BOOL bA2PC /* #IN# TRUE if Atari to PC conversion is requested, otherwise FALSE */
)
{
	char szSrcFile[ MAX_PATH ] = { '\0' };
	/* Choose a text file to load */
	if( PickFileName( TRUE, szSrcFile, "Select text file to convert",
					  PF_TXT_FILTER, "txt", PF_LOAD_FLAGS ) &&
		*szSrcFile != '\0' )
	{
		/* Select the conversion type */
		CConvertTypeDlg dlgConvertType( bA2PC );

		if( IDOK == dlgConvertType.DoModal() )
		{
			char szDstFile[ MAX_PATH ] = { '\0' };
			/* Choose a text file name to save */
			if( PickFileName( FALSE, szDstFile, "Name of text file to save",
							  PF_TXT_FILTER, "txt", PF_SAVE_FLAGS, FALSE ) &&
				*szDstFile != '\0' )
			{
				/* Do the convertion work */
				CFile cfSrc, cfDst;
				/* We'll use a CFileException object to get error information */
				CFileException e;
				char szError[ 1024 ];

				if( !cfSrc.Open( szSrcFile, CFile::modeRead | CFile::shareDenyWrite, &e ) )
				{
					/* Complain if an error happened no need to delete the e object */
					e.GetErrorMessage( szError, 1024 );
					DisplayMessage( NULL, IDS_ERROR_A2PC_READ, 0, MB_ICONEXCLAMATION | MB_OK, szError );
					return;
				}
				else
				{
					if( !cfDst.Open( szDstFile, CFile::modeWrite | CFile::shareExclusive | CFile::modeCreate, &e ) )
					{
						e.GetErrorMessage( szError, 1024 );
						DisplayMessage( NULL, IDS_ERROR_A2PC_SAVE, 0, MB_ICONEXCLAMATION | MB_OK, szError );

						cfSrc.Close();
						return;
					}
					UBYTE ubSrcBuf[ 4096 ];
					UBYTE ubDstBuf[ 8192 ];
					DWORD dwSrcLen, dwDstLen;

					/* Read in 4096-byte blocks.
					   This loop ends when there are no more bytes to read */
					do
					{
						dwSrcLen = cfSrc.Read( ubSrcBuf, 4096 );

						/* Convert EOLs */
						dwDstLen = dlgConvertType.Atascii2Ascii( ubSrcBuf, ubDstBuf, dwSrcLen, bA2PC );

						/* Convert national "bushes" if requested */
						dlgConvertType.A8Std2Win1250( ubDstBuf,
													  ubDstBuf,
													  dwDstLen,
													  dlgConvertType.GetConvertType(),
													  bA2PC );

						/* Write the bunch to the output */
						cfDst.Write( ubDstBuf, dwDstLen );
					}
					while( dwSrcLen > 0 );

					/* Close both files */
					cfDst.Close();
					cfSrc.Close();
				}
			}
		}
	}
}
