/****************************************************************************
File    : DriveDlg.cpp
/*
@(#) #SY# Atari800Win
@(#) #IS# CDriveDlg implementation file
@(#) #BY# Richard Lawrence, Tomasz Szymankowski
@(#) #LM# 16.10.2000
*/

/*
Copyright (c) 1998 Richard Lawrence

This program is free software; you can redistribute it and/or modify it under the terms 
of the GNU General Public License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later version. This program is 
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details. You should have received a copy of the GNU
General Public License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "StdAfx.h"
#include "Atari800Win.h"
#include "Helpers.h"
#include "NewDiskImageDlg.h"
#include "DriveDlg.h"

//#include "core.h"			// AtariWin core

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#define CSD_READERROR		0
#define CSD_COMPRESSED		1
#define CSD_XFDREADWRITE	2
#define CSD_ATRREADONLY		3
#define CSD_ATRREADWRITE	4
#define CSD_PCREADONLY		5
#define CSD_DRIVEOFF		6
#define CSD_NODISKIMAGE		7

#define IDC_DRIVE_FIRST		IDC_DRIVE_BUTTON1
#define IDC_DRIVE_LAST		IDC_DRIVE_CANCEL


/////////////////////////////////////////////////////////////////////////////
// Static objects

static CDriveDlg::DiskData_t s_aDiskData[ MAX_DRIVES ] =
{
	{ sio_filename[ 0 ], "Off", 0, 0, 0, Off },
	{ sio_filename[ 1 ], "Off", 0, 0, 0, Off },
	{ sio_filename[ 2 ], "Off", 0, 0, 0, Off },
	{ sio_filename[ 3 ], "Off", 0, 0, 0, Off },
	{ sio_filename[ 4 ], "Off", 0, 0, 0, Off },
	{ sio_filename[ 5 ], "Off", 0, 0, 0, Off },
	{ sio_filename[ 6 ], "Off", 0, 0, 0, Off },
	{ sio_filename[ 7 ], "Off", 0, 0, 0, Off }
};


/////////////////////////////////////////////////////////////////////////////
// CDriveDlg dialog

BEGIN_MESSAGE_MAP(CDriveDlg, CDialog)
	//{{AFX_MSG_MAP(CDriveDlg)
	ON_BN_CLICKED(IDC_DRIVE_CLEARALL, OnClearAll)
	ON_BN_CLICKED(IDC_DRIVE_NEWIMAGE, OnNewImage)
	ON_CBN_SELCHANGE(IDC_DRIVE_NUMBER, OnSelchangeNumber)
	ON_WM_HELPINFO()
	//}}AFX_MSG_MAP
	ON_BN_CLICKED(IDC_DRIVE_OK, OnOK)
	ON_BN_CLICKED(IDC_DRIVE_CANCEL, CDialog::OnCancel)
END_MESSAGE_MAP()

/*========================================================
Method   : CDriveDlg::CDriveDlg
=========================================================*/
/* #FN#
   Standard constructor */
CDriveDlg::
CDriveDlg(
	CWnd *pParent /*=NULL*/ /* #IN# Pointer to the parent window */
)
	: CDialog( (g_ulScreenMode & SM_ATTR_SMALL_DLG ? IDD_DRIVE_SMALL : CDriveDlg::IDD), pParent )
{
	//{{AFX_DATA_INIT(CDriveDlg)
		// NOTE: the ClassWizard will add member initialization here
	//}}AFX_DATA_INIT

	m_pDiskData = s_aDiskData;
} /* #OF# CDriveDlg::CDriveDlg */

/*========================================================
Method   : CDriveDlg::DoDataExchange
=========================================================*/
/* #FN#
   Dynamic Data Exchange (not using) */
void
/* #AS#
   Nothing */
CDriveDlg::
DoDataExchange(
	CDataExchange *pDX /* #IN# Pointer to CDataExchange object */
)
{
	CDialog::DoDataExchange( pDX );
	//{{AFX_DATA_MAP(CDriveDlg)
		// NOTE: the ClassWizard will add DDX and DDV calls here
	//}}AFX_DATA_MAP
} /* #OF# CDriveDlg::DoDataExchange */


/////////////////////////////////////////////////////////////////////////////
// CDriveDlg implementation

/*========================================================
Method   : CDriveDlg::SetDlgState
=========================================================*/
/* #FN#
   We set the drive edit boxes to the the appropriate values,
   and set the status ComboBox */
void
/* #AS#
   Nothing */
CDriveDlg::
SetDlgState()
{
	CComboBox *pCombo = NULL;
	CEdit     *pEdit  = NULL;
	int        nDrive = 0;

	if( m_bSmallMode )
	{
		pCombo = (CComboBox *)GetDlgItem( IDC_DRIVE_NUMBER );
		ASSERT(pCombo);
		nDrive = pCombo->GetCurSel();
	}

	for( int i = 0; i < MAX_DRIVES; i++ )
	{
		if( !m_bSmallMode || (m_bSmallMode && i == nDrive ) )
		{
			/* The szNewName field is not really needed */
			SetDlgItemText( m_pDiskData[ i ].nEditID, m_pDiskData[ i ].szNewName );

			pCombo = (CComboBox *)GetDlgItem( m_pDiskData[ i ].nComboID );
			ASSERT(pCombo);
			pEdit = (CEdit *)GetDlgItem( m_pDiskData[ i ].nEditID );
			ASSERT(pEdit);
			
			switch( m_pDiskData[ i ].usStatus )
			{
				case Off:
					pCombo->SetCurSel( 0 );
					pEdit->SetReadOnly();
					break;

				case ReadWrite:
					pCombo->SetCurSel( 1 );
					pEdit->SetReadOnly( FALSE );
					break;

				case ReadOnly:
					pCombo->SetCurSel( 2 );
					pEdit->SetReadOnly( FALSE );
					break;

				case NoDisk:
					pCombo->SetCurSel( 3 );
					pEdit->SetReadOnly( FALSE );
					break;
			}
		}
	}
} /* #OF# CDriveDlg::SetDlgState */


/////////////////////////////////////////////////////////////////////////////
// CDriveDlg message handlers

/*========================================================
Method   : CDriveDlg::OnInitDialog
=========================================================*/
/* #FN#
   Performs special processing when the dialog box is initialized */
BOOL
/* #AS#
   TRUE unless you set the focus to a control */
CDriveDlg::
OnInitDialog()
{
	CDialog::OnInitDialog();

	const UINT anCtrlsLarge[ MAX_DRIVES ][ 3 ] =
	{
		{ IDC_DRIVE_BUTTON1, IDC_DRIVE_EDIT1, IDC_DRIVE_COMBO1 },
		{ IDC_DRIVE_BUTTON2, IDC_DRIVE_EDIT2, IDC_DRIVE_COMBO2 },
		{ IDC_DRIVE_BUTTON3, IDC_DRIVE_EDIT3, IDC_DRIVE_COMBO3 },
		{ IDC_DRIVE_BUTTON4, IDC_DRIVE_EDIT4, IDC_DRIVE_COMBO4 },
		{ IDC_DRIVE_BUTTON5, IDC_DRIVE_EDIT5, IDC_DRIVE_COMBO5 },
		{ IDC_DRIVE_BUTTON6, IDC_DRIVE_EDIT6, IDC_DRIVE_COMBO6 },
		{ IDC_DRIVE_BUTTON7, IDC_DRIVE_EDIT7, IDC_DRIVE_COMBO7 },
		{ IDC_DRIVE_BUTTON8, IDC_DRIVE_EDIT8, IDC_DRIVE_COMBO8 }
	};

	m_bSmallMode = g_ulScreenMode & SM_ATTR_SMALL_DLG;

	for( int i = 0; i < MAX_DRIVES; i++ )
	{
		/* Set the appropriate controls IDs */
		m_pDiskData[ i ].nButtonID = (m_bSmallMode ? IDC_DRIVE_BUTTON : anCtrlsLarge[ i ][ 0 ]);
		m_pDiskData[ i ].nEditID   = (m_bSmallMode ? IDC_DRIVE_EDIT   : anCtrlsLarge[ i ][ 1 ]);
		m_pDiskData[ i ].nComboID  = (m_bSmallMode ? IDC_DRIVE_COMBO  : anCtrlsLarge[ i ][ 2 ]);
		/* Backup drives paths */
		strncpy( m_pDiskData[ i ].szNewName, m_pDiskData[ i ].pszName, MAX_PATH );
		m_pDiskData[ i ].usStatus = drive_status[ i ];
	}
	if( m_bSmallMode )
	{
		CComboBox *pCombo = (CComboBox *)GetDlgItem( IDC_DRIVE_NUMBER );
		ASSERT(pCombo);
		pCombo->SetCurSel( 0 );
	}
	/* Unfortunately, the context help is available only in windowed modes */
	if( g_ulScreenMode & SM_MODE_WIND )
		SetWindowLong( GetSafeHwnd(), GWL_EXSTYLE, WS_EX_CONTEXTHELP | GetWindowLong( GetSafeHwnd(), GWL_EXSTYLE ) );

	SetDlgState();

	return TRUE;  // return TRUE unless you set the focus to a control
	              // EXCEPTION: OCX Property Pages should return FALSE
} /* #OF# CDriveDlg::OnInitDialog */

UnitStatus
CDriveDlg::
CheckSelectedDisk(
	LPCSTR pszDiskName,
	UINT  *pInfo /*= NULL*/
)
{
	UnitStatus usResult = ReadOnly;
	UINT unInfo = CSD_READERROR;

	if( strcmp( pszDiskName, "Off" ) == 0 )
	{
		usResult = Off;
		unInfo = CSD_DRIVEOFF;
	}
	else
	{
		CFileStatus fsStatus;
		CFile       cfDisk;
		/* That's a static method, don't panic */
		if( cfDisk.GetStatus( pszDiskName, fsStatus ) )
		{
			if( !IsCompressedFile( pszDiskName ) )
			{
				if( !(fsStatus.m_attribute & CFile::readOnly) )
				{
					int fd = open( pszDiskName, O_RDONLY | O_BINARY, 0777 );
					/* If there is an ATR image check header's writeprotect member */
					if( fd >= 0 )
					{
						ATR_Header header;
						lseek( fd, 0L, SEEK_SET );
						if( (-1 != read( fd, &header, sizeof(ATR_Header) )) &&
							header.magic1 == MAGIC1 && header.magic2 == MAGIC2 )
						{
							if( !header.writeprotect )
							{
								usResult = ReadWrite;
								unInfo = CSD_ATRREADWRITE;
							}
							else
								unInfo = CSD_ATRREADONLY;
						}
						else
						{
							usResult = ReadWrite;
							unInfo = CSD_XFDREADWRITE;
						}
						close( fd );
					}
				}
				else
					unInfo = CSD_PCREADONLY;
			}
			else
				unInfo = CSD_COMPRESSED;
		}
		else
		{
			usResult = NoDisk;
			unInfo = CSD_NODISKIMAGE;
		}
	}
	if( pInfo )
		*pInfo = unInfo;

	return usResult;
} /* #OF# CDriveDlg::CheckSelectedDisk */

/* GetDiskImage will call up a file dialog, and if the user oks the file will copy
   it's name into the appropriate edit box. The drives are not unmounted/mounted 
   until the OK button is pushed, in case the user decides to cancel the entire 
   operation. So the edit boxes are being used as temporary buffers, basically */
BOOL
CDriveDlg::
GetDiskImage(
	int nDrive
)
{
	CString	strDiskName = m_pDiskData[ nDrive ].szNewName;
	char	szPrompt[ 64 ];
	BOOL	bResult = FALSE;

	sprintf( szPrompt, "Select disk for drive %d", nDrive + 1 );

	if( PickFileName( TRUE, strDiskName, szPrompt,
					  PF_DSK_FILTER, NULL, PF_LOAD_FLAGS, TRUE, "None", this ) &&
		!strDiskName.IsEmpty() )
	{
		SetDlgItemText( m_pDiskData[ nDrive ].nEditID, strDiskName );
		strncpy( m_pDiskData[ nDrive ].szNewName, strDiskName, MAX_PATH );
		m_pDiskData[ nDrive ].usStatus = CheckSelectedDisk( strDiskName );

		bResult = TRUE;
	}
	return bResult;
} /* #OF# CDriveDlg::GetDiskImage */

BOOL
CDriveDlg::
DriveButton(
	UINT nButtonID,
	HWND hwndButton
)
{
	int nDrive = -1;

	if( m_bSmallMode )
	{
		if( nButtonID == IDC_DRIVE_BUTTON )
		{
			CComboBox *pCombo = (CComboBox *)GetDlgItem( IDC_DRIVE_NUMBER );
			ASSERT(pCombo);
			nDrive = pCombo->GetCurSel();
		}
	}
	else
	{
		for( int i = 0; i < MAX_DRIVES; i++ )
		{
			if( nButtonID == m_pDiskData[ i ].nButtonID ) 
			{
				nDrive = i;
				break;
			}
		}
	}
	if( -1 == nDrive || CB_ERR == nDrive ) /* It wasn't a drive button */
		return FALSE;

	if( GetDiskImage( nDrive ) )
		SetDlgState();

	return TRUE;
} /* #OF# CDriveDlg::DriveButton */

BOOL
CDriveDlg::
StatusSelChange(
	UINT nComboID,
	HWND hwndCombo
)
{
	CComboBox *pCombo = NULL;
	UINT       unDiskInf =  0;
	UnitStatus usRealMode;
	int        nDrive    = -1;

	if( m_bSmallMode )
	{
		if( nComboID == IDC_DRIVE_COMBO )
		{
			pCombo = (CComboBox *)GetDlgItem( IDC_DRIVE_NUMBER );
			ASSERT(pCombo);
			nDrive = pCombo->GetCurSel();
		}
	}
	else
	{
		for( int i = 0; i < MAX_DRIVES; i++ )
		{
			if( nComboID == m_pDiskData[ i ].nComboID )
			{
				nDrive = i;
				break;
			}
		}
	}
	if( -1 == nDrive || CB_ERR == nDrive ) /* It wasn't a drive button */
		return FALSE;

	usRealMode = CheckSelectedDisk( m_pDiskData[ nDrive ].szNewName, &unDiskInf );

	pCombo = (CComboBox *)GetDlgItem( m_pDiskData[ nDrive ].nComboID );
	ASSERT(pCombo);

	switch( pCombo->GetCurSel() )
	{
		case 0: /* Combo box indicates "Off" */
		{
			m_pDiskData[ nDrive ].usStatus = Off;
			break;
		}
		case 1: /* Combo box indicates "Read/Write" */
		{
			if( usRealMode == ReadOnly )
			{
				switch( unDiskInf )
				{
					case CSD_COMPRESSED:
						DisplayMessage( GetSafeHwnd(), IDS_DRIVE_MSG1, 0, MB_ICONINFORMATION | MB_OK );
						break;

					case CSD_ATRREADONLY:
						if( IDYES == DisplayMessage( GetSafeHwnd(), IDS_DRIVE_MSG2, 0, MB_ICONQUESTION | MB_YESNO ) )
						{
							int fd = open( m_pDiskData[ nDrive ].szNewName, O_WRONLY | O_BINARY, 0777 );
							if( fd >= 0 )
							{
								if( -1 != lseek( fd, 15L, SEEK_SET ) &&
									-1 != write( fd, "\000", 1 ) )
								{
									usRealMode = ReadWrite;
								}
								close( fd );
							}
						}
						break;

					case CSD_PCREADONLY:
						DisplayMessage( GetSafeHwnd(), IDS_DRIVE_MSG3, 0, MB_ICONINFORMATION | MB_OK );
						break;
				}
			}
			m_pDiskData[ nDrive ].usStatus = usRealMode;
			break;
		}
		case 2: /* Combo box indicates "Read Only" */
		{
			if( usRealMode == ReadWrite )
			{
				if( CSD_ATRREADWRITE == unDiskInf )
				{
					if( IDYES == DisplayMessage( GetSafeHwnd(), IDS_DRIVE_MSG4, 0, MB_ICONQUESTION | MB_YESNO ) )
					{
						int fd = open( m_pDiskData[ nDrive ].szNewName, O_WRONLY | O_BINARY, 0777 );
						if( fd >= 0 )
						{
							if( -1 != lseek( fd, 15L, SEEK_SET ) &&
								-1 != write( fd, "\001", 1 ) )
							{
								usRealMode = ReadOnly;
							}
							close( fd );
						}
					}
				}
				else
					DisplayMessage( GetSafeHwnd(), IDS_DRIVE_MSG5, 0, MB_ICONINFORMATION | MB_OK );
			}
			m_pDiskData[ nDrive ].usStatus = usRealMode;
			break;
		}
		case 3: /* Combo box indicates "No Disk" */
		{
			m_pDiskData[ nDrive ].usStatus = NoDisk;
			break;
		}
	}
	SetDlgState();

	return TRUE;
} /* #OF# CDriveDlg::StatusSelChange */

BOOL
CDriveDlg::
KillfocusEditDrive(
	UINT nEditID,
	HWND hwndEdit
)
{
	char szDiskName[ MAX_PATH ];
	int  nDrive = -1;

	if( m_bSmallMode )
	{
		if( nEditID == IDC_DRIVE_EDIT )
		{
			CComboBox *pCombo = (CComboBox *)GetDlgItem( IDC_DRIVE_NUMBER );
			ASSERT(pCombo);
			nDrive = pCombo->GetCurSel();
		}
	}
	else
	{
		for( int i = 0; i < MAX_DRIVES; i++ )
		{
			if( nEditID == m_pDiskData[ i ].nEditID )
			{
				nDrive = i;
				break;
			}
		}
	}
	if( -1 == nDrive || CB_ERR == nDrive ) /* It wasn't a drive edit */
		return FALSE;

	GetDlgItemText( m_pDiskData[ nDrive ].nEditID, szDiskName, MAX_PATH );
	strcpy( m_pDiskData[ nDrive ].szNewName, szDiskName );
	m_pDiskData[ nDrive ].usStatus = CheckSelectedDisk( szDiskName );

	SetDlgState();

	return TRUE;
} /* #OF# CDriveDlg::KillfocusEditDrive */

/*========================================================
Method   : CDriveDlg::OnCommand
=========================================================*/
/* #FN#
   The framework calls this member function when the user selects an item
   from a menu, when a child control sends a notification message, or when
   an accelerator keystroke is translated */
BOOL
/* #AS#
   Nonzero if the application processes this message; otherwise 0 */
CDriveDlg::
OnCommand(
	WPARAM wParam,
	LPARAM lParam
)
{
	/* This could also be handled by eight OnSelChanged plus eight ButtonClicked
	   individual procedures, but this seems a little more elegant */
	switch( HIWORD(wParam) )
	{
		case BN_CLICKED:
		{
			if( DriveButton( (UINT)LOWORD(wParam), (HWND)lParam ) )
				return TRUE;
			break;
		}
		case EN_KILLFOCUS:
		{
			if( KillfocusEditDrive( (UINT)LOWORD(wParam), (HWND)lParam ) )
				return TRUE;
			break;
		}
		case CBN_SELCHANGE:
		{
			if( StatusSelChange( (UINT)LOWORD(wParam), (HWND)lParam ) )
				return TRUE;
			break;
		}
	}
	return CDialog::OnCommand( wParam, lParam );
} /* #OF# CDriveDlg::OnCommand */

/*========================================================
Method   : CDriveDlg::OnClearAll
=========================================================*/
/* #FN#
   Removes disks from all drives and turns them off */
void
/* #AS#
   Nothing */
CDriveDlg::
OnClearAll()
{
	for( int i = 0; i < MAX_DRIVES; i++ )
	{
		m_pDiskData[ i ].usStatus = Off;
		strcpy( m_pDiskData[ i ].szNewName, "Off" );
	}
	SetDlgState();
} /* #OF# CDriveDlg::OnClearAll */

/*========================================================
Method   : CDriveDlg::OnNewImage
=========================================================*/
/* #FN#
   Displays New Disk Image dialog box */
void
/* #AS#
   Nothing */
CDriveDlg::
OnNewImage()
{
	CNewDiskImageDlg dlgNewDiskImage;

	if( IDOK == dlgNewDiskImage.DoModal() &&
		/* Attach disk image to any drive if it was expected */
		dlgNewDiskImage.m_bAttachDrive )
	{
		strncpy( m_pDiskData[ dlgNewDiskImage.m_nDriveNumber ].szNewName, dlgNewDiskImage.m_szImageName, MAX_PATH );
		m_pDiskData[ dlgNewDiskImage.m_nDriveNumber ].usStatus = CheckSelectedDisk( dlgNewDiskImage.m_szImageName );
	}
	SetDlgState();
} /* #OF# CDriveDlg::OnNewImage */

void
CDriveDlg::
OnSelchangeNumber()
{
	SetDlgState();
}

/*========================================================
Method   : CDriveDlg::ReceiveFocused
=========================================================*/
/* #FN#
   Receive the edit controls content again. The user could press
   'Enter' or 'Alt-O' and then all changes he's made in the last
   edited control would be lost. */
void
/* #AS#
   Nothing */
CDriveDlg::
ReceiveFocused()
{
	CWnd *pWnd    = GetFocus();
	UINT  nCtrlID = pWnd ? pWnd->GetDlgCtrlID() : 0;

	for( int i = 0; i < MAX_DRIVES; i++ )
	{
		/* Receive the edit contents again. A user could use 'Enter' or 'Alt-O'
		   and then all changes made in the last edited control would be lost */
		if( m_pDiskData[ i ].nEditID == nCtrlID )
		{
			KillfocusEditDrive( nCtrlID, (HWND)NULL );
			break;
		}
	}
} /* #OF# CDriveDlg::ReceiveFocused */

/*========================================================
Method   : CDriveDlg::OnOK
=========================================================*/
/* #FN#
   Called when the user clicks the OK button */
void
/* #AS#
   Nothing */
CDriveDlg::
OnOK()
{
	char szFileName[ MAX_PATH ];
	BOOL bChanged = FALSE;

	/* Unfortunately, edit controls do not lose the focus before
	   handling this when the user uses accelerators */
	ReceiveFocused();

	for( int i = 0; i < MAX_DRIVES; i++ )
	{
		if( m_pDiskData[ i ].usStatus == Off )
		{
			strcpy( szFileName, "Off" );
		}
		else if( m_pDiskData[ i ].usStatus == NoDisk )
		{
			strcpy( szFileName, "Empty" );
		}
		else
			strcpy( szFileName, m_pDiskData[ i ].szNewName );

		/* The reason for doing this here instead of just doing it in
		   GetDiskImage is to preserve the ability to cancel the entire
		   drive operation with one click of the cancel button. */
		if( _stricmp( szFileName, m_pDiskData[ i ].pszName ) != 0 )
		{
			SIO_Dismount( i + 1 );
			strcpy( m_pDiskData[ i ].pszName, szFileName );

			if( strcmp( szFileName, "Empty" ) != 0 &&
				strcmp( szFileName, "Off" ) != 0 )
			{
				if( !SIO_Mount( i + 1, szFileName, FALSE ) )
				{
					/* There was an error with SIO mounting */
					strcpy( m_pDiskData[ i ].pszName, "Empty" );
					m_pDiskData[ i ].usStatus = NoDisk;
				}
			}
			bChanged = TRUE;
		}
		if( m_pDiskData[ i ].usStatus != drive_status[ i ] )
		{
			drive_status[ i ] = m_pDiskData[ i ].usStatus;
			bChanged = TRUE;
		}
	}
	if( bChanged )
		WriteRegDrives( NULL );

	CDialog::OnOK();
} /* #OF# CDriveDlg::OnOK */

BOOL
CDriveDlg::
OnHelpInfo(
	HELPINFO *pHelpInfo
)
{
	if( g_ulScreenMode & SM_MODE_WIND )
		HelpInfo( pHelpInfo,
				  GetSafeHwnd(),
				  IDD_DRIVE,
				  IDC_DRIVE_FIRST, IDC_DRIVE_LAST );

	return TRUE; //CDialog::OnHelpInfo(pHelpInfo);
}
