/****************************************************************************
File    : sound_win.c
/*
@(#) #SY# Atari800Win
@(#) #IS# Sound implementation for Win32 platforms
@(#) #BY# Richard Lawrence, Tomasz Szymankowski
@(#) #LM# 01.10.2000
*/

/*
Copyright (c) 1998 Richard Lawrence

This program is free software; you can redistribute it and/or modify it under the terms 
of the GNU General Public License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later version. This program is 
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details. You should have received a copy of the GNU
General Public License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdio.h>
#include <windows.h>
#include <mmsystem.h>
#include <limits.h>
#include <crtdbg.h>
#include <dsound.h>
#include "WinConfig.h"
#include "Resource.h"
#include "atari800.h"
#include "globals.h"
#include "registry.h"
#include "display_win.h"
#include "misc_win.h"
#include "sound_win.h"


/* Public objects */

BOOL  g_bTimerRollover = FALSE;
FILE *g_pfSndOutput    = NULL;

/* Private objects */

static WAVEHDR      s_arrWaveHDR[ NUM_SOUND_BUFS ];
static WAVEFORMATEX s_wfxWaveFormat;
static HWAVEOUT     s_hWaveOut               = 0;
static char        *s_pcSndBuffer            = NULL;
static char        *s_pcCurSndBuffer         = NULL;
static UINT         s_unSampleSize           = 0;
static UINT         s_unSwitchBuffer         = 1;
static UINT         s_unFrameCount           = 1;
static UINT         s_unUpdateCount          = 1;
static UINT         s_unSamplePos;
static UINT         s_unRealUpdatesPerSample = 262 * SOUND_LATENCY_IN_FRAMES / DEF_SKIP_UPDATE;
static UINT         s_unSkipPokeyUpdate      = DEF_SKIP_UPDATE;
static DWORD        s_dwStartVolume          = 0;
static BOOL         s_bSoundIsPaused         = FALSE;

#ifdef WIN_USE_DSOUND
static LPDIRECTSOUND        s_lpDirectSound  = NULL;
static LPDIRECTSOUNDBUFFER  s_lpDSBuffer     = NULL;
#endif /*WIN_USE_DSOUND*/

static void SndPlay_MMSound( void );
static void SndMngr_MMSound( void );
#ifdef WIN_USE_DSOUND
static void SndPlay_DSound ( void );
static void SndMngr_DSound ( void );
#endif /*WIN_USE_DSOUND*/
static void SndMngr_NoSound( void );

void (*Atari_PlaySound)(void) = SndMngr_NoSound;

#ifdef WIN_USE_DSOUND
static BOOL GetDSErrorString( HRESULT  hResult,  LPSTR lpszErrorBuff, DWORD dwError );
#endif /*WIN_USE_DSOUND*/
static BOOL GetMMErrorString( MMRESULT mmResult, LPSTR lpszErrorBuff, DWORD dwError );

#ifdef _DEBUG
#define ServeMMError( nUID, hResult, bQuit ) \
		ShowMMError( nUID, hResult, bQuit, __FILE__, __LINE__ )
#ifdef WIN_USE_DSOUND
#define ServeDSError( nUID, hResult, bQuit ) \
		ShowDSError( nUID, hResult, bQuit, __FILE__, __LINE__ )
#endif /*WIN_USE_DSOUND*/
#else /*_DEBUG*/
#define ServeMMError( nUID, hResult, bQuit ) \
		ShowMMError( nUID, hResult, bQuit )
#ifdef WIN_USE_DSOUND
#define ServeDSError( nUID, hResult, bQuit ) \
		ShowDSError( nUID, hResult, bQuit )
#endif /*WIN_USE_DSOUND*/
#endif /*_DEBUG*/


/*========================================================
Function : ShowMMError
=========================================================*/
/* #FN#
   Displays a multimedia sound error description */
static
void
/* #AS#
   Nothing */
ShowMMError( UINT     nUID,
			 MMRESULT mmResult,
			 BOOL     bQuit
#ifdef _DEBUG
		   , char   *pszFile,
			 DWORD   dwLine
#endif /*_DEBUG*/
)
{
	char szError [ LOADSTRING_STRING_SIZE ];
	char szAction[ LOADSTRING_STRING_SIZE ];
	
#ifdef _DEBUG
	Aprint( "Multimedia Sound error: %s@%ld", pszFile, dwLine );
#endif /*_DEBUG*/

	/* Get us back to a GDI display and disable sound */
	SetSafeDisplay( FALSE );
	DisableSound( TRUE );
	
	/* Get the error string and present it to the user */
	GetMMErrorString( mmResult, szError, LOADSTRING_STRING_SIZE );
	LoadString( NULL, nUID, szAction, LOADSTRING_STRING_SIZE );
	DisplayMessage( NULL, IDS_MMERR_PROMPT, IDS_MMERR_HDR, MB_ICONSTOP | MB_OK, szAction, szError );

	/* Start a quit (this will end up in Atari_Exit()) */
	if( bQuit )
	{
		/* Make sure the atari is turned off */
		g_ulAtariState = ATARI_UNINITIALIZED;
		PostMessage( g_hMainWnd, WM_CLOSE, 0, 0L );
	}
} /* #OF# ShowMMError */

#ifdef WIN_USE_DSOUND
/*========================================================
Function : ShowMMError
=========================================================*/
/* #FN#
   Displays a direct sound error description */
static
void
/* #AS#
   Nothing */
ShowDSError( UINT    nUID,
			 HRESULT hResult,
			 BOOL    bQuit
#ifdef _DEBUG
		   , char   *pszFile,
			 DWORD   dwLine
#endif /*_DEBUG*/
)
{
	char szError [ LOADSTRING_STRING_SIZE ];
	char szAction[ LOADSTRING_STRING_SIZE ];

#ifdef _DEBUG
	Aprint( "DirectSound error: %s@%ld", pszFile, dwLine );
#endif /*_DEBUG*/

	/* Get us back to a GDI display and disable sound */
	SetSafeDisplay( FALSE );
	DisableSound( TRUE );

	/* Get the error string and present it to the user */
	GetDSErrorString( hResult, szError, LOADSTRING_STRING_SIZE );
	LoadString( NULL, nUID, szAction, LOADSTRING_STRING_SIZE );
	DisplayMessage( NULL, IDS_DSERR_PROMPT, IDS_DSERR_HDR, MB_ICONSTOP | MB_OK, szAction, szError );
	
	/* Start a quit (this will end up in Atari_Exit()) */
	if( bQuit )
	{
		/* Make sure the atari is turned off */
		g_ulAtariState = ATARI_UNINITIALIZED;
		PostMessage( g_hMainWnd, WM_CLOSE, 0, 0L );
	}
} /* #OF# ShowDSError */
#endif /*WIN_USE_DSOUND*/

/*========================================================
Function : wait_for_vbi
=========================================================*/
/* #FN#
   The main timing function an emulated Atari is based on */
static
void
/* #AS#
   Nothing */
wait_for_vbi( void )
{
	long  lSpareTicks;
	ULONG ulTimerLastVal = g_ulAtariHWNextTime;
	LARGE_INTEGER lnTicks;

	QueryPerformanceCounter( &lnTicks );
	if( g_bTimerRollover )
	{
		while( lnTicks.LowPart > g_ulAtariHWNextTime && (g_ulAtariHWNextTime - lnTicks.LowPart < g_ulDeltaT) )
		{
			QueryPerformanceCounter( &lnTicks );
			lSpareTicks = ULONG_MAX - lnTicks.LowPart;
			_ASSERT(lSpareTicks <= (long)g_ulDeltaT);
			if( lSpareTicks > g_nSleepThreshold )
			{
				SleepEx( SLEEP_TIME_IN_MS, TRUE );
				QueryPerformanceCounter( &lnTicks );
			}
		}
		g_bTimerRollover = FALSE;
	}
	lSpareTicks = (long)(g_ulAtariHWNextTime - lnTicks.LowPart);

	if( lSpareTicks > 0L )
	{
		if( !(g_ulMiscStates & MS_FULL_SPEED) )
		{
			while( lSpareTicks > g_nSleepThreshold )
			{
				SleepEx( SLEEP_TIME_IN_MS, TRUE );
				QueryPerformanceCounter( &lnTicks );
				lSpareTicks = (long)(g_ulAtariHWNextTime - lnTicks.LowPart);
			}
			while( g_ulAtariHWNextTime > lnTicks.LowPart && lnTicks.LowPart > ulTimerLastVal )
				QueryPerformanceCounter( &lnTicks );
			g_ulAtariHWNextTime += g_ulDeltaT;
		}
		else
		{
			if( lSpareTicks > (long)g_ulDeltaT )
				g_ulAtariHWNextTime = lnTicks.LowPart + g_ulDeltaT;
			else
				g_ulAtariHWNextTime += g_ulDeltaT;
		}
	}
	else
	{
		if( -lSpareTicks > (long)g_ulDeltaT )
			g_ulAtariHWNextTime = lnTicks.LowPart + g_ulDeltaT;
		else
			g_ulAtariHWNextTime += g_ulDeltaT;
	}
	if( ulTimerLastVal > g_ulAtariHWNextTime )
		g_bTimerRollover = TRUE;
} /* #OF# wait_for_vbi */

#ifdef WIN_USE_DSOUND
static
BOOL
DetermineHardwareCaps( LPDIRECTSOUND lpDirectSound,
					   BOOL          bPrimary )
{
	DSCAPS  dscaps;
	char	szFailed[ LOADSTRING_STRING_SIZE * 2 ];
	char	szError [ LOADSTRING_STRING_SIZE ];
	HRESULT hResult;

	dscaps.dwSize = sizeof(DSCAPS);
	*szFailed = '\0';
	
	hResult = IDirectSound_GetCaps( lpDirectSound, &dscaps );
	
	if( SUCCEEDED(hResult) ) 
	{
		if( bPrimary )
		{	
			if( !(dscaps.dwFlags & DSCAPS_PRIMARY8BIT) )
			{
				LoadString( NULL, IDS_DSERR_NO_PRIMARY, szError, LOADSTRING_STRING_SIZE );
				strcat( szFailed, szError );
			}
			if( !(dscaps.dwFlags & DSCAPS_PRIMARYMONO) )
			{
				LoadString( NULL, IDS_DSERR_NO_MONO, szError, LOADSTRING_STRING_SIZE );
				strcat( szFailed, szError );
			}
			if( stereo_enabled && !(dscaps.dwFlags & DSCAPS_PRIMARYSTEREO) )
			{
				LoadString( NULL, IDS_DSERR_NO_STEREO, szError, LOADSTRING_STRING_SIZE );
				strcat( szFailed, szError );
			}
		}
		if( dscaps.dwFlags & DSCAPS_EMULDRIVER  )
		{
			LoadString( NULL, IDS_DSERR_NO_DRIVER, szError, LOADSTRING_STRING_SIZE );
			strcat( szFailed, szError );
		}
		
		if( *szFailed )
		{
			DisableSound( TRUE );
			DisplayMessage( NULL, IDS_DSERR_INIT, IDS_DSERR_HDR, MB_ICONEXCLAMATION | MB_OK, szFailed );
			return FALSE;
		}
	}
	else
	{
		ServeDSError( IDS_DSERR_QUERY, hResult, FALSE );
		return FALSE;
	}
	return TRUE;
} /* #OF# DetermineHardwareCaps */
#endif /*WIN_USE_DSOUND*/

int
InitialiseSound( void )
{
	short nChannels = stereo_enabled + 1;

	_ASSERT(stereo_enabled == 0 || stereo_enabled == 1);

	s_bSoundIsPaused    = FALSE; //(g_ulMiscStates & MS_FULL_SPEED ? TRUE : FALSE);
	s_unSkipPokeyUpdate = g_nSkipUpdate;
	s_unSamplePos       = 0;
	s_unUpdateCount     = 1;
	s_unSwitchBuffer    = 1;

	if( !s_pcSndBuffer &&
		!(s_pcSndBuffer = calloc( 1, NUM_SOUND_BUFS * DEFAULT_SOUND_BUFFER_SIZE)) )
	{
		DisableSound( TRUE );
		return 0;
	}
	s_pcCurSndBuffer = s_pcSndBuffer;

	if( tv_mode == TV_PAL )
	{
		s_unSampleSize = (g_nSoundRate / g_nPalFreq) * SOUND_LATENCY_IN_FRAMES * nChannels;
		Pokey_sound_init( FREQ_17_EXACT,
			(short)g_nSoundRate, (char)nChannels );
		s_unRealUpdatesPerSample = 312 * SOUND_LATENCY_IN_FRAMES / s_unSkipPokeyUpdate;
	}
	else
	{
		s_unSampleSize = (g_nSoundRate / g_nNtscFreq) * SOUND_LATENCY_IN_FRAMES * nChannels;
		Pokey_sound_init( FREQ_17_EXACT,
			(short)g_nSoundRate, (char)nChannels );
		s_unRealUpdatesPerSample = 262 * SOUND_LATENCY_IN_FRAMES / s_unSkipPokeyUpdate;
	}

	if( g_ulSoundState & SOUND_NOSOUND )
	{
		Atari_PlaySound = SndMngr_NoSound;
		return 1; /* Exit if SOUND_NOSOUND flag is set */
	}

	/* Set this up for PCM, 1/2 channels, 8 bits unsigned samples */
	ZeroMemory( &s_wfxWaveFormat, sizeof(WAVEFORMATEX) );
	s_wfxWaveFormat.wFormatTag      = WAVE_FORMAT_PCM; /* The only tag valid with DirectSound */
	s_wfxWaveFormat.nChannels       = nChannels;
	s_wfxWaveFormat.nSamplesPerSec  = g_nSoundRate;
	s_wfxWaveFormat.wBitsPerSample  = 8;
	s_wfxWaveFormat.nBlockAlign     = (s_wfxWaveFormat.wBitsPerSample * s_wfxWaveFormat.nChannels) / 8;
	s_wfxWaveFormat.nAvgBytesPerSec = s_wfxWaveFormat.nSamplesPerSec * s_wfxWaveFormat.nBlockAlign;
	s_wfxWaveFormat.cbSize          = 0; /* This member is always zero for PCM formats */

	if( g_ulSoundState & SOUND_MMSOUND )
	{
		MMRESULT mmResult = MMSYSERR_NOERROR; /* That initialization is important! */
		int i; /* Loop counter */

		for( i = 0; i < NUM_SOUND_BUFS; i++ )
			ZeroMemory( &s_arrWaveHDR[ i ], sizeof(WAVEHDR) );

		if( !s_hWaveOut )
			mmResult = waveOutOpen( &s_hWaveOut, WAVE_MAPPER, &s_wfxWaveFormat, 0, 0, CALLBACK_NULL );

		if( mmResult != MMSYSERR_NOERROR )
		{
			ServeMMError( IDS_MMERR_OPEN, mmResult, FALSE );
			return 0;
		}
		for( i = 0; i < NUM_SOUND_BUFS; i++ )
		{
			s_arrWaveHDR[ i ].lpData          = &s_pcSndBuffer[ i * DEFAULT_SOUND_BUFFER_SIZE ];
			s_arrWaveHDR[ i ].dwBufferLength  = s_unSampleSize;
			s_arrWaveHDR[ i ].dwBytesRecorded = 0;
			s_arrWaveHDR[ i ].dwUser          = 0;
			s_arrWaveHDR[ i ].dwFlags         = 0;
			s_arrWaveHDR[ i ].dwLoops         = 1;
		}
		mmResult = waveOutPrepareHeader( s_hWaveOut, &s_arrWaveHDR[ s_unSwitchBuffer ], sizeof(s_arrWaveHDR) );
		if( mmResult != MMSYSERR_NOERROR )
		{
			ServeMMError( IDS_MMERR_INIT_HDR, mmResult, FALSE );
			return 0;
		}
		waveOutUnprepareHeader( s_hWaveOut, &s_arrWaveHDR[ s_unSwitchBuffer ], sizeof(WAVEHDR) );

		Atari_PlaySound = SndMngr_MMSound;
	}

#ifdef WIN_USE_DSOUND
	if( g_ulSoundState & SOUND_DIRECTSOUND )
	{
		DSBUFFERDESC dsbdesc;
		HRESULT	     hResult;

		/* Create IDirectSound using the primary sound device */
		if( !s_lpDirectSound && FAILED(
			hResult = DirectSoundCreate( NULL, &s_lpDirectSound, NULL )) )
		{
			ServeDSError( IDS_DSERR_CREATE_OBJ, hResult, FALSE );
			return 0;
		}
		_ASSERT(s_lpDirectSound);

		if( !DetermineHardwareCaps( s_lpDirectSound, g_ulSoundState & SOUND_CUSTOM_RATE ) )
			return 0;

		/* Set up DSBUFFERDESC structure */
		ZeroMemory( &dsbdesc, sizeof(DSBUFFERDESC) );
		dsbdesc.dwSize  = sizeof(DSBUFFERDESC);
		dsbdesc.dwFlags = DSBCAPS_PRIMARYBUFFER;

		if( g_ulSoundState & SOUND_CUSTOM_RATE )
		{
			LPDIRECTSOUNDBUFFER lpDSBPrimary;

			/* Obtain priority cooperative level */
			hResult = IDirectSound_SetCooperativeLevel( s_lpDirectSound, g_hMainWnd, DSSCL_PRIORITY );
			if( FAILED(hResult) )
			{
				ServeDSError( IDS_DSERR_COOP_PRIORITY, hResult, FALSE );
				return 0;
			}

			/* The DirectSound mixer produces the best sound quality if all
			   application's sounds use the same wave format and the hardware
			   output format is matched to the format of the sounds. If this
			   is done, the mixer need not perform any format conversion.
			   Note that this primary buffer is for control purposes only;
			   creating it is not the same as obtaining write access to the
			   primary buffer */

			/* Try to create primary buffer */
			hResult = IDirectSound_CreateSoundBuffer( s_lpDirectSound, &dsbdesc, &lpDSBPrimary, NULL );
			if( FAILED(hResult) )
			{
				lpDSBPrimary = NULL;
				ServeDSError( IDS_DSERR_CREATE_PRIMARY, hResult, FALSE );
				return 0;
			}
			/* Set primary buffer to desired format */
			hResult = IDirectSoundBuffer_SetFormat( lpDSBPrimary, &s_wfxWaveFormat );
			if( FAILED(hResult) )
			{
				ServeDSError( IDS_DSERR_FORMAT_PRIMARY, hResult, FALSE );
				return 0;
			}
//			if( stereo_enabled )
//			{
//				hResult = IDirectSoundBuffer_SetPan( lpDSBPrimary, DSBPAN_CENTER );
//				if( FAILED(hResult) )
//				{
//					ServeDSError( IDS_DSERR_FORMAT_PRIMARY, hResult, FALSE );
//					return 0;
//				}
//			}
			/* Release primary buffer */
			IDirectSoundBuffer_Release( lpDSBPrimary );
		}
		else
		{
			/* Obtain normal cooperative level (22kHz) */
			hResult = IDirectSound_SetCooperativeLevel( s_lpDirectSound, g_hMainWnd, DSSCL_NORMAL );
			if( FAILED(hResult) )
			{
				ServeDSError( IDS_DSERR_COOP_NORMAL, hResult, FALSE );
				return 0;
			}
		}
		/* Create a secondary DirectSound buffers */

		/* Set up DSBUFFERDESC structure */
		ZeroMemory( &dsbdesc, sizeof(DSBUFFERDESC) );
		dsbdesc.dwSize = sizeof(DSBUFFERDESC);
		/* Need default controls (volume, frequency) */
		dsbdesc.dwFlags = DSBCAPS_CTRLVOLUME | DSBCAPS_CTRLFREQUENCY | DSBCAPS_GETCURRENTPOSITION2;
		/* 2-sample buffer (put some extra on the end because we'll
		   rarely refresh at exactly the right time */
		dsbdesc.dwBufferBytes = s_unSampleSize;
		dsbdesc.lpwfxFormat   = &s_wfxWaveFormat;

		hResult = IDirectSound_CreateSoundBuffer( s_lpDirectSound, &dsbdesc, &s_lpDSBuffer, NULL );
		if( FAILED(hResult) )
		{ 
			s_lpDSBuffer = NULL;
			ServeDSError( IDS_DSERR_CREATE_BUFF, hResult, FALSE );
			return 0;
		}
		/* Set current play position of secondary buffer to zero */
		IDirectSoundBuffer_SetCurrentPosition( s_lpDSBuffer, 0 );

		Atari_PlaySound = SndMngr_DSound;
	}
#endif /*WIN_USE_DSOUND*/

	/* Set sound volume */
	Sound_SetVolume();

	if( g_ulMiscStates & MS_FULL_SPEED )
		ClearSound( FALSE );

	return 1;
} /* #OF# InitialiseSound */

/*========================================================
Function : pokey_update
=========================================================*/
/* #FN#
   This function is called by an Atari800 kernel */
void
/* #AS#
   Nothing */
pokey_update( void )
{
	UINT unNewSamplePos = 0;

	if( --s_unSkipPokeyUpdate )
		return;
	else
		s_unSkipPokeyUpdate = g_nSkipUpdate;

	if( s_bSoundIsPaused || !s_pcCurSndBuffer /* Not really needed */ )
		return;

	unNewSamplePos = (s_unSampleSize * s_unUpdateCount / s_unRealUpdatesPerSample) & 0xfffffffe;
	if( unNewSamplePos > s_unSampleSize )
		unNewSamplePos = s_unSampleSize;

	_ASSERT(s_pcCurSndBuffer);

#ifdef WIN_USE_DSOUND
	if( g_ulSoundState & SOUND_DIRECTSOUND )
	{
		LPVOID  lpvPtr1;
		DWORD   dwBytes1;
//		DWORD   dwPlayCursor, dwWriteCursor;
		HRESULT hResult =

//		hResult = IDirectSoundBuffer_GetCurrentPosition( s_lpDSBuffer, &dwPlayCursor, &dwWriteCursor );
//	    if( dwPlayCursor && (dwPlayCursor >= s_unSamplePos) && (dwPlayCursor < unNewSamplePos) )
//			return;
			
		/* Obtain a valid write pointer to the sound buffer's audio data */
		IDirectSoundBuffer_Lock( s_lpDSBuffer, s_unSamplePos, unNewSamplePos - s_unSamplePos, &lpvPtr1, &dwBytes1, 0, 0, 0 );
		/* If DSERR_BUFFERLOST is returned, restore and retry lock */
		if( hResult == DSERR_BUFFERLOST )
		{ 
			hResult = IDirectSoundBuffer_Restore( s_lpDSBuffer );
			if( FAILED(hResult) )
			{
				ServeDSError( IDS_DSERR_RESTORE, hResult, FALSE );
				return;
			}
			hResult = IDirectSoundBuffer_Lock( s_lpDSBuffer, s_unSamplePos, unNewSamplePos - s_unSamplePos, &lpvPtr1, &dwBytes1, 0, 0, 0 );
		}
		if( FAILED(hResult) )
		{
			ServeDSError( IDS_DSERR_LOCK, hResult, FALSE );
			return;
		}
		/* Write the audio data to the buffer */
		Pokey_process( lpvPtr1, (short)dwBytes1 );
		if( g_pfSndOutput )
			CopyMemory( s_pcCurSndBuffer + s_unSamplePos, lpvPtr1, dwBytes1 );

		/* Release the data back to DirectSound */
		hResult = IDirectSoundBuffer_Unlock( s_lpDSBuffer, lpvPtr1, dwBytes1, 0, 0 );
		if( FAILED(hResult) )
		{
			ServeDSError( IDS_DSERR_UNLOCK, hResult, FALSE );
			return;
		}
	}
	else
#endif /*WIN_USE_DSOUND*/
		/* Write the part of audio data to the buffer */
		Pokey_process( s_pcCurSndBuffer + s_unSamplePos, (short)(unNewSamplePos - s_unSamplePos) );

	s_unSamplePos = unNewSamplePos;
	s_unUpdateCount++;
} /* #OF# pokey_update */

static
void
SndMngr_NoSound( void )
{
	if( s_bSoundIsPaused )
	{
		wait_for_vbi();
		return;
	}
	if( ++s_unFrameCount > SOUND_LATENCY_IN_FRAMES )
	{
		_ASSERT(s_pcCurSndBuffer);

		if( s_unSamplePos < s_unSampleSize )
			Pokey_process( s_pcCurSndBuffer + s_unSamplePos, (short)(s_unSampleSize - s_unSamplePos) );

		if( g_pfSndOutput )
			fwrite( s_pcCurSndBuffer, s_unSampleSize, 1, g_pfSndOutput );

		if( ++s_unSwitchBuffer == NUM_SOUND_BUFS )
			s_unSwitchBuffer = 0;

		s_pcCurSndBuffer = &s_pcSndBuffer[ s_unSwitchBuffer * DEFAULT_SOUND_BUFFER_SIZE ];
		s_unFrameCount   = 1;
		s_unUpdateCount  = 1;
		s_unSamplePos    = 0;
	}
	wait_for_vbi();
} /* #OF# SndMngr_NoSound */

__inline
static
void
SndPlay_MMSound()
{
	MMRESULT mmResult;

	_ASSERT(s_unSamplePos < DEFAULT_SOUND_BUFFER_SIZE);
	_ASSERT(s_pcCurSndBuffer);

	if( s_unSamplePos < s_unSampleSize )
		/* Write the audio data to the buffer if it is not full */
		Pokey_process( s_pcCurSndBuffer + s_unSamplePos, (short)(s_unSampleSize - s_unSamplePos) );

	if( g_pfSndOutput )
		fwrite( s_pcCurSndBuffer, s_unSampleSize, 1, g_pfSndOutput );
	
	s_arrWaveHDR[ s_unSwitchBuffer ].dwBufferLength = s_unSampleSize;
	waveOutUnprepareHeader( s_hWaveOut, &s_arrWaveHDR[ s_unSwitchBuffer ], sizeof(WAVEHDR) );

	mmResult = waveOutPrepareHeader( s_hWaveOut, &s_arrWaveHDR[ s_unSwitchBuffer ], sizeof(WAVEHDR) );
	if( mmResult != MMSYSERR_NOERROR )
		ServeMMError( IDS_MMERR_PREP_HDR, mmResult, FALSE );

	wait_for_vbi();

	waveOutWrite( s_hWaveOut, &s_arrWaveHDR[ s_unSwitchBuffer ], sizeof(WAVEHDR) );
} /* #OF# SndPlay_MMSound */

static
void
SndMngr_MMSound( void )
{
	_ASSERT(s_pcCurSndBuffer < &s_pcSndBuffer[ 0 ] + s_unSwitchBuffer * DEFAULT_SOUND_BUFFER_SIZE + DEFAULT_SOUND_BUFFER_SIZE);
	_ASSERT(SOUND_LATENCY_IN_FRAMES + 1 >= s_unFrameCount);

	if( ++s_unFrameCount > SOUND_LATENCY_IN_FRAMES )
	{
		if( !s_bSoundIsPaused )
		{
			_ASSERT(s_unSamplePos);
			SndPlay_MMSound();
		}
		if( ++s_unSwitchBuffer == NUM_SOUND_BUFS )
			s_unSwitchBuffer = 0;

		s_pcCurSndBuffer = &s_pcSndBuffer[ s_unSwitchBuffer * DEFAULT_SOUND_BUFFER_SIZE ];
		s_unFrameCount   = 1;
		s_unUpdateCount  = 1;
		s_unSamplePos    = 0;
	}
	else
		wait_for_vbi();
} /* #OF# SndMngr_MMSound */

#ifdef WIN_USE_DSOUND
__inline
static
void
SndPlay_DSound( void )
{
	_ASSERT(s_unSamplePos < DEFAULT_SOUND_BUFFER_SIZE);
	_ASSERT(s_lpDSBuffer);

	if( s_unSamplePos < s_unSampleSize )
	{
		LPVOID  lpvPtr1;
		DWORD   dwBytes1;
		HRESULT hResult =
		/* Obtain a valid write pointer to the sound buffer's audio data */
		IDirectSoundBuffer_Lock( s_lpDSBuffer, s_unSamplePos, s_unSampleSize - s_unSamplePos, &lpvPtr1, &dwBytes1, 0, 0, 0 );
		/* If DSERR_BUFFERLOST is returned, restore and retry lock */
		if( hResult == DSERR_BUFFERLOST )
		{ 
			hResult = IDirectSoundBuffer_Restore( s_lpDSBuffer );
			if( FAILED(hResult) )
			{
				ServeDSError( IDS_DSERR_RESTORE, hResult, FALSE );
				return;
			}
			hResult = IDirectSoundBuffer_Lock( s_lpDSBuffer, s_unSamplePos, s_unSampleSize - s_unSamplePos, &lpvPtr1, &dwBytes1, 0, 0, 0 );
		}
		if( FAILED(hResult) )
		{
			ServeDSError( IDS_DSERR_LOCK, hResult, FALSE );
			return;
		}
		/* Write the audio data to the buffer if it is not full */
		Pokey_process( lpvPtr1, (short)dwBytes1 );
		if( g_pfSndOutput )
			CopyMemory( s_pcCurSndBuffer + s_unSamplePos, lpvPtr1, dwBytes1 );

		/* Release the data back to DirectSound */
		hResult = IDirectSoundBuffer_Unlock( s_lpDSBuffer, lpvPtr1, dwBytes1, 0, 0 );
		if( FAILED(hResult) )
		{
			ServeDSError( IDS_DSERR_UNLOCK, hResult, FALSE );
			return;
		}
	}
	if( g_pfSndOutput )
		fwrite( s_pcCurSndBuffer, s_unSampleSize, 1, g_pfSndOutput );

	wait_for_vbi();

	IDirectSoundBuffer_Play( s_lpDSBuffer, 0, 0, DSBPLAY_LOOPING );
} /* #OF# SndPlay_DSound */

static
void
SndMngr_DSound( void )
{
	if( ++s_unFrameCount > SOUND_LATENCY_IN_FRAMES )
	{
		if( !s_bSoundIsPaused )
		{
			_ASSERT(s_unSamplePos);
			SndPlay_DSound();
		}
		s_unFrameCount  = 1;
		s_unUpdateCount = 1;
		s_unSamplePos   = 0;
	}
	else
		wait_for_vbi();
} /* #OF# SndMngr_DSound */
#endif /*WIN_USE_DSOUND*/

/*========================================================
Function : Sound_VolumeCapable
=========================================================*/
/* #FN#
   Checks if a sound output device supports a volume control */
BOOL
/* #AS#
   TRUE if volume control is supported, otherwise FALSE */
Sound_VolumeCapable( void )
{
	BOOL bReturn = TRUE;

	if( g_ulSoundState & SOUND_MMSOUND )
	{
		WAVEOUTCAPS woc;
		memset( &woc, 0, sizeof(WAVEOUTCAPS) );

		if( s_hWaveOut )
		{
			waveOutGetDevCaps( (unsigned int)s_hWaveOut, &woc, sizeof(WAVEOUTCAPS) );
		}
		else
		{
			HWAVEOUT hLocalWave;
			MMRESULT mmResult;

			mmResult = waveOutOpen( &hLocalWave, WAVE_MAPPER, &s_wfxWaveFormat, 0, 0, CALLBACK_NULL );
			if( mmResult == MMSYSERR_NOERROR )
			{
				waveOutGetDevCaps( (unsigned int)hLocalWave, &woc, sizeof(WAVEOUTCAPS) );
				waveOutClose( hLocalWave );
			}
		}
		bReturn = (woc.dwSupport & WAVECAPS_VOLUME ? TRUE : FALSE);
	}
	return bReturn;
} /* #OF# Sound_VolumeCapable */

/*========================================================
Function : Sound_SetVolume
=========================================================*/
/* #FN#
   Sets volume of a sound playback */
void
/* #AS#
   Nothing */
Sound_SetVolume( void )
{
	if( g_ulSoundState & SOUND_MMSOUND && s_hWaveOut )
	{
		if( Sound_VolumeCapable() )
		{
			MMRESULT mmResult;
			DWORD    dwVolume, dwTempVolume;

			/* Save the volume setting so we can put it back later */
			if( !s_dwStartVolume )
				mmResult = waveOutGetVolume( s_hWaveOut, &s_dwStartVolume );

			/* Hiword is the right channel, low word is the left channel */
			dwVolume = (HIWORD(s_dwStartVolume) + HIWORD(s_dwStartVolume) / 100 * g_nSoundVol) * 65536;
			dwVolume += LOWORD(s_dwStartVolume) + LOWORD(s_dwStartVolume) / 100 * g_nSoundVol;

			mmResult = waveOutSetVolume( s_hWaveOut, dwVolume );
			if( mmResult != MMSYSERR_NOERROR )
			{
				ServeMMError( IDS_MMERR_SET_VOLUME, mmResult, FALSE );
				return;
			}

			/* It's possible this wave device doesn't support 16 bits of volume control,
			   so we'll check the result of the set with waveOutGetVolume, if it's less
			   than what we set, we know the new max and we'll scale to that */

			waveOutGetVolume( s_hWaveOut, &dwTempVolume );
			if( dwTempVolume < dwVolume )
			{
				float fPercentage = ((float)dwTempVolume / dwVolume);
				dwVolume = (unsigned long)(dwVolume * fPercentage);
				waveOutSetVolume( s_hWaveOut, dwVolume );
			}
		}
	}
#ifdef WIN_USE_DSOUND
	if( g_ulSoundState & SOUND_DIRECTSOUND && s_lpDSBuffer )
		IDirectSoundBuffer_SetVolume( s_lpDSBuffer, g_nSoundVol * 22 );
#endif /*WIN_USE_DSOUND*/
} /* #OF# Sound_SetVolume */

/*========================================================
Function : ClearSound
=========================================================*/
/* #FN#
   Closes/mutes a sound output device */
void
/* #AS#
   Nothing */
ClearSound( BOOL bPermanent )
{
	if( !s_bSoundIsPaused || bPermanent )
	{
		s_bSoundIsPaused = TRUE;

		/* Are we shutting down everything, or just killing the 
		   sound that is playing currently? */
		if( bPermanent )
		{
			int i;
			Atari_PlaySound = SndMngr_NoSound;

			if( g_pfSndOutput )
				/* Close Sound Output file */
				CloseSndOutput();

			/* Clear Multimedia stuff */
			if( s_hWaveOut )
			{
				if( s_dwStartVolume )
				{
					waveOutSetVolume( s_hWaveOut, s_dwStartVolume );
					s_dwStartVolume = 0;
				}
				waveOutReset( s_hWaveOut );
				for( i = 0; i < NUM_SOUND_BUFS; i++ )
				{
					waveOutUnprepareHeader( s_hWaveOut, &s_arrWaveHDR[ i ], sizeof(WAVEHDR) );
					s_arrWaveHDR[ i ].lpData = NULL;
				}
				waveOutClose( s_hWaveOut );
				s_hWaveOut = 0;
			}
			if( s_pcSndBuffer )
			{
				free( s_pcSndBuffer );
				s_pcSndBuffer = NULL;
			}
			s_pcCurSndBuffer = NULL;

#ifdef WIN_USE_DSOUND
			/* Clear DirectSound stuff */
			if( s_lpDSBuffer )
			{
				IDirectSoundBuffer_Release( s_lpDSBuffer );
				s_lpDSBuffer = NULL;
			}
			if( s_lpDirectSound )
			{
				IDirectSound_Release( s_lpDirectSound );
				s_lpDirectSound = NULL;
			}
#endif /*WIN_USE_DSOUND*/
		}
		else
		{
			/* Stop playback and clear waveOut buffer */
			if( s_hWaveOut )
				waveOutPause( s_hWaveOut );

//			if( s_pcSndBuffer )
//				ZeroMemory( s_pcSndBuffer,
//					NUM_SOUND_BUFS * DEFAULT_SOUND_BUFFER_SIZE );

#ifdef WIN_USE_DSOUND
			/* Clear DirectSound buffer and stop playback */
			if( s_lpDSBuffer )
				IDirectSoundBuffer_Stop( s_lpDSBuffer );
#endif /*WIN_USE_DSOUND*/
		}
	}
} /* #OF# ClearSound */

/*========================================================
Function : RestartSound
=========================================================*/
/* #FN#
   Resumes playback on a paused sound output device */
void
/* #AS#
   Nothing */
RestartSound( void )
{
	if( g_ulMiscStates & MS_FULL_SPEED )
		return;

	if( s_bSoundIsPaused )
	{
		if( s_hWaveOut )
		{
			waveOutRestart( s_hWaveOut );
			s_bSoundIsPaused = FALSE;
		}
#ifdef WIN_USE_DSOUND
//		if( s_lpDSBuffer )
//		{
//			IDirectSoundBuffer_Play( s_lpDSBuffer, 0, 0, DSBPLAY_LOOPING );
//			s_bSoundIsPaused = FALSE;
//		}
#endif /*WIN_USE_DSOUND*/
	}
} /* #OF# RestartSound */

/*========================================================
Function : OpenSndOutput
=========================================================*/
/* #FN#
   Opens a sound file */
void
/* #AS#
   Nothing */
OpenSndOutput( char *pszOutFileName )
{
	WAVEFORMAT wf;

	if( g_pfSndOutput )
		/* Close Sound Output file */
		CloseSndOutput();

	g_pfSndOutput = fopen( pszOutFileName, "wb" );

	_ASSERT(stereo_enabled == 0 || stereo_enabled == 1);

//	wf.wFormatTag      = WAVE_FORMAT_PCM; 
	wf.nChannels       = stereo_enabled + 1; 
	wf.nSamplesPerSec  = g_nSoundRate;
	wf.nBlockAlign     = wf.nChannels;
	wf.nAvgBytesPerSec = wf.nSamplesPerSec * wf.nBlockAlign;
	/*
	  Offset  Length   Contents

	The RIFF header: 
	  0       4 bytes  'RIFF'
	  4       4 bytes  <file length - 8>
	  8       4 bytes  'WAVE'
	
	The format (fmt) chunk: 
	  12      4 bytes  'fmt '
	  16      4 bytes  0x00000010     // Length of the fmt data (16 bytes)
	  20      2 bytes  0x0001         // Format tag: 1 = PCM
	  22      2 bytes  <channels>     // Channels: 1 = mono, 2 = stereo
	  24      4 bytes  <sample rate>  // Samples per second: e.g., 44100
	  28      4 bytes  <bytes/second> // sample rate * block align
	  32      2 bytes  <block align>  // channels * bits/sample / 8
	  34      2 bytes  <bits/sample>  // 8 or 16

	The data chunk: 
	  36      4 bytes  'data'
	  40      4 bytes  <length of the data block>
	  44        bytes  <sample data>
	*/
	fwrite( "RIFF\000\000\000\000WAVEfmt\040\020\000\000\000\001\000", 22, 1, g_pfSndOutput );
	fwrite( &wf.nChannels,       2, 1, g_pfSndOutput );
	fwrite( &wf.nSamplesPerSec,  4, 1, g_pfSndOutput );
	fwrite( &wf.nAvgBytesPerSec, 4, 1, g_pfSndOutput );
	fwrite( &wf.nBlockAlign,     2, 1, g_pfSndOutput );
	fwrite( "\010\000data\000\000\000\000", 10, 1, g_pfSndOutput );
} /* #OF# OpenSndOutput */

/*========================================================
Function : CloseSndOutput
=========================================================*/
/* #FN#
   Closes a sound file */
void
/* #AS#
   Nothing */
CloseSndOutput( void )
{
	if( g_pfSndOutput )
	{
		UINT unPos = 0;

		/* Sound file is finished, so modify header and close it */
		unPos = ftell( g_pfSndOutput ) - 8;
		fseek ( g_pfSndOutput, 4, SEEK_SET );	// Seek past RIFF
		fwrite( &unPos, 4, 1, g_pfSndOutput );	// Write out size of entire data chunk
		fseek ( g_pfSndOutput, 40, SEEK_SET );
		unPos -= 36;						
		fwrite( &unPos, 4, 1, g_pfSndOutput );	// Write out size of just sample data
		fclose( g_pfSndOutput );
		g_pfSndOutput = NULL;

		DisplayMessage( NULL, IDS_SFX_FILE_CLOSED, 0, MB_ICONINFORMATION | MB_OK );
	}
} /* #OF# CloseSndOutput */

/*========================================================
Function : DisableSound
=========================================================*/
/* #FN#
   Turns off a sound playback (disable sound) */
int
/* #AS#
   A status returned by InitialiseSound function */
DisableSound( BOOL bClearSound )
{
	if( bClearSound )
		ClearSound( TRUE );

	/* Modify sound state of emulator */
//	g_ulSoundState &= ~(SOUND_DIRECTSOUND | SOUND_MMSOUND); /* Sound/Mute wants these flags */
	g_ulSoundState |= SOUND_NOSOUND;
	WriteRegDWORD( NULL, REG_SOUND_STATE, g_ulSoundState );

	/* It always will be succeeded for SOUND_NOSOUND */
	return InitialiseSound();
} /* #OF# DisableSound */

#ifdef WIN_USE_DSOUND
/*========================================================
Function : GetDSErrorString
=========================================================*/
/* #FN#
   Outputs a debug string to debugger */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
GetDSErrorString( HRESULT hResult,
				  LPSTR   lpszErrorBuff,
				  DWORD   dwError )
{
	DWORD cLen;
	LPSTR lpszError;
	TCHAR szMsg[ 256 ];

	/* Check parameters */
	if( !lpszErrorBuff || !dwError )
	{
		/* Error, invalid parameters */
		return FALSE;
	}

	switch( hResult )
	{
		/* The request completed successfully */
		case DS_OK:
			lpszError = TEXT("DS_OK");
			break;

#ifdef WIN_TRANSLATE_ERRORS

		/* The request failed because resources, such as a priority level, were already in use by another caller */
		case DSERR_ALLOCATED:
			lpszError = TEXT("DSERR_ALLOCATED");
			break;

		/* The object is already initialized */
		case DSERR_ALREADYINITIALIZED:
			lpszError = TEXT("DSERR_ALREADYINITIALIZED");
			break;

		/* The specified wave format is not supported */
		case DSERR_BADFORMAT:
			lpszError = TEXT("DSERR_BADFORMAT");
			break;

		/* The buffer memory has been lost and must be restored */
		case DSERR_BUFFERLOST:
			lpszError = TEXT("DSERR_BUFFERLOST");
			break;

		/* The control (volume, pan, and so forth) requested by the caller is not available */
		case DSERR_CONTROLUNAVAIL:
			lpszError = TEXT("DSERR_CONTROLUNAVAIL");
			break;

		/* An undetermined error occurred inside the DirectSound subsystem */
		case DSERR_GENERIC:
			lpszError = TEXT("DSERR_GENERIC");
			break;

		/* This function is not valid for the current state of this object */
		case DSERR_INVALIDCALL:
			lpszError = TEXT("DSERR_INVALIDCALL");
			break;

		/* An invalid parameter was passed to the returning function */
		case DSERR_INVALIDPARAM:
			lpszError = TEXT("DSERR_INVALIDPARAM");
			break;
		
		/* The object does not support aggregation */
		case DSERR_NOAGGREGATION:
			lpszError = TEXT("DSERR_NOAGGREGATION");
			break;

		/* No sound driver is available for use */
		case DSERR_NODRIVER:
			lpszError = TEXT("DSERR_NODRIVER");
			break;

		/* The requested COM interface is not available */
		case DSERR_NOINTERFACE:
			lpszError = TEXT("DSERR_NOINTERFACE");
			break;
			
		/* Another application has a higher priority level, preventing this call from succeeding */
		case DSERR_OTHERAPPHASPRIO:
			lpszError = TEXT("DSERR_OTHERAPPHASPRIO");
			break;

		/* The DirectSound subsystem could not allocate sufficient memory to complete the caller's request */
		case DSERR_OUTOFMEMORY:
			lpszError = TEXT("DSERR_OUTOFMEMORY");
			break;

		/* The caller does not have the priority level required for the function to succeed */
		case DSERR_PRIOLEVELNEEDED:
			lpszError = TEXT("DSERR_PRIOLEVELNEEDED");
			break;

		/* The IDirectSound::Initialize method has not been called or has not been called successfully before other methods were called */
		case DSERR_UNINITIALIZED:
			lpszError = TEXT("DSERR_UNINITIALIZED");
			break;

		/* The function called is not supported at this time */
		case DSERR_UNSUPPORTED:
			lpszError = TEXT("DSERR_UNSUPPORTED");
			break;

#endif /*WIN_TRANSLATE_ERRORS*/

		/* Unknown DS Error */
		default:
			wsprintf( szMsg, "Error #%ld", (DWORD)(hResult & 0x0000FFFFL) );
			lpszError = szMsg;
			break;
	}
	/* Copy DS Error string to buff */
	cLen = strlen( lpszError );
	if (cLen >= dwError)
	{
		cLen = dwError - 1;
	}
	if( cLen )
	{
		strncpy( lpszErrorBuff, lpszError, cLen );
		lpszErrorBuff[ cLen ] = 0;
	}
	return TRUE;
} /* #OF# GetDSErrorString */
#endif /*WIN_USE_DSOUND*/

/*========================================================
Function : GetMMErrorString
=========================================================*/
/* #FN#
   Outputs a debug string to debugger */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
GetMMErrorString( MMRESULT mmResult,
				  LPSTR    lpszErrorBuff,
				  DWORD    dwError )
{
	DWORD cLen;
	LPSTR lpszError;
	TCHAR szMsg[ 256 ];

	/* Check parameters */
	if( !lpszErrorBuff || !dwError )
	{
		/* Error, invalid parameters */
		return FALSE;
	}

	switch( mmResult )
	{
		/* No error */ 
		case MMSYSERR_NOERROR:
			lpszError = TEXT("MMSYSERR_NOERROR");
			break;

#ifdef WIN_TRANSLATE_ERRORS

		/* Wave header is not prepared for output */
		case WAVERR_UNPREPARED:
			lpszError = TEXT( "WAVERR_UNPREPARED");
			break;

		/* Unspecified error */
		case MMSYSERR_ERROR:
			lpszError = TEXT("MMSYSERR_ERROR");
			break;

		/* Device ID out of range */
		case MMSYSERR_BADDEVICEID:
			lpszError = TEXT("MMSYSERR_BADDEVICEID");
			break;

		/* Driver failed enable */
		case MMSYSERR_NOTENABLED:
			lpszError = TEXT("MMSYSERR_NOTENABLED");
			break;

		/* Device already allocated */
		case MMSYSERR_ALLOCATED:
			lpszError = TEXT("MMSYSERR_ALLOCATED");
			break;

		/* Device handle is invalid */
		case MMSYSERR_INVALHANDLE:
			lpszError = TEXT("MMSYSERR_INVALHANDLE");
			break;

		/* No device driver present */
		case MMSYSERR_NODRIVER:
			lpszError = TEXT("MMSYSERR_NODRIVER");
			break;

		/* Memory allocation error */
		case MMSYSERR_NOMEM:
			lpszError = TEXT("MMSYSERR_NOMEM");
			break;

		/* Function isn't supported */
		case MMSYSERR_NOTSUPPORTED:
			lpszError = TEXT("MMSYSERR_NOTSUPPORTED");
			break;

		/* Error value out of range */
		case MMSYSERR_BADERRNUM:
			lpszError = TEXT("MMSYSERR_BADERRNUM");
			break;

		/* Invalid flag passed */
		case MMSYSERR_INVALFLAG:
			lpszError = TEXT("MMSYSERR_INVALFLAG");
			break;

		/* Invalid parameter passed */
		case MMSYSERR_INVALPARAM:
			lpszError = TEXT("MMSYSERR_INVALPARAM");
			break;

		/* Handle being used */
		case MMSYSERR_HANDLEBUSY:
			lpszError = TEXT("MMSYSERR_HANDLEBUSY");
			break;

		/* Specified alias not found */
		case MMSYSERR_INVALIDALIAS:
			lpszError = TEXT("MMSYSERR_INVALIDALIAS");
			break;

		/* Bad registry database */
		case MMSYSERR_BADDB:
			lpszError = TEXT("MMSYSERR_BADDB");
			break;

		/* Registry key not found */
		case MMSYSERR_KEYNOTFOUND:
			lpszError = TEXT("MMSYSERR_KEYNOTFOUND");
			break;

		/* Registry read error */
		case MMSYSERR_READERROR:
			lpszError = TEXT("MMSYSERR_READERROR");
			break;

		/* Registry write error */
		case MMSYSERR_WRITEERROR:
			lpszError = TEXT("MMSYSERR_WRITEERROR");
			break;

		/* Registry delete error */
		case MMSYSERR_DELETEERROR:
			lpszError = TEXT("MMSYSERR_DELETEERROR");
			break;

		/* Registry value not found */
		case MMSYSERR_VALNOTFOUND:
			lpszError = TEXT("MMSYSERR_VALNOTFOUND");
			break;

		/* Driver does not call DriverCallback */
		case MMSYSERR_NODRIVERCB:
			lpszError = TEXT("MMSYSERR_NODRIVERCB");
			break;

#endif /*WIN_TRANSLATE_ERRORS*/

		/* Unknown MM Error */
		default:
			wsprintf( szMsg, "Unknown Error #%ld", (DWORD)(mmResult) );
			lpszError = szMsg;
			break;
	}

	/* Copy MM Error string to buff */
	cLen = strlen (lpszError);
	if( cLen >= dwError )
	{
		cLen = dwError - 1;
	}
	if( cLen )
	{
		strncpy( lpszErrorBuff, lpszError, cLen );
		lpszErrorBuff[ cLen ] = 0;
	}
	return TRUE;
} /* #OF# GetMMErrorString */
