/****************************************************************************
File    : misc_win.c
/*
@(#) #SY# Atari800Win
@(#) #IS# Miscellanous stuff implementation for Win32 platforms
@(#) #BY# Richard Lawrence, Tomasz Szymankowski
@(#) #LM# 01.12.2000
*/

/*
Copyright (c) 1998 Richard Lawrence

This program is free software; you can redistribute it and/or modify it under the terms 
of the GNU General Public License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later version. This program is 
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details. You should have received a copy of the GNU
General Public License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#define _WIN32_WINNT	0x0400  /* for LowLevelKeyboardProc purpose */

#include <windows.h>
#include <shellapi.h>
#include <stdlib.h>
#include <crtdbg.h>
#include <stdio.h>
#include <io.h>
#include <fcntl.h>
#include <mmsystem.h>
#include "WinConfig.h"
#include "Resource.h"
#include "atari800.h"
#include "globals.h"
#include "macros.h"
#include "crc.h"
#include "zlib.h"
#include "registry.h"
#include "display_win.h"
#include "sound_win.h"
#include "input_win.h"
#include "misc_win.h"

/* Constants definition */

#define FCT_CHECK_16STD		0x01
#define FCT_CHECK_16AGS		0x02
#define FCT_CHECK_16OSS		0x04
#define FCT_CHECK_16ROM		(FCT_CHECK_16STD | FCT_CHECK_16AGS | FCT_CHECK_16OSS)
#define FCT_CHECK_32AGS		0x08
#define FCT_CHECK_32DB		0x10
#define FCT_CHECK_32ROM		(FCT_CHECK_32AGS | FCT_CHECK_32DB)

#define SYS_WIN_95			0x01
#define SYS_WIN_98			0x02
#define SYS_WIN_9x			0x0f
#define SYS_WIN_NT4			0x10
#define SYS_WIN_NT5			0x20
#define SYS_WIN_NT			0xf0

#define	EC_BUFFER_LEN		32
#define EC_MAX_LINE_LEN		2048

/* Public objects */

char g_szSystemDesc[ 64 ];

FARPROC	pgzread      = NULL;
FARPROC	pgzopen      = NULL;
FARPROC	pgzclose     = NULL;
FARPROC	pgzwrite     = NULL;
FARPROC	pgzerror     = NULL;

LPCSTR TITLE_STOPPED = "Stopped";
LPCSTR TITLE_NOTRUN  = "Not running";
LPCSTR TITLE_PAUSED  = "Paused (F9 to continue)";
LPCSTR TITLE_FAILURE = "Initialization failure";
LPCSTR TITLE_MONITOR = "Running monitor";

UINT g_unSystemInfo = 0;
int  g_nPalFreq     = 50;
int  g_nNtscFreq    = 60;

struct RomTypeInfo_t g_aRomTypeInfo[] =
{
	{ RTI_OSA, 2370568491L, "Rev.A  \t| 4/800\t| PAL" },
	{ RTI_OSB, 3252116993L, "Rev.B  \t| 4/800\t| PAL" },
	{ RTI_OSB, 4051249634L, "Rev.B  \t| 4/800\t| NTSC" },
	{ RTI_XLE, 2613326695L, "Rev.1  \t| 600XL\t| 03/11/1983" },
	{ RTI_XLE, 3764596111L, "Rev.2  \t| XL/XE\t| 05/10/1983" },
	{ RTI_XLE, 3591293960L, "Rev.3  \t| 800XE\t| 03/01/1985" },
	{ RTI_XLE, 3780165629L, "Rev.4  \t| XEGS \t| 05/07/1987" },
	{ RTI_XLE, 2766230028L, "Rev.3  \t| ARGS \t| ---" },
	{ RTI_XLE, 1577346053L, "Rev.2.3\t| QMEG \t| ---" },
	{ RTI_XLE, 1932228838L, "Rev.3.2\t| QMEG \t| ---" },
	{ RTI_XLE, 2265792253L, "Rev.3.8\t| QMEG \t| ---" },
	{ RTI_XLE, 2603811756L, "Rev.4.2\t| QMEG \t| ---" },
	{ RTI_XLE, 1105050917L, "Rev.4.3\t| QMEG \t| ---" },
	{ RTI_XLE, 2258206520L, "---    \t| TOMS \t| ---" },
	{ RTI_A52, 3182898204L, "---    \t| 5200 \t| ---" },
//	{ RTI_A52, 2155009960L, "FoxOS  \t| 5200 \t| ---" },
	{ RTI_BAS, 3021189661L, "Rev.A  \t| ---  \t| ---" },
	{ RTI_BAS, 266326092L,  "Rev.B  \t| ---  \t| ---" },
	{ RTI_BAS, 2190982779L, "Rev.C  \t| ---  \t| ---" }
};

const int g_nRomTypeInfoNo = sizeof(g_aRomTypeInfo)/sizeof(g_aRomTypeInfo[0]);

/* Private objects */

static BOOL   s_bCheckZlib     = TRUE;
static HANDLE s_hZlib          = NULL;
static UINT   s_unCartCheck    = FCT_CHECK_16ROM;
static BOOL   s_bBreakSearch   = FALSE;
static int    s_nSSSRPrevState = 1;
static HHOOK  s_hKeyboardHook  = NULL;

static LARGE_INTEGER s_lnTimeFreq;


/*========================================================
Function : LowLevelKeyboardProc
=========================================================*/
/* #FN#
   The system calls this function every time a new keyboard input
   event is about to be posted into a thread input queue */
static
LRESULT CALLBACK
/* #AS#
   If nCode is less than zero, the hook procedure must return the
   value returned by CallNextHookEx */
LowLevelKeyboardProc(
	INT    nCode,
	WPARAM wParam,
	LPARAM lParam
)
{
	/* By returning a non-zero value from the hook procedure, the
	   message does not get passed to the target window */
	KBDLLHOOKSTRUCT *pkbhs = (KBDLLHOOKSTRUCT *)lParam;
	BOOL bControlKeyDown = 0;

	switch( nCode )
	{
		case HC_ACTION:
		{
			/* Check to see if the CTRL key is pressed */
			bControlKeyDown = GetAsyncKeyState( VK_CONTROL ) >> ((sizeof(SHORT) * 8) - 1);

			/* Disable CTRL+ESC */
			if( pkbhs->vkCode == VK_ESCAPE && bControlKeyDown )
				return 1;
			/* Disable ALT+TAB */
//			if( pkbhs->vkCode == VK_TAB && pkbhs->flags & LLKHF_ALTDOWN )
//				return 1;
			/* Disable ALT+ESC */
//			if( pkbhs->vkCode == VK_ESCAPE && pkbhs->flags & LLKHF_ALTDOWN )
//				return 1;
		}
	}
	return CallNextHookEx( s_hKeyboardHook, nCode, wParam, lParam );
} /* #OF# LowLevelKeyboardProc */

/*========================================================
Function : GetSystemVersion
=========================================================*/
/* #FN#
   Gets information about the system an emulator is run on */
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
GetSystemVersion(
	UINT *pSystemInfo
)
{
	OSVERSIONINFO osvi;
	*pSystemInfo = 0;

	ZeroMemory( &osvi, sizeof(OSVERSIONINFO) );
	osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);

	if( !GetVersionEx( (OSVERSIONINFO *)&osvi ) )
		return FALSE;

	switch( osvi.dwPlatformId )
	{
		case VER_PLATFORM_WIN32_NT:
		{
			/* Test for the product */
			if( osvi.dwMajorVersion == 4 )
				*pSystemInfo = SYS_WIN_NT4;

			if( osvi.dwMajorVersion == 5 )
				*pSystemInfo = SYS_WIN_NT5;

			break;
		}

		case VER_PLATFORM_WIN32_WINDOWS:
		{
			if( osvi.dwMajorVersion  > 4 || 
			   (osvi.dwMajorVersion == 4 && osvi.dwMinorVersion > 0) )
			{
				*pSystemInfo = SYS_WIN_98;
			}
			else
				*pSystemInfo = SYS_WIN_95;

			break;
		}
	}
	return TRUE; 
} /* #OF# GetSystemVersion */

/*========================================================
Function : ReadDisabledROMs
=========================================================*/
/* #FN#
   Reads ROM files from disk instead from state image */
int
/* #AS#
   TRUE if succeeded, otherwise FALSE */
ReadDisabledROMs( void )
{
	int	nRomFile;

	nRomFile = _open( atari_basic_filename, O_BINARY | O_RDONLY, 0777 );
	if( nRomFile == -1 )
	{
		Aprint( "Could not open %s for reading.", atari_basic_filename );
		return FALSE;
	}

	if( _read( nRomFile, atari_basic, 8192 ) < 8192 )
	{
		Aprint( "Could not read all of atari basic from %s.", atari_basic_filename );
		return FALSE;
	}
	_close( nRomFile );

	nRomFile = _open( atari_xlxe_filename, O_BINARY | O_RDONLY, 0777 );
	if( nRomFile == -1 )
	{
		Aprint( "Could not open %s for reading.", atari_xlxe_filename );
		return FALSE;
	}

	if( _read( nRomFile, atari_xlxe_filename, 16384 ) < 16384 )
	{
		Aprint( "Could not read entire atari ROM from %s.", atari_xlxe_filename );
		return FALSE;
	}
	_close( nRomFile );

	return TRUE;
} /* #OF# ReadDisabledROMs */

/*========================================================
Function : prepend_tmpfile_path
=========================================================*/
/* #FN#
   Inserts path to a temp dir to supplied buffer */
int
/* #AS#
   A length of the path */
prepend_tmpfile_path(
	char *pszBuffer
)
{
	int	nOffset = 0;

	nOffset += GetTempPath( MAX_PATH, pszBuffer ) - 1;
	if( nOffset == -1 || pszBuffer[ nOffset ] != '\\' )
		nOffset++;

	return nOffset;
} /* #OF# prepend_tmpfile_path */

/*========================================================
Function : zlib_capable
=========================================================*/
/* #FN#
   Checks if the ZLIB library is available */
int
/* #AS#
   1 if ZLIB is available, otherwise -1 */
zlib_capable( void )
{
	if( !s_hZlib )
	{
		Aprint( "Cannot gzip/ungzip without gzip.dll loaded properly." );
		return -1;
	}
	return 1;
} /* #OF# zlib_capable */

/*========================================================
Function : StartAtariTimer
=========================================================*/
/* #FN#
   Runs a timer an emulated Atari is based on */
void
/* #AS#
   Nothing */
StartAtariTimer( void )
{
	LARGE_INTEGER lnCurrentTime;

	QueryPerformanceCounter( &lnCurrentTime );
	g_ulAtariHWNextTime = lnCurrentTime.LowPart + g_ulDeltaT;
} /* #OF# StartAtariTimer */

/*========================================================
Function : ResetAtariTimer
=========================================================*/
/* #FN#
   Re-starts the timer an emulated Atari is based on */
void
/* #AS#
   Nothing */
ResetAtariTimer( void )
{
	memset( &s_lnTimeFreq, 0, sizeof(s_lnTimeFreq) );
	g_ulAtariHWNextTime = 0L;

	QueryPerformanceFrequency( &s_lnTimeFreq );
	g_ulDeltaT = s_lnTimeFreq.LowPart / (TV_PAL == tv_mode ? g_nPalFreq : g_nNtscFreq);
	g_nSleepThreshold  = s_lnTimeFreq.LowPart / 1000L;
	g_nSleepThreshold *= (SLEEP_TIME_IN_MS + 1);
} /* #OF# ResetAtariTimer */

/*========================================================
Function : Atari_Initialise
=========================================================*/
/* #FN#
   Prepares a Windows stuff for emulation; this function is invoked
   by Atari800 kernel */
void
/* #AS#
   Nothing */
Atari_Initialise(
	int  *argc,
	char *argv[]
)
{
	int i;

	memset( g_szSystemDesc, 0, 64 );

	if( s_bCheckZlib )
	{
		s_bCheckZlib = FALSE;
		if( !s_hZlib )
		{
			s_hZlib = LoadLibrary( "ZLIB.DLL" );
			if( !s_hZlib )
				DisplayMessage( NULL, IDS_ERROR_ZLIB_LOAD, 0, MB_ICONEXCLAMATION | MB_OK );
			else
			{
				pgzread  = GetProcAddress( s_hZlib, "gzread" );
				pgzopen  = GetProcAddress( s_hZlib, "gzopen" );
				pgzclose = GetProcAddress( s_hZlib, "gzclose" );
				pgzwrite = GetProcAddress( s_hZlib, "gzwrite" );
				pgzerror = GetProcAddress( s_hZlib, "gzerror" );
				if( !pgzread || !pgzopen || !pgzclose || !pgzwrite || !pgzerror)
				{
					FreeLibrary( s_hZlib );
					s_hZlib = NULL;
					DisplayMessage( NULL, IDS_ERROR_ZLIB_USE, 0, MB_ICONEXCLAMATION | MB_OK );
				}
			}
		}
	}
	/* Clean up the Atari timer stuff */
	ResetAtariTimer();

	if( !g_ulModesAvail )
		CheckDDrawModes();

	if( !g_lpbmi )
	{
		/* Only initialize this the first time it's allocated, since we'll come through here
		   on changing hardware types and it will already be set up correctly at that point */
		g_lpbmi = (LPBITMAPINFO)calloc( 1, (sizeof( BITMAPINFOHEADER ) + sizeof( RGBQUAD ) * 256) );

		if( !g_lpbmi )
			Atari_Exit( 1 );

		g_lpbmi->bmiHeader.biSize        = sizeof(BITMAPINFOHEADER);
		g_lpbmi->bmiHeader.biWidth       =  ATARI_VIS_WIDTH;
		g_lpbmi->bmiHeader.biHeight      = -ATARI_HEIGHT;	/* Negative because we are a top-down bitmap */
		g_lpbmi->bmiHeader.biPlanes      = 1;
		g_lpbmi->bmiHeader.biBitCount    = 8;				/* Each byte stands for a color value */
		g_lpbmi->bmiHeader.biCompression = BI_RGB;			/* Uncompressed format */

		UseAtariPalette();
	}
	ResetInput();

	for( i = 0; i < 4; i++ ) /* For autofire */
	{
		if( g_ulAutofireSticks & (1 << i) )
			TRIG_auto[ i ] = g_nAutofireMode;
		else
			TRIG_auto[ i ] = AUTOFIRE_OFF;
	}
	nframes = 0;

	if( MOUSE_PEN == g_nMouseMode )
		light_pen_enabled = TRUE;	/* Enable emulation in antic.c */
} /* #OF# Atari_Initialise */

/*========================================================
Function : UseAtariPalette
=========================================================*/
/* #FN#
   Forces using Atari800 palette in Windows environment */
void
/* #AS#
   Nothing */
UseAtariPalette( void )
{
	int i;

	_ASSERT(g_lpbmi);

	for( i = 0; i < 256; i++ )
	{
		int	rgb = colortable[ i ];
		
		g_lpbmi->bmiColors[ i ].rgbRed      = g_pe[ i ].peRed   = (rgb & 0x00ff0000) >> 16;
		g_lpbmi->bmiColors[ i ].rgbGreen    = g_pe[ i ].peGreen = (rgb & 0x0000ff00) >> 8;
		g_lpbmi->bmiColors[ i ].rgbBlue     = g_pe[ i ].peBlue  = rgb & 0x000000ff;
		g_lpbmi->bmiColors[ i ].rgbReserved = g_pe[ i ].peFlags = 0;
	}
} /* #OF# UseAtariPalette */

/*========================================================
Function : TogglePause
=========================================================*/
/* #FN#
   Toggles between the pause on and off */
void
/* #AS#
   Nothing */
TogglePause( void )
{
	if( g_ulAtariState & ATARI_RUNNING )
	{
		if( g_ulAtariState & ATARI_PAUSED )
		{
			g_ulAtariState &= ~ATARI_PAUSED;
			RestartSound();

			if( g_ulScreenMode & SM_MODE_FULL && g_ulScreenMode & SM_OPTN_FLIP_BUFFERS )
				ClearScreen( FALSE );
 		}
		else
		{
			g_ulAtariState |= ATARI_PAUSED;
			ClearSound( FALSE );

			if( g_hMainWnd )
				SetTimer( g_hMainWnd, TIMER_READ_JOYSTICK, 100, NULL ); 

			ShowMousePointer( TRUE );
			/* Get surface for GDI if necessary */
			FlipToGDI();
			DrawPausedScreen( (UBYTE *)atari_screen );
		}
	}
} /* #OF# TogglePause */

/*========================================================
Function : ToggleFullSpeed
=========================================================*/
/* #FN#
   Toggles between the full speed mode on and off */
void
/* #AS#
   Nothing */
ToggleFullSpeed( void )
{
	if( g_ulAtariState & ATARI_RUNNING )
	{
		if( g_ulMiscStates & MS_FULL_SPEED )
		{
			g_ulMiscStates &= ~MS_FULL_SPEED;
			RestartSound();
		}
		else
		{
			g_ulMiscStates |= MS_FULL_SPEED;
			ClearSound( FALSE );
		}
		WriteRegDWORD( NULL, REG_MISC_STATES, g_ulMiscStates );
	}
} /* #OF# ToggleFullSpeed */

/*========================================================
Function : ToggleSIOPatch
=========================================================*/
/* #FN#
   Toggles between the SIO patch mode on and off */
void
/* #AS#
   Nothing */
ToggleSIOPatch( void )
{
	if( enable_rom_patches )
	{
		if( enable_sio_patch )
		{
			enable_sio_patch = 0;
			RestoreSIO();
		}
		else
		{
			enable_sio_patch = 1;
			SetSIOEsc();
		}
		WriteRegDWORD( NULL, REG_ENABLE_SIO_PATCH, enable_sio_patch );
	}
} /* #OF# ToggleSIOPatch */

/*========================================================
Function : ToggleEscCapture
=========================================================*/
/* #FN#
   Toggles between Ctrl+Esc key sequence capture on and off */
void
/* #AS#
   Nothing */
ToggleEscCapture( void )
{
	if( g_ulMiscStates & MS_CAPTURE_CTRLESC )
	{
		EnableEscCapture( FALSE );
		g_ulMiscStates &= ~MS_CAPTURE_CTRLESC;
	}
	else
	{
		EnableEscCapture( TRUE );
		g_ulMiscStates |= MS_CAPTURE_CTRLESC;
	}
	WriteRegDWORD( NULL, REG_MISC_STATES, g_ulMiscStates );
} /* #OF# ToggleEscCapture */

/*========================================================
Function : EnableEscCapture
=========================================================*/
/* #FN#
   Enables or disables Ctrl+Esc key sequence */
void
/* #AS#
   Nothing */
EnableEscCapture(
	BOOL bEnable /* #IN# Enable/disable capture flag */
)
{
	/*
	  Windows 95 and Windows 98: applications can enable and
	  disable ALT+TAB and CTRL+ESC, for example, by calling
	  SystemParametersInfo(SPI_SETSCREENSAVERRUNNING). To disable
	  ALT+TAB and CTRL+ESC, set the uiParam parameter to TRUE;
	  to enable the key combinations, set the parameter to FALSE: 

	  NOTE: Applications that use SystemParametersInfo
	  (SPI_SETSCREENSAVERRUNNING) to disable task switching must
	  enable task switching before exiting or task switching
	  remains disabled after the process terminates.

	  Windows NT 4.0 Service Pack 2 and Earlier, Windows NT 3.51
	  and Earlier Applications can disable CTRL+ESC system-wide
	  by replacing the Windows NT Task Manager, but this is not
	  recommended.

	  Windows NT 4.0 Service Pack 3 and Later and Windows 2000
	  Applications can disable ALT+TAB or CTRL+ESC by installing
	  a low-level keyboard hook. A low-level keyboard hook
	  (WH_KEYBOARD_LL) is installed by calling SetWindowsHookEx.

	  Because low-level keyboard hooks are a feature specific to
	  Windows NT 4.0 Service Pack 3 and later, define _WIN32_WINNT
	  >= 0x0400 prior to including winuser.h (or windows.h).
	*/
	_ASSERT(g_unSystemInfo);

	if( bEnable )
	{
		/* Disables task switching */
		if( g_unSystemInfo & SYS_WIN_NT )
		{
			if( s_hKeyboardHook )
				EnableEscCapture( FALSE );

			s_hKeyboardHook =
				SetWindowsHookEx( WH_KEYBOARD_LL, LowLevelKeyboardProc,
								  g_hInstance, GetCurrentThreadId() );
 		}
		else
		{
			if( s_nSSSRPrevState )
				EnableEscCapture( FALSE );

			SystemParametersInfo( SPI_SETSCREENSAVERRUNNING, TRUE,
								  &s_nSSSRPrevState, 0 );
		}
	}
	else
	{
		/* Enables task switching */
		if( g_unSystemInfo & SYS_WIN_NT )
		{
			if( s_hKeyboardHook )
				UnhookWindowsHookEx( s_hKeyboardHook );

			s_hKeyboardHook = NULL;
		}
		else if( !s_nSSSRPrevState )
			SystemParametersInfo( SPI_SETSCREENSAVERRUNNING, FALSE,
								  &s_nSSSRPrevState, 0 );
	}
} /* #OF# EnableEscCapture */

/*========================================================
Function : RefreshViewBounds
=========================================================*/
/* #FN#
   Cashes view area bondaries */
void
/* #AS#
   Nothing */
RefreshViewBounds( BOOL bClipCursor )
{
	GetWindowRect( g_hViewWnd, &g_rcViewBounds );
	/* Deflate the rect boundaries */
	g_rcViewBounds.left += 2; g_rcViewBounds.right -= 2;
	g_rcViewBounds.top += 2; g_rcViewBounds.bottom -= 2;

	if( g_ulMiscStates & MS_MOUSE_CAPTURED && bClipCursor )
	{
		/* Re-confine the cursor to a rectangular area on the view */
		ClipCursor( NULL );
		ClipCursor( &g_rcViewBounds );
		/* Initialise the mouse emulation settings */
		ResetMouse( TRUE );//MOUSE_JOY == g_nMouseMode && Atari5200 == machine );
	}
} /* #OF# RefreshViewBounds */

/*========================================================
Function : ShowMousePointer
=========================================================*/
/* #FN#
   Makes the mouse pointer visible or hidden */
void
/* #AS#
   Nothing */
ShowMousePointer(
	BOOL bShow /* #IN# Show mouse pointer flag */
)
{
	if( g_ulScreenMode & SM_MODE_FULL && g_ulScreenMode & SM_OPTN_HIDE_CURSOR ||
		g_ulMiscStates & MS_MOUSE_CAPTURED )
	{
		if( bShow )
		{
			if( g_nShowCursor <= 0 )
				ShowCursor( TRUE );

			if( g_ulMiscStates & MS_MOUSE_CAPTURED )
			{
				ClipCursor( NULL );
				g_nShowCursor = 10;
			}
			else
				g_nShowCursor = DEF_SHOW_CURSOR;
		}
		else
		{
			ShowCursor( FALSE ); /* Hide the mouse cursor */
			/* Re-capture the cursor */
			RefreshViewBounds( TRUE );
		}
	}
} /* #OF# ShowMousePointer */

/*========================================================
Function : MonitorThreadProc
=========================================================*/
/* #FN#
   A separate thread procedure for a monitor console */
static
DWORD WINAPI
/* #AS#
   Thread return code (success or failure) */
MonitorThreadProc(
	LPVOID lpParameter
)
{
	ExitThread( monitor() );
	return 0;
} /* #OF# MonitorThreadProc */

/*========================================================
Function : LaunchMonitor
=========================================================*/
/* #FN#
   Launches the emulator monitor console */
int
/* #AS#
   An integer value: 1 to go back to emulation, 0 to exit */
LaunchMonitor( void )
{
	DWORD  dwThreadId   = 0; 
	DWORD  dwExitCode   = 0;
	HANDLE hThread      = NULL;
	ULONG  ulScreenMode = 0L;

	/* CAtari800WinView::OnKillFocus will invoke that */
//	ClearSound( FALSE );

	/* Unfortunately, there are some problems with automatically return to
	   flipped full-screen mode, go to windowed and then restore instead */
	if( g_ulScreenMode & SM_MODE_FULL && g_ulScreenMode & SM_OPTN_FLIP_BUFFERS )
	{
		ulScreenMode = g_ulScreenMode;

		g_ulScreenMode &= ~SM_MODE_MASK;
		/* Set a windowed mode */
		g_ulScreenMode |= SM_MODE_WIND;
		if( !InitialiseScreen( TRUE ) )
		{
			SetSafeDisplay( TRUE );
			ulScreenMode = 0L;
		}
	}
	/* Disable the main window */
	g_ulAtariState |= ATARI_MONITOR;
	EnableWindow( g_hMainWnd, FALSE );

	/* Launch an emulated Atari monitor */
	if( (hThread = CreateThread( NULL,
								 0,
								 MonitorThreadProc,
								 (LPVOID)NULL,
								 0,
								 &dwThreadId )) )
	{
		/* This message loop will solve some problems with
           the main window repainting */
		do
		{
			MSG msg;
			if( PeekMessage( &msg, 0, 0, 0, PM_REMOVE ) )
			{
				TranslateMessage( &msg );
				DispatchMessage( &msg );
			}
			Sleep( 1 ); /* Wait a moment, please */
			GetExitCodeThread( hThread, &dwExitCode );
		}
		while( STILL_ACTIVE == dwExitCode );
	}
	else
		/* We had few luck to create an additional thread */
		dwExitCode = monitor();

	/* Enable the main window */
	g_ulAtariState &= ~ATARI_MONITOR;
	EnableWindow( g_hMainWnd, TRUE );

	if( dwExitCode )
	{
		HWND hPopupWnd = GetLastActivePopup( g_hMainWnd );

		BringWindowToTop( g_hMainWnd );
		if( IsIconic( hPopupWnd ) )
		{
			ShowWindow( hPopupWnd, SW_RESTORE );
		}
		else
			SetForegroundWindow( hPopupWnd );

		if( ulScreenMode )
		{
			g_ulScreenMode = ulScreenMode;
			if( !InitialiseScreen( TRUE ) )
				SetSafeDisplay( TRUE );
		}
		return 1;	/* Go back to the emulation */
	}
	else
		/* Start a quit (this will end up in Atari_Exit(0)) */
		PostMessage( g_hMainWnd, WM_CLOSE, 0, 0L );

	return 0;
} /* #OF# LaunchMonitor */

/*========================================================
Function : Atari_Exit
=========================================================*/
/* #FN#
   This function is called by Atari800 kernel when emulation is
   about exit */
int
/* #AS#
   Nonzero to go back to emulation, 0 to exit */
Atari_Exit(
	int nPanic
)
{
	if( nPanic && g_ulAtariState & ATARI_RUNNING )
		g_ulAtariState |= ATARI_CRASHED;
	
	if( ATARI_CRASHED == g_ulAtariState )
	{
		wsync_halt = 1;	/* turn off CPU */
		return 0;
	}
	/* Get surface for GDI if necessary */
	FlipToGDI();

	if( nPanic && g_ulAtariState & ATARI_RUNNING )
	{
		int nResult;

		if( ST_MENUBAR_HIDDEN )
			ShowMenuBar( TRUE );

		nResult = DisplayMessage( NULL, IDS_WARN_ATARI_PANIC, 0, MB_ICONSTOP | MB_YESNO );

		InvalidateRect( g_hMainWnd, NULL, FALSE );

		if( IDYES == nResult &&
			LaunchMonitor() /* Launch the monitor */ )
		{
			g_ulAtariState &= ~ATARI_CRASHED; /* For the CIM service */
			return 1;
		}
		g_ulAtariState = ATARI_CRASHED;
		wsync_halt = 1;	/* turn off CPU */
		g_nTestVal = 32767;
	}
	else
	{
		/* Reset everything DirectDraw */
		ClearScreen( TRUE );

		/* Reset everything MultiMedia/DirectSound */
		ClearSound( TRUE );

		/* Reset everything DirectInput */
		ClearInput();

		g_ulAtariState = ATARI_UNINITIALIZED | ATARI_CLOSING;

		Remove_ROM();

		if( g_lpbmi )
			free( g_lpbmi );
		g_lpbmi = NULL;

		Clear_Temp_Files();

		if( s_hZlib )
		{
			FreeLibrary( s_hZlib );
   			pgzopen = pgzclose = pgzread = pgzwrite = pgzerror = NULL;
		}
	}
	return 0;
} /* #OF# Atari_Exit */

/*========================================================
Function : FindAppWindow
=========================================================*/
/* #FN#
   Tries to find an emulator window */
HWND
/* #AS#
   Handle of the window if it has been found, otherwise NULL */
FindAppWindow( void )
{
	LPCSTR szPostfix[] = { TITLE_STOPPED, TITLE_NOTRUN, TITLE_PAUSED, TITLE_FAILURE, TITLE_MONITOR };
	char   szTitle[ 64 ];
	HWND   hAppWnd = NULL;
	int    i;

	for( i = 0; i < sizeof(szPostfix)/sizeof(szPostfix[0]); i++ )
	{
		strcpy( szTitle, "Atari800Win: " );
		strcat( szTitle, szPostfix[ i ] );

		if( hAppWnd = FindWindow( NULL, szTitle ) )
			return hAppWnd;
	}
	return NULL;
} /* #OF# FindAppWindow */

/*========================================================
Function : GetHomeDirectory
=========================================================*/
/* #FN#
   Retrieves the path for the Atari800Win home directory */
BOOL
/* #AS#
   TRUE if the path was retrieved, otherwise FALSE */
GetHomeDirectory(
	LPSTR pszHomeDir
)
{
	BOOL bResult = FALSE;

	if( GetModuleFileName( NULL, pszHomeDir, MAX_PATH ) )
	{
		size_t i;
		if( i = strlen( pszHomeDir ) )
		{
			while( i && pszHomeDir[ i ] != '\\' )
				i--;
			if( i )
				pszHomeDir[ i ] = '\0';
			bResult = TRUE;
		}
	}
	return bResult;
} /* #OF# GetHomeDirectory */

/*========================================================
Function : TestAndSetPath
=========================================================*/
/* #FN#
   Searches the Atari system ROMs in home folder and its subfolders */
static
BOOL
/* #AS#
   TRUE if the ROM files has been found, otherwise FALSE */
TestAndSetPath(
	LPSTR pszFileName,
	enum  RomType rtType,
	LPSTR pszPath
)
{
	char   szStartPath[ MAX_PATH ];
	char   szImagePath[ MAX_PATH ];
	char   szPattern  [ MAX_PATH ];
	ULONG  ulCRC;
	int    nResult;

	WIN32_FIND_DATA fileData;
	HANDLE hFile;

	if( NULL == pszPath )
	{
		/* We begin searching at an Atari800Win home directory */
		GetHomeDirectory( szStartPath );
	}
	else
		strcpy( szStartPath, pszPath );

	if( szStartPath[ strlen( szStartPath ) - 1 ] != '\\' )
		strcat( szStartPath, "\\" );

	strcpy( szPattern, szStartPath );
	strcat( szPattern, "*.rom" );

	hFile = FindFirstFile( szPattern, &fileData );
	/* If any file has been found... */
	while( INVALID_HANDLE_VALUE != hFile && !s_bBreakSearch )
	{
		if( !(fileData.dwFileAttributes & (FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_SYSTEM | FILE_ATTRIBUTE_DIRECTORY)) )
		{
			strcpy( szImagePath, szStartPath );
			strcat( szImagePath, fileData.cFileName );

			ulCRC = CheckFile( szImagePath, &nResult );
			if( ulCRC != 0 )
			{
				int i;
				for( i = 0; i < g_nRomTypeInfoNo; i++ )
				{
					if( g_aRomTypeInfo[ i ].rtType == rtType &&
						g_aRomTypeInfo[ i ].ulCRC  == ulCRC )
					{
						/* Again, strncpy gives the strange effects here */
						_ASSERT(strlen( szImagePath ) < MAX_PATH);
						strcpy( pszFileName, szImagePath );
						/* The searched file has been found */
						return TRUE;
					}
				}
			}
		}
		/* Try to find a next file */
		if( !FindNextFile( hFile, &fileData ) )//&& (GetLastError() == ERROR_NO_MORE_FILES) )
		{
			FindClose( hFile );
			hFile = INVALID_HANDLE_VALUE; /* Ending the loop */
		}
	}
	strcpy( szPattern, szStartPath );
	strcat( szPattern, "*.*" );

	hFile = FindFirstFile( szPattern, &fileData );
	/* If any folder has been found... */
	while( INVALID_HANDLE_VALUE != hFile && !s_bBreakSearch )
	{
		if( !(fileData.dwFileAttributes & (FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_SYSTEM)) &&
			 (fileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) && (*fileData.cFileName != '.') )
		{
			strcpy( szImagePath, szStartPath );
			strcat( szImagePath, fileData.cFileName );

			/* Search the ROM file in this subfolder */
			if( TestAndSetPath( pszFileName, rtType, szImagePath ) )
				return TRUE;
		}
		/* Try to find a next folder */
		if( !FindNextFile( hFile, &fileData ) )//&& (GetLastError() == ERROR_NO_MORE_FILES) )
		{
			FindClose( hFile );
			hFile = INVALID_HANDLE_VALUE; /* Ending the loop */
		}
	}
	return FALSE;
} /* #OF# TestAndSetPath */

/*========================================================
Function : RomSearchingDlgProc
=========================================================*/
/* #FN#
   The "ROM Searching" dialog box procedure */
static
int CALLBACK
/* #AS#
   TRUE if the message has been handled, otherwise FALSE */
RomSearchingDlgProc(
	HWND hDialog,
	UINT uiMsg,
	UINT wParam,
	LONG lParam
)
{
	LPCSTR pszProgress[] = { "|", "/", "-", "\\" };
	static int i = -1;

	switch( uiMsg )
	{
		case WM_USER:
			if( ++i > 3 ) i = 0;
			SetWindowText( hDialog, pszProgress[ i ] );
			return TRUE;

		case WM_COMMAND:
			s_bBreakSearch = TRUE;
			DestroyWindow( hDialog );
			return TRUE;
	}
	return FALSE;
} /* #OF# RomSearchingDlgProc */

/*========================================================
Function : SearchingThreadProc
=========================================================*/
/* #FN#
   A separate thread procedure for ROM searching */
DWORD WINAPI
/* #AS#
   Thread return code (success or failure) */
SearchingThreadProc(
	LPVOID lpParameter
)
{
	LPSTR pszStartPath = (LPSTR)lpParameter;
	BOOL  bFoundRom = FALSE;

	if(	!s_bBreakSearch &&
		TestAndSetPath( atari_osa_filename,   RTI_OSA, pszStartPath ) )
		bFoundRom = TRUE;
	if(	!s_bBreakSearch &&
		TestAndSetPath( atari_osb_filename,   RTI_OSB, pszStartPath ) )
		bFoundRom = TRUE;
	if(	!s_bBreakSearch &&
		TestAndSetPath( atari_xlxe_filename,  RTI_XLE, pszStartPath ) )
		bFoundRom = TRUE;
	if(	!s_bBreakSearch &&
		TestAndSetPath( atari_5200_filename,  RTI_A52, pszStartPath ) )
		bFoundRom = TRUE;
	if(	!s_bBreakSearch &&
		TestAndSetPath( atari_basic_filename, RTI_BAS, pszStartPath ) )
		bFoundRom = TRUE;

	ExitThread( bFoundRom );
	return 0;
} /* #OF# SearchingThreadProc */

/*========================================================
Function : TestRomPaths
=========================================================*/
/* #FN#
   Routine tries to locate the system ROMs in either the home directory
   or a subdirectory "ROMs" under the home directory */
BOOL
/* #AS#
   TRUE if at least one ROM file was found, otherwise FALSE */
TestRomPaths(
	LPSTR pszStartPath,
	HWND  hWnd
)
{
	HWND   hDialog = NULL;
	HANDLE hThread = NULL;
	DWORD  dwThreadId, dwExitCode = 0;

	s_bBreakSearch = FALSE;

	/* Install the Abort Dialog handler and display the window */
	if( hWnd )
	{
		EnableWindow( hWnd, FALSE );
		hDialog = CreateDialog( g_hInstance, MAKEINTRESOURCE(IDD_ROMSEARCHING),
								hWnd, (DLGPROC)RomSearchingDlgProc );
	}
	/* The searching procedure may take a long time until it
	   ends its activity. */
	if( (hThread = CreateThread( NULL,
								 0,
								 SearchingThreadProc,
								 (LPVOID)pszStartPath,
								 0,
								 &dwThreadId )) )
	{
		int nCount = 0;
		/* This message loop will solve some problems with Abort
		   Dialog activity and the main window repainting */
		do
		{
			MSG msg;
			if( PeekMessage( &msg, 0, 0, 0, PM_REMOVE ) )
				if( !hDialog || !IsDialogMessage( hDialog, &msg ) )
				{
					TranslateMessage( &msg );
					DispatchMessage( &msg );
				}
			if( nCount++ == 20 && hDialog )
			{
				SendMessage( hDialog, WM_USER, 0, 0 );
				nCount = 0;
			}
			Sleep( 10 ); /* Wait a moment, please */
			GetExitCodeThread( hThread, &dwExitCode );
		}
		while( STILL_ACTIVE == dwExitCode );
	}
	if(	hWnd )
	{
		EnableWindow( hWnd, TRUE );
		/* There is the Search window enabled, so we can destroy the
		   Abort box now */
		if( !s_bBreakSearch && hDialog )
			DestroyWindow( hDialog );

//		SetFocus( hWnd );
	}
	return (BOOL)dwExitCode;
} /* #OF# TestRomPaths */

/*========================================================
Function : GetRomInfo
=========================================================*/
/* #FN#
   Returns a description of the ROM image */
LPCSTR
/* #AS#
   Description of the ROM image if was found, otherwise NULL */
GetRomInfo(
	LPSTR pszFileName,
	enum RomType rtType
)
{
	ULONG ulCRC;
	int   nResult;

	if( NULL != pszFileName )
	{
		ulCRC = CheckFile( pszFileName, &nResult );

		if( ulCRC != 0 )
		{
			int i;
			for( i = 0; i < g_nRomTypeInfoNo; i++ )
			{
				if( g_aRomTypeInfo[ i ].rtType == rtType &&
					g_aRomTypeInfo[ i ].ulCRC  == ulCRC )
				{
					return g_aRomTypeInfo[ i ].pszDesc;
				}
			}
		}
	}
	return NULL;
} /* #OF# GetRomInfo */

/*========================================================
Function : RomTypeDlgProc
=========================================================*/
/* #FN#
   The "ROM Type" dialog box procedure. Since the one doesn't use MFC,
   it may be invoked from the pure "C" functions */
static
int CALLBACK
/* #AS#
   TRUE if the message has been handled, otherwise FALSE */
RomTypeDlgProc(
	HWND hDialog,
	UINT uiMsg,
	UINT wParam,
	LONG lParam
)
{
	UINT aunRadioIDs [ ROM_TYPES_NO ] = { IDC_ROMTYPE_16STD, IDC_ROMTYPE_16AGS, IDC_ROMTYPE_16OSS, IDC_ROMTYPE_32AGS, IDC_ROMTYPE_32DB };
	UINT aunCartTypes[ ROM_TYPES_NO ] = { NORMAL16_CART, AGS32_CART, OSS_SUPERCART, AGS32_CART, DB_SUPERCART };
	char szCartDesc  [ LOADSTRING_STRING_SIZE ];
	int  i;

	switch( uiMsg )
	{
		case WM_INITDIALOG:
			for( i = 0; i < ROM_TYPES_NO; i++ )
				EnableWindow( GetDlgItem( hDialog, aunRadioIDs[ i ] ), (BOOL)(s_unCartCheck & (1 << i)) );

			/* Check a appropriate radio button */
			CheckRadioButton( hDialog, IDC_ROMTYPE_16STD, IDC_ROMTYPE_32DB,
							  (s_unCartCheck & FCT_CHECK_16ROM ?
							  (default_system == 6 ? IDC_ROMTYPE_16AGS : IDC_ROMTYPE_16STD) :
							  (default_system == 6 ? IDC_ROMTYPE_32AGS : IDC_ROMTYPE_32DB)) );
			/* Set description of the cartridge type */
			LoadString( NULL, (s_unCartCheck & FCT_CHECK_16ROM ?
							  (default_system == 6 ? IDS_ROM_16AGS_DESC : IDS_ROM_16STD_DESC) :
							  (default_system == 6 ? IDS_ROM_32AGS_DESC : IDS_ROM_32DB_DESC)),
							  szCartDesc, LOADSTRING_STRING_SIZE );
			SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
			return TRUE;

		case WM_CLOSE:
			wParam = IDCANCEL;

		case WM_COMMAND:
			switch( wParam )
			{
				case IDC_ROMTYPE_16STD:
					LoadString( NULL, IDS_ROM_16STD_DESC, szCartDesc, LOADSTRING_STRING_SIZE );
					SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
					return TRUE;

				case IDC_ROMTYPE_16AGS:
					LoadString( NULL, IDS_ROM_16AGS_DESC, szCartDesc, LOADSTRING_STRING_SIZE );
					SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
					return TRUE;

				case IDC_ROMTYPE_16OSS:
					LoadString( NULL, IDS_ROM_16OSS_DESC, szCartDesc, LOADSTRING_STRING_SIZE );
					SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
					return TRUE;

				case IDC_ROMTYPE_32AGS:
					LoadString( NULL, IDS_ROM_32AGS_DESC, szCartDesc, LOADSTRING_STRING_SIZE );
					SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
					return TRUE;

				case IDC_ROMTYPE_32DB:
					LoadString( NULL, IDS_ROM_32DB_DESC, szCartDesc, LOADSTRING_STRING_SIZE );
					SetWindowText( GetDlgItem( hDialog, IDC_ROMTYPE_DESCRIPTION ), szCartDesc );
					return TRUE;

				case IDOK:
					for( i = 0; i < ROM_TYPES_NO; i++ )
						if( BST_CHECKED == IsDlgButtonChecked( hDialog, aunRadioIDs[ i ] ) )
							break;
					EndDialog( hDialog, aunCartTypes[ i ] );
					return TRUE;

				case IDCANCEL:
					EndDialog( hDialog, NO_CART );
					return TRUE;
			}
			break;
	}
	return FALSE;
} /* #OF# RomTypeDlgProc */

/*========================================================
Function : FindCartType
=========================================================*/
/* #FN#
   Checks the pointed cartridge type */
int
/* #AS#
   Cartridge type */
FindCartType(
	LPCSTR pszFileName, /* #IN# Name of the file to test          */
	BOOL   bAutoDetect, /* #IN# Don't ask user for cart type      */
	BOOL   bCheckExt    /* #IN# Check if the file is an Atari one */
)
{
	UINT unFileType = IAF_CRT_IMAGE | IAF_ROM_IMAGE;
	int  nCartType  = NO_CART;

	if( !IsAtariFile( pszFileName, &unFileType ) )
	{
		if( !bCheckExt )
		{
			/* It is not a good idea to ignore the IsAtariFile()
               function result but it's the only way to load
			   a ROM file without an appropriate extension */
			unFileType = IAF_ROM_IMAGE;
		}
		else
			return NO_CART;
	}
	if( IAF_ROM_IMAGE == unFileType )
	{
		int	nFileLen = 0;
		int fd = _open( pszFileName, _O_RDONLY | _O_BINARY, 0 );
		if( -1 == fd )
			return NO_CART;

		nFileLen = _filelength( fd );
		_close( fd );

		if( nFileLen < 8193 )
		{
			nCartType = NORMAL8_CART;
		}
		else
		if( nFileLen < 16385 )
		{
			if( bAutoDetect )
				nCartType = (default_system == 6 ? AGS32_CART : NORMAL16_CART);
			else
			{
				/* Ask the user for 16K cart type */
				s_unCartCheck = FCT_CHECK_16ROM;
				nCartType = DialogBox( g_hInstance, MAKEINTRESOURCE(IDD_ROMTYPE),
									   g_hMainWnd, (DLGPROC)RomTypeDlgProc );
			}
		}
		else
		if( nFileLen < 32769 )
		{
			if( bAutoDetect )
				nCartType = (default_system == 6 ? AGS32_CART : DB_SUPERCART);
			else
			{
				/* Ask the user for 32K cart type */
				s_unCartCheck = FCT_CHECK_32ROM;
				nCartType = DialogBox( g_hInstance, MAKEINTRESOURCE(IDD_ROMTYPE),
									   g_hMainWnd, (DLGPROC)RomTypeDlgProc );
			}
		}
		else
		if( nFileLen < 40961 )
			/* BountyBob 5200 cart has got the size greater than 32K */
			nCartType = AGS32_CART;
	}
	else
	if( IAF_CRT_IMAGE == unFileType )
		nCartType = CARTRIDGE;

	return nCartType;
} /* #OF# FindCartType */

/*========================================================
Function : IsAtariFile
=========================================================*/
/* #FN#
   Checks if the pointed image is known Atari file */
BOOL
/* #AS#
   TRUE if the image might be an Atari file, otherwise FALSE */
IsAtariFile(
	LPCSTR  pszFileName, /* #IN# Name of the file to test */
	UINT   *pFileType    /* #IN/OUT# The file type */
)
{
	char szTempFile[ MAX_PATH ];
	int  fd;

	strcpy( szTempFile, pszFileName );
	strupr(	szTempFile );

	if( IAF_DSK_IMAGE & *pFileType )
	{
		if( strstr( szTempFile, ".GZ"  ) ||
			strstr( szTempFile, ".ATZ" ) ||
			strstr( szTempFile, ".XFZ" ) ||
			strstr( szTempFile, ".DCM" ) ||
			strstr( szTempFile, ".ATR" ) ||
			strstr( szTempFile, ".XFD" ) )
		{
			*pFileType = IAF_DSK_IMAGE;
			return TRUE;
		}
	}
	if( IAF_ROM_IMAGE & *pFileType )
	{
		if( strstr( szTempFile, ".ROM" ) ||
			strstr( szTempFile, ".BIN" ) )
		{
			*pFileType = IAF_ROM_IMAGE;
			return TRUE;
		}
	}
	if( IAF_CRT_IMAGE & *pFileType &&
		(fd = _open( pszFileName, _O_RDONLY | _O_BINARY, 0 )) != -1 )
	{
		char cBuffer[ 4 ];
		int  nBytesRead = 0;

		_lseek( fd, 0L, SEEK_SET );
		nBytesRead = _read( fd, cBuffer, 4 );
		_close( fd );

		if( 4 == nBytesRead &&
			'C' == cBuffer[ 0 ] &&
			'A' == cBuffer[ 1 ] &&
			'R' == cBuffer[ 2 ] &&
			'T' == cBuffer[ 3 ] )
		{
			*pFileType = IAF_CRT_IMAGE;
			return TRUE;
		}
	}
	if( IAF_A8S_IMAGE & *pFileType )
	{
		if( strstr( szTempFile, ".A8S" ) )
		{
			*pFileType = IAF_A8S_IMAGE;
			return TRUE;
		}
	}
	/* That's the last check because some carts have $FFFF header */
	if( IAF_BIN_IMAGE & *pFileType &&
		(fd = _open( pszFileName, _O_RDONLY | _O_BINARY, 0 )) != -1 )
	{
		char cBuffer[ 2 ];
		int  nBytesRead = 0;

		_lseek( fd, 0L, SEEK_SET );
		nBytesRead = _read( fd, cBuffer, 2 );
		_close( fd );

		if( 2 == nBytesRead &&
			0xff == (BYTE)cBuffer[ 0 ] &&
			0xff == (BYTE)cBuffer[ 1 ] )
		{
			*pFileType = IAF_BIN_IMAGE;
			return TRUE;
		}
	}
	*pFileType = 0;

	return FALSE;
} /* #OF# IsAtariFile */

/*========================================================
Function : IsCompressedFile
=========================================================*/
/* #FN#
   Simple routine to check if the given filename is any one of the various
   compressed types. Currently the only test for this is by extension, also
   the only way you can actually read a compressed file in, so that seems
   valid enough. */
BOOL
/* #AS#
   TRUE if the file is compressed, otherwise FALSE */
IsCompressedFile(
	LPCSTR pszFileName /* #IN# Name of the file to test */
)
{
	char szTempFile[ MAX_PATH ];

	strcpy( szTempFile, pszFileName );
	strupr(	szTempFile );

	if( strstr( szTempFile, ".GZ"  ) ||
		strstr( szTempFile, ".ATZ" ) ||
		strstr( szTempFile, ".XFZ" ) ||
		strstr( szTempFile, ".DCM" ) )
		return TRUE;

	return FALSE;
} /* #OF# IsCompressedFile */

/*========================================================
Function : RunAtariExe
=========================================================*/
/* #FN#
   Loads and executes the pointed Atari 8-bit executable file */
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
RunAtariExe(
	LPSTR pszFileName /* #IN# Name of the executable file to run */
)
{
	char szNewDir[ MAX_PATH ];
	BOOL bResult;

	strncpy( szNewDir, pszFileName, MAX_PATH );
	GetFolderPath( szNewDir, NULL );
	if( _stricmp( szNewDir, atari_exe_dir ) != 0 )
	{
		strcpy( atari_exe_dir, szNewDir );
		WriteRegString( NULL, REG_EXE_PATH, atari_exe_dir );
	}
	bResult = BIN_loader( pszFileName );

	return bResult;
} /* #OF# RunAtariExe */

/*========================================================
Function : GetFolderPath
=========================================================*/
/* #FN#
   Extracts the folder from full file path */
void
/* #AS#
   Nothing */
GetFolderPath(
	LPSTR pszPathName, /* #IN/OUT# Full file path/file folder name */
	LPSTR pszFileName  /* #OUT#    File path */
)
{
	int nPathLen = strlen( pszPathName ) - 1;
	int i;

	for( i = nPathLen; i > 0 && pszPathName[ i ] != '\\'; i-- );
	if( i > 0 || pszPathName[ i ] == '\\' )
	{
		pszPathName[ i++ ] = '\0';
	}
	if( pszFileName != NULL )
		strncpy( pszFileName, &pszPathName[ i ], MAX_PATH );

	/* Add ending backslash to drive name */
	if( strlen( pszPathName ) == 2 && pszPathName[ 1 ] == ':' )
		strcat( pszPathName, "\\" );
	
} /* #OF# GetFolderPath */

/*========================================================
Function : ConsoleHandlerRoutine
=========================================================*/
/* #FN#
   A console process uses this function to handle control signals received
   by the process */
static
BOOL WINAPI
/* #AS#
   If the function handles the control signal, it should return TRUE. If it
   returns FALSE, the next handler function in the list of handlers for this
   process is used */
ConsoleHandlerRoutine(
	DWORD dwCtrlType /* #IN# Control signal type */
)
{
	/* On Windows 95, a console application that has installed a control
	   signal handler function only gets called for the CTRL_C_EVENT and
	   CTRL_BREAK_EVENT signals; the signal handler function is never
	   called for the CTRL_SHUTDOWN_EVENT, CTRL_LOGOFF_EVENT, and
	   CTRL_CLOSE_EVENT signals. */
	switch( dwCtrlType )
	{
		case CTRL_CLOSE_EVENT:
		case CTRL_BREAK_EVENT:
		case CTRL_C_EVENT:
			break;

		default:
			/* Unknown type, better pass it on */
			return FALSE;
	}
	/* Handled all known events */
	return TRUE;
} /* #OF# ConsoleHandlerRoutine */

/*========================================================
Function : FreeMonitorConsole
=========================================================*/
/* #FN#
   Invokes a monitor console */
BOOL
/* #AS#
   BOOL if succeeded, otherwise FALSE */
AllocMonitorConsole(
	FILE **pOutput,
	FILE **pInput
)
{
    if( !AllocConsole() )
		return FALSE;

	SetConsoleTitle( "Atari800Win Monitor" );
	*pOutput = fopen( "CON", "wt" );
	*pInput  = fopen( "CON", "rt" );
	/* Disable Ctrl+C and Ctrl+Break keystrokes */
	SetConsoleCtrlHandler( ConsoleHandlerRoutine, TRUE );

	return TRUE;
} /* #OF# AllocMonitorConsole */

/*========================================================
Function : FreeMonitorConsole
=========================================================*/
/* #FN#
   Closes a monitor console */
void
/* #AS#
   Nothing */
FreeMonitorConsole( void )
{
	/* Detach Console Routine */
	SetConsoleCtrlHandler( ConsoleHandlerRoutine, FALSE );
	FreeConsole();
} /* #OF# FreeMonitorConsole */

/*========================================================
Function : DisplayMessage
=========================================================*/
/* #FN#
   Creates, displays, and operates a message box */
int
/* #AS#
   The MessageBox API function return value */
DisplayMessage(
	HWND hWnd,	    /* #IN# Parent window     */
	UINT uFormat,	/* #IN# Text resource id  */
	UINT uCaption,	/* #IN# Title resource id */
	UINT uType,		/* #IN# Message box style */
	...
)
{
	va_list argList;

	char szFormat [ LOADSTRING_STRING_SIZE ];
	char szCaption[ LOADSTRING_STRING_SIZE ];
	char szMessage[ LOADSTRING_STRING_SIZE * 3 ];
	int  nResult;

	if( !(uFormat &&
		LoadString( NULL, uFormat, szFormat, LOADSTRING_STRING_SIZE )) )
	{
		strcpy( szFormat, "Resource load error!" );
	}
	/* Format the message string */
	va_start( argList, uType );
	vsprintf( szMessage, szFormat, argList );

	if( !(uCaption &&
		LoadString( NULL, uCaption, szCaption, LOADSTRING_STRING_SIZE )) )
	{
		strcpy( szCaption, "Atari800Win" );
	}
	if( !uType )
		uType = MB_ICONSTOP | MB_OK;

	ShowMousePointer( TRUE );
	nResult = MessageBox( (hWnd ? hWnd : g_hMainWnd), szMessage, szCaption, uType );

	va_end( argList );

	return nResult;
} /* #OF# DisplayMessage */

/*========================================================
Function : ExecuteCmd
=========================================================*/
/* #FN#
   Executes a system command */
BOOL
/* #AS#
   TRUE if succeded, otherwise FALSE */
ExecuteCmd(
	LPSTR pszCommand, /* #IN# System command to execute */
	BOOL  bGetOutput  /* #IN# Output stream to error log */
)
{	
	BOOL   bProcess   = FALSE;
	BOOL   bPipe      = FALSE;
	HANDLE hReadPipe  = NULL;
	HANDLE hWritePipe = NULL;

	PROCESS_INFORMATION pi;
	STARTUPINFO si;

	SECURITY_ATTRIBUTES sa;
	sa.nLength              = sizeof(sa);
	sa.lpSecurityDescriptor = NULL;
	sa.bInheritHandle       = TRUE;

	if( bGetOutput )
		bPipe = CreatePipe( &hReadPipe, &hWritePipe, &sa, 0 );

	/* Run a child process */
	ZeroMemory( &si, sizeof(si) );
	si.cb = sizeof(si);
	si.wShowWindow = SW_SHOWNORMAL; /* Do not display the console window */
	si.dwFlags = STARTF_USESHOWWINDOW;

	if( bPipe )
	{
		si.dwFlags   |= STARTF_USESTDHANDLES;
		si.hStdInput  = GetStdHandle( STD_INPUT_HANDLE );
		si.hStdOutput = hWritePipe;
		si.hStdError  = hWritePipe;
	}

	/* Run system command */
	bProcess =
		CreateProcess( NULL,		// no module name (use command line)
					   pszCommand,	// command line
					   NULL,		// process handle not inheritable
					   NULL,		// threat handle not inheritable
					   TRUE,		// set handle inheritance to TRUE
					   0,			// no creation flags
					   NULL,		// use parent's environment block
					   NULL,		// use parent's starting directory
					   &si,
					   &pi );

	/* We have to close the reading end of the pipe */
	if( bPipe )
		CloseHandle( hWritePipe );

	if( bProcess )
	{
		if( bPipe )
		{
			char  szBuffer[ EC_BUFFER_LEN ];	/* Buffer that receives data */
			char  szText  [ EC_MAX_LINE_LEN ];	/* Buffer for output line */
			char *pszOut = szText;
			char  c;
			DWORD dwBytesRead;					/* Number of bytes read */
			BOOL  bReadResult;

			/* Output stream to Error View buffer */
			do
			{
				/* Read the executed process output message */
				bReadResult = ReadFile( hReadPipe, (LPVOID)&szBuffer, EC_BUFFER_LEN, &dwBytesRead, NULL );
				if( bReadResult )
				{
					char *pszIn = szBuffer;
					int   i;
					for( i = 0; i < (int)dwBytesRead; i++ )
					{
						c = *(pszIn++);
						if( c >= ' ' && c <= 'z' )
							*(pszOut++) = c;
						if( c == '\0' || c == '\n' )
						{
							*pszOut = '\0';
							/* Print message line to the Error View buffer */
							Aprint( szText );
							pszOut = szText;
						}
					}
				}
			}
			while( bReadResult && dwBytesRead != 0 );
		}
		if( bPipe )
			/* Wait for ending the executed process */
			WaitForSingleObject( pi.hProcess, INFINITE );

		if( pi.hProcess && pi.hProcess != INVALID_HANDLE_VALUE )
			CloseHandle( pi.hProcess );
		if( pi.hThread && pi.hThread != INVALID_HANDLE_VALUE )
			CloseHandle( pi.hThread );
	}

	if( bPipe )
		CloseHandle( hReadPipe );

	return bProcess;
} /* #OF# ExecuteCmd */

/*========================================================
Function : ExecutePrintCmd
=========================================================*/
/* #FN#
   Executes a print command */
BOOL
/* #AS#
   TRUE if succeded, otherwise FALSE */
ExecutePrintCmd(
	LPSTR pszPrintFile /* #IN# Print command to execute */
)
{
	BOOL bResult = TRUE;

	/* Unfortunately, there are some problems with automatically return to
	   flipped full-screen mode, go to windowed instead */
	if( g_ulScreenMode & SM_MODE_FULL /*&& g_ulScreenMode & SM_OPTN_FLIP_BUFFERS*/ )
	{
		/* The only safe method to doing it here */
		PostMessage( g_hMainWnd, WM_COMMAND, ID_GRAPHICS_TOGGLEMODES, 0L );
	}

	if( !(g_ulMiscStates & MS_USE_PRINT_COMMAND) )
	{
		char szPath[ MAX_PATH ];
		char szFile[ MAX_PATH ];
		int  nResult;

		strncpy( szPath, pszPrintFile, MAX_PATH );
		GetFolderPath( szPath, szFile );

		if( (nResult =
			(int)ShellExecute( g_hMainWnd,
							   "print",
							   szFile,
							   NULL,
							   szPath,
							   SW_HIDE )) < 32 )
		{
			Aprint( "Printing error (shell execution code: %d)", nResult );
			bResult = FALSE;
		}
	}
	else
	{
		char szPrintCmd[ MAX_PATH ];

		sprintf( szPrintCmd, print_command, pszPrintFile );
		if( !ExecuteCmd( szPrintCmd, FALSE ) )
			bResult = FALSE;
	}
	g_ulAtariState &= ~ATARI_PAUSED;

	return bResult;
} /* #OF# ExecutePrintCmd */
