/****************************************************************************
File    : input_win.c
/*
@(#) #SY# Atari800Win
@(#) #IS# Input implementation for Win32 platforms
@(#) #BY# Richard Lawrence, Tomasz Szymankowski
@(#) #LM# 01.10.2000
*/

/*
Copyright (c) 1998 Richard Lawrence

This program is free software; you can redistribute it and/or modify it under the terms
of the GNU General Public License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version. This program is
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details. You should have received a copy of the GNU
General Public License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdio.h>
#include <dinput.h>
#include <crtdbg.h>
#include "WinConfig.h"
#include "Resource.h"
#include "atari800.h"
#include "globals.h"
#include "display_win.h"
#include "misc_win.h"
#include "sound_win.h"
#include "registry.h"
#include "input_win.h"

#define MAX_BUTTONS_NO	4

/* Public objects */

HINSTANCE g_hInstance   = NULL;
int       g_nCurrentKey = AKEY_NONE;
int       g_nCurrentVK  = 0;
int       g_nNewKey     = AKEY_NONE;
int       g_nNewVK      = 0;
int       g_nNumDIDevs  = 0;
BOOL      g_bKBJoystick = FALSE;
int       CTRL_KEY;

char g_acInputDevNames      [ MAX_INPUT_DEVICES  ][ INPUT_DEV_NAMELEN ];
int  g_anSelectedInputDevice[ MAX_ATARI_JOYPORTS ] =
{
	-4, 127, 127, 127
};

#define VK_NUMPADENTER	95 /* For temporary use */

BYTE g_anExtendedKeyTab[ 256 ] =
{
/*000*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, VK_NUMPADENTER, 0, 0,
/*016*/ 0, VK_RCONTROL, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*032*/ 0, VK_PRIOR, VK_NEXT, VK_END, VK_HOME, VK_LEFT, VK_UP, VK_RIGHT, VK_DOWN, 0, 0, 0, 0, VK_INSERT, VK_DELETE, 0,
/*048*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*064*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*080*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, VK_LWIN, VK_RWIN, 0, 0, 0,
/*096*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, VK_DIVIDE,
/*112*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*128*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*144*/ VK_NUMLOCK, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*160*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*176*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*192*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*208*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*224*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*240*/ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

WORD g_anKeyset[ NUM_KBJOY_DEVICES ][ NUM_KBJOY_KEYS ] =
{
	/* Default is numpad "joystick" */
	{ VK_NUMPAD7, VK_NUMPAD8, VK_NUMPAD9, VK_NUMPAD6, VK_NUMPAD3, VK_NUMPAD2, VK_NUMPAD1, VK_NUMPAD4, VK_NUMPAD5, VK_NUMPAD0 },
	/* Cursor Keys + RIGHT CTRL */
	{ 0, VK_UP, 0, VK_RIGHT, 0, VK_DOWN, 0, VK_LEFT, 0, VK_RCONTROL },
	/* User defined A & B Keysets */
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }
};

/* Private objects */

static LPDIRECTINPUT        s_lpDInput = NULL;
static LPDIRECTINPUTDEVICE2 s_alpDevFound[ MAX_INPUT_DEVICES ];
static LPDIRECTINPUTDEVICE2 s_alpJoyPorts[ MAX_ATARI_JOYPORTS ] =
{
	NULL, NULL, NULL, NULL
};

static int	 s_nJoyCount = 0;
static int   s_nConsol;
static BOOL	 s_bBtn2Flip;
static BOOL	 s_bBtn3Flip;
static BOOL	 s_bBtn4Flip;

static int   s_nCurrentMouseX = 0; /* Used in AMIGA, ST and JOY modes */
static int   s_nCurrentMouseY = 0;
static int   s_nMouseJoy      = 0;

static int   s_anStick    [ MAX_ATARI_JOYPORTS ];
static int   s_anTrig     [ MAX_ATARI_JOYPORTS ];

static UINT  s_aunKeystick[ NUM_KBJOY_DEVICES ] = { STICK_CENTRE, STICK_CENTRE, STICK_CENTRE, STICK_CENTRE };
static UINT  s_aunKeytrig [ NUM_KBJOY_DEVICES ] = { 1, 1, 1, 1 };
static int   s_anPot      [ 4 ][ 2 ];

static const int s_nJoy5200Min    = 15;
static const int s_nJoy5200Center = 120;
static const int s_nJoy5200Max    = 220;

static const int s_anStickTable[ NUM_KBJOY_KEYS - 1 ] =
{
	STICK_UL,
	STICK_FORWARD,
	STICK_UR,
	STICK_RIGHT,
	STICK_LR,
	STICK_BACK,
	STICK_LL,
	STICK_LEFT,
	STICK_CENTRE
};

static BOOL GetDIErrorString( HRESULT hResult, LPSTR lpszErrorBuff, DWORD cchError );

#ifdef _DEBUG
#define ServeDInputError( nUID, hResult, bQuit ) \
		ShowDInputError( nUID, hResult, bQuit, __FILE__, __LINE__ )
#else /*_DEBUG*/
#define ServeDInputError( nUID, hResult, bQuit ) \
		ShowDInputError( nUID, hResult, bQuit )
#endif /*_DEBUG*/


/*========================================================
Function : ShowDInputError
=========================================================*/
/* #FN#
   Fatal DirectInput error function. Since we cannot continue, bring us back
   to a known working state before displaying the error, then start an exit.
   Make sure to shut down the Atari layer on the way. */
static
void
/* #AS#
   Nothing */
ShowDInputError( UINT    nUID,    /* #IN# */
				 HRESULT hResult, /* #IN# */
				 BOOL    bQuit    /* #IN# */
#ifdef _DEBUG
			   , char   *pszFile,
				 DWORD   dwLine
#endif /*_DEBUG*/
)
{
	char szError [ LOADSTRING_STRING_SIZE ];
	char szAction[ LOADSTRING_STRING_SIZE ];

#ifdef _DEBUG
	Aprint( "DirectInput error: %s@%ld", pszFile, dwLine );
#endif /*_DEBUG*/

	/* Get us back to a GDI display and stop making noises */
	SetSafeDisplay( FALSE );
	ClearSound( FALSE );

	/* Get the error string and present it to the user */
	GetDIErrorString( hResult, szError, LOADSTRING_STRING_SIZE );
	LoadString( NULL, nUID, szAction, LOADSTRING_STRING_SIZE );
	DisplayMessage( IDS_DIERR_PROMPT, IDS_DIERR_HDR, MB_ICONSTOP | MB_OK, szAction, szError );

	/* Start a quit (this will end up in Atari_Exit()) */
	if( bQuit )
	{
		/* Make sure the atari is turned off */
		g_ulAtariState = ATARI_UNINITIALIZED;
		PostMessage( g_hMainWnd, WM_CLOSE, 0, 0L );
	}
	else
		RestartSound();
} /* #OF# ShowDInputError */

/*========================================================
Function : ResetKeys
=========================================================*/
/* #FN#
   Resets the keys state to default */
void
/* #AS#
   Nothing */
ResetKeys( void )
{
	int i;
	for( i = 0; i < NUM_KBJOY_DEVICES; i++ )
	{
		s_aunKeystick[ i ] = STICK_CENTRE;
		s_aunKeytrig [ i ] = 1;
	}
	g_nCurrentKey = g_nNewKey = AKEY_NONE;
	g_nCurrentVK  = g_nNewVK  = 0;
	SHIFT_KEY     = CTRL_KEY  = FALSE;
	KEYPRESSED    = FALSE;
	s_nConsol     = 0x07;
} /* #OF# ResetKeys */

/*========================================================
Function : ResetInput
=========================================================*/
/* #FN#
   Resets the input (keys and joys) state to default */
void
/* #AS#
   Nothing */
ResetInput( void )
{
	int i;

	s_bBtn2Flip = FALSE;
	s_bBtn3Flip = FALSE;
	s_bBtn4Flip = FALSE;

	ResetKeys();
	for( i = 0; i < MAX_ATARI_JOYPORTS; i++ )
	{
		s_anTrig [ i ] = 1;
		s_anStick[ i ] = STICK_CENTRE;
	}
} /* #OF# ResetInput */

/*========================================================
Function : SetDIDwordProperty
=========================================================*/
/* #FN#
   Helper function just makes sure the Property struct is set correctly */
static
HRESULT
/* #AS#
   DI_OK, DI_PROPNOEFFECT, if the methods succeeds. DIERR_OBJECTNOTFOUND,
   DIERR_UNSUPPORTED, DIERR_NOTINITIALIZED, DIERR_INVALIDPARAM, if the
   method fails */
SetDIDwordProperty( LPDIRECTINPUTDEVICE lpInputDev,   /* #IN# */
					REFGUID             guidProperty, /* #IN# */
					DWORD               dwObject,     /* #IN# */
					DWORD               dwHow,        /* #IN# */
					DWORD               dwValue       /* #IN# */ )
{
	DIPROPDWORD dipdw;

	dipdw.diph.dwSize       = sizeof(dipdw);
	dipdw.diph.dwHeaderSize = sizeof(dipdw.diph);
	dipdw.diph.dwObj        = dwObject;
	dipdw.diph.dwHow        = dwHow;
	dipdw.dwData            = dwValue;

	return IDirectInputDevice_SetProperty( lpInputDev, guidProperty, &dipdw.diph );
} /* #OF# SetDIDwordProperty */

/*========================================================
Function : InitJoystickInput
=========================================================*/
/* #FN#
   Callback function that will be called with a description of each
   DirectInput device */
static
BOOL CALLBACK
/* #AS#
   Returns DIENUM_CONTINUE to continue the enumeration or DIENUM_STOP
   to stop the enumeration */
InitJoystickInput( LPCDIDEVICEINSTANCE lpDevInstance,
				   LPVOID pvRef )
{
	LPDIRECTINPUT       lpDirectInput = pvRef;
	LPDIRECTINPUTDEVICE pdev;
	DIPROPRANGE         diprg;
	HRESULT             hResult;

	/* Create the DirectInput joystick device */
	hResult = IDirectInput_CreateDevice( lpDirectInput, &lpDevInstance->guidInstance, &pdev, NULL );
	if( FAILED(hResult) )
	{
		ServeDInputError( IDS_DIERR_CREATING_DEVICE, hResult, FALSE );
		return DIENUM_STOP;/*DIENUM_CONTINUE;*/
	}
	/* Set joystick data format */
	hResult = IDirectInputDevice_SetDataFormat( pdev, &c_dfDIJoystick );
	if( FAILED(hResult) )
	{
		ServeDInputError( IDS_DIERR_SETTING_JOY_FORMAT, hResult, FALSE );
		IDirectInputDevice_Release( pdev );
		return DIENUM_STOP;
	}
	/* Set the cooperative level */
	hResult = IDirectInputDevice_SetCooperativeLevel( pdev, g_hMainWnd, DISCL_NONEXCLUSIVE | DISCL_BACKGROUND );
	if( FAILED(hResult) )
	{
		ServeDInputError( IDS_DIERR_SETTING_COOP, hResult, FALSE );
		IDirectInputDevice_Release( pdev );
		return DIENUM_STOP;
	}
	/* Set X-axis range to (-1000 ... +1000)
	   This lets us test against 0 to see which way the s_anStick is pointed. */
	diprg.diph.dwSize       = sizeof(diprg);
	diprg.diph.dwHeaderSize = sizeof(diprg.diph);
	diprg.diph.dwObj        = DIJOFS_X;
	diprg.diph.dwHow        = DIPH_BYOFFSET;
	diprg.lMin              = -1000;
	diprg.lMax              = +1000;

	hResult = IDirectInputDevice_SetProperty( pdev, DIPROP_RANGE, &diprg.diph );
	if( FAILED(hResult) )
	{
		ServeDInputError( IDS_DIERR_SETTING_PROPS, hResult, FALSE );
		IDirectInputDevice_Release( pdev );
		return FALSE;
	}

	/* And again for Y-axis range */
	diprg.diph.dwObj = DIJOFS_Y;

	hResult = IDirectInputDevice_SetProperty( pdev, DIPROP_RANGE, &diprg.diph );
	if( FAILED(hResult) )
	{
		ServeDInputError( IDS_DIERR_SETTING_PROPS, hResult, FALSE );
		IDirectInputDevice_Release( pdev );
		return FALSE;
	}
	/* Set X axis dead zone to 50% (to avoid accidental turning)
	   Units are ten thousandths, so 50% = 5000/10000. */
	hResult = SetDIDwordProperty( pdev, DIPROP_DEADZONE, DIJOFS_X, DIPH_BYOFFSET, 5000 );
	if( FAILED(hResult) )
	{
		ServeDInputError( IDS_DIERR_SETTING_DEADZONE, hResult, FALSE );
		IDirectInputDevice_Release( pdev );
		return FALSE;
	}
	/* Set Y axis dead zone to 50% (to avoid accidental thrust)
	   Units are ten thousandths, so 50% = 5000/10000. */
	hResult = SetDIDwordProperty( pdev, DIPROP_DEADZONE, DIJOFS_Y, DIPH_BYOFFSET, 5000 );
	if( FAILED(hResult) )
	{
		ServeDInputError( IDS_DIERR_SETTING_DEADZONE, hResult, FALSE );
		IDirectInputDevice_Release( pdev );
		return FALSE;
	}
	/* Add it to our list of devices */
	hResult = IDirectInputDevice_QueryInterface( pdev, &IID_IDirectInputDevice2, (LPVOID *)&s_alpDevFound[ g_nNumDIDevs ] );
	if( FAILED(hResult) )
	{
		ServeDInputError( IDS_DIERR_ADDING_DEVICE, hResult, FALSE );
		return FALSE;
	}
	strncpy( g_acInputDevNames[ g_nNumDIDevs++ ], lpDevInstance->tszInstanceName, 32 );
	IDirectInputDevice_Release( pdev );

	return DIENUM_CONTINUE;
} /* #OF# InitJoystickInput */

/*========================================================
Function : ReacquireInput
=========================================================*/
/* #FN#
   */
static
BOOL
/* #AS#
   TRUE if succeeded, FALSE if failed */
ReacquireInput( int nStickNum )
{
    /* If we have a current device */
    if( s_alpJoyPorts[ nStickNum ] )
    {
		/* Acquire the device */
		if( FAILED(
			IDirectInputDevice_Acquire( s_alpJoyPorts[ nStickNum ] )) )
			return FALSE;	/* acquisition failed */
    }
    else	/* We don't have a current device */
		return FALSE;

	return TRUE;
} /* #OF# ReacquireInput */

/*========================================================
Function : PickJoystick
=========================================================*/
/* #FN#
   */
static
void
/* #AS#
   Nothing */
PickJoystick( int nStickNum,
			  int nJoystick )
{
	int i;

	for( i = 0; i < MAX_ATARI_JOYPORTS; i++ )
	{
		if( s_alpJoyPorts[ i ] != NULL &&
			(nJoystick >= 0 && nJoystick < MAX_INPUT_DEVICES) &&
			s_alpJoyPorts[ i ] == s_alpDevFound[ nJoystick ] )
		{
			IDirectInputDevice_Unacquire( s_alpJoyPorts[ i ] );
			s_alpJoyPorts[ i ] = NULL;
		}
	}

	g_anSelectedInputDevice[ nStickNum ] = nJoystick;

	if( s_alpJoyPorts[ nStickNum ] )
		IDirectInputDevice_Unacquire( s_alpJoyPorts[ nStickNum ] );
	s_alpJoyPorts[ nStickNum ] = NULL;

	g_bKBJoystick = FALSE;
	for( i = 0; i < MAX_ATARI_JOYPORTS; i++ )
	{
		if( g_anSelectedInputDevice[ i ] < 0 )
			g_bKBJoystick = TRUE;
	}

	/* (nJoystick < 0) means to use the keyboard instead */
	if( nJoystick < 0 )
		return;

	if( nJoystick < MAX_INPUT_DEVICES && s_alpDevFound[ nJoystick ] )
	{
		s_alpJoyPorts[ nStickNum ] = s_alpDevFound[ nJoystick ];
		if( !ReacquireInput( nStickNum ) )
			ServeDInputError( IDS_DIERR_ACQUIRE_INIT, DIERR_NOTACQUIRED, FALSE );
	}
} /* #OF# PickJoystick */

/*========================================================
Function : InitialiseInput
=========================================================*/
/* #FN#
   */
int
/* #AS#
   */
InitialiseInput( void )
{
	char    cValue;
	int     nStick;
	HRESULT	hResult;

	g_nCurrentKey = g_nNewKey = AKEY_NONE;
	g_nCurrentVK  = g_nNewVK  = 0;
	KEYPRESSED    = FALSE;

	ResetInput();

	if( !s_lpDInput )
	{
		ZeroMemory( s_alpDevFound, sizeof(LPDIRECTINPUTDEVICE) * MAX_INPUT_DEVICES );

		/* Create the DirectInput 5.0 interface object */
		hResult = DirectInputCreate( g_hInstance, DIRECTINPUT_VERSION, &s_lpDInput, NULL );
		if( FAILED(hResult) )
		{
			if( hResult == DIERR_OLDDIRECTINPUTVERSION )
			{
				s_lpDInput = NULL;
			}
			else
			{
				ServeDInputError( IDS_DIERR_CREATING_OBJ, hResult, FALSE );
				return 0;
			}
		}
		else
		{
			/* Enumerate the joystick devices */
			hResult = IDirectInput_EnumDevices( s_lpDInput, DIDEVTYPE_JOYSTICK, InitJoystickInput, s_lpDInput, DIEDFL_ATTACHEDONLY );
			if( FAILED(hResult) )
			{
				ServeDInputError( IDS_DIERR_ENUMERATING, hResult, FALSE );
				return 0;
			}
			IDirectInput_Release( s_lpDInput );
		}
	}
	cValue = (char)(g_ulJoystickSelects & 0x000000ff);
	nStick = cValue;
	PickJoystick( 0, nStick );

	nStick = g_ulJoystickSelects & 0x0000ff00;
	nStick >>= 8;
	cValue = (char)nStick;
	nStick = cValue;
	if( nStick == g_anSelectedInputDevice[ 0 ] )
		nStick = NO_JOYSTICK;
	PickJoystick( 1, nStick );

	nStick = g_ulJoystickSelects & 0x00ff0000;
	nStick >>= 16;
	cValue = (char)nStick;
	nStick = cValue;
	if( nStick == g_anSelectedInputDevice[ 0 ] || nStick == g_anSelectedInputDevice[ 1 ] )
		nStick = NO_JOYSTICK;
	PickJoystick( 2, nStick );

	nStick = g_ulJoystickSelects & 0xff000000;
	nStick >>= 24;
	cValue = (char)nStick;
	nStick = cValue;
	if( nStick == g_anSelectedInputDevice[ 0 ] || nStick == g_anSelectedInputDevice[ 1 ] || nStick == g_anSelectedInputDevice[ 2 ] )
		nStick = NO_JOYSTICK;
	PickJoystick( 3, nStick );

	s_nJoyCount = g_nJoyRepeat;

	return 1;
} /* #OF# InitialiseInput */

/*========================================================
Function : ClearInput
=========================================================*/
/* #FN#
   */
void
/* #AS#
   Nothing */
ClearInput( void )
{
	int i;
	/* Make sure the device is unacquired it doesn't harm
	   to unacquire a device that isn't acquired */
	for( i = 0; i < MAX_ATARI_JOYPORTS; i++ )
	{
		if( s_alpJoyPorts[ i ] )
		{
			IDirectInputDevice_Unacquire( s_alpJoyPorts[ i ] );
			s_alpJoyPorts[ i ] = NULL;
		}
	}
	/* Release all the devices we created */
	for( i = 0; i < g_nNumDIDevs; i++ )
	{
		if( s_alpDevFound[ i ] )
		{
			IDirectInputDevice_Release( s_alpDevFound[ i ] );
			s_alpDevFound[ i ] = NULL;
		}
	}
} /* #OF# ClearInput */

/*========================================================
Function : ReadJoystickInput
=========================================================*/
/* #FN#
   */
int
/* #AS#
   */
ReadJoystickInput( int  nStickNum,
				   BOOL bTestButton )
{
	DIJOYSTATE js;
	int        nStickVal = 0;
	HRESULT    hResult;

	if( nStickNum >= MAX_INPUT_DEVICES || !s_alpJoyPorts[ nStickNum ] )
		return STICK_CENTRE;

	/* Poll the joystick to read the current state */
	IDirectInputDevice2_Poll( s_alpJoyPorts[ nStickNum ] );

	/* Get data from the joystick */
	hResult = IDirectInputDevice_GetDeviceState( s_alpJoyPorts[ nStickNum ], sizeof(DIJOYSTATE), &js );
	if( FAILED(hResult) )
	{
		/* did the read fail because we lost input for some reason?
		   if so, then attempt to reacquire.  If the second acquire
		   fails, then the error from GetDeviceData will be
		   DIERR_NOTACQUIRED, so we won't get stuck an infinite loop. */
		if( hResult == DIERR_INPUTLOST )
		{
			ReacquireInput( nStickNum );
			/* return the fact that we did not read any data */
			return 0;
		}
		ServeDInputError( IDS_DIERR_ACQUIRE, DIERR_NOTACQUIRED, FALSE );
	}

	if( Atari5200 == machine )
	{
		int i;
		for( i = 0; i < MAX_BUTTONS_NO; i++ )
		{
			if( js.rgbButtons[ i ] & 0x80 )
				s_anTrig[ i ] = 0;
			else
				s_anTrig[ i ] = 1;
		}
	}
	else
	{
		if( !(g_ulMiscStates & MS_JOY_FIRE_ONLY) )
		{
			if( !s_bBtn2Flip )
			{
				if( js.rgbButtons[ 1 ] & 0x80 )
				{
					/* Description should be changed */
					PostMessage( g_hMainWnd, WM_COMMAND, ID_ATARI_PAUSE, 0L );
//					TogglePause();
					s_bBtn2Flip = TRUE;
				}
			}
			else
			{
				if( !(js.rgbButtons[ 1 ] & 0x80) )
					s_bBtn2Flip = FALSE;
			}

			if( !s_bBtn3Flip )
			{
				if( js.rgbButtons[ 2 ] & 0x80 )
				{
					/* Description should be changed */
					PostMessage( g_hMainWnd, WM_COMMAND, ID_ATARI_FULLSPEED, 0L );
//					ToggleFullSpeed();
					s_bBtn3Flip = TRUE;
				}
			}
			else
			{
				if( !(js.rgbButtons[ 2 ] & 0x80) )
					s_bBtn3Flip = FALSE;
			}

			if( !s_bBtn4Flip )
			{
				if( js.rgbButtons[ 3 ] & 0x80 )
				{
					/* Description should be changed */
					PostMessage( g_hMainWnd, WM_COMMAND, ID_ATARI_SIOPATCH, 0L );
//					ToggleSIOPatch();
					s_bBtn4Flip = TRUE;
				}
			}
			else
			{
				if( !(js.rgbButtons[ 3 ] & 0x80) )
					s_bBtn4Flip = FALSE;
			}
		}

		if( bTestButton )
		{
			/* See if the repeat/skip options are being used for holding button down */
			if( g_nJoyRepeat )
			{
				if( js.rgbButtons[ 0 ] & 0x80 ||
					(g_ulMiscStates & MS_JOY_FIRE_ONLY && ((js.rgbButtons[ 1 ] | js.rgbButtons[ 2 ] | js.rgbButtons[ 3 ]) & 0x80)) )
				{
					/* Count down the number of repeat cycles and return button up for those */
					if( --s_nJoyCount > 0 )
						s_anTrig[ nStickNum ] = 1;

					/* If we've counted the number of skip cycles, start over and send button down */
					if( s_nJoyCount < -g_nJoySkip )
					{
						s_nJoyCount = g_nJoyRepeat;
						s_anTrig[ nStickNum ] = 0;
					}
				}
				else
				{
					/* Otherwise continue skipping, sending button up */
					s_nJoyCount = g_nJoyRepeat;
					s_anTrig[ nStickNum ] = 1;
				}
			}
			else
			{
				int i;
				/* No special handling, just return whether it is held down */
				for( i = 0; i < (g_ulMiscStates & MS_JOY_FIRE_ONLY ? MAX_BUTTONS_NO : 1); i++ )
				{
					if( js.rgbButtons[ i ] & 0x80 )
					{
						s_anTrig[ nStickNum ] = 0;
						break;
					}
					else
						s_anTrig[ nStickNum ] = 1;
				}
			}
		}
	}

	if( js.lX < 0 )
	{
		if( js.lY < 0 )
			nStickVal |= STICK_UL;
		else
		if( js.lY > 0 )
			nStickVal |= STICK_LL;
		else
		if( js.lY == 0 )
			nStickVal |= STICK_LEFT;
	}
	else
	if( js.lX > 0 )
	{
		if( js.lY < 0 )
			nStickVal |= STICK_UR;
		else
		if( js.lY > 0 )
			nStickVal |= STICK_LR;
		else
		if( js.lY == 0 )
			nStickVal |= STICK_RIGHT;
	}
	else
	if( js.lX == 0 )
	{
		if( js.lY < 0 )
			nStickVal |= STICK_FORWARD;
		else
		if( js.lY > 0 )
			nStickVal |= STICK_BACK;
		else
		if( js.lY == 0 )
			nStickVal |= STICK_CENTRE;
	}
	s_anStick[ nStickNum ] = nStickVal;

	return 1;
} /* #OF# ReadJoystickInput */

/*========================================================
Function : Atari_Keyboard
=========================================================*/
/* #FN#
   In other Atari800 versions, keyboard handling is here. In Atari800Win
   I handle the actual key processing in the mainframe of the app and by
   the time it gets here we already have the appropriate Atari key code
   (except for keycodes that require multiple logic operations, like the
   s_anStick) */
int
/* #AS#
   Current Atari key value */
Atari_Keyboard( void )
{
	if( AKEY_NONE != g_nCurrentKey )
	{
		if( GetAsyncKeyState( g_nCurrentVK ) >= 0 )
		{
			KEYPRESSED = FALSE;
			g_nCurrentKey = AKEY_NONE;
		}
	}
	if( AKEY_NONE == g_nNewKey )
		return g_nCurrentKey;

	if( Atari5200 == machine )
	{
		switch( g_nNewKey )
		{
			case AKEY_STARTDWN:
				g_nNewKey = 0x39;	/* start */
				break;

			case AKEY_p:
				g_nNewKey = 0x31;	/* pause */
				break;

			case AKEY_r:
				g_nNewKey = 0x29;	/* reset (5200 has no warmstart) */
				break;

			case AKEY_0:
				g_nNewKey = 0x25;	/* controller numpad keys (0-9) */
				break;

			case AKEY_1:
				g_nNewKey = 0x3f;
				break;

			case AKEY_2:
				g_nNewKey = 0x3d;
				break;

			case AKEY_3:
				g_nNewKey = 0x3b;
				break;

			case AKEY_4:
				g_nNewKey = 0x37;
				break;

			case AKEY_5:
				g_nNewKey = 0x35;
				break;

			case AKEY_6:
				g_nNewKey = 0x33;
				break;

			case AKEY_7:
				g_nNewKey = 0x2f;
				break;

			case AKEY_8:
				g_nNewKey = 0x2d;
				break;

			case AKEY_9:
				g_nNewKey = 0x2b;
				break;

			case AKEY_MINUS:
				g_nNewKey = 0x23;	/* # key on 5200 controller */
				break;

			case AKEY_ASTERISK:
				g_nNewKey = 0x27;	/* * key on 5200 controller */
				break;

			default:
				g_nNewKey = AKEY_NONE;
				break;
		}
	}

	if( g_nNewKey & SPECIAL_HANDLER_MASK )
	{
		int	nKeyTemp = g_nNewKey;
		g_nNewKey = AKEY_NONE;

		switch( nKeyTemp )
		{
			case (AKEY_HELP | SPECIAL_HANDLER_MASK):
				if( mach_xlxe )
					g_nNewKey = nKeyTemp & ~SPECIAL_HANDLER_MASK;
				break;

//			case (AKEY_F1 | SPECIAL_HANDLER_MASK):
//			case (AKEY_F2 | SPECIAL_HANDLER_MASK):
//			case (AKEY_F3 | SPECIAL_HANDLER_MASK):
//			case (AKEY_F4 | SPECIAL_HANDLER_MASK):
//				if( mach_xlxe )
//					g_nNewKey = nKeyTemp & ~SPECIAL_HANDLER_MASK;
//				break;

			case AKEY_BREAK:
				IRQST &= ~0x80;
				if( IRQEN & 0x80 )
					GenerateIRQ();
				break;
		}
		if( g_nNewKey == AKEY_NONE )
			g_nNewVK = 0;
	}

	if( g_nNewKey != AKEY_NONE && g_nCurrentKey == AKEY_NONE )
	{
		KEYPRESSED   = TRUE;
		KBCODE       = g_nCurrentKey = g_nNewKey;
		g_nCurrentVK = g_nNewVK;
		g_nNewKey    = AKEY_NONE;
		g_nNewVK     = 0;

		IRQST &= ~0x40;
		if( IRQEN & 0x40 )
			GenerateIRQ();
	}
	return g_nCurrentKey;
} /* #OF# Atari_Keyboard */

/*========================================================
Function : GetJoystickInput
=========================================================*/
/* #FN#
   */
void
/* #AS#
   Nothing */
GetJoystickInput( void )
{
	int i;

	for( i = 0; i < MAX_ATARI_JOYPORTS; i++ )
	{
		if( s_alpJoyPorts[ i ] )
			ReadJoystickInput( i, TRUE );
		else
		{
			if( g_anSelectedInputDevice[ i ] < 0 )
			{
				s_anStick[ i ] = s_aunKeystick[ g_anSelectedInputDevice[ i ] + NUM_KBJOY_DEVICES ];
				s_anTrig [ i ] = s_aunKeytrig [ g_anSelectedInputDevice[ i ] + NUM_KBJOY_DEVICES ];
			}
			else
			{
				s_anStick[ i ] = STICK_CENTRE;
				s_anTrig [ i ] = 1;
			}
		}
	}
} /* #OF# GetJoystickInput */

/*========================================================
Function : UpdateCurrentMouse
=========================================================*/
/* #FN#
   */
static
int
/* #AS#
   */
UpdateCurrentMouse( void )
{
	int nVal = 0xff;
	if( s_nCurrentMouseX < g_nMouseX )
	{
		s_nCurrentMouseX++;
		nVal &= ~8;	/* Right */
	}
	else
	if( s_nCurrentMouseX > g_nMouseX )
	{
		s_nCurrentMouseX--;
		nVal &= ~4;	/* Left */
	}
	if( s_nCurrentMouseY < g_nMouseY )
	{
		s_nCurrentMouseY++;
		nVal &= ~2;	/* Down */
	}
	else
	if( s_nCurrentMouseY > g_nMouseY )
	{
		s_nCurrentMouseY--;
		nVal &= ~1;	/* Up */
	}
	return nVal;
} /* #OF# UpdateCurrentMouse */

/*========================================================
Function : Atari_PORT
=========================================================*/
/* #FN#
   This function is called by the Atari800 kernel */
int
/* #AS#
   Two stick port states for requested controller port */
Atari_PORT( int nNum )
{
	int nVal = (s_anStick[ (nNum << 1) + 1 ] << 4) | s_anStick[ nNum << 1 ];

	if( g_ulMiscStates & MS_MOUSE_CAPTURED &&
		nNum == g_nMousePort >> 1 )
	{
		switch( g_nMouseMode )
		{
			case MOUSE_PAD:
				return nVal & ~((g_nMouseButtons & 3) << (g_nMousePort % 2 ? 6 : 2));

			case MOUSE_PEN:
//				if( g_nMousePort )
//					break; /* The light pen is assigned to port #0 in ANTIC.c */
//				return nVal & ~(g_nMouseButtons & 1);
				return nVal & ~(g_nMousePort % 2 ? ((g_nMouseButtons & 1) << 4) : g_nMouseButtons & 1);

			case MOUSE_AMIGA:
				UpdateCurrentMouse();
				nVal = ((UBYTE *) "\0\2\xa\x8")[ s_nCurrentMouseX & 3 ] | ((UBYTE *) "\0\1\5\4")[ s_nCurrentMouseY & 3 ];
				return (g_nMousePort % 2 ?
					(nVal << 4) | s_anStick[ nNum << 1 ] :
					(s_anStick[ (nNum << 1) + 1 ] << 4) | nVal);

			case MOUSE_ST:
				UpdateCurrentMouse();
				nVal = ((UBYTE *) "\0\2\3\1")[ s_nCurrentMouseX & 3 ] | ((UBYTE *) "\0\x8\xc\4")[ s_nCurrentMouseY & 3 ];
				return (g_nMousePort % 2 ?
					(nVal << 4) | s_anStick[ nNum << 1 ] :
					(s_anStick[ (nNum << 1) + 1 ] << 4) | nVal);

			case MOUSE_JOY:
				return nVal & (g_nMousePort % 2 ? (s_nMouseJoy << 4) | 0xf : s_nMouseJoy);
		}
	}
	return nVal;
} /* #OF# Atari_PORT */

/*========================================================
Function : Atari_TRIG
=========================================================*/
/* #FN#
   This function is called by the Atari800 kernel */
int
/* #AS#
   0 if the trigger is pressed, otherwise 1 */
Atari_TRIG( int nNum )
{
/* The trigger inputs, TRIG0 through TRIG3, are wired to the controller
   ports, one to a port. The bottom button on either side of the leftmost
   controller zeroes the TRIG0 register when pressed, and likewise for
   the other ports.
*/
	if( nNum >= 0 && nNum <= 3 )
	{
		int nVal = s_anTrig[ nNum ];

		if( g_ulMiscStates & MS_MOUSE_CAPTURED &&
			nNum == g_nMousePort )
		{
			switch( g_nMouseMode )
			{       
				case MOUSE_PAD:
					if( Atari5200 != machine )
						break;	/* Paddles button is TRIG only in 5200 mode! */
				case MOUSE_AMIGA:
				case MOUSE_ST:
				case MOUSE_JOY:
					nVal &= ((~g_nMouseButtons) & 1);
			}
		}
		return nVal;
    }
    return 0;
} /* #OF# Atari_TRIG */

/*========================================================
Function : Atari_POT
=========================================================*/
/* #FN#
   This function is called by the Atari800 kernel */
int
/* #AS#
   The position of range from 1 to 228 */
Atari_POT( int nNum )
{
/* The eight pot inputs used for paddles in the 400/800 are wired to the
   5200's controller ports, two to a port. Each controller has an analog
   joystick, using one pot to sense horizontal position and a second pot
   for vertical position. The even pots (POT0-POT6) give the horizontal
   positions of range from 1 to 228; the maximum readings are to the right
   for the horizontal pot and at the bottom for the vertical pot.
*/
	int nVal = 228;

	if( nNum >= 0 && nNum <= 7 )
	{
		if( Atari5200 == machine )
			nVal = s_nJoy5200Center;

		if( g_ulMiscStates & MS_MOUSE_CAPTURED &&
			g_nMousePort == nNum >> 1 )
		{
			if( MOUSE_PAD == g_nMouseMode )
			{
				nVal = (nNum % 2 ? g_nMouseY : g_nMouseX);
				/* Left upper corner is 0,0 on 5200 and 228,228 on others */
				nVal = (Atari5200 == machine ?
						nVal : 228 - nVal);
			}
			else
			if( MOUSE_JOY == g_nMouseMode &&
				Atari5200 == machine )
			{
				nVal = s_anPot[ g_nMousePort ][ nNum % 2 ? 1 : 0 ];
			}
		}
		if( Atari5200 == machine )
		{
			if( !(s_anStick[ nNum >> 1 ] & (nNum % 2 ? 0x01 : 0x04)) )
				if( nVal > s_nJoy5200Min )
					nVal = s_nJoy5200Min;
			if( !(s_anStick[ nNum >> 1 ] & (nNum % 2 ? 0x02 : 0x08)) )
				if( nVal < s_nJoy5200Max )
					nVal = s_nJoy5200Max;
		}
	}
 	return nVal;
} /* #OF# Atari_POT */

/*========================================================
Function : Atari_PEN
=========================================================*/
/* #FN#
   This function is called by the Atari800 kernel */
int
/* #AS#
   A value of range from 0 to 255 */
Atari_PEN( int vertical )
{
	if( MOUSE_PEN == g_nMouseMode )
		return vertical ?
			/* The pen_h = 44 and pen_v = 4 (exactly 2)
			   are a minimum delay */
			g_nMouseY +  4 + g_nPenOffsetY :
			g_nMouseX + 44 + g_nPenOffsetX; /* % 228 (?) */

	return vertical ? 0xff : 0;
} /* #OF# Atari_PEN */

/*========================================================
Function : UpdateMouseState
=========================================================*/
/* #FN#
   Updates variables are used for a devices emulation using the mouse */
void
/* #AS#
   Nothing */
UpdateMouseState( void )
{
	if( (g_ulScreenMode & SM_OPTN_HIDE_CURSOR && g_ulScreenMode & SM_MODE_FULL ||
		 g_ulMiscStates & MS_MOUSE_CAPTURED) &&
		 g_nShowCursor > -1 && --g_nShowCursor == 0 )
	{
		ShowMousePointer( FALSE ); /* Hide the mouse cursor */
	}

	if( g_ulMiscStates & MS_MOUSE_CAPTURED &&
		MOUSE_OFF != g_nMouseMode )
	{
		int nMaxX, nMaxY;

		POINT pt;
		GetCursorPos( &pt ); /* Read the pointer position */

		if( MOUSE_PAD == g_nMouseMode )
		{
			nMaxX = g_nPaddleRange;
			nMaxY = g_nPaddleRange;
		}
		else
		if( MOUSE_PEN == g_nMouseMode )
		{
			nMaxX = 167;
			nMaxY = 119;
		}
		else /* The Amiga/ST mouse or an emulated joystick */
		{
			int nDivisor = (MOUSE_JOY == g_nMouseMode ? 1 : g_nMouseSpeed);

			nMaxX = (g_rcViewBounds.right - g_rcViewBounds.left) / nDivisor;
			nMaxY = (g_rcViewBounds.bottom - g_rcViewBounds.top) / nDivisor;

			if( pt.x >= g_rcViewBounds.right - 1 )
			{
				pt.x = g_rcViewBounds.left + 2;
				SetCursorPos( pt.x, pt.y );
				s_nCurrentMouseX = (pt.x - g_rcViewBounds.left) / nDivisor;
			}
			else
			if( pt.x <= g_rcViewBounds.left + 1 )
			{
				pt.x = g_rcViewBounds.right - 2;
				SetCursorPos( pt.x, pt.y );
				s_nCurrentMouseX = (pt.x - g_rcViewBounds.left) / nDivisor;
			}
			if( pt.y >= g_rcViewBounds.bottom - 1 )
			{
				pt.y = g_rcViewBounds.top + 2;
				SetCursorPos( pt.x, pt.y );
				s_nCurrentMouseY = (pt.y - g_rcViewBounds.top) / nDivisor;
			}
			else
			if( pt.y <= g_rcViewBounds.top + 1 )
			{
				pt.y = g_rcViewBounds.bottom - 2;
				SetCursorPos( pt.x, pt.y );
				s_nCurrentMouseY = (pt.y - g_rcViewBounds.top) / nDivisor;
			}
		}
		g_nMouseX = MulDiv( pt.x - g_rcViewBounds.left, nMaxX, g_rcViewBounds.right - g_rcViewBounds.left - 1 );
		g_nMouseY = MulDiv( pt.y - g_rcViewBounds.top,  nMaxY, g_rcViewBounds.bottom - g_rcViewBounds.top - 1 );

		if( MOUSE_JOY == g_nMouseMode )
		{
			if( Atari5200 == machine )
			{
				s_anPot[ g_nMousePort ][ 0 ] = s_nJoy5200Center + g_nMouseX - s_nCurrentMouseX;
				if( s_anPot[ g_nMousePort ][ 0 ] < s_nJoy5200Min )
					s_anPot[ g_nMousePort ][ 0 ] = s_nJoy5200Min;
				else
				if( s_anPot[ g_nMousePort ][ 0 ] > s_nJoy5200Max )
					s_anPot[ g_nMousePort ][ 0 ] = s_nJoy5200Max;

				s_anPot[ g_nMousePort ][ 1 ] = s_nJoy5200Center + g_nMouseY - s_nCurrentMouseY;
				if( s_anPot[ g_nMousePort ][ 1 ] < s_nJoy5200Min )
					s_anPot[ g_nMousePort ][ 1 ] = s_nJoy5200Min;
				else
				if( s_anPot[ g_nMousePort ][ 1 ] > s_nJoy5200Max )
					s_anPot[ g_nMousePort ][ 1 ] = s_nJoy5200Max;

				s_nCurrentMouseX = g_nMouseX;
				s_nCurrentMouseY = g_nMouseY;
			}
			else
			{
				if( g_nMouseX < s_nCurrentMouseX - g_nJoystickInertia )
					s_nCurrentMouseX = g_nMouseX + g_nJoystickInertia;
				if( g_nMouseX > s_nCurrentMouseX + g_nJoystickInertia )
					s_nCurrentMouseX = g_nMouseX - g_nJoystickInertia;
				if( g_nMouseY < s_nCurrentMouseY - g_nJoystickInertia )
					s_nCurrentMouseY = g_nMouseY + g_nJoystickInertia;
				if( g_nMouseY > s_nCurrentMouseY + g_nJoystickInertia )
					s_nCurrentMouseY = g_nMouseY - g_nJoystickInertia;

				s_nMouseJoy = UpdateCurrentMouse();
			}
		}
	}
} /* #OF# UpdateMouseState */

/*========================================================
Function : ResetMouse
=========================================================*/
/* #FN#
   Cleans up after the device emulation with using the mouse */
void
/* #AS#
   Nothing */
ResetMouse( BOOL bCenter )
{
	if( bCenter )
	{
		s_nCurrentMouseX = g_nMouseX = (g_rcViewBounds.right - g_rcViewBounds.left) / 2;
		s_nCurrentMouseY = g_nMouseY = (g_rcViewBounds.bottom - g_rcViewBounds.top) / 2;

		SetCursorPos( g_rcViewBounds.left + g_nMouseX, g_rcViewBounds.top + g_nMouseY );

		s_anPot[ 0 ][ 0 ] = s_anPot[ 0 ][ 1 ] = s_nJoy5200Center;
		s_anPot[ 1 ][ 0 ] = s_anPot[ 1 ][ 1 ] = s_nJoy5200Center;
	}
	/* This is necessary because the joystick cursor moves across
	   the screen after disabling mouse capture option (F12) */
	s_nMouseJoy = 0xff;
	/* There were some problems with light pen cursor, fixed */
	g_nMouseButtons = 0;
} /* #OF# ResetMouse */

/*========================================================
Function : ToggleMouseCapture
=========================================================*/
/* #FN#
   Toggles between the mouse capture mode on and off */
void
/* #AS#
   Nothing */
ToggleMouseCapture( void )
{
	if( g_ulMiscStates & MS_MOUSE_CAPTURED )
	{
		ShowMousePointer( TRUE );

		g_ulMiscStates &= ~MS_MOUSE_CAPTURED;
		ClipCursor( NULL );

		/* Disable emulation in antic.c */
		light_pen_enabled = FALSE;
		/* Clean up the mouse emulation settings */
		ResetMouse( FALSE );
	}
	else
	{
		g_ulMiscStates |= MS_MOUSE_CAPTURED;
		ClipCursor( &g_rcViewBounds );

		if( MOUSE_PEN == g_nMouseMode )
			/* Enable emulation in antic.c */
			light_pen_enabled = TRUE;

		ShowMousePointer( TRUE );
		/* The mouse will be hidden by UpdateMouseState function */
		g_nShowCursor = 1;

		/* Initialise the mouse emulation settings */
		ResetMouse( MOUSE_JOY == g_nMouseMode &&
					Atari5200 == machine );
	}
	WriteRegDWORD( NULL, REG_MISC_STATES, g_ulMiscStates );
	WriteRegDWORD( NULL, REG_MOUSE_MODE, g_nMouseMode );
} /* #OF# ToggleMouseCapture */

/*========================================================
Function : Atari_CONSOL
=========================================================*/
/* #FN#
   This function is called by the Atari800 kernel */
int
/* #AS#
   The consol keys state */
Atari_CONSOL( void )
{
	return s_nConsol;
} /* #OF# Atari_CONSOL */

/*========================================================
Function : Atari_Set_CONSOL
=========================================================*/
/* #FN#
   Sets a consol keys status */
static
void
/* #AS#
   Nothing */
Atari_Set_CONSOL( int nNewConsol )
{
	switch( nNewConsol )
	{
		case AKEY_OPTIONDWN:	/* OPTION */
			s_nConsol &= 0x03;
			break;

		case AKEY_SELECTDWN:	/* SELECT */
			s_nConsol &= 0x05;
			break;

		case AKEY_STARTDWN:		/* START */
			s_nConsol &= 0x06;
			break;

		case AKEY_OPTIONUP:
			s_nConsol |= 0x04;
			break;

		case AKEY_SELECTUP:
			s_nConsol |= 0x02;
			break;

		case AKEY_STARTUP:
			s_nConsol |= 0x01;
			break;
	}
} /* #OF# Atari_Set_CONSOL */

/*========================================================
Function : ServeKBJoystickDown
=========================================================*/
/* #FN#
   Handles a joystick key down event */
static
BOOL
/* #AS#
   TRUE for ending the key down event handling, otherwise FALSE */
ServeKBJoystickDown( WPARAM wp, LPARAM lp )
{
	int nIDevice;
	register int i, j;

	/* Translate extended codes */
	if( lp & 0x1000000/*(1 << 24)*/ )
	{
		wp = g_anExtendedKeyTab[ wp ];
	}
	else if( wp == VK_SHIFT && lp & 0x100000 )
		wp = VK_RSHIFT;

	for( i = 0; i < MAX_ATARI_JOYPORTS; i++ )
	{
		/* Check a keyboard joystick */
		if( g_anSelectedInputDevice[ i ] < 0 )
		{
			nIDevice = g_anSelectedInputDevice[ i ] + NUM_KBJOY_DEVICES;
			if( wp == g_anKeyset[ nIDevice ][ KEYSET_FIRE ] )
			{
				s_aunKeytrig[ nIDevice ] = 0;
				return TRUE;
			}
			else
			{
				for( j = 0; j < NUM_KBJOY_KEYS - 1/* The last is KEYSET_FIRE */; j++ )
					if( wp == g_anKeyset[ nIDevice ][ j ] )
					{
						s_aunKeystick[ nIDevice ] &= s_anStickTable[ j ];
						return TRUE;
					}
			}
		}
	}
	return FALSE;
} /* #OF# ServeKBJoystickDown */

/*========================================================
Function : ServeKBJoystickUp
=========================================================*/
/* #FN#
   Handles a joystick key up event */
static
BOOL
/* #AS#
   TRUE for ending the key up event handling, otherwise FALSE */
ServeKBJoystickUp( WPARAM wp, LPARAM lp )
{
	int nIDevice;
	register int i, j;

	/* Translate extended codes */
	if( lp & 0x1000000/*(1 << 24)*/ )
	{
		wp = g_anExtendedKeyTab[ wp ];
	}
	else if( wp == VK_SHIFT && lp & 0x100000 )
		wp = VK_RSHIFT;

	for( i = 0; i < MAX_ATARI_JOYPORTS; i++ )
	{
		/* Check a keyboard joystick */
		if( g_anSelectedInputDevice[ i ] < 0 )
		{
			nIDevice = g_anSelectedInputDevice[ i ] + NUM_KBJOY_DEVICES;
			if( wp == g_anKeyset[ nIDevice ][ KEYSET_FIRE ] )
			{
				s_aunKeytrig[ nIDevice ] = 1;
				return TRUE;
			}
			else if( g_ulMiscStates & MS_STICK_RELEASE )
			{
				for( j = 0; j < NUM_KBJOY_KEYS - 1/* The last is KEYSET_FIRE */; j++ )
					if( wp == g_anKeyset[ nIDevice ][ j ] )
					{
						/* Turn off bits for this key */
						s_aunKeystick[ nIDevice ] |= (~s_anStickTable[ j ]) & 0x0f;
						return TRUE;
					}
			}
		}
	}
	return FALSE;
} /* #OF# ServeKBJoystickUp */

/*========================================================
Function : ToggleKeyDown
=========================================================*/
/* #FN#
   Handles a toggling key down event */
BOOL
/* #AS#
   TRUE for ending key down event handling, FALSE for continue */
ToggleKeyDown( WPARAM wp, LPARAM lp )
{
	if( g_bKBJoystick && ServeKBJoystickDown( wp, lp ) )
		return TRUE; /* End the key down event handling */

	switch( wp )
	{
		case VK_SHIFT:
			SHIFT_KEY = TRUE;
			return TRUE;

		case VK_CONTROL:
			CTRL_KEY = TRUE;
			return TRUE;

		/* Console keys */
		case VK_F2:
			Atari_Set_CONSOL( AKEY_OPTIONDWN );
			break;

		case VK_F3:
			Atari_Set_CONSOL( AKEY_SELECTDWN );
			break;

		case VK_F4:
			Atari_Set_CONSOL( AKEY_STARTDWN );
			break;
	}
	return FALSE;
} /* #OF# ToggleKeyDown */

/*========================================================
Function : ToggleKeyUp
=========================================================*/
/* #FN#
   Handles a toggling key up event */
BOOL
/* #AS#
   TRUE for ending key up event handling, FALSE for continue */
ToggleKeyUp( WPARAM wp, LPARAM lp )
{
	if( g_bKBJoystick && ServeKBJoystickUp( wp, lp ) )
		return TRUE; /* End the key up event handling */

	switch( wp )
	{
		case VK_SHIFT:
			SHIFT_KEY = FALSE;
			return TRUE;

		case VK_CONTROL:
			CTRL_KEY = FALSE;
			return TRUE;

		/* Console keys */
		case VK_F2:
			Atari_Set_CONSOL( AKEY_OPTIONUP );
			break;

		case VK_F3:
			Atari_Set_CONSOL( AKEY_SELECTUP );
			break;

		case VK_F4:
			Atari_Set_CONSOL( AKEY_STARTUP );
			break;
	}
	return FALSE;
} /* #OF# ToggleKeyUp */

/*========================================================
Function : GetDIErrorString
=========================================================*/
/* #FN#
   Outputs a debug string to debugger */
static
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
GetDIErrorString( HRESULT hResult, LPSTR lpszErrorBuff, DWORD cchError )
{
    DWORD cLen;
    LPSTR lpszError;
    TCHAR szMsg[ 256 ];

    /* Check parameters */
    if( !lpszErrorBuff || !cchError )
    {
        /* Error, invalid parameters */
        return FALSE;
    }

    switch( hResult )
    {
		/* The operation completed successfully. This value is equal to the S_OK standard COM return value */
		case DI_OK:
			lpszError = TEXT("DI_OK");
			break;

#ifdef WIN_TRANSLATE_ERRORS

		/* The device buffer overflowed and some input was lost. This value is equal to the S_FALSE standard COM return value */
		case DI_BUFFEROVERFLOW:
			lpszError = TEXT("DI_BUFFEROVERFLOW");
			break;

		/* The parameters of the effect were successfully updated, but the effect could not be downloaded because the associated
		   device was not acquired in exclusive mode */
		case DI_DOWNLOADSKIPPED:
			lpszError = TEXT("DI_DOWNLOADSKIPPED");
			break;

		/* The effect was stopped, the parameters were updated, and the effect was restarted */
		case DI_EFFECTRESTARTED:
			lpszError = TEXT("DI_EFFECTRESTARTED");
			break;

		/* The operation had no effect. This value is equal to the S_FALSE standard COM return value */
//		case DI_NOEFFECT:
//			lpszError = TEXT("DI_NOEFFECT");
//			break;

		/* The device exists but is not currently attached. This value is equal to the S_FALSE standard COM return value */
//		case DI_NOTATTACHED:
//			lpszError = TEXT("DI_NOTATTACHED");
//			break;

		/* The device is a polled device. As a result, device buffering will not collect any data and event notifications will not be signaled until the IDirectInputDevice2::Poll method is called */
		case DI_POLLEDDEVICE:
			lpszError = TEXT("DI_POLLEDDEVICE");
			break;

		/* The change in device properties had no effect. This value is equal to the S_FALSE standard COM return value */
//		case DI_PROPNOEFFECT:
//			lpszError = TEXT("DI_PROPNOEFFECT");
//			break;

		/* The parameters of the effect were successfully updated, but some of them were beyond the capabilities of the device and were truncated to the nearest supported value */
		case DI_TRUNCATED:
			lpszError = TEXT("DI_TRUNCATED");
			break;

		/* Equal to DI_EFFECTRESTARTED | DI_TRUNCATED */
		case DI_TRUNCATEDANDRESTARTED:
			lpszError = TEXT("DI_TRUNCATEDANDRESTARTED");
			break;

		/* The operation cannot be performed while the device is acquired */
		case DIERR_ACQUIRED:
			lpszError = TEXT("DIERR_ACQUIRED");
			break;

		/* This object is already initialized */
		case DIERR_ALREADYINITIALIZED:
			lpszError = TEXT("DIERR_ALREADYINITIALIZED");
			break;

		/* The object could not be created due to an incompatible driver version or mismatched or incomplete driver components */
		case DIERR_BADDRIVERVER:
			lpszError = TEXT("DIERR_BADDRIVERVER");
			break;

		/* The application was written for an unsupported prerelease version of DirectInput */
		case DIERR_BETADIRECTINPUTVERSION:
			lpszError = TEXT("DIERR_BETADIRECTINPUTVERSION");
			break;

		/* The device is full */
		case DIERR_DEVICEFULL:
			lpszError = TEXT("DIERR_DEVICEFULL");
			break;

		/* The device or device instance is not registered with DirectInput. This value is equal to the REGDB_E_CLASSNOTREG standard COM return value */
		case DIERR_DEVICENOTREG:
			lpszError = TEXT("DIERR_DEVICENOTREG");
			break;

		/* The parameters were updated in memory but were not downloaded to the device because the device does not support updating an effect while it is still playing */
		case DIERR_EFFECTPLAYING:
			lpszError = TEXT("DIERR_EFFECTPLAYING");
			break;

		/* The device cannot be reinitialized because there are still effects attached to it */
		case DIERR_HASEFFECTS:
			lpszError = TEXT("DIERR_HASEFFECTS");
			break;

		/* An undetermined error occurred inside the DirectInput subsystem. This value is equal to the E_FAIL standard COM return value */
		case DIERR_GENERIC:
			lpszError = TEXT("DIERR_GENERIC");
			break;

		/* The device already has an event notification associated with it. This value is equal to the E_ACCESSDENIED standard COM return value */
		case DIERR_HANDLEEXISTS:
			lpszError = TEXT("DIERR_HANDLEEXISTS");
			break;

		/* The effect could not be downloaded because essential information is missing. For example, no axes have been associated with the effect, or no type-specific information has been supplied */
		case DIERR_INCOMPLETEEFFECT:
			lpszError = TEXT("DIERR_INCOMPLETEEFFECT");
			break;

		/* Access to the input device has been lost. It must be reacquired */
		case DIERR_INPUTLOST:
			lpszError = TEXT("DIERR_INPUTLOST");
			break;

		/* An invalid parameter was passed to the returning function, or the object was not in a state that permitted the function to be called. This value is equal to the E_INVALIDARG standard COM return value */
		case DIERR_INVALIDPARAM:
			lpszError = TEXT("DIERR_INVALIDPARAM");
			break;

		/* Not all the requested information fitted into the buffe */
		case DIERR_MOREDATA:
			lpszError = TEXT("DIERR_MOREDATA");
			break;

		/* This object does not support aggregation */
		case DIERR_NOAGGREGATION:
			lpszError = TEXT("DIERR_NOAGGREGATION");
			break;

		/* The specified interface is not supported by the object. This value is equal to the E_NOINTERFACE standard COM return value */
		case DIERR_NOINTERFACE:
			lpszError = TEXT("DIERR_NOINTERFACE");
			break;

		/* The operation cannot be performed unless the device is acquired */
		case DIERR_NOTACQUIRED:
			lpszError = TEXT("DIERR_NOTACQUIRED");
			break;

		/* The device is not buffered. Set the DIPROP_BUFFERSIZE property to enable bufferin */
		case DIERR_NOTBUFFERED:
			lpszError = TEXT("DIERR_NOTBUFFERED");
			break;

		/* The effect is not downloade */
		case DIERR_NOTDOWNLOADED:
			lpszError = TEXT("DIERR_NOTDOWNLOADED");
			break;

		/* The operation cannot be performed unless the device is acquired in DISCL_EXCLUSIVE mode */
		case DIERR_NOTEXCLUSIVEACQUIRED:
			lpszError = TEXT("DIERR_NOTEXCLUSIVEACQUIRED");
			break;

		/* The requested object does not exist */
		case DIERR_NOTFOUND:
			lpszError = TEXT("DIERR_NOTFOUND");
			break;

		/* This object has not been initialized */
		case DIERR_NOTINITIALIZED:
			lpszError = TEXT("DIERR_NOTINITIALIZED");
			break;

		/* The requested object does not exist */
//		case DIERR_OBJECTNOTFOUND:
//			lpszError = TEXT("DIERR_OBJECTNOTFOUND");
//			break;

		/* The application requires a newer version of DirectInput */
		case DIERR_OLDDIRECTINPUTVERSION:
			lpszError = TEXT("DIERR_OLDDIRECTINPUTVERSION");
			break;

		/* Another application has a higher priority level, preventing this call from succeeding. This value is equal to the E_ACCESSDENIED standard COM return value.
		   This error can be returned when an application has only foreground access to a device but is attempting to acquire the device while in the background */
//		case DIERR_OTHERAPPHASPRIO:
//			lpszError = TEXT("DIERR_OTHERAPPHASPRIO");
//			break;

		/* The DirectInput subsystem couldn't allocate sufficient memory to complete the call. This value is equal to the E_OUTOFMEMORY standard COM return value */
		case DIERR_OUTOFMEMORY:
			lpszError = TEXT("DIERR_OUTOFMEMORY");
			break;

		/* The specified property cannot be changed. This value is equal to the E_ACCESSDENIED standard COM return value */
//		case DIERR_READONLY:
//			lpszError = TEXT("DIERR_READONLY");
//			break;

		/* The function called is not supported at this time. This value is equal to the E_NOTIMPL standard COM return value */
		case DIERR_UNSUPPORTED:
			lpszError = TEXT("DIERR_UNSUPPORTED");
			break;

		/* Data is not yet availabl */
		case E_PENDING:
			lpszError = TEXT("E_PENDING");
			break;

#endif /*WIN_TRANSLATE_ERRORS*/

		/* Unknown DI Error */
		default:
			wsprintf( szMsg, "Error #%ld", (DWORD)(hResult & 0x0000FFFFL) );
			lpszError = szMsg;
			break;
	}
    /* Copy DI Error string to buff */
    cLen = strlen( lpszError );
    if( cLen >= cchError )
    {
        cLen = cchError - 1;
    }
	if( cLen )
    {
        strncpy( lpszErrorBuff, lpszError, cLen );
        lpszErrorBuff[ cLen ] = 0;
    }
    return TRUE;
} /* #OF# GetDIErrorString */
