/****************************************************************************
File    : PaletteDlg.cpp
/*
@(#) #SY# Atari800Win
@(#) #IS# CPaletteDlg implementation file
@(#) #BY# Tomasz Szymankowski
@(#) #LM# 13.05.2000
*/

/*
Copyright (c) 2000 Tomasz Szymankowski

This program is free software; you can redistribute it and/or modify it under the terms 
of the GNU General Public License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later version. This program is 
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details. You should have received a copy of the GNU
General Public License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "StdAfx.h"
#include "Atari800Win.h"
#include "Helpers.h"
#include "PaletteDlg.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#define PAL_ENTRIES_NO	256
#define PAL_BARS_NO		2
#define BAR_ENTRIES_NO	PAL_ENTRIES_NO / PAL_BARS_NO

/* Note: afxData.cxBorder and afxData.cyBorder aren't used anymore */
#define CX_BORDER   2
#define CY_BORDER   2

/////////////////////////////////////////////////////////////////////////////
// CPaletteDlg dialog


CPaletteDlg::
CPaletteDlg( CWnd *pParent /*=NULL*/ )
: CDialog( (g_ulScreenMode & SM_MODE_FULL ? IDD_PALETTE_SMALL : CPaletteDlg::IDD), pParent)
{
	//{{AFX_DATA_INIT(CPaletteDlg)
		// NOTE: the ClassWizard will add member initialization here
	//}}AFX_DATA_INIT
}


void CPaletteDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CPaletteDlg)
		// NOTE: the ClassWizard will add DDX and DDV calls here
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CPaletteDlg, CDialog)
	//{{AFX_MSG_MAP(CPaletteDlg)
	ON_BN_CLICKED(IDC_PALETTE_BROWSE, OnBrowse)
	ON_EN_KILLFOCUS(IDC_PALETTE_EDIT, OnKillfocusEdit)
	ON_BN_CLICKED(IDC_PALETTE_LOAD, OnLoad)
	ON_WM_PAINT()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


/////////////////////////////////////////////////////////////////////////////
// CPaletteDlg message handlers

BOOL
CPaletteDlg::
OnInitDialog()
{
	CButton *pButton = NULL;

	CDialog::OnInitDialog();
	
	m_bSmallMode   = g_ulScreenMode & SM_MODE_FULL;
	m_ulMiscStates = g_ulMiscStates;
	strncpy( m_szPaletteFile, g_szPaletteFile, MAX_PATH );
	
	SetDlgItemText( IDC_PALETTE_EDIT, m_szPaletteFile );

	pButton = (CButton *)GetDlgItem( IDC_PALETTE_LOAD );
	ASSERT(pButton);
	pButton->SetCheck( (m_ulMiscStates & MS_USE_PALETTE) != 0 );

	if( !m_bSmallMode )
	{
		CWnd *pStatic = NULL;

		if( !(m_ulMiscStates & MS_USE_PALETTE) &&
			!read_palette( m_szPaletteFile ) )
		{
			LoadString( NULL, IDS_READ_PALETTE_ERROR, g_tszErrorString, LOADSTRING_STRING_SIZE );
			AfxGetMainWnd()->MessageBox( g_tszErrorString, "Atari800Win", MB_ICONEXCLAMATION | MB_OK );
		}
		pStatic = GetDlgItem( IDC_PALETTE_STATIC1 );
		ASSERT(pStatic);
		pStatic->GetWindowRect( m_rcPalBar1 ); ScreenToClient( m_rcPalBar1 );
		pStatic->SetWindowPos( NULL, 0, 0, (g_bLargeFonts ? 256 : 128) + 2 * CX_BORDER, m_rcPalBar1.Height(),
			SWP_NOMOVE | SWP_NOZORDER | SWP_NOACTIVATE );
		m_rcPalBar1.DeflateRect( CX_BORDER, CY_BORDER );

		pStatic = GetDlgItem( IDC_PALETTE_STATIC2 );
		ASSERT(pStatic);
		pStatic->GetWindowRect( m_rcPalBar2 ); ScreenToClient( m_rcPalBar2 );
		pStatic->SetWindowPos( NULL, 0, 0, (g_bLargeFonts ? 256 : 128) + 2 * CX_BORDER, m_rcPalBar2.Height(),
			SWP_NOMOVE | SWP_NOZORDER | SWP_NOACTIVATE );
		m_rcPalBar2.DeflateRect( CX_BORDER, CY_BORDER );
	}
	return TRUE;  // return TRUE unless you set the focus to a control
	              // EXCEPTION: OCX Property Pages should return FALSE
}

void
CPaletteDlg::
OnLoad()
{
	CButton *pButton = (CButton *)GetDlgItem( IDC_PALETTE_LOAD );
	ASSERT(pButton);

	if( pButton->GetCheck() )
		m_ulMiscStates |= MS_USE_PALETTE;
	else
		m_ulMiscStates &= ~MS_USE_PALETTE;
}

/*========================================================
Method   : CPaletteDlg::OnBrowsePalette
=========================================================*/
/* #FN#
   Allows to select palette file using FileDialog window */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnBrowse()
{
	char szPaletteOld[ MAX_PATH ];

	strcpy( szPaletteOld, m_szPaletteFile );

	if( PickFileName( TRUE, m_szPaletteFile, "Load Atari palette file",
					  PF_ACT_FILTER, "act", PF_LOAD_FLAGS ) &&
		*m_szPaletteFile != '\0' &&
		_stricmp( szPaletteOld, m_szPaletteFile ) != 0 )
	{
		SetDlgItemText( IDC_PALETTE_EDIT, m_szPaletteFile );
		ReadPaletteFile( m_szPaletteFile, FALSE ); /* Read palette from file */
	}
}

/*========================================================
Method   : CPaletteDlg::ReadPaletteFile
=========================================================*/
/* #FN#
   Reads an external palette from file */
BOOL
/* #AS#
   Nothing */
CPaletteDlg::
ReadPaletteFile( LPSTR pszPaletteFile,
				 BOOL  bCheckIfExists /*=TRUE*/ )
{
	BOOL bResult = FALSE;

	if( *pszPaletteFile != '\0' )
	{
		CFileStatus fsStatus;
		CFile       fDisk;

		if( !bCheckIfExists || fDisk.GetStatus( pszPaletteFile, fsStatus ) )
		{
			if( !read_palette( pszPaletteFile ) )
			{
				LoadString( NULL, IDS_READ_PALETTE_ERROR, g_tszErrorString, LOADSTRING_STRING_SIZE );
				MessageBox( g_tszErrorString, "Atari800Win", MB_ICONEXCLAMATION | MB_OK );
			}
			else
				bResult = TRUE;

			/* Palette bar is drawn in windowed modes only */
			if( !m_bSmallMode )
				/* This will make the changes visible */
				Invalidate();
		}
	}
	return bResult;
}

/*========================================================
Method   : CPaletteDlg::OnKillfocusEdit
=========================================================*/
/* #FN#
   The framework calls this function before edit losing the input focus */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnKillfocusEdit()
{
	char szPaletteOld[ MAX_PATH ];

	strcpy( szPaletteOld, m_szPaletteFile );
	GetDlgItemText( IDC_PALETTE_EDIT, m_szPaletteFile, MAX_PATH );

	if( _stricmp( szPaletteOld, m_szPaletteFile ) != 0 )
		ReadPaletteFile( m_szPaletteFile ); /* Read palette from file */
}

void
CPaletteDlg::
OnOK()
{
	if( m_ulMiscStates != g_ulMiscStates )
	{
		g_ulMiscStates = m_ulMiscStates;
		WriteRegDWORD( NULL, REG_MISC_STATES, g_ulMiscStates );
	}	
	/* Get the external palette name from edit (OnKillfocusEdit()
	   didn't work when user had pressed 'Enter' or 'Alt-O' */
	GetDlgItemText( IDC_PALETTE_EDIT, m_szPaletteFile, MAX_PATH );
	/* Windows NT doesn't like empty strings in Registry */
	if( *m_szPaletteFile == '\0' )
		strcpy( m_szPaletteFile, DEFAULT_ACT );

	if( _stricmp( g_szPaletteFile, m_szPaletteFile ) != 0 )
	{
		strncpy( g_szPaletteFile, m_szPaletteFile, MAX_PATH );
		WriteRegString( NULL, REG_EXT_PALETTE, g_szPaletteFile );

		if( ReadPaletteFile( g_szPaletteFile, FALSE ) ) /* Read palette from file */
		{
			/* Apply the changes to the main window */
			UseAtariPalette();
			if( m_bSmallMode )
				InitialiseScreen( TRUE ); /* Force initialization */
		}
	}
	CDialog::OnOK();
}

void
CPaletteDlg::
OnCancel()
{
	if( _stricmp( g_szPaletteFile, m_szPaletteFile ) != 0 )
	{
		int	nRGB;

		strncpy( m_szPaletteFile, g_szPaletteFile, MAX_PATH );
		SetDlgItemText( IDC_PALETTE_EDIT, g_szPaletteFile );
		/* Restore old palette */
		for( int i = 0; i < PAL_ENTRIES_NO; i++ )
		{
			nRGB  = (g_pe[ i ].peRed   & 0xff) << 16;
			nRGB |= (g_pe[ i ].peGreen & 0xff) <<  8;
			nRGB |= (g_pe[ i ].peBlue  & 0xff);

			colortable[ i ] = nRGB;
		}
	}
	CDialog::OnCancel();
}

void
CPaletteDlg::
OnPaint() 
{
	CPaintDC dc( this ); /* Device context for painting */

	if( !m_bSmallMode )
		PaintPalette( &dc );
}

BOOL
CPaletteDlg::
PaintPalette( CDC *pDC )
{
	CPalette     palPalette;
	CPalette    *pOldPalette;
	CBrush      *pBrush;
	CBrush      *pOldBrush;
	int          nWidth = (g_bLargeFonts ? 2 : 1);
	int          nHeight;
	int	         nRGB;
	HANDLE       hLogPal;    /* Handle to a logical palette */
	LPLOGPALETTE lpPal;      /* Pointer to a logical palette */
	BOOL         bResult;

	/* Allocate memory block for logical palette */
	hLogPal = ::GlobalAlloc( GHND/*LPTR*/, sizeof(LOGPALETTE) + sizeof(PALETTEENTRY) * PAL_ENTRIES_NO );
	/* If not enough memory, clean up and return NULL */
	if( !hLogPal )
		return FALSE;

	lpPal = (LPLOGPALETTE)::GlobalLock( (HGLOBAL)hLogPal );
	/* Init palette header info */
	lpPal->palVersion    = 0x300;
	lpPal->palNumEntries = (WORD)PAL_ENTRIES_NO;

	/* Copy entries */
	for( int i = 0; i < PAL_ENTRIES_NO; i++ )
	{
		nRGB = colortable[ i ];
		
		lpPal->palPalEntry[ i ].peRed   = (nRGB & 0x00ff0000) >> 16;
		lpPal->palPalEntry[ i ].peGreen = (nRGB & 0x0000ff00) >> 8;
		lpPal->palPalEntry[ i ].peBlue  = nRGB & 0x000000ff;
		lpPal->palPalEntry[ i ].peFlags = 0;
	}
	/* Create palette */
	bResult = (BOOL)(palPalette.CreatePalette( lpPal ));

	::GlobalUnlock( (HGLOBAL) hLogPal );
	::GlobalFree( (HGLOBAL) hLogPal );

	if( bResult )
	{
		CRect *aprcPalBars[ PAL_BARS_NO ] = { &m_rcPalBar1, &m_rcPalBar2 };
		CRect *pPalBar;

		/* Select and realize palette */
		pOldPalette = pDC->SelectPalette( &palPalette, TRUE );
		pDC->RealizePalette();

		for( int i = 0; i < PAL_BARS_NO; i++ ) /* Two Palette bars */
		{
			pPalBar = aprcPalBars[ i ];

			nHeight = pPalBar->Height();
//			pDC->IntersectClipRect( pPalBar );

			/* Draw each stripe */
			for( int j = 0; j < BAR_ENTRIES_NO; j++ )
			{
				pBrush = new CBrush;
				/* A palette-relative pBrush is used for drawing */
				if( TRUE == pBrush->CreateSolidBrush( PALETTEINDEX( j + (i * BAR_ENTRIES_NO) ) ) )
				{
					/* Setup pBrush */
					pOldBrush = pDC->SelectObject( pBrush );
					pBrush->UnrealizeObject();

					/* Draw stripe */
					pDC->PatBlt( pPalBar->left + j * nWidth, pPalBar->top, nWidth, nHeight, PATCOPY );
					pDC->SelectObject( pOldBrush );
				}
				delete pBrush;
			}
		}
		/* Back to old palette */
		pDC->SelectPalette( pOldPalette, TRUE );
		pDC->RealizePalette();
	}
	return bResult;
}
