/****************************************************************************
File    : DriveDlg.cpp
/*
@(#) #SY# Atari800Win
@(#) #IS# CDriveDlg implementation file
@(#) #BY# Richard Lawrence, Tomasz Szymankowski
@(#) #LM# 03.07.2000
*/

/*
Copyright (c) 1998 Richard Lawrence

This program is free software; you can redistribute it and/or modify it under the terms 
of the GNU General Public License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later version. This program is 
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details. You should have received a copy of the GNU
General Public License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "StdAfx.h"
#include "Atari800Win.h"
#include "Helpers.h"
#include "NewDiskImageDlg.h"
#include "DriveDlg.h"

//#include "core.h"			// AtariWin core

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
// Static objects

static UnitStatus s_ausUnitStatus[ MAX_DRIVES ];
static char       s_acSioFileName[ MAX_DRIVES ][ MAX_PATH ];

static int s_nDlgEditID[ MAX_DRIVES ] =
{
	IDC_DRIVE_EDITDRIVE1,
	IDC_DRIVE_EDITDRIVE2,
	IDC_DRIVE_EDITDRIVE3,
	IDC_DRIVE_EDITDRIVE4,
	IDC_DRIVE_EDITDRIVE5,
	IDC_DRIVE_EDITDRIVE6,
	IDC_DRIVE_EDITDRIVE7,
	IDC_DRIVE_EDITDRIVE8
};

static int s_nDlgComboID[ MAX_DRIVES ] =
{
	IDC_DRIVE_COMBO1,
	IDC_DRIVE_COMBO2,
	IDC_DRIVE_COMBO3,
	IDC_DRIVE_COMBO4,
	IDC_DRIVE_COMBO5,
	IDC_DRIVE_COMBO6,
	IDC_DRIVE_COMBO7,
	IDC_DRIVE_COMBO8
};

static int s_nDlgButtonID[ MAX_DRIVES ] =
{
	IDC_DRIVE_DRIVE1,
	IDC_DRIVE_DRIVE2,
	IDC_DRIVE_DRIVE3,
	IDC_DRIVE_DRIVE4,
	IDC_DRIVE_DRIVE5,
	IDC_DRIVE_DRIVE6,
	IDC_DRIVE_DRIVE7,
	IDC_DRIVE_DRIVE8
};


/////////////////////////////////////////////////////////////////////////////
// CDriveDlg dialog

BEGIN_MESSAGE_MAP(CDriveDlg, CDialog)
	//{{AFX_MSG_MAP(CDriveDlg)
	ON_BN_CLICKED(IDC_DRIVE_CLEARDRIVES, OnClearDrives)
	ON_BN_CLICKED(IDC_DRIVE_NEWIMAGE, OnDriveNewImage)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

CDriveDlg::
CDriveDlg( CWnd *pParent /*=NULL*/ )
	: CDialog( CDriveDlg::IDD, pParent )
{
	//{{AFX_DATA_INIT(CDriveDlg)
		// NOTE: the ClassWizard will add member initialization here
	//}}AFX_DATA_INIT
}


/////////////////////////////////////////////////////////////////////////////
// CDriveDlg message handlers

/* We set the drive edit boxes to the the appropriate values, and set
   the status ComboBox */
void
CDriveDlg::
SetDlgState( void )
{
	CComboBox *pComboBox = NULL;
	CEdit     *pEdit     = NULL;

	for( int i = 0; i < MAX_DRIVES; i++ )
	{
		/* The s_acSioFileName table is not really needed */
		SetDlgItemText( s_nDlgEditID[ i ], s_acSioFileName[ i ] );

		pComboBox = (CComboBox *)GetDlgItem( s_nDlgComboID[ i ] );
		ASSERT(pComboBox);
		pEdit = (CEdit *)GetDlgItem( s_nDlgEditID[ i ] );
		ASSERT(pEdit);
		
		switch( s_ausUnitStatus[ i ] )
		{
			case Off:
				pComboBox->SetCurSel( 0 );
				pEdit->SetReadOnly();
				break;

			case NoDisk:
				pComboBox->SetCurSel( 3 );
				pEdit->SetReadOnly( FALSE );
				break;

			case ReadOnly:
				pComboBox->SetCurSel( 2 );
				pEdit->SetReadOnly( FALSE );
				break;

			case ReadWrite:
				pComboBox->SetCurSel( 1 );
				pEdit->SetReadOnly( FALSE );
				break;
		}
	}
} /* #OF# CDriveDlg::SetDlgState */

BOOL
CDriveDlg::
OnInitDialog()
{
	CDialog::OnInitDialog();

	for( int i = 0; i < MAX_DRIVES; i++ )
	{
		strncpy( s_acSioFileName[ i ], sio_filename[ i ], MAX_PATH );
		s_ausUnitStatus[ i ] = drive_status[ i ];
	}
	SetDlgState();

	return TRUE;  // return TRUE unless you set the focus to a control
	              // EXCEPTION: OCX Property Pages should return FALSE
} /* #OF# CDriveDlg::OnInitDialog */

UnitStatus
CDriveDlg::
CheckSelectedDisk( LPSTR pszDiskName,
				   UINT *pInfo /*= NULL*/ )
{
	UnitStatus usResult = ReadOnly;
	UINT unInfo = CSD_READERROR;

	if( strcmp( pszDiskName, "Off" ) == 0 )
	{
		usResult = Off;
		unInfo = CSD_DRIVEOFF;
	}
	else
	{
		CFileStatus fsStatus;
		CFile       cfDisk;
		/* That's a static method, don't panic */
		if( cfDisk.GetStatus( pszDiskName, fsStatus ) )
		{
			if( !IsCompressedFile( pszDiskName ) )
			{
				if( !(fsStatus.m_attribute & CFile::readOnly) )
				{
					int fd = open( pszDiskName, O_RDONLY | O_BINARY, 0777 );
					/* If there is an ATR image check header's writeprotect member */
					if( fd >= 0 )
					{
						ATR_Header header;
						lseek( fd, 0L, SEEK_SET );
						if( (-1 != read( fd, &header, sizeof(ATR_Header) )) &&
							header.magic1 == MAGIC1 && header.magic2 == MAGIC2 )
						{
							if( !header.writeprotect )
							{
								usResult = ReadWrite;
								unInfo = CSD_ATRREADWRITE;
							}
							else
								unInfo = CSD_ATRREADONLY;
						}
						else
						{
							usResult = ReadWrite;
							unInfo = CSD_XFDREADWRITE;
						}
						close( fd );
					}
				}
				else
					unInfo = CSD_PCREADONLY;
			}
			else
				unInfo = CSD_COMPRESSED;
		}
		else
		{
			usResult = NoDisk;
			unInfo = CSD_NODISKIMAGE;
		}
	}
	if( pInfo )
		*pInfo = unInfo;

	return usResult;
} /* #OF# CDriveDlg::CheckSelectedDisk */

/* GetDiskImage will call up a file dialog, and if the user oks the file will copy
   it's name into the appropriate edit box. The drives are not unmounted/mounted 
   until the OK button is pushed, in case the user decides to cancel the entire 
   operation. So the edit boxes are being used as temporary buffers, basically */
BOOL
CDriveDlg::
GetDiskImage( int nDrive )
{
	char	szPrompt[ 64 ];
	CString	strDiskName = s_acSioFileName[ nDrive ];
	BOOL	bResult = FALSE;

	sprintf( szPrompt, "Select disk for drive %d", nDrive + 1 );

	if( PickFileName( TRUE, strDiskName, szPrompt,
					  PF_DSK_FILTER, NULL, PF_LOAD_FLAGS ) &&
		!strDiskName.IsEmpty() )
	{
		SetDlgItemText( s_nDlgEditID[ nDrive ], (LPSTR)(LPCSTR)strDiskName );
		strncpy( s_acSioFileName[ nDrive ], strDiskName, MAX_PATH );
		s_ausUnitStatus[ nDrive ] = CheckSelectedDisk( (LPSTR)(LPCSTR)strDiskName );

		bResult = TRUE;
	}
	return bResult;
} /* #OF# CDriveDlg::GetDiskImage */

BOOL
CDriveDlg::
DriveButton( int  nButtonID,
			 HWND hwndButton )
{
	int nDrive = -1;

	for( int i = 0; i < MAX_DRIVES; i++ )
	{
		if( nButtonID == s_nDlgButtonID[ i ] ) 
			nDrive = i;
	}
	if( nDrive == -1 ) /* It wasn't a drive button */
		return FALSE;

	if( GetDiskImage( nDrive ) )
		SetDlgState();

	return TRUE;
} /* #OF# CDriveDlg::DriveButton */

void
CDriveDlg::
StatusSelChange( int  nComboID,
				 HWND hwndCombo )
{
	CComboBox *pComboBox = NULL;
	UINT       unDiskInf =  0;
	int        nDrive    = -1;
	UnitStatus usRealMode;

	for( int i = 0; i < MAX_DRIVES; i++ )
	{
		if( nComboID == s_nDlgComboID[ i ] )
			nDrive = i;
	}
	if( nDrive == -1 )	/* It wasn't a drive button */
		return;

	usRealMode = CheckSelectedDisk( s_acSioFileName[ nDrive ], &unDiskInf );

	pComboBox = (CComboBox *)GetDlgItem( s_nDlgComboID[ nDrive ] );
	ASSERT(pComboBox);

	switch( pComboBox->GetCurSel() )
	{
		case 0:	/* Combo box indicates "Off" */
		{
			s_ausUnitStatus[ nDrive ] = Off;
			break;
		}
		case 1: /* Combo box indicates "Read/Write" */
		{
			if( usRealMode == ReadOnly )
			{
				switch( unDiskInf )
				{
					case CSD_COMPRESSED:
						MessageBox( "Cannot write to a compressed disk image, mode remains Read Only.", "Atari800Win", MB_ICONINFORMATION | MB_OK );
						break;
					case CSD_ATRREADONLY:
						if( IDYES == MessageBox( "Do you want to remove write-protection for the Atari disk image?", "Atari800Win", MB_ICONQUESTION | MB_YESNO ) )
						{
							int fd = open( s_acSioFileName[ nDrive ], O_WRONLY | O_BINARY, 0777 );
							if( fd >= 0 )
							{
								if( -1 != lseek( fd, 15L, SEEK_SET ) &&
									-1 != write( fd, "\000", 1 ) )
								{
									usRealMode = ReadWrite;
								}
								close( fd );
							}
						}
						break;
					case CSD_PCREADONLY:
						MessageBox( "Could not set to Read/Write because the file is marked Read Only.", "Atari800Win", MB_ICONINFORMATION | MB_OK );
						break;
					default:
						break;
				}
			}
			s_ausUnitStatus[ nDrive ] = usRealMode;
			break;
		}
		case 2:	/* Combo box indicates "Read Only" */
		{
			if( usRealMode == ReadWrite )
			{
				if( CSD_ATRREADWRITE == unDiskInf )
				{
					if( IDYES == MessageBox( "Do you want to enable write-protection for the Atari disk image?", "Atari800Win", MB_ICONQUESTION | MB_YESNO ) )
					{
						int fd = open( s_acSioFileName[ nDrive ], O_WRONLY | O_BINARY, 0777 );
						if( fd >= 0 )
						{
							if( -1 != lseek( fd, 15L, SEEK_SET ) &&
								-1 != write( fd, "\001", 1 ) )
							{
								usRealMode = ReadOnly;
							}
							close( fd );
						}
					}
				}
				else
					MessageBox( "Could not set write-protection because the file format doesn't support this.", "Atari800Win", MB_ICONINFORMATION | MB_OK );
			}
			s_ausUnitStatus[ nDrive ] = usRealMode;
			break;
		}
		case 3: /* Combo box indicates "No Disk" */
		{
			s_ausUnitStatus[ nDrive ] = NoDisk;
			break;
		}
		default:
			break;
	}
	SetDlgState();
} /* #OF# CDriveDlg::StatusSelChange */

BOOL
CDriveDlg::
KillfocusEditDrive( int  nEditID,
					HWND hwndEdit )
{
	char szDiskName[ MAX_PATH ];
	int  nDrive = -1;

	for( int i = 0; i < MAX_DRIVES; i++ )
	{
		if( nEditID == s_nDlgEditID[ i ] ) 
			nDrive = i;
	}
	if( nEditID == -1 ) /* It wasn't a drive edit */
		return FALSE;

	GetDlgItemText( s_nDlgEditID[ nDrive ], szDiskName, MAX_PATH );
	strcpy( s_acSioFileName[ nDrive ], szDiskName );
	s_ausUnitStatus[ nDrive ] = CheckSelectedDisk( szDiskName );

	SetDlgState();

	return TRUE;
} /* #OF# CDriveDlg::KillfocusEditDrive */

/*========================================================
Method   : CDriveDlg::OnClearDrives
=========================================================*/
/* #FN#
   The framework calls this member function when the user selects an item
   from a menu, when a child control sends a notification message, or when
   an accelerator keystroke is translated */
BOOL
/* #AS#
   Nonzero if the application processes this message; otherwise 0 */
CDriveDlg::
OnCommand( WPARAM wParam,
		   LPARAM lParam )
{
	/* This could also be handled by eight OnSelChanged plus eight ButtonClicked
	   individual procedures, but this seems a little more elegant */
	switch( HIWORD(wParam) )
	{
		case BN_CLICKED:
		{
			if( DriveButton( (int)LOWORD(wParam), (HWND)lParam ) )
				return TRUE;
			break;
		}
		case EN_KILLFOCUS:
		{
			if( KillfocusEditDrive( (int)LOWORD(wParam), (HWND)lParam ) )
				return TRUE;
			break;
		}
		case CBN_SELCHANGE:
		{
			StatusSelChange( (int)LOWORD(wParam), (HWND)lParam );
			return TRUE;
			break;
		}
		default:
			break;
	}
	return CDialog::OnCommand( wParam, lParam );
} /* #OF# CDriveDlg::OnCommand */

/*========================================================
Method   : CDriveDlg::OnClearDrives
=========================================================*/
/* #FN#
   Removes disks from all drives and turns them off */
void
/* #AS#
   Nothing */
CDriveDlg::
OnClearDrives()
{
	for( int i = 0; i < MAX_DRIVES; i++ )
	{
		s_ausUnitStatus[ i ] = Off;
		memset( s_acSioFileName[ i ], 0, MAX_PATH );
		strcpy( s_acSioFileName[ i ], "Off" );
	}
	SetDlgState();
} /* #OF# CDriveDlg::OnClearDrives */

/*========================================================
Method   : CDriveDlg::OnDriveNewImage
=========================================================*/
/* #FN#
   Displays New Disk Image dialog box */
void
/* #AS#
   Nothing */
CDriveDlg::
OnDriveNewImage()
{
	CNewDiskImageDlg dlgNewDiskImage;

	if( IDOK == dlgNewDiskImage.DoModal() &&
		/* Attach disk image to any drive if it was expected */
		dlgNewDiskImage.m_bAttachDrive )
	{
		strncpy( s_acSioFileName[ dlgNewDiskImage.m_nDriveNumber ], dlgNewDiskImage.m_szImageName, MAX_PATH );
		s_ausUnitStatus[ dlgNewDiskImage.m_nDriveNumber ] = CheckSelectedDisk( dlgNewDiskImage.m_szImageName );
	}
	SetDlgState();
} /* #OF# CDriveDlg::OnDriveNewImage */

/*========================================================
Method   : CDriveDlg::OnOK
=========================================================*/
/* #FN#
   Called when the user clicks the OK button */
void
/* #AS#
   Nothing */
CDriveDlg::
OnOK()
{
	char  szFileName[ MAX_PATH ];
	CWnd *pWnd           = GetFocus();
	int   nCtrlID        = pWnd ? pWnd->GetDlgCtrlID() : 0;
	BOOL  bDrivesChanged = FALSE;

	for( int i = 0; i < MAX_DRIVES; i++ )
	{
		/* Receive the edit contents again. A user could use 'Enter' or 'Alt-O'
		   and then all changes made in the last edited control would be lost */
		if( s_nDlgEditID[ i ] == nCtrlID )
			KillfocusEditDrive( nCtrlID, (HWND)NULL );
		
		if( s_ausUnitStatus[ i ] == Off )
		{
			strcpy( szFileName, "Off" );
		}
		else if( s_ausUnitStatus[ i ] == NoDisk )
		{
			strcpy( szFileName, "Empty" );
		}
		else
			strcpy( szFileName, s_acSioFileName[ i ] );

		/* The reason for doing this here instead of just doing it in
		   GetDiskImage is to preserve the ability to cancel the entire
		   drive operation with one click of the cancel button. */
		if( _stricmp( szFileName, sio_filename[ i ] ) != 0 )
		{
			SIO_Dismount( i + 1 );
			strcpy( sio_filename[ i ], szFileName );

			if( strcmp( szFileName, "Empty" ) != 0 &&
				strcmp( szFileName, "Off" ) != 0 )
			{
				if( !SIO_Mount( i + 1, szFileName ) )
				{
					/* There was an error with SIO mounting */
					strcpy( sio_filename[ i ], "Empty" );
					s_ausUnitStatus[ i ] = NoDisk;
				}
			}
			bDrivesChanged = TRUE;
		}
		if( s_ausUnitStatus[ i ] != drive_status[ i ] )
		{
			drive_status[ i ] = s_ausUnitStatus[ i ];
			bDrivesChanged = TRUE;
		}
	}
	if( bDrivesChanged )
		WriteRegDrives( NULL );

	CDialog::OnOK();
} /* #OF# CDriveDlg::OnOK */
