/****************************************************************************
File    : display_win.c
/*
@(#) #SY# Atari800Win
@(#) #IS# Display implementation for Win32 platforms
@(#) #BY# Richard Lawrence, Tomasz Szymankowski
@(#) #LM# 17.04.2000
*/

/*
Copyright (c) 1998 Richard Lawrence

This program is free software; you can redistribute it and/or modify it under the terms 
of the GNU General Public License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later version. This program is 
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details. You should have received a copy of the GNU
General Public License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdio.h>
#include <crtdbg.h>
#include <ddraw.h>
#include <mmsystem.h>
#include "WinConfig.h"
#include "Resource.h"
#include "atari800.h"
#include "platform.h" /* There is the pointer to Atari_DisplayScreen definition */
#include "globals.h"
#include "registry.h"
#include "sound_win.h"
#include "display_win.h"


/* Public objects */

UCHAR *g_screenbuff  = NULL;	/* screen buffer for windowed modes */
HDC    g_hScreenDC   = NULL;
int    g_nStatusSize = 18;

/* Private objects */

static UCHAR *s_screentemp    = NULL;	/* temporary buffer used when windowed screen with scanlines is created */
static UCHAR *s_offscrsurf    = NULL;	/* pointer to DirectDraw offscreen surface */
static ULONG  s_ulScreenMode  = 0;
static DWORD  s_dwVbiFlags    = 0;
static DWORD  s_dwBltFlags    = DDBLT_ASYNC;
static long   s_lOrigWndStyle = 0;
static ULONG  s_ulLEDOffset   = 0;
static RECT   s_rcSource;
static RECT   s_rcDest;
static RECT   s_rcSrcClip;
#ifdef WIN_USE_FLIP_BUFFER
static BOOL   s_bFlipBuffers  = FALSE;
#endif /*WIN_USE_FLIP_BUFFER*/

static LPDIRECTDRAW         s_lpDirectDraw = NULL;
/* Don't use SURFACE3, it doesn't work with NT 4.0 */
static LPDIRECTDRAWSURFACE  s_lpDDSPrimary = NULL;
static LPDIRECTDRAWSURFACE  s_lpDDSOffscrn = NULL;
#ifdef WIN_USE_FLIP_BUFFER
static LPDIRECTDRAWSURFACE  s_lpDDSBackBuf = NULL;
#endif /*WIN_USE_FLIP_BUFFER*/

static UCHAR s_aucAsciiToScreen[ 128 ] =
{
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
	0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
	0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
	0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
	0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f
};

static UINT s_aunDarklineTranslate[ 256 ];

static BOOL GetDDErrorString ( HRESULT hResult, LPSTR lpszErrorBuff, DWORD cchError );

static void Screen_DDraw                    ( UBYTE *screen );
static void Screen_DDraw_Stretch            ( UBYTE *screen );
static void Screen_DDraw_Stretch_Scanlines  ( UBYTE *screen );
static void Screen_DDraw_Full_Blt           ( UBYTE *screen );
static void Screen_DDraw_Full_Blt_Scanlines ( UBYTE *screen );
static void Screen_DDraw_1024_Scanlines     ( UBYTE *screen );
static void Screen_GDI_Stretch              ( UBYTE *screen );
static void Screen_GDI_Stretch_Scanlines    ( UBYTE *screen );
static void Screen_GDI                      ( UBYTE *screen );

#ifdef _DEBUG
#define ServeDDrawError( nUID, hResult, bQuit ) \
		ShowDDrawError( nUID, hResult, bQuit, __FILE__, __LINE__ )
#else /*_DEBUG*/
#define ServeDDrawError( nUID, hResult, bQuit ) \
		ShowDDrawError( nUID, hResult, bQuit )
#endif /*_DEBUG*/


/*========================================================
Function : ShowDDrawError
=========================================================*/
/* #FN#
   Shows DirectDraw error */
static
void
/* #AS#
   Nothing */
ShowDDrawError( UINT    nUID,    /* #IN# */
				HRESULT hResult, /* #IN# */
				BOOL    bQuit    /* #IN# */
#ifdef _DEBUG
			  , char   *pszFile,
				DWORD   dwLine
#endif /*_DEBUG*/
)
{
	TCHAR tszError     [ LOADSTRING_STRING_SIZE ];
	TCHAR tszFullString[ LOADSTRING_STRING_SIZE ];
	TCHAR tszAction    [ LOADSTRING_STRING_SIZE ];

#ifdef _DEBUG
	Aprint( "DirectDraw error: %s@%ld", pszFile, dwLine );
#endif /*_DEBUG*/

	/* Get us back to a GDI display and stop making noises */
	SetSafeDisplay( TRUE );
	ClearSound( FALSE );

	GetDDErrorString( hResult, tszError, LOADSTRING_STRING_SIZE );
	LoadString( NULL, IDS_DDRAW_ERROR_PROMPT, g_tszErrorString, LOADSTRING_STRING_SIZE );
	LoadString( NULL, nUID, tszAction, LOADSTRING_STRING_SIZE );
	wsprintf( tszFullString, g_tszErrorString, tszAction, tszError );
	LoadString( NULL, IDS_DDRAW_ERROR_HDR, g_tszErrorString, LOADSTRING_STRING_SIZE );
	MessageBox( g_hWnd, tszFullString, g_tszErrorString, MB_ICONSTOP | MB_OK );

	/* Start a quit (this will end up in Atari_Exit()) */
	if( bQuit )
	{
		/* Make sure the atari is turned off */
		g_ulAtariState = ATARI_UNINITIALIZED | ATARI_PAUSED;
		PostMessage( g_hMainWnd, WM_CLOSE, 0, 0L );
	}
	else
		RestartSound();
} /* #OF# ShowDDrawError */

/*========================================================
Function : AtariPlot
=========================================================*/
/* #FN#
   Routines for plotting on the Atari screen (taken out of ui.c, which
   I don't use currently */
static
void
/* #AS#
   Nothing */
AtariPlot( UBYTE *pScreen, /* #IN# */
		   int    nFg,     /* #IN# */
		   int    nBg,     /* #IN# */
		   int    nCh,     /* #IN# */
		   int    x,       /* #IN# */
		   int    y        /* #IN# */ )
{
	int   nOffset = s_aucAsciiToScreen[ (nCh & 0x07f) ] * 8;
	int   i;
	int   j;
	char *pAtariCharset;
	char *ptr;

	if( !g_ulAtariState & ATARI_RUNNING )
		return;

	if( mach_xlxe != 0 )
		pAtariCharset = atarixl_os + 0x2000;
	else
		pAtariCharset = memory + 0xe000;

	ptr = &pScreen[ 24 * ATARI_WIDTH + 32 ];

	for( i = 0; i < 8; i++ )
	{
		UBYTE data = pAtariCharset[ nOffset++ ];

		for( j = 0; j < 8; j++ )
		{
			int nPixel;

			if( data & 0x80 )
				nPixel = nFg;
			else
				nPixel = nBg;

			ptr[ (y * 8 + i) * ATARI_WIDTH + (x * 8 + j) ] = nPixel;

			data = data << 1;
		}
	}
} /* #OF# AtariPlot */

/*========================================================
Function : AtariPrint
=========================================================*/
/* #FN#
   Routines for printing on the Atari screen (taken out of ui.c, which
   I don't use currently */
static
void
/* #AS#
   Nothing */
AtariPrint( UBYTE *pScreen,   /* #IN# */
			int    nFg,       /* #IN# */
			int    nBg,       /* #IN# */
			char  *pszString, /* #IN# */
			int    x,         /* #IN# */
			int    y          /* #IN# */ )
{
	while( *pszString )
	{
		AtariPlot( pScreen, nFg, nBg, *pszString++, x, y );
		x++;
	}
} /* #OF# AtariPrint */

/*========================================================
Function : CenterPrint
=========================================================*/
/* #FN#
   Routines for printing on the Atari screen (taken out of ui.c, which
   I don't use currently */
static
void
/* #AS#
   Nothing */
CenterPrint( UBYTE *pScreen,   /* #IN# */
			 int    nFg,       /* #IN# */
			 int    nBg,       /* #IN# */
			 char  *pszString, /* #IN# */
			 int    y          /* #IN# */ )
{
	AtariPrint( pScreen, nFg, nBg, pszString, (40 - strlen( pszString )) / 2, y );
} /* #OF# CenterPrint */

/*========================================================
Function : CreateDirectDrawObject
=========================================================*/
/* #FN#
   Creates DirectDraw object */
static
BOOL
/* #AS#
   TRUE if succeeded, FALSE if failed */
CreateDirectDrawObject( void ) 
{
	/* Get the object if not already allocated */
	if( !s_lpDirectDraw )
	{
		HRESULT hResult;

		hResult = DirectDrawCreate( NULL, &s_lpDirectDraw, 0 );
		if( FAILED(hResult) )
		{
			ServeDDrawError( IDS_DDERR_ALLOC_OBJ, hResult, FALSE );
			g_ulAtariState = ATARI_UNINITIALIZED | ATARI_PAUSED;
			return FALSE;
		}
	}
	return TRUE;
} /* #OF# CreateDirectDrawObject */

/*========================================================
Function : SetupPrimarySurface
=========================================================*/
/* #FN#
   Sets up DirectDraw Primary Surface */
static
BOOL
/* #AS#
   TRUE if succeeded, FALSE if failed */
SetupPrimarySurface( void )
{
/*	LPDIRECTDRAWSURFACE lpDDSTemp;*/
	DDSURFACEDESC ddsd;
	HRESULT       hResult;

	ZeroMemory( &ddsd, sizeof(ddsd) );
	ddsd.dwSize         = sizeof(ddsd);
	ddsd.dwFlags        = DDSD_CAPS;
	ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;

#ifdef WIN_USE_FLIP_BUFFER
	if( s_bFlipBuffers )
	{
		ddsd.dwFlags            |= DDSD_BACKBUFFERCOUNT;
		ddsd.dwBackBufferCount   = 1;
		ddsd.ddsCaps.dwCaps     |= DDSCAPS_FLIP | DDSCAPS_COMPLEX;
	}
#endif /*WIN_USE_FLIP_BUFFER*/

	_ASSERT(!s_lpDDSPrimary);

	hResult = IDirectDraw_CreateSurface( s_lpDirectDraw, &ddsd, &s_lpDDSPrimary, NULL );
	if( FAILED(hResult) )
	{
		if( s_lpDDSPrimary )
		{
			IDirectDrawSurface_Release( s_lpDDSPrimary );
			s_lpDDSPrimary = NULL;
		}
		ServeDDrawError( IDS_DDERR_ALLOC_PRIMARY, hResult, FALSE );
		return FALSE;
	}
#ifdef WIN_USE_FLIP_BUFFER
	if( s_bFlipBuffers )
	{
		DDSCAPS caps;
		ZeroMemory( &caps, sizeof(caps) );
		caps.dwCaps = DDSCAPS_BACKBUFFER;

		hResult = IDirectDrawSurface_GetAttachedSurface( s_lpDDSPrimary, &caps, &s_lpDDSBackBuf );
		if( FAILED(hResult) )
		{
			IDirectDrawSurface_Release( s_lpDDSPrimary );
			s_lpDDSPrimary = NULL;

			ServeDDrawError( IDS_DDERR_ALLOC_PRIMARY, hResult, FALSE );
			return FALSE;
		}
	}
#endif /*WIN_USE_FLIP_BUFFER*/
	return TRUE;
} /* #OF# SetupPrimarySurface */

/*========================================================
Function : SetupMemorySurface
=========================================================*/
/* #FN#
   Sets up DirectDraw Memory Surface */
static
BOOL
/* #AS#
   TRUE if succeeded, FALSE if failed */
SetupMemorySurface( void )
{
	DDSURFACEDESC ddsd;
	HRESULT hResult;

	_ASSERT(!s_lpDDSOffscrn);

	ZeroMemory( &ddsd, sizeof(ddsd) );
	ddsd.dwSize = sizeof(ddsd);

	if( g_ulScreenMode & SM_MODE_FULL )
	{
		ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH | DDSD_PIXELFORMAT;
		ddsd.ddpfPixelFormat.dwSize  = sizeof(DDPIXELFORMAT);
		ddsd.ddpfPixelFormat.dwFlags = DDPF_RGB | DDPF_PALETTEINDEXED8;
		ddsd.ddpfPixelFormat.dwRGBBitCount = 8;
	}
	else
		ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;

	ddsd.dwHeight = (ATARI_HEIGHT + 16) * 2;
	if( g_ulScreenMode & SM_MODE_FULL && g_ulScreenMode & SM_ATTR_SCANLINE_CAP && g_ulScreenMode & SM_OPTN_SHOW_SCANLINES )
		ddsd.dwHeight = (ATARI_HEIGHT + 16) * 3;
	ddsd.dwWidth = ATARI_WIDTH;

	if( g_ulScreenMode & SM_OPTN_USE_VIDEOMEM )
	{
		ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_VIDEOMEMORY | DDSCAPS_LOCALVIDMEM;

		hResult = IDirectDraw_CreateSurface( s_lpDirectDraw, &ddsd, &s_lpDDSOffscrn, NULL );
		if( FAILED(hResult) )
			Aprint( "Video memory allocation failed, trying system memory..." );
	}
	if( !(g_ulScreenMode & SM_OPTN_USE_VIDEOMEM) || FAILED(hResult) )
	{
		ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;

		hResult = IDirectDraw_CreateSurface( s_lpDirectDraw, &ddsd, &s_lpDDSOffscrn, NULL );
		if( FAILED(hResult) )
		{
			Aprint( "Could not allocate system memory surface!" );
			s_offscrsurf = (UCHAR *)s_lpDDSOffscrn = NULL;
			ServeDDrawError( IDS_DDERR_ALLOC_SYSMEM_SURFACE, hResult, FALSE );
			return FALSE;
		}
	}
	if( g_ulScreenMode & SM_MODE_FULL )
	{
		ZeroMemory( &ddsd, sizeof(ddsd) );
		ddsd.dwSize  = sizeof(ddsd);
		IDirectDrawSurface_Lock( s_lpDDSOffscrn, NULL, &ddsd, 0, NULL );
		atari_screen = (ULONG *)ddsd.lpSurface;
		s_offscrsurf = (UCHAR *)ddsd.lpSurface;
		ZeroMemory( ddsd.lpSurface, ddsd.dwWidth * ddsd.dwHeight );
		IDirectDrawSurface_Unlock( s_lpDDSOffscrn, &ddsd );

		if( g_ulScreenMode & SM_OPTN_SHOW_SCANLINES && g_ulScreenMode & SM_ATTR_SCANLINE_CAP )
			atari_screen = (ULONG *)g_screenbuff;
	}
	else
		atari_screen = (ULONG *)g_screenbuff;

	return TRUE;
} /* #OF# SetupMemorySurface */

/*========================================================
Function : ReleaseAllSurfaces
=========================================================*/
/* #FN#
   Releases all the DirectDraw Surfaces */
static
HRESULT
/* #AS#
   DD_OK if succeeded */
ReleaseAllSurfaces( BOOL bReleaseObj /* #IN# */ )
{
	HRESULT hResult  = DD_OK;
	HRESULT hFailure = DD_OK;

	/* Release Memory Surface */
	if( s_lpDDSOffscrn )
	{
		hResult = IDirectDrawSurface_Release( s_lpDDSOffscrn );
		if( FAILED(hResult) )
			hFailure = hResult;
		s_offscrsurf = (UCHAR *)s_lpDDSOffscrn = NULL;
	}
	/* Release Primary Surface */
	if( s_lpDDSPrimary )
	{
		hResult = IDirectDrawSurface_Release( s_lpDDSPrimary );
		if( FAILED(hResult) )
			hFailure = hResult;
		s_lpDDSPrimary = NULL;
#ifdef WIN_USE_FLIP_BUFFER
		if( s_bFlipBuffers )
			s_lpDDSBackBuf = NULL;
#endif /*WIN_USE_FLIP_BUFFER*/
	}
	/* Release DirectDraw object and restore display mode */
	if( bReleaseObj && s_lpDirectDraw )
	{
		IDirectDraw_RestoreDisplayMode( s_lpDirectDraw );
		hResult = IDirectDraw_Release( s_lpDirectDraw );
		if( FAILED(hResult) )
			hFailure = hResult;
		s_lpDirectDraw = NULL;
	}
	return hFailure;
} /* #OF# ReleaseAllSurfaces */

/*========================================================
Function : RestoreSurfaces
=========================================================*/
/* #FN#
   Restores all the DirectDraw Surfaces */
static
BOOL
/* #AS#
   TRUE if succeeded, FALSE if failed */
RestoreSurfaces( void )
{
	HRESULT hResult; 

	/* Restore Memory Surface */
	if( s_lpDDSOffscrn )
	{
		hResult = IDirectDrawSurface_IsLost( s_lpDDSOffscrn );
		if( FAILED(hResult) )
		{
			hResult = IDirectDrawSurface_Restore( s_lpDDSOffscrn );
			if( FAILED(hResult) )
				return FALSE;
		}
	}
	/* Restore Primary Surface */
	if( s_lpDDSPrimary )
	{
		hResult = IDirectDrawSurface_IsLost( s_lpDDSPrimary );
		if( FAILED(hResult) ) 
		{
			hResult = IDirectDrawSurface_Restore( s_lpDDSPrimary );
			if( FAILED(hResult) )
				return FALSE;
		}
	}
	return TRUE;
} /* #OF# RestoreSurfaces */

/*========================================================
Function : EnumModesCallback
=========================================================*/
/* #FN#
   Checks available display modes */
static
HRESULT WINAPI
/* #AS#
   Always DDENUMRET_OK */
EnumModesCallback( LPDDSURFACEDESC lpDDSurfaceDesc, /* #IN# */
				   LPVOID          lpContext        /* #IN# */ )
{
	_ASSERT(lpDDSurfaceDesc);

	if( lpDDSurfaceDesc->ddpfPixelFormat.dwRGBBitCount == 8 )
	{
		if( lpDDSurfaceDesc->dwHeight == 768 && lpDDSurfaceDesc->dwWidth == 1024 )
			g_ulModesAvail |= MODE_1024_768;
		if( lpDDSurfaceDesc->dwHeight == 600 && lpDDSurfaceDesc->dwWidth == 800 )
			g_ulModesAvail |= MODE_800_600;
		if( lpDDSurfaceDesc->dwHeight == 480 && lpDDSurfaceDesc->dwWidth == 640 )
			g_ulModesAvail |= MODE_640_480;
		if( lpDDSurfaceDesc->dwHeight == 400 && lpDDSurfaceDesc->dwWidth == 640 )
			g_ulModesAvail |= MODE_640_400;
		if( lpDDSurfaceDesc->dwHeight == 384 && lpDDSurfaceDesc->dwWidth == 512 )
			g_ulModesAvail |= MODE_512_384;
		if( lpDDSurfaceDesc->dwHeight == 240 && lpDDSurfaceDesc->dwWidth == 320 )
			g_ulModesAvail |= MODE_320_240;
		if( lpDDSurfaceDesc->dwHeight == 200 && lpDDSurfaceDesc->dwWidth == 320 )
			g_ulModesAvail |= MODE_320_200;
	}
	return DDENUMRET_OK;
} /* #OF# EnumModesCallback */

#ifdef WIN_USE_FLIP_BUFFER
/*========================================================
Function : FlipToGDI
=========================================================*/
/* #FN#
   Makes the surface that GDI writes to the primary surface */
void
/* #AS#
   Nothing */
FlipToGDI( void )
{
	if( s_bFlipBuffers && s_lpDirectDraw )
	{
		IDirectDraw_FlipToGDISurface( s_lpDirectDraw );
	}
} /* #OF# FlipToGDI */
#endif /*WIN_USE_FLIP_BUFFER*/

/*========================================================
Function : RedrawScreen
=========================================================*/
/* #FN#
   Redraws windowed mode display */
void
RedrawScreen( HDC hDC )
{
	int nWidth  = g_lpbmi->bmiHeader.biWidth;
	int nHeight = g_lpbmi->bmiHeader.biHeight;
	HDC hBackDC = g_hScreenDC;

	g_hScreenDC = hDC;

	if( g_ulScreenMode & SM_WRES_STRETCH )
	{
		g_lpbmi->bmiHeader.biWidth  = ATARI_STRETCH_VIS_WIDTH;
		g_lpbmi->bmiHeader.biHeight = -ATARI_STRETCH_HEIGHT;

		if( g_ulScreenMode & SM_OPTN_SHOW_SCANLINES )
		{
			Screen_GDI_Stretch_Scanlines( (UBYTE *)atari_screen );
		}
		else
			Screen_GDI_Stretch( (UBYTE *)atari_screen );
	}
	else
	{
		g_lpbmi->bmiHeader.biWidth  = ATARI_VIS_WIDTH;
		g_lpbmi->bmiHeader.biHeight = -ATARI_HEIGHT;

		Screen_GDI( (UBYTE *)atari_screen );
	}
	g_lpbmi->bmiHeader.biWidth  = nWidth;
	g_lpbmi->bmiHeader.biHeight = nHeight;

	g_hScreenDC = hBackDC;
} /* #OF# RedrawScreen */

/*========================================================
Function : SetupWindowedDisplay
=========================================================*/
/* #FN#
   Sets up windowed mode display */
static
BOOL
/* #AS#
   TRUE if succeeded, FALSE if failed */
SetupWindowedDisplay( void )
{
	_ASSERT(g_ulScreenMode & SM_MODE_WIND);

	if( g_ulScreenMode & SM_OPTN_USE_GDI && g_ulScreenMode & SM_WRES_STRETCH ) /* GDI stretch windowed modes */
	{
		g_lpbmi->bmiHeader.biWidth  = ATARI_STRETCH_VIS_WIDTH;
		g_lpbmi->bmiHeader.biHeight = -ATARI_STRETCH_HEIGHT;	/* Negative because we are a top-down bitmap */
	}
	/* If we are recovering from a DirectDraw mode, the window frame will be
	   whatever size that mode was instead of what we want it to be, so set
	   window size here */
	if( g_hMainWnd )
	{
		RECT rcPos;
		int	 nHeight, nWidth;

		GetWindowRect( g_hMainWnd, &rcPos );
		if( ((rcPos.left != 0) && (rcPos.left != g_nStartX)) || ((rcPos.top != 0) && (rcPos.top != g_nStartY)) )
		{
			g_nStartX = rcPos.left;
			g_nStartY = rcPos.top;
		}
		if( g_ulScreenMode & SM_WRES_NORMAL )
		{
			nHeight = ATARI_HEIGHT;
			nWidth  = ATARI_VIS_WIDTH;
		}
		else
		{
			nHeight = ATARI_STRETCH_HEIGHT;
			nWidth  = ATARI_STRETCH_VIS_WIDTH;
		}
		/* Set main window size and position */
		SetWindowPos( g_hMainWnd, HWND_NOTOPMOST, g_nStartX, g_nStartY, 
			nWidth + GetSystemMetrics( SM_CXDLGFRAME ) * 2 + GetSystemMetrics( SM_CXEDGE ) * 2,
			nHeight + GetSystemMetrics( SM_CYMENU ) + g_nStatusSize + GetSystemMetrics( SM_CYDLGFRAME ) * 2 +
			GetSystemMetrics( SM_CYCAPTION ) + GetSystemMetrics( SM_CYEDGE ) * 2,
			SWP_SHOWWINDOW );
	}
	
	if( g_ulScreenMode & SM_OPTN_USE_GDI )
	{
		/* Specifics for non-DirectDraw windowed modes */
		if( g_ulScreenMode & SM_WRES_STRETCH )
		{
			if( g_ulScreenMode & SM_OPTN_SHOW_SCANLINES )
			{
				Atari_DisplayScreen = Screen_GDI_Stretch_Scanlines;
			}
			else
				Atari_DisplayScreen = Screen_GDI_Stretch;
		}
		else
			Atari_DisplayScreen = Screen_GDI;
	}
	else
	{
		/* Specifics for DirectDraw windowed modes */
		if( g_ulScreenMode & SM_WRES_STRETCH )
		{
			if( g_ulScreenMode & SM_OPTN_SHOW_SCANLINES )
			{
				s_rcSource.bottom = ATARI_STRETCH_HEIGHT;
				g_lpbmi->bmiHeader.biHeight = -ATARI_STRETCH_HEIGHT;

				Atari_DisplayScreen = Screen_DDraw_Stretch_Scanlines;
			}
			else
				Atari_DisplayScreen = Screen_DDraw_Stretch;
		}
		else
			Atari_DisplayScreen = Screen_DDraw;

		ComputeClipArea();
	}
	return TRUE;
} /* #OF# SetupWindowedDisplay */

/*========================================================
Function : SetupFullScreenDisplay
=========================================================*/
/* #FN#
   Sets up DirectDraw full screen mode display */
static
BOOL
/* #AS#
   TRUE if succeeded, FALSE if failed */
SetupFullScreenDisplay( void )
{
	HRESULT hResult;

	_ASSERT(g_ulScreenMode & SM_MODE_FULL);

	Atari_DisplayScreen = Screen_DDraw_Full_Blt;

	if( g_ulScreenMode & SM_FRES_320_200 )
	{
		hResult = IDirectDraw_SetDisplayMode( s_lpDirectDraw, 320, 200, 8 );
		if( FAILED(hResult) )
		{
			ServeDDrawError( IDS_DDERR_INIT_RES, hResult, FALSE );
			return FALSE;
		}
		s_rcDest.left     = 0;
		s_rcDest.top      = 0;
		s_rcDest.right    = 320;
		s_rcDest.bottom   = 200;
		s_rcSource.left   = ATARI_HORZ_CLIP + ((ATARI_VIS_WIDTH - 320) >> 1);
		s_rcSource.right  = ATARI_WIDTH - s_rcSource.left;
		s_rcSource.top    = (ATARI_HEIGHT - 200) >> 1;
		s_rcSource.bottom = ATARI_HEIGHT - s_rcSource.top;
		g_ulScreenMode   |= SM_ATTR_SMALL_DLG | SM_ATTR_NO_MENU;
		s_ulLEDOffset     = (s_rcSource.bottom - DISKLED_FONT_HEIGHT) * ATARI_WIDTH + s_rcSource.right - DISKLED_FONT_WIDTH;
	}

	if( g_ulScreenMode & SM_FRES_320_240 )
	{
		hResult = IDirectDraw_SetDisplayMode( s_lpDirectDraw, 320, 240, 8 );
		if( FAILED(hResult) )
		{
			ServeDDrawError( IDS_DDERR_INIT_RES, hResult, FALSE );
			return FALSE;
		}
		s_rcDest.left     = 0;
		s_rcDest.top      = 0;
		s_rcDest.right    = 320;
		s_rcDest.bottom   = 240;
		s_rcSource.left   = ATARI_HORZ_CLIP + ((ATARI_VIS_WIDTH - 320) >> 1);
		s_rcSource.right  = ATARI_WIDTH - s_rcSource.left;
		s_rcSource.top    = 0;
		s_rcSource.bottom = ATARI_HEIGHT;
		g_ulScreenMode   |= SM_ATTR_SMALL_DLG | SM_ATTR_NO_MENU;
		s_ulLEDOffset     = (s_rcSource.bottom - DISKLED_FONT_HEIGHT) * ATARI_WIDTH + s_rcSource.right - DISKLED_FONT_WIDTH;
	}

	if( g_ulScreenMode & SM_FRES_512_384 )
	{
		hResult = IDirectDraw_SetDisplayMode( s_lpDirectDraw, 512, 384, 8 );
		if( FAILED(hResult) )
		{
			ServeDDrawError( IDS_DDERR_INIT_RES, hResult, FALSE );
			return FALSE;
		}
		s_rcDest.left    = (512 - ATARI_VIS_WIDTH) >> 1;
		s_rcDest.top     = (384 - ATARI_HEIGHT) >> 1;
		s_rcDest.right   = s_rcDest.left + ATARI_VIS_WIDTH;
		s_rcDest.bottom  = s_rcDest.top  + ATARI_HEIGHT;
		s_rcSource.left  = ATARI_HORZ_CLIP;
		s_rcSource.right = ATARI_WIDTH - ATARI_HORZ_CLIP;
	}

	if( g_ulScreenMode & SM_FRES_640_400 )
	{
		hResult = IDirectDraw_SetDisplayMode( s_lpDirectDraw, 640, 400, 8 );
		if( FAILED(hResult) )
		{
			ServeDDrawError( IDS_DDERR_INIT_RES, hResult, FALSE );
			return FALSE;
		}
		s_rcDest.left     = 0;
		s_rcDest.top      = 0;
		s_rcDest.right    = 640;
		s_rcDest.bottom   = 400;
		s_rcSource.left   = ATARI_HORZ_CLIP + ((ATARI_VIS_WIDTH - 320) >> 1);
		s_rcSource.right  = ATARI_WIDTH - s_rcSource.left;
		s_rcSource.top    = (ATARI_HEIGHT - 200) >> 1;
		s_rcSource.bottom = ATARI_HEIGHT - s_rcSource.top;
		g_ulScreenMode   |= SM_ATTR_SCANLINE_CAP | SM_ATTR_NO_MENU;
		s_ulLEDOffset     = (s_rcSource.bottom - DISKLED_FONT_HEIGHT) * ATARI_WIDTH + s_rcSource.right - DISKLED_FONT_WIDTH;

		if( g_ulScreenMode & SM_OPTN_SHOW_SCANLINES )
		{
			s_rcSource.top    = (ATARI_STRETCH_HEIGHT - 400) >> 1;
			s_rcSource.bottom = ATARI_STRETCH_HEIGHT - s_rcSource.top;

			Atari_DisplayScreen = Screen_DDraw_Full_Blt_Scanlines;
		}
	}

	if( g_ulScreenMode & SM_FRES_640_480 )
	{
		hResult = IDirectDraw_SetDisplayMode( s_lpDirectDraw, 640, 480, 8 );
		if( FAILED(hResult) )
		{
			ServeDDrawError( IDS_DDERR_INIT_RES, hResult, FALSE );
			return FALSE;
		}
		s_rcDest.left     = 0;
		s_rcDest.top      = 0;
		s_rcDest.right    = 640;
		s_rcDest.bottom   = 480;
		s_rcSource.left   = ATARI_HORZ_CLIP + ((ATARI_VIS_WIDTH - 320) >> 1);
		s_rcSource.right  = ATARI_WIDTH - s_rcSource.left;
		s_rcSource.top    = 0;
		s_rcSource.bottom = ATARI_HEIGHT;
		g_ulScreenMode   |= SM_ATTR_SCANLINE_CAP | SM_ATTR_NO_MENU;
		s_ulLEDOffset     = (s_rcSource.bottom - DISKLED_FONT_HEIGHT) * ATARI_WIDTH + s_rcSource.right - DISKLED_FONT_WIDTH;

		if( g_ulScreenMode & SM_OPTN_SHOW_SCANLINES )
		{
			s_rcSource.bottom = ATARI_STRETCH_HEIGHT;

			Atari_DisplayScreen = Screen_DDraw_Full_Blt_Scanlines;
		}
	}

	if( g_ulScreenMode & SM_FRES_800_600 )
	{
		hResult = IDirectDraw_SetDisplayMode( s_lpDirectDraw, 800, 600, 8 );
		if( FAILED(hResult) )
		{
			ServeDDrawError( IDS_DDERR_INIT_RES, hResult, FALSE );
			return FALSE;
		}
		s_rcDest.left    = (800 - ATARI_STRETCH_VIS_WIDTH) >> 1;
		s_rcDest.top     = (600 - ATARI_STRETCH_HEIGHT) >> 1;
		s_rcDest.right   = s_rcDest.left + ATARI_STRETCH_VIS_WIDTH;
		s_rcDest.bottom  = s_rcDest.top + ATARI_STRETCH_HEIGHT;
		s_rcSource.left  = ATARI_HORZ_CLIP;
		s_rcSource.right = ATARI_WIDTH - ATARI_HORZ_CLIP;
		g_ulScreenMode  |= SM_ATTR_SCANLINE_CAP;

		if( g_ulScreenMode & SM_OPTN_SHOW_SCANLINES )
		{
			s_rcSource.bottom = ATARI_STRETCH_HEIGHT;

			Atari_DisplayScreen = Screen_DDraw_Full_Blt_Scanlines;
		}
	}

	if( g_ulScreenMode & SM_FRES_1024_768 )
	{
		hResult = IDirectDraw_SetDisplayMode( s_lpDirectDraw, 1024, 768, 8 );
		if( FAILED(hResult) )
		{
			ServeDDrawError( IDS_DDERR_INIT_RES, hResult, FALSE );
			return FALSE;
		}
		s_rcDest.left    = (1024 - (ATARI_VIS_WIDTH * 3)) >> 1;
		s_rcDest.top     = (768 - (ATARI_HEIGHT * 3)) >> 1;
		s_rcDest.right   = s_rcDest.left + (ATARI_VIS_WIDTH * 3);
		s_rcDest.bottom  = s_rcDest.top + (ATARI_HEIGHT * 3);
		s_rcSource.left  = ATARI_HORZ_CLIP;
		s_rcSource.right = ATARI_WIDTH - ATARI_HORZ_CLIP;
		g_ulScreenMode  |= SM_ATTR_SCANLINE_CAP;

		if( g_ulScreenMode & SM_OPTN_SHOW_SCANLINES )
		{
			s_rcSource.bottom = ATARI_HEIGHT * 3;

			Atari_DisplayScreen = Screen_DDraw_1024_Scanlines;
		}
	}
	return TRUE;
} /* #OF# SetupFullScreenDisplay */
  
/*========================================================
Function : InitialiseScreen
=========================================================*/
/* #FN#
   Initializes DirectDraw/GDI display */
int
/* #AS#
   1 if succeeded, 0 if failed */
InitialiseScreen( BOOL bForceInit /* #IN# Force initialization */ )
{
	LPDIRECTDRAWPALETTE	lpDDPalette;
	HRESULT	hResult;

	if( !g_screenbuff )
		g_screenbuff = malloc( ATARI_WIDTH * (ATARI_HEIGHT + 16) );
	if( !s_screentemp )
		s_screentemp = malloc( ATARI_STRETCH_VIS_SCREEN_SIZE + 1 );

	if( !s_screentemp || !g_screenbuff )
		return 0;

	ClearScreen( FALSE );

	if( !bForceInit && s_lpDirectDraw && s_ulScreenMode == g_ulScreenMode )
		return 1;

	s_ulScreenMode  = g_ulScreenMode;
	g_ulScreenMode &= ~SM_ATTR_MASK;

	/* Generic LED offset */
	s_ulLEDOffset = (ATARI_HEIGHT - DISKLED_FONT_HEIGHT) * ATARI_WIDTH + ATARI_WIDTH - ATARI_HORZ_CLIP - DISKLED_FONT_WIDTH;

	s_rcSource.left   = 0;
	s_rcSource.top    = 0;
	s_rcSource.right  = ATARI_VIS_WIDTH;
	s_rcSource.bottom = ATARI_HEIGHT;

	g_lpbmi->bmiHeader.biWidth  = ATARI_VIS_WIDTH;
	g_lpbmi->bmiHeader.biHeight = -ATARI_HEIGHT;	/* Negative because we are a top-down bitmap */
	
	atari_screen = (ULONG *)g_screenbuff;

	/* Sometimes when changing from window to full screen, DirectSound becomes very
	   confused, so stop it here and start it re-init it again when we have finished
	   the screen change */
//	ClearSound( TRUE );

	if( !s_lOrigWndStyle )
		s_lOrigWndStyle = GetWindowLong( g_hMainWnd, GWL_STYLE );

	/* Release also DirectDraw object if windowed non-DirectDraw mode is choosen */
	ReleaseAllSurfaces( g_ulScreenMode & SM_MODE_WIND && g_ulScreenMode & SM_OPTN_USE_GDI );

	s_dwBltFlags = (g_ulScreenMode & SM_OPTN_DDBLT_WAIT ? DDBLT_WAIT : DDBLT_ASYNC);
	/* If flipped mode won't be active, we will just check if s_dwVbiFlags != 0 */
	s_dwVbiFlags = (g_ulScreenMode & SM_OPTN_DDVBL_WAIT ? DDFLIP_WAIT : 0);

#ifdef WIN_USE_FLIP_BUFFER
	s_bFlipBuffers = FALSE;
#endif /*WIN_USE_FLIP_BUFFER*/

	/* Handle windowed DirectDraw or non-DirectDraw modes */
	if( g_ulScreenMode & SM_MODE_WIND )
	{
		/* Restore oryginal window style */
		if( g_hMainWnd && s_lOrigWndStyle )
			SetWindowLong( g_hMainWnd, GWL_STYLE, s_lOrigWndStyle );

		/* Restore display mode */
		if( s_lpDirectDraw )
			IDirectDraw_RestoreDisplayMode( s_lpDirectDraw );

		/* Specifics for DirectDraw windowed modes */
		if( !(g_ulScreenMode & SM_OPTN_USE_GDI) )
		{
			if( !CreateDirectDrawObject() )
				return 0;

			_ASSERT(!s_lpDDSPrimary && !s_lpDDSOffscrn);
			_ASSERT(s_lpDirectDraw);

			hResult = IDirectDraw_SetCooperativeLevel( s_lpDirectDraw, NULL, DDSCL_NORMAL );
			if( FAILED(hResult) )
			{
				ServeDDrawError( IDS_DDERR_SET_MODE, hResult, FALSE );
				return 0;
			}
		}
		if( !SetupWindowedDisplay() )
			return 0;

		if( !(g_ulScreenMode & SM_OPTN_USE_GDI) )
		{
			/* Primary surface for windowed DirectDraw modes */
			if( !SetupPrimarySurface() )
				return 0;
		}
		else /* No more processing required for GDI modes */
			return 1;
	}

	/* Handle the various FULL (exclusive screen) modes */
	if( g_ulScreenMode & SM_MODE_FULL )
	{
		if( !CreateDirectDrawObject() )
			return 0;

		if( g_hMainWnd )
			SetWindowLong( g_hMainWnd, GWL_STYLE, WS_BORDER );

		/* Set DirectDraw cooperative level */
		hResult = IDirectDraw_SetCooperativeLevel( s_lpDirectDraw, g_hMainWnd, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN | DDSCL_ALLOWREBOOT );
		if( FAILED(hResult) )
		{
			ServeDDrawError( IDS_DDERR_SET_MODE, hResult, FALSE );
			return 0;
		}
		if( !SetupFullScreenDisplay() )
			return 0;

#ifdef WIN_USE_FLIP_BUFFER
		/* Flipped buffers are available for full screen without menu only */
		s_bFlipBuffers = g_ulScreenMode & SM_OPTN_FLIP_BUFFERS &&
						 g_ulScreenMode & SM_ATTR_NO_MENU;
#endif /*WIN_USE_FLIP_BUFFER*/
	} 
	/* Now that the cooperative mode is set, create a palette */
	if( s_lpDirectDraw )
	{
		hResult = IDirectDraw_CreatePalette( s_lpDirectDraw, DDPCAPS_8BIT | DDPCAPS_ALLOW256, g_pe, &lpDDPalette, NULL );
		if( FAILED(hResult) )
		{
			ServeDDrawError( IDS_DDERR_ALLOC_PALETTE, hResult, FALSE );
			return 0;
		}
	}
	else
		return 0; /* Something went wrong, e.g. registry are corrupted */

	if( g_ulScreenMode & SM_MODE_FULL )
	{
		if( !SetupPrimarySurface() )
			return 0;

		hResult = IDirectDrawSurface_SetPalette( s_lpDDSPrimary, lpDDPalette );
		if( FAILED(hResult) )
		{
			ServeDDrawError( IDS_DDERR_SETPAL_PRIMARY, hResult, FALSE );
			return 0;
		}
	}
	if( g_ulScreenMode & SM_MODE_FULL || (g_ulScreenMode & SM_WRES_STRETCH && !(g_ulScreenMode & SM_OPTN_USE_GDI)) )
	{
		/* Memory surface is only used for full screen, stretched window modes */
		if( !SetupMemorySurface() )
			return 0;
	}
	/* Don't actually want to set a palette for this if it's stretched window
	   DDRAW, because that's what we use this surface for - a colorspace
	   conversion to whatever the primary surface is via StretchDIB */
	if( g_ulScreenMode & SM_MODE_FULL )
	{
		hResult = IDirectDrawSurface_SetPalette( s_lpDDSOffscrn, lpDDPalette );
		if( FAILED(hResult) )
		{
			ServeDDrawError( IDS_DDERR_SETPAL_MEMORY, hResult, FALSE );
			return 0;
		}
	}
	ClearScreen( FALSE );

//	InitialiseSound();
	return 1;
} /* #OF# InitialiseScreen */

/*========================================================
Function : ComputeClipArea
=========================================================*/
/* #FN#
   Computes clip area */
void
/* #AS#
   Nothing */
ComputeClipArea( void )
{
	RECT rcWind, rcDesktop;
	int  nDivisor;

	if( g_ulScreenMode & SM_OPTN_USE_GDI || g_ulScreenMode & SM_MODE_FULL )
		return;	/* Don't need to worry about this for GDI screens */

	/* The following section covers clip behavior for DDraw windowed modes
	   (important, because the primary surface is the desktop). These are
	   computed once here and then every time a WM_MOVE is received in
	   Atari800WinView. I never had much luck with ClipObjects... */
	SystemParametersInfo( SPI_GETWORKAREA, 0, (LPVOID)&rcDesktop, 0 );
	GetWindowRect( g_hWnd, &rcWind );
	rcWind.left += GetSystemMetrics( SM_CXEDGE );
	rcWind.top  += GetSystemMetrics( SM_CYEDGE );

	if( g_ulScreenMode & SM_WRES_STRETCH )
		nDivisor = 2;
	else
		nDivisor = 1;

	memcpy( &s_rcSrcClip, &s_rcSource, sizeof(RECT) );

	s_rcDest.left = rcWind.left;
	if( s_rcDest.left < rcDesktop.left )
	{
		s_rcSrcClip.left = (rcDesktop.left - s_rcDest.left) / nDivisor;
		s_rcDest.left    = rcDesktop.left;
		s_rcDest.right   = s_rcDest.left + ATARI_VIS_WIDTH * nDivisor - s_rcSrcClip.left * nDivisor;
	}
	else
	{
		s_rcDest.right = s_rcDest.left + ATARI_VIS_WIDTH * nDivisor;
		if( s_rcDest.right > rcDesktop.right )
		{
			s_rcSrcClip.right -= (s_rcDest.right - rcDesktop.right) / nDivisor;
			s_rcDest.right     = rcDesktop.right;
		}
	}
	s_rcDest.top = rcWind.top - rcDesktop.top;
	if( s_rcDest.top < 0 )
	{
		s_rcSrcClip.top = (rcDesktop.top - s_rcDest.top) / nDivisor;
		s_rcDest.top    = rcDesktop.top;
		s_rcDest.bottom = s_rcDest.top + ATARI_HEIGHT * nDivisor - s_rcSrcClip.top * nDivisor;
	}
	else
	{
		s_rcDest.bottom = s_rcDest.top + ATARI_HEIGHT * nDivisor;
		if( s_rcDest.bottom > rcDesktop.bottom )
		{   /* Sigh..scanline mode is different, since it is already stretched */
			if( nDivisor == 2 && g_ulScreenMode & SM_ATTR_SCANLINE_CAP && g_ulScreenMode & SM_OPTN_SHOW_SCANLINES ) 
				s_rcSrcClip.bottom -= (s_rcDest.bottom - rcDesktop.bottom);
			else
				s_rcSrcClip.bottom -= (s_rcDest.bottom - rcDesktop.bottom) / nDivisor;
			s_rcDest.bottom = rcDesktop.bottom ;
		}
	}
	/* Negative because we are a top-down bitmap. Note that it is CORRECT to
	   only adjust height of the src bitmap. This is to prevent windows from
	   blitting the wrong vertical portion (from the bottom up) if the
	   bmiHeader.biHeight doesn't match the source rectangle. However you
	   always want width the FULL width, otherwise a blit will assume the
	   memory area described has shorter rows and will unevenly blit. Confused?
	   good, we're not finished. In DDRAW windowed modes the SRC rectangle
	   will usually be 1X Atari size, because we stretch it to memory first
	   then just directly BLT. The exception is "scan-line" modes, which have
	   to be 2X Atari Height to accomodate the extra information (or lack
	   thereof, more appropriately) */
	if( g_lpbmi && nDivisor == 1 )
		g_lpbmi->bmiHeader.biHeight = s_rcSrcClip.top - s_rcSrcClip.bottom;
} /* #OF# ComputeClipArea */

/*========================================================
Function : DrawPausedScreen
=========================================================*/
/* #FN#
   Draws paused screen */
void
/* #AS#
   Nothing */
DrawPausedScreen( UBYTE *pScreen )
{
	int    i, j;
	UBYTE *blank;
	
	for( i = 0; i < ATARI_HEIGHT; i++ )
	{
		blank = pScreen + i * ATARI_WIDTH + i % 2;
		for( j = 0; j < ATARI_WIDTH; j += 2 )
		{
			*blank = 0;
			 blank +=2;
		}
	}
	CenterPrint( pScreen, 255, 0, "Paused: F9 to continue", 11 );
	InvalidateRect( g_hWnd, NULL, FALSE );
	Atari_DisplayScreen( pScreen );
} /* #OF# DrawPausedScreen */

/*========================================================
Function : SetSafeDisplay
=========================================================*/
/* #FN#
   Sets safe display screen for compatibility */
int
/* #AS#
   The value returned by InitialiseScreen function */
SetSafeDisplay( BOOL bForceGDI /* #IN# Use GDI flag */ )
{
	g_ulScreenMode &= ~SM_MODE_MASK;
	/* Set windowed GDI mode */
	g_ulScreenMode |= SM_MODE_WIND;
	if( bForceGDI )
		g_ulScreenMode |= SM_OPTN_USE_GDI;
	WriteRegDWORD( NULL, REG_DDRAW_MODE, g_ulScreenMode );

	return InitialiseScreen( FALSE );
} /* #OF# SetSafeDisplay */

/*========================================================
Function : SafeShowScreen
=========================================================*/
/* #FN#
   Shows Atari display screen, if available */
void
/* #AS#
   Nothing */
SafeShowScreen( void )
{
	if( Atari_DisplayScreen != NULL )
		Atari_DisplayScreen( (UCHAR *)atari_screen );
} /* #OF# SafeShowScreen */

/*========================================================
Function : ClearScreen
=========================================================*/
/* #FN#
   Clears screen buffers */
void
/* #AS#
   Nothing */
ClearScreen( BOOL bPermanent )
{
	if( g_screenbuff )
		memset( g_screenbuff, 0, (ATARI_WIDTH * (ATARI_HEIGHT + 16)) );
	if( s_screentemp )
		memset( s_screentemp, 0, ATARI_STRETCH_VIS_SCREEN_SIZE );

	if( s_lpDDSOffscrn )
	{
		DDSURFACEDESC ddsd;
		HRESULT       hResult;
		ZeroMemory( &ddsd, sizeof(ddsd) );
		ddsd.dwSize = sizeof(ddsd);

		hResult = IDirectDrawSurface_Lock( s_lpDDSOffscrn, NULL, &ddsd, 0, NULL );
		if( SUCCEEDED(hResult))
		{
			ZeroMemory( ddsd.lpSurface, ddsd.dwHeight * ddsd.dwWidth );
			IDirectDrawSurface_Unlock( s_lpDDSOffscrn, &ddsd );
		}
	}
	if( s_lpDDSPrimary && g_ulScreenMode & SM_MODE_FULL )
	{
		DDSURFACEDESC ddsd;
		HRESULT       hResult;

		ZeroMemory( &ddsd, sizeof(ddsd) );
		ddsd.dwSize = sizeof(ddsd);
		hResult = IDirectDrawSurface_Lock( s_lpDDSPrimary, NULL, &ddsd, 0, NULL );
		if( SUCCEEDED(hResult) ) 
		{
			RECT    rcTemp;
			DDBLTFX ddbltfx;

			rcTemp.top    = GetSystemMetrics( SM_CYMENU );
			rcTemp.left   = 0;
			rcTemp.right  = ddsd.dwWidth;
			rcTemp.bottom = ddsd.dwHeight - rcTemp.top;
			IDirectDrawSurface_Unlock( s_lpDDSPrimary, &ddsd );

			ZeroMemory( &ddbltfx, sizeof(ddbltfx) );
			ddbltfx.dwSize      = sizeof(ddbltfx);
			ddbltfx.dwFillColor = 0;

			IDirectDrawSurface_Blt( s_lpDDSPrimary, &rcTemp, NULL, NULL, s_dwBltFlags | DDBLT_COLORFILL , &ddbltfx);
		}
	}
	if( bPermanent )
	{
		/* Release DirectDraw Surfaces */
		HRESULT hResult = ReleaseAllSurfaces( TRUE );
		if( FAILED(hResult) )
		{
			LoadString( NULL, IDS_DDRAW_ERROR_RELEASE_ALL, g_tszErrorString, LOADSTRING_STRING_SIZE );
			MessageBox( g_hMainWnd, g_tszErrorString, "Atari800Win", MB_ICONEXCLAMATION | MB_OK );
		}
		if( s_screentemp )
		{
			free( s_screentemp );
			s_screentemp = NULL;
		}
		if( g_screenbuff )
		{
			free( g_screenbuff );
			g_screenbuff = NULL;
		}
	}
} /* #OF# ClearScreen */

/*========================================================
Function : CheckDDrawModes
=========================================================*/
/* #FN#
   Checks DirectDraw modes */
void
/* #AS#
   Nothing */
CheckDDrawModes( void )
{
	if( !s_lpDirectDraw )
	{
		if( SUCCEEDED(DirectDrawCreate( NULL, &s_lpDirectDraw, 0 )) )
		{
			IDirectDraw_EnumDisplayModes( s_lpDirectDraw, 0, NULL, NULL, EnumModesCallback );
			IDirectDraw_Release( s_lpDirectDraw );
		}
		s_lpDirectDraw = NULL;
		return;
	}
	IDirectDraw_EnumDisplayModes( s_lpDirectDraw, 0, NULL, NULL, EnumModesCallback );
} /* #OF# CheckDDrawModes */

/*========================================================
Function : ScreenBenchmark
=========================================================*/
/* #FN#
   Renders speed of the current mode */
void
/* #AS#
   Nothing */
ScreenBenchmark( void )
{
	char   szReport[ LOADSTRING_STRING_SIZE ];
	ULONG  starttime, rendertime;
	int    i;
	double lfResult;

	LoadString( NULL, IDS_GRAPHICS_TEST_BEGIN, g_tszErrorString, LOADSTRING_STRING_SIZE );
	MessageBox( g_hWnd, g_tszErrorString, "Atari800Win", MB_ICONINFORMATION | MB_OK );
	starttime = timeGetTime();

	for( i = 0; i < 30; i++ )
		Atari_DisplayScreen( (UBYTE *)atari_screen );

	rendertime = timeGetTime() - starttime;
	lfResult   = (double)rendertime / 30.0;

	LoadString( NULL, IDS_GRAPHICS_TEST_RESULT, g_tszErrorString, LOADSTRING_STRING_SIZE );
	sprintf( szReport, g_tszErrorString, rendertime, lfResult );
	if( lfResult > 20.0 )
	{
		LoadString( NULL, IDS_SPEED_WARNING, g_tszErrorString, LOADSTRING_STRING_SIZE );
		strcat( szReport, g_tszErrorString );
	}
	LoadString( NULL, IDS_BENCHMARK_RESULTS, g_tszErrorString, LOADSTRING_STRING_SIZE );
	MessageBox( g_hWnd, szReport, g_tszErrorString, MB_ICONINFORMATION | MB_OK );
} /* #OF# ScreenBenchmark */

/*========================================================
Function : Screen_DDraw
=========================================================*/
/* #FN#
   Draws Atari screen in DirectDraw windowed mode x 1 */
static
void
/* #AS#
   Nothing */
Screen_DDraw( UBYTE *screen /* #IN# Pointer to Atari screen */ )
{
	HDC hdc;
	HRESULT hResult;

	if( g_ulAtariState & ATARI_PAUSED )
		return;

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	Update_LED( (UBYTE *)g_screenbuff + s_ulLEDOffset );
#endif

#ifdef WIN_INLINE_ASM
	/* We have to copy the full size atari screen to another buffer to chop
	   off the left and right overflow (reason: allowing StretchDIB to cut the
	   rectangle for us is VERY slow). It so happens everything involved is
	   DWORD aligned, and the offsets and such are constants, so I'm sticking
	   this asm routine in here...roughly equivalent C source code is below */
	_asm
	{
        mov     esi, dword ptr [g_screenbuff]	;; the source pointer (g_screenbuff + ATARI_HORZ_CLIP)
		add		esi, ATARI_HORZ_CLIP
        mov     edi, dword ptr [s_screentemp]	;; the dest pointer
        mov     eax, ATARI_HEIGHT				;; number of lines
scan_line:
        mov     ecx, 054h       ;; Our count of DWORDs to copy (ATARI_VIS_WIDTH/4)
        rep     movsd			;; Move that string
		add		esi, 030h		;; add in the offset to the next scan line for src

        dec     eax
        jnz     scan_line
	}
#else /*WIN_INLINE_ASM*/
	{
		UCHAR *dest, *src, *stop;

		dest = s_screentemp;
		src  = g_screenbuff + ATARI_HORZ_CLIP;
		stop = s_screentemp + ATARI_VIS_SCREEN_SIZE;
		while( dest < stop )
		{
			memcpy( dest, src, ATARI_VIS_WIDTH );
			dest += ATARI_VIS_WIDTH;
			src  += ATARI_WIDTH;
		}
	}
#endif /*WIN_INLINE_ASM*/
	if( DD_OK == (hResult = IDirectDrawSurface_GetDC( s_lpDDSPrimary, &hdc )) )
	{
		StretchDIBits( hdc, s_rcDest.left, s_rcDest.top, s_rcDest.right - s_rcDest.left, s_rcDest.bottom - s_rcDest.top, 
			s_rcSrcClip.left, s_rcSrcClip.top, s_rcSrcClip.right - s_rcSrcClip.left, s_rcSrcClip.bottom - s_rcSrcClip.top, 
			s_screentemp, g_lpbmi, DIB_RGB_COLORS, SRCCOPY);
		IDirectDrawSurface_ReleaseDC( s_lpDDSPrimary, hdc );
	}
	else
	{
		ServeDDrawError( IDS_DDERR_SURFACE_LOCK, hResult, FALSE );
		return;
	}
	if( s_dwVbiFlags )
		IDirectDraw_WaitForVerticalBlank( s_lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );
} /* #OF# Screen_DDraw */

/*========================================================
Function : Screen_DDraw_Stretch
=========================================================*/
/* #FN#
   Draws Atari screen in DirectDraw windowed mode x 2 */
static
void
/* #AS#
   Nothing */
Screen_DDraw_Stretch( UBYTE *screen /* #IN# Pointer to Atari screen */ )
{
	HDC	hdc;
	HRESULT	hResult;

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	Update_LED( (UBYTE *)g_screenbuff + s_ulLEDOffset );
#endif

#ifdef WIN_INLINE_ASM
	_asm
	{
        mov     esi, dword ptr [g_screenbuff]	;; the source pointer (g_screenbuff + ATARI_HORZ_CLIP)
		add		esi, ATARI_HORZ_CLIP
        mov     edi, dword ptr [s_screentemp]	;; the dest pointer
        mov     eax, ATARI_HEIGHT				;; number of lines
scan_line:
        mov     ecx, 054h       ;; Our count of DWORDs to copy (ATARI_VIS_WIDTH/4)
        rep     movsd			;; Move that string
		add		esi, 030h		;; add in the offset to the next scan line for src

        dec     eax
        jnz     scan_line
	}
#else /*WIN_INLINE_ASM*/
	{
		UCHAR *dest, *src, *stop;

		dest = s_screentemp;
		src  = g_screenbuff + ATARI_HORZ_CLIP;
		stop = s_screentemp + ATARI_VIS_SCREEN_SIZE;
		while( dest < stop )
		{
			memcpy( dest, src, ATARI_VIS_WIDTH );
			dest += ATARI_VIS_WIDTH;
			src  += ATARI_WIDTH;
		}
	}
#endif /*WIN_INLINE_ASM*/
	_ASSERT(s_lpDDSOffscrn);

	if( DD_OK == (hResult = IDirectDrawSurface_GetDC( s_lpDDSOffscrn, &hdc )) )
	{
		/* This StretchDIB basically does only the color space conversion to the memory surface */
		StretchDIBits( hdc, 0, 0, ATARI_VIS_WIDTH, ATARI_HEIGHT,
			0, 0, ATARI_VIS_WIDTH, ATARI_HEIGHT,
			s_screentemp, g_lpbmi, DIB_RGB_COLORS, SRCCOPY );
		IDirectDrawSurface_ReleaseDC( s_lpDDSOffscrn, hdc );
	}
	else
	{
		ServeDDrawError( IDS_DDERR_SURFACE_LOCK, hResult, FALSE );
		return;
	}
	if( s_dwVbiFlags )
		IDirectDraw_WaitForVerticalBlank( s_lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );
	/* Blit to primary surface */
	if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_lpDDSPrimary, &s_rcDest, s_lpDDSOffscrn, &s_rcSrcClip, s_dwBltFlags, NULL )) )
		return;

	if( hResult == DDERR_WASSTILLDRAWING )
		return;

	ServeDDrawError( IDS_DDERR_PRIMARY_BLT, hResult, FALSE );
} /* #OF# Screen_DDraw_Stretch */

/*========================================================
Function : Screen_DDraw_Stretch_Scanlines
=========================================================*/
/* #FN#
   Draws Atari screen in windowed mode x 2 with scan lines */
static
void
/* #AS#
   Nothing */
Screen_DDraw_Stretch_Scanlines( UBYTE *screen /* #IN# Pointer to Atari screen */ )
{
	HDC	hdc;
	HRESULT	hResult;

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	Update_LED( (UBYTE *)g_screenbuff + s_ulLEDOffset );
#endif

#ifdef WIN_INLINE_ASM
	_asm
	{
        mov     esi, dword ptr [g_screenbuff]	;; the source pointer (g_screenbuff + ATARI_HORZ_CLIP)
		add		esi, ATARI_HORZ_CLIP
        mov     edi, dword ptr [s_screentemp]	;; the dest pointer
        mov     eax, ATARI_HEIGHT				;; number of line pairs (ATARI_HEIGHT)
new_line:
        mov     ecx, 054h       ;; Our count of DWORDs to copy (ATARI_VIS_WIDTH/4)
        rep     movsd			;; Move that string

		sub		esi, ATARI_VIS_WIDTH			;; Move back to origin of original scan line
		mov		edx, ATARI_VIS_WIDTH			;; The pixel counter
dark_line:
		mov		bl, byte ptr [esi]				;; Make copy of original pixel
		mov		cl, bl							;; Another copy of original pixel
		and		bl, 0x0f						;; Bottom nybble of this pixel
		shr		bl, 1							;; nybble / 2
		and		cl, 0xf0						;; Top nybble of this pixel
		add		cl, bl							;; Make darkened pixel by adding nybbles
		inc		esi								;; move to next source pixel
		mov		byte ptr [edi], cl				;; Copy darkened pixel into destination point
		inc		edi
		dec		edx     						;; decrement our pixel counter
		jnz		dark_line						;; back to start of pixel operation

		add		esi, 030h						;; add in the offset to the next scan line for src
        dec     eax
        jnz     new_line
	}
#else /*WIN_INLINE_ASM*/
	{
		UCHAR *dest, *src, *stop;

		stop = s_screentemp + (ATARI_STRETCH_HEIGHT * ATARI_VIS_WIDTH);
		dest = s_screentemp;
		src  = g_screenbuff + ATARI_HORZ_CLIP;
		while( dest < stop )
		{
			memcpy( dest, src, ATARI_VIS_WIDTH );
			dest += ATARI_VIS_WIDTH * 2;
			src  += ATARI_WIDTH;
		}
	}
#endif /*WIN_INLINE_ASM*/
	_ASSERT(s_lpDDSOffscrn);

	if( DD_OK == (hResult = IDirectDrawSurface_GetDC( s_lpDDSOffscrn, &hdc )) )
	{
		/* This StretchDIB basically does only the color space conversion to the memory surface */
		StretchDIBits( hdc, 0, 0, ATARI_VIS_WIDTH, ATARI_STRETCH_HEIGHT,
			0, 0, ATARI_VIS_WIDTH, ATARI_STRETCH_HEIGHT,
			s_screentemp, g_lpbmi, DIB_RGB_COLORS, SRCCOPY );
		IDirectDrawSurface_ReleaseDC( s_lpDDSOffscrn, hdc );
	}
	else
	{
		ServeDDrawError( IDS_DDERR_SURFACE_LOCK, hResult, FALSE );
		return;
	}
	if( s_dwVbiFlags )
		IDirectDraw_WaitForVerticalBlank( s_lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );
	/* Blit to primary surface */
	if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_lpDDSPrimary, &s_rcDest, s_lpDDSOffscrn, &s_rcSrcClip, s_dwBltFlags, NULL )) )
		return;

	if( hResult == DDERR_WASSTILLDRAWING )
		return;

	ServeDDrawError( IDS_DDERR_PRIMARY_BLT, hResult, FALSE );
} /* #OF# Screen_DDraw_Stretch_Scanlines */

/*========================================================
Function : Screen_DDraw_Full_Blt
=========================================================*/
/* #FN#
   Draws Atari screen in full screen mode */
static
void
/* #AS#
   Nothing */
Screen_DDraw_Full_Blt( UBYTE *screen /* #IN# Pointer to Atari screen */ )
{
	HRESULT	hResult;

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	Update_LED( screen + s_ulLEDOffset );
#endif

#ifdef WIN_USE_FLIP_BUFFER
	if( s_bFlipBuffers )
	{
		IDirectDrawSurface_Blt( s_lpDDSBackBuf, &s_rcDest, s_lpDDSOffscrn, &s_rcSource, s_dwBltFlags, NULL );
		/* Flip the DirectDraw surfaces */
		if( DD_OK == (hResult = IDirectDrawSurface_Flip( s_lpDDSPrimary, NULL, s_dwVbiFlags )) )
			return;
	}
	else
#endif
	{
		if( s_dwVbiFlags )
			IDirectDraw_WaitForVerticalBlank( s_lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );
		/* Blit to primary surface */
		if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_lpDDSPrimary, &s_rcDest, s_lpDDSOffscrn, &s_rcSource, s_dwBltFlags, NULL )) )
			return;
	}
	if( hResult == DDERR_WASSTILLDRAWING )
		return;

    if( hResult == DDERR_SURFACELOST )
	{
		if( !RestoreSurfaces() )
			ServeDDrawError( IDS_DDERR_RESTORE_SURFACE, hResult, FALSE );
	}
	else
		ServeDDrawError( IDS_DDERR_PRIMARY_BLT, hResult, FALSE );
} /* #OF# Screen_DDraw_Full_Blt */

/*========================================================
Function : Screen_DDraw_Full_Blt_Scanlines
=========================================================*/
/* #FN#
   Draws Atari screen in full mode with scan lines */
static
void
/* #AS#
   Nothing */
Screen_DDraw_Full_Blt_Scanlines( UBYTE *screen /* #IN# Pointer to Atari screen */ )
{
	HRESULT	hResult;

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	Update_LED( screen + s_ulLEDOffset );
#endif

#ifdef WIN_INLINE_ASM
	_asm
	{
        mov     esi, dword ptr [g_screenbuff]	;; the source pointer 
        mov     edi, dword ptr [s_offscrsurf]	;; the dest pointer
        mov     eax, ATARI_HEIGHT				;; number of line pairs (ATARI_HEIGHT)
new_line:
        mov     ecx, 060h						;; Our count of DWORDs to copy (ATARI_WIDTH/4)
        rep     movsd							;; Move that string

		sub		esi, ATARI_WIDTH				;; Move back to origin of original scan line
		mov		edx, ATARI_WIDTH				;; The pixel counter
dark_line:
		mov		bl, byte ptr [esi]				;; Make copy of original pixel
		mov		cl, bl							;; Another copy of original pixel
		and		bl, 0x0f						;; Bottom nybble of this pixel
		shr		bl, 1							;; nybble / 2
		and		cl, 0xf0						;; Top nybble of this pixel
		add		cl, bl							;; Make darkened pixel by adding nybbles
		inc		esi								;; move to next source pixel
		mov		byte ptr [edi], cl				;; Copy darkened pixel into destination point
		inc		edi
		dec		edx     						;; decrement our pixel counter
		jnz		dark_line						;; back to start of pixel operation

        dec     eax
        jnz     new_line
	}
#else /*WIN_INLINE_ASM*/
	{
		UCHAR *dest, *src, *stop;

		dest = s_offscrsurf + ATARI_HORZ_CLIP;
		src  = g_screenbuff + ATARI_HORZ_CLIP;
		stop = s_offscrsurf + (ATARI_STRETCH_HEIGHT * ATARI_WIDTH);
		while( dest < stop )
		{
			memcpy( dest, src, ATARI_VIS_WIDTH );
			dest += ATARI_WIDTH * 2;
			src  += ATARI_WIDTH;
		}
	}
#endif /*WIN_INLINE_ASM*/

#ifdef WIN_USE_FLIP_BUFFER
	if( s_bFlipBuffers )
	{
		IDirectDrawSurface_Blt( s_lpDDSBackBuf, &s_rcDest, s_lpDDSOffscrn, &s_rcSource, s_dwBltFlags, NULL );
		/* Flip the DirectDraw surfaces */
		if( DD_OK == (hResult = IDirectDrawSurface_Flip( s_lpDDSPrimary, NULL, s_dwVbiFlags )) )
			return;
	}
	else
#endif
	{
		if( s_dwVbiFlags )
			IDirectDraw_WaitForVerticalBlank( s_lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );
		/* Blit to primary surface */
		if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_lpDDSPrimary, &s_rcDest, s_lpDDSOffscrn, &s_rcSource, s_dwBltFlags, NULL )) )
			return;
	}
	if( hResult == DDERR_WASSTILLDRAWING )
		return;

    if( hResult == DDERR_SURFACELOST )
	{
		if( !RestoreSurfaces() )
			ServeDDrawError( IDS_DDERR_RESTORE_SURFACE, hResult, FALSE );
	}
	else
		ServeDDrawError( IDS_DDERR_PRIMARY_BLT, hResult, FALSE );
} /* #OF# Screen_DDraw_Full_Blt_Scanlines */

/*========================================================
Function : Screen_DDraw_1024_Scanlines
=========================================================*/
/* #FN#
   Draws Atari screen in full 1024 mode with scan lines */
static
void
/* #AS#
   Nothing */
Screen_DDraw_1024_Scanlines( UBYTE *screen /* #IN# Pointer to Atari screen */ )
{
	HRESULT	hResult;

#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	Update_LED( screen + s_ulLEDOffset );
#endif

#ifdef WIN_INLINE_ASM
	_asm
	{
		;; Order in this routine has been manipulated to maximize pairing on a Pentium processor
        mov     esi, dword ptr [g_screenbuff]	;; the source pointer 
        mov     edi, dword ptr [s_offscrsurf]	;; the dest pointer
        mov     eax, ATARI_HEIGHT				;; number of line pairs (ATARI_HEIGHT)
new_line:
        mov     ecx, 060h						;; Our count of DWORDs to copy (ATARI_WIDTH/4)
        rep     movsd							;; Move that string
		sub		esi, ATARI_WIDTH				;; Move back to origin of this scan line
		mov		ecx, 060h						;; Going to move another set of words...
		rep		movsd							;; Make another copy of scanline

		sub		esi, ATARI_WIDTH				;; Move back to origin of original scan line
		mov		edx, ATARI_WIDTH				;; The pixel counter
dark_line:
		mov		bl, byte ptr [esi]				;; Make copy of original pixel
		mov		cl, bl							;; Another copy of original pixel
		and		bl, 0x0f						;; Bottom nybble of this pixel
		shr		bl, 1							;; nybble / 2
		and		cl, 0xf0						;; Top nybble of this pixel
		add		cl, bl							;; Make darkened pixel by adding nybbles
		inc		esi								;; move to next source pixel
		mov		byte ptr [edi], cl				;; Copy darkened pixel into destination point
		inc		edi
		dec		edx     						;; decrement our pixel counter
		jnz		dark_line						;; back to start of pixel operation


        dec     eax								;; decrement our line counter
        jnz     new_line
	}
#else /*WIN_INLINE_ASM*/
	{
		UCHAR *dest, *src, *stop;

		dest = s_offscrsurf + ATARI_HORZ_CLIP;
		src  = g_screenbuff + ATARI_HORZ_CLIP;
		stop = s_offscrsurf + (ATARI_HEIGHT * 3 * ATARI_WIDTH);
		while( dest < stop )
		{
			memcpy( dest, src, ATARI_VIS_WIDTH );
			dest += ATARI_WIDTH;
			memcpy( dest, src, ATARI_VIS_WIDTH );
			dest += ATARI_WIDTH * 2;
			src  += ATARI_WIDTH;
		}
	}
#endif /*WIN_INLINE_ASM*/

#ifdef WIN_USE_FLIP_BUFFER
	if( s_bFlipBuffers )
	{
		IDirectDrawSurface_Blt( s_lpDDSBackBuf, &s_rcDest, s_lpDDSOffscrn, &s_rcSource, s_dwBltFlags, NULL );
		/* Flip the DirectDraw surfaces */
		if( DD_OK == (hResult = IDirectDrawSurface_Flip( s_lpDDSPrimary, NULL, s_dwVbiFlags )) )
			return;
	}
	else
#endif
	{
		if( s_dwVbiFlags )
			IDirectDraw_WaitForVerticalBlank( s_lpDirectDraw, DDWAITVB_BLOCKBEGIN, NULL );
		/* Blit to primary surface */
		if( DD_OK == (hResult = IDirectDrawSurface_Blt( s_lpDDSPrimary, &s_rcDest, s_lpDDSOffscrn, &s_rcSource, s_dwBltFlags, NULL )) )
			return;
	}
	if( hResult == DDERR_WASSTILLDRAWING )
		return;

    if( hResult == DDERR_SURFACELOST )
	{
		if( !RestoreSurfaces() )
			ServeDDrawError( IDS_DDERR_RESTORE_SURFACE, hResult, FALSE );
	}
	else
		ServeDDrawError( IDS_DDERR_PRIMARY_BLT, hResult, FALSE );
} /* #OF# Screen_DDraw_Full_Blt_Scanlines */

/*========================================================
Function : Screen_GDI_Stretch
=========================================================*/
/* #FN#
   Draws Atari screen in GDI windowed mode x 2 */
static
void
/* #AS#
   Nothing */
Screen_GDI_Stretch( UBYTE *screen /* #IN# Pointer to Atari screen */ )
{
#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	Update_LED( screen + s_ulLEDOffset );
#endif

#ifdef WIN_INLINE_ASM
	_asm
	{
		;; Order in this routine has been manipulated to maximize pairing on a Pentium processor
        mov     ebx, dword ptr [g_screenbuff]	;; the source pointer (g_screenbuff + ATARI_HORZ_CLIP)
		add		ebx, ATARI_HORZ_CLIP
		mov     edi, dword ptr [s_screentemp]	;; the dest pointer
		mov		eax, 0f00000h					;; number of scan lines in the source (shifted 16 bits)
new_line_b:
		mov		ecx, 0150h						;; We're going to copy ATARI_VIS_WIDTH pixel pairs
copy_bytes_b:
		mov		dl, byte ptr [ebx]				;; Get this pixel
		mov		byte ptr [edi], dl				;; Store copy 1
		inc		edi								
		inc		ebx								;; Next source
		mov		byte ptr [edi], dl				;; Store copy 2
		inc		edi								;; Next destination

		dec		ecx
		jnz 	copy_bytes_b					;; do this until ATARI_VIS_WIDTH pairs are done

		mov		esi, edi						;; Current destination pointer into source index
		mov     ecx, 0a8h						;; Our count of DWORDs to copy (ATARI_VIS_WIDTH * 2 / 4)
		sub		esi, 02a0h						;; move back ATARI_VIS_WIDTH * 2 (beginning of pixel-doubled scan line)
		add		ebx, 030h						;; add in the offset to the next scan line for src
		rep     movsd							;; Move that string

		sub		eax, 10000h						;; decrement high word of eax
		jnz		new_line_b						;; Jump if we're not done
	}
#else /*WIN_INLINE_ASM*/
	{
		UCHAR *dest, *src, *stop;

		dest = s_screentemp;
		src  = g_screenbuff + ATARI_HORZ_CLIP;
		stop = dest + ATARI_STRETCH_VIS_SCREEN_SIZE;
		while( dest < stop )
		{
			char *linep = dest;
			while( linep < dest + ATARI_STRETCH_VIS_WIDTH )
			{
				*linep++ = *src;
				*linep++ = *src++;
			}
			memcpy( linep, linep - ATARI_STRETCH_VIS_WIDTH, ATARI_STRETCH_VIS_WIDTH );
			dest += ATARI_STRETCH_VIS_WIDTH * 2;
			src  += ATARI_HORZ_CLIP * 2;
		}
	}
#endif /*WIN_INLINE_ASM*/
	StretchDIBits( g_hScreenDC, 0, 0, ATARI_STRETCH_VIS_WIDTH, ATARI_STRETCH_HEIGHT,
		0, 0, ATARI_STRETCH_VIS_WIDTH, ATARI_STRETCH_HEIGHT,
		s_screentemp, g_lpbmi, DIB_RGB_COLORS, SRCCOPY );
} /* #OF# Screen_GDI_Stretch */

/*========================================================
Function : Screen_GDI_Stretch_Scanlines
=========================================================*/
/* #FN#
   Draws Atari screen in GDI windowed mode x 2 with scan lines */
static
void
/* #AS#
   Nothing */
Screen_GDI_Stretch_Scanlines( UBYTE *screen /* #IN# Pointer to Atari screen */ )
{
#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	Update_LED( screen + s_ulLEDOffset );
#endif

#ifdef WIN_INLINE_ASM
	_asm
	{
		;; Order in this routine has been manipulated to maximize pairing on a Pentium processor
        mov     esi, dword ptr [g_screenbuff]	;; the source pointer (g_screenbuff + ATARI_HORZ_CLIP)
		add		esi, ATARI_HORZ_CLIP
		mov     edi, dword ptr [s_screentemp]	;; the dest pointer
		dec		edi
		mov		eax, 0f00000h					;; number of scan lines in the source (shifted 16 bits)
new_line:
		mov		ax, ATARI_VIS_WIDTH				;; We're going to copy ATARI_VIS_WIDTH pixel pairs
copy_scanline:
		mov		dl, byte ptr [esi]								;; Get this pixel
		inc		edi												;; Next destination
		mov		cl, dl											;; Make a copy of original pixel
		mov		bl, dl											;; Another copy of original pixel
		and		bl, 0x0f										;; Bottom nybble of this pixel
		mov		byte ptr [edi], dl								;; Store copy 1 of regular pixel
		shr		bl, 1											;; bits / 2
		and     cl, 0xf0										;; Top nybble of this pixel
		add		bl, cl											;; (pixel val / 16) * 16 + (val % 16) / 2
		mov		byte ptr [edi + ATARI_STRETCH_VIS_WIDTH], bl	;; Store darkline pixel 1
		inc		esi												;; Next source pixel
		inc		edi												;; Next destination
		mov		byte ptr [edi], dl								;; Store copy 2 of regular pixel
		mov		byte ptr [edi + ATARI_STRETCH_VIS_WIDTH], bl	;; Store darkline pixel 2

		dec		ax												;; Decrement counter
		jnz		copy_scanline					;; do this until ATARI_VIS_WIDTH pairs are done

		add		edi, 02a0h						;; skip over the darkline
		add		esi, 030h						;; add in the offset to the next scan line for src

		sub		eax, 10000h						;; decrement high word of eax
		jnz		new_line						;; Jump if we're not done
	}
#else /*WIN_INLINE_ASM*/
	{
		UCHAR *dest, *src, *stop;

		dest = s_screentemp;
		src  = g_screenbuff + ATARI_HORZ_CLIP;
		stop = dest + ATARI_STRETCH_VIS_SCREEN_SIZE;
		while( dest < stop )
		{
			char *linep    = dest;
			char *darkline = linep + ATARI_STRETCH_VIS_WIDTH;
			while( linep < dest + ATARI_STRETCH_VIS_WIDTH )
			{
				*linep++ = *src;
				*darkline++ = (char)s_aunDarklineTranslate[ *src ];
				*linep++ = *src;
				*darkline++ = (char)s_aunDarklineTranslate[ *src++ ];
			}
			dest += ATARI_STRETCH_VIS_WIDTH * 2;
			src  += ATARI_HORZ_CLIP * 2;
		}
	}
#endif /*WIN_INLINE_ASM*/
	StretchDIBits( g_hScreenDC, 0, 0, ATARI_STRETCH_VIS_WIDTH, ATARI_STRETCH_HEIGHT,
		0, 0, ATARI_STRETCH_VIS_WIDTH, ATARI_STRETCH_HEIGHT,
		s_screentemp, g_lpbmi, DIB_RGB_COLORS, SRCCOPY );
} /* #OF# Screen_GDI_Stretch_Scanlines */

/*========================================================
Function : Screen_GDI
=========================================================*/
/* #FN#
   Draws Atari screen in GDI windowed mode x 1 */
static
void
/* #AS#
   Nothing */
Screen_GDI( UBYTE *screen /* #IN# Pointer to Atari screen */ )
{
#if defined(SET_LED) && !defined(NO_LED_ON_SCREEN)
	Update_LED( screen + s_ulLEDOffset );
#endif

#ifdef WIN_INLINE_ASM
	_asm
	{
        mov     esi, dword ptr [g_screenbuff]	;; the source pointer (g_screenbuff + ATARI_HORZ_CLIP)
		add		esi, ATARI_HORZ_CLIP
        mov     edi, dword ptr [s_screentemp]	;; the dest pointer
        mov     eax, ATARI_HEIGHT				;; number of lines
scan_line:
        mov     ecx, 054h       ;; Our count of DWORDs to copy (ATARI_VIS_WIDTH/4)
        rep     movsd			;; Move that string
		add		esi, 030h		;; add in the offset to the next scan line for src

        dec     eax
        jnz     scan_line
	}
#else /*WIN_INLINE_ASM*/
	{
		UCHAR *dest, *src, *stop;

		dest = s_screentemp;
		src  = g_screenbuff + ATARI_HORZ_CLIP;
		stop = s_screentemp + ATARI_VIS_SCREEN_SIZE;
		while( dest < stop )
		{
			memcpy( dest, src, ATARI_VIS_WIDTH );
			dest += ATARI_VIS_WIDTH;
			src  += ATARI_WIDTH;
		}
	}
#endif /*WIN_INLINE_ASM*/
	StretchDIBits( g_hScreenDC, 0, 0, ATARI_VIS_WIDTH, ATARI_HEIGHT, 
		0, 0, ATARI_VIS_WIDTH, ATARI_HEIGHT, 
		s_screentemp, g_lpbmi, DIB_RGB_COLORS, SRCCOPY );
} /* #OF# Screen_GDI */

/*========================================================
Function : GetDDErrorString
=========================================================*/
/* #FN#
   Outputs a debug string to debugger */
static
BOOL
/* #AS#
   TRUE if succeeded, FALSE if failed */
GetDDErrorString( HRESULT hResult,       /* #IN# */
				  LPSTR   lpszErrorBuff, /* #IN# */
				  DWORD   cchError       /* #IN# */ )
{
    DWORD cLen;
    LPSTR lpszError;
    TCHAR szMsg[ 256 ];
	
    /* Check parameters */
    if( !lpszErrorBuff || !cchError )
    {
        /* Error, invalid parameters */
        return FALSE;
    }
	
    switch( hResult )
    {
		/* The request completed successfully */
		case DD_OK:
			lpszError = TEXT("DD_OK");
			break;

#ifdef WIN_TRANSLATE_ERRORS

		/* This object is already initialized */
		case DDERR_ALREADYINITIALIZED:
			lpszError = TEXT("DDERR_ALREADYINITIALIZED");
			break;
		
		/* This surface can not be attached to the requested surface */
		case DDERR_CANNOTATTACHSURFACE:
			lpszError = TEXT("DDERR_CANNOTATTACHSURFACE");
			break;
		
		/* This surface can not be detached from the requested surface */
		case DDERR_CANNOTDETACHSURFACE:
			lpszError = TEXT("DDERR_CANNOTDETACHSURFACE");
			break;
		
		/* Support is currently not available */
		case DDERR_CURRENTLYNOTAVAIL:
			lpszError = TEXT("DDERR_CURRENTLYNOTAVAIL");
			break;
		
		/* An exception was encountered while performing the requested operation */
		case DDERR_EXCEPTION:
			lpszError = TEXT("DDERR_EXCEPTION");
			break;
		
		/* Generic failure */
		case DDERR_GENERIC:
			lpszError = TEXT("DDERR_GENERIC");
			break;
		
		/* Height of rectangle provided is not a multiple of reqd alignment */
		case DDERR_HEIGHTALIGN:
			lpszError = TEXT("DDERR_HEIGHTALIGN");
			break;
		
		/* Unable to match primary surface creation request with existing
		   primary surface */
		case DDERR_INCOMPATIBLEPRIMARY:
			lpszError = TEXT("DDERR_INCOMPATIBLEPRIMARY");
			break;
		
		/* One or more of the caps bits passed to the callback are incorrect */
		case DDERR_INVALIDCAPS:
			lpszError = TEXT("DDERR_INVALIDCAPS");
			break;
		
		/* DirectDraw does not support provided Cliplist */
		case DDERR_INVALIDCLIPLIST:
			lpszError = TEXT("DDERR_INVALIDCLIPLIST");
			break;
		
		/* DirectDraw does not support the requested mode */
		case DDERR_INVALIDMODE:
			lpszError = TEXT("DDERR_INVALIDMODE");
			break;
		
		/* DirectDraw received a pointer that was an invalid DIRECTDRAW object */
		case DDERR_INVALIDOBJECT:
			lpszError = TEXT("DDERR_INVALIDOBJECT");
			break;
		
		/* One or more of the parameters passed to the callback function are
		   incorrect */
		case DDERR_INVALIDPARAMS:
			lpszError = TEXT("DDERR_INVALIDPARAMS");
			break;
		
		/* Pixel format was invalid as specified */
		case DDERR_INVALIDPIXELFORMAT:
			lpszError = TEXT("DDERR_INVALIDPIXELFORMAT");
			break;
		
		/* Rectangle provided was invalid */
		case DDERR_INVALIDRECT:
			lpszError = TEXT("DDERR_INVALIDRECT");
			break;
		
		/* Operation could not be carried out because one or more surfaces are locked */
		case DDERR_LOCKEDSURFACES:
			lpszError = TEXT("DDERR_LOCKEDSURFACES");
			break;
		
		/* There is no 3D present. */
		case DDERR_NO3D:
			lpszError = TEXT("DDERR_NO3D");
			break;
		
		/* Operation could not be carried out because there is no alpha accleration
		   hardware present or available */
		case DDERR_NOALPHAHW:
			lpszError = TEXT("DDERR_NOALPHAHW");
			break;
		
		/* no clip list available */
		case DDERR_NOCLIPLIST:
			lpszError = TEXT("DDERR_NOCLIPLIST");
			break;
		
		/* Operation could not be carried out because there is no color conversion
		   hardware present or available */
		case DDERR_NOCOLORCONVHW:
			lpszError = TEXT("DDERR_NOCOLORCONVHW");
			break;
		
		/* Create function called without DirectDraw object method SetCooperativeLevel
		   being called */
		case DDERR_NOCOOPERATIVELEVELSET:
			lpszError = TEXT("DDERR_NOCOOPERATIVELEVELSET");
			break;
		
		/* Surface doesn't currently have a color key */
		case DDERR_NOCOLORKEY:
			lpszError = TEXT("DDERR_NOCOLORKEY");
			break;
		
		/* Operation could not be carried out because there is no hardware support
		   of the dest color key */
		case DDERR_NOCOLORKEYHW:
			lpszError = TEXT("DDERR_NOCOLORKEYHW");
			break;
		
		/* No DirectDraw support possible with current display driver */
		case DDERR_NODIRECTDRAWSUPPORT:
			lpszError = TEXT("DDERR_NODIRECTDRAWSUPPORT");
			break;
		
		/* Operation requires the application to have exclusive mode but the
		   application does not have exclusive mode */
		case DDERR_NOEXCLUSIVEMODE:
			lpszError = TEXT("DDERR_NOEXCLUSIVEMODE");
			break;
		
		/* Flipping visible surfaces is not supported */
		case DDERR_NOFLIPHW:
			lpszError = TEXT("DDERR_NOFLIPHW");
			break;
		
		/* There is no GDI present */
		case DDERR_NOGDI:
			lpszError = TEXT("DDERR_NOGDI");
			break;
		
		/* Operation could not be carried out because there is no hardware present
		   or available */
		case DDERR_NOMIRRORHW:
			lpszError = TEXT("DDERR_NOMIRRORHW");
			break;
		
		/* Requested item was not found */
		case DDERR_NOTFOUND:
			lpszError = TEXT("DDERR_NOTFOUND");
			break;
		
		/* Operation could not be carried out because there is no overlay hardware
		   present or available */
		case DDERR_NOOVERLAYHW:
			lpszError = TEXT("DDERR_NOOVERLAYHW");
			break;
		
		/* Operation could not be carried out because there is no appropriate raster
		   op hardware present or available */
		case DDERR_NORASTEROPHW:
			lpszError = TEXT("DDERR_NORASTEROPHW");
			break;
		
		/* Operation could not be carried out because there is no rotation hardware
		   present or available */
		case DDERR_NOROTATIONHW:
			lpszError = TEXT("DDERR_NOROTATIONHW");
			break;
		
		/* Operation could not be carried out because there is no hardware support
		   for stretching */
		case DDERR_NOSTRETCHHW:
			lpszError = TEXT("DDERR_NOSTRETCHHW");
			break;
		
		/* DirectDrawSurface is not in 4 bit color palette and the requested operation
		   requires 4 bit color palette */
		case DDERR_NOT4BITCOLOR:
			lpszError = TEXT("DDERR_NOT4BITCOLOR");
			break;
		
		/* DirectDrawSurface is not in 4 bit color index palette and the requested
		   operation requires 4 bit color index palette */
		case DDERR_NOT4BITCOLORINDEX:
			lpszError = TEXT("DDERR_NOT4BITCOLORINDEX");
			break;
		
		/* DirectDraw Surface is not in 8 bit color mode and the requested operation
		   requires 8 bit color */
		case DDERR_NOT8BITCOLOR:
			lpszError = TEXT("DDERR_NOT8BITCOLOR");
			break;
		
		/* Operation could not be carried out because there is no texture mapping
		   hardware present or available */
		case DDERR_NOTEXTUREHW:
			lpszError = TEXT("DDERR_NOTEXTUREHW");
			break;
		
		/* Operation could not be carried out because there is no hardware support
		   for vertical blank synchronized operations */
		case DDERR_NOVSYNCHW:
			lpszError = TEXT("DDERR_NOVSYNCHW");
			break;
		
		/* Operation could not be carried out because there is no hardware support
		   for zbuffer blting */
		case DDERR_NOZBUFFERHW:
			lpszError = TEXT("DDERR_NOZBUFFERHW");
			break;
		
		/* Overlay surfaces could not be z layered based on their BltOrder because
		   the hardware does not support z layering of overlays */
		case DDERR_NOZOVERLAYHW:
			lpszError = TEXT("DDERR_NOZOVERLAYHW");
			break;
		
		/* The hardware needed for the requested operation has already been
		   allocated */
		case DDERR_OUTOFCAPS:
			lpszError = TEXT("DDERR_OUTOFCAPS");
			break;
		
		/* DirectDraw does not have enough memory to perform the operation */
		case DDERR_OUTOFMEMORY:
			lpszError = TEXT("DDERR_OUTOFMEMORY");
			break;
		
		/* DirectDraw does not have enough memory to perform the operation */
		case DDERR_OUTOFVIDEOMEMORY:
			lpszError = TEXT("DDERR_OUTOFVIDEOMEMORY");
			break;
		
		/* Hardware does not support clipped overlays */
		case DDERR_OVERLAYCANTCLIP:
			lpszError = TEXT("DDERR_OVERLAYCANTCLIP");
			break;
		
		/* Can only have ony color key active at one time for overlays */
		case DDERR_OVERLAYCOLORKEYONLYONEACTIVE:
			lpszError = TEXT("DDERR_OVERLAYCOLORKEYONLYONEACTIVE");
			break;
		
		/* Access to this palette is being refused because the palette is already
		   locked by another thread */
		case DDERR_PALETTEBUSY:
			lpszError = TEXT("DDERR_PALETTEBUSY");
			break;
		
		/* No src color key specified for this operation */
		case DDERR_COLORKEYNOTSET:
			lpszError = TEXT("DDERR_COLORKEYNOTSET");
			break;
		
		/* This surface is already attached to the surface it is being attached to */
		case DDERR_SURFACEALREADYATTACHED:
			lpszError = TEXT("DDERR_SURFACEALREADYATTACHED");
			break;
		
		/* This surface is already a dependency of the surface it is being made a
		   dependency of */
		case DDERR_SURFACEALREADYDEPENDENT:
			lpszError = TEXT("DDERR_SURFACEALREADYDEPENDENT");
			break;
		
		/* Access to this surface is being refused because the surface is already
		   locked by another thread */
		case DDERR_SURFACEBUSY:
			lpszError = TEXT("DDERR_SURFACEBUSY");
			break;
		
		/* Access to this surface is being refused because no driver exists
		   which can supply a pointer to the surface.
		   This is most likely to happen when attempting to lock the primary
		   surface when no DCI provider is present.
		   Will also happen on attempts to lock an optimized surface. */
		case DDERR_CANTLOCKSURFACE:
			lpszError = TEXT("DDERR_CANTLOCKSURFACE");
			break;
		
		/* Access to Surface refused because Surface is obscured */
		case DDERR_SURFACEISOBSCURED:
			lpszError = TEXT("DDERR_SURFACEISOBSCURED");
			break;
		
		/* Access to this surface is being refused because the surface is gone.
		   The DIRECTDRAWSURFACE object representing this surface should
		   have Restore called on it. */
		case DDERR_SURFACELOST:
			lpszError = TEXT("DDERR_SURFACELOST");
			break;
		
		/* The requested surface is not attached */
		case DDERR_SURFACENOTATTACHED:
			lpszError = TEXT("DDERR_SURFACENOTATTACHED");
			break;
		
		/* Height requested by DirectDraw is too large */
		case DDERR_TOOBIGHEIGHT:
			lpszError = TEXT("DDERR_TOOBIGHEIGHT");
			break;
		
		/* Size requested by DirectDraw is too large. The individual height and
		   width are OK */
		case DDERR_TOOBIGSIZE:
			lpszError = TEXT("DDERR_TOOBIGSIZE");
			break;
		
		/* Width requested by DirectDraw is too large */
		case DDERR_TOOBIGWIDTH:
			lpszError = TEXT("DDERR_TOOBIGWIDTH");
			break;
		
		/* Action not supported */
		case DDERR_UNSUPPORTED:
			lpszError = TEXT("DDERR_UNSUPPORTED");
			break;
		
		/* FOURCC format requested is unsupported by DirectDraw */
		case DDERR_UNSUPPORTEDFORMAT:
			lpszError = TEXT("DDERR_UNSUPPORTEDFORMAT");
			break;
		
		/* Bitmask in the pixel format requested is unsupported by DirectDraw */
		case DDERR_UNSUPPORTEDMASK:
			lpszError = TEXT("DDERR_UNSUPPORTEDMASK");
			break;
		
		/* Vertical blank is in progress */
		case DDERR_VERTICALBLANKINPROGRESS:
			lpszError = TEXT("DDERR_VERTICALBLANKINPROGRESS");
			break;
		
		/* Informs DirectDraw that the previous Blt which is transfering information
		   to or from this Surface is incomplete */
		case DDERR_WASSTILLDRAWING:
			lpszError = TEXT("DDERR_WASSTILLDRAWING");
			break;
		
		/* Rectangle provided was not horizontally aligned on reqd. boundary */
		case DDERR_XALIGN:
			lpszError = TEXT("DDERR_XALIGN");
			break;
		
		/* The GUID passed to DirectDrawCreate is not a valid DirectDraw driver
		   identifier */
		case DDERR_INVALIDDIRECTDRAWGUID:
			lpszError = TEXT("DDERR_INVALIDDIRECTDRAWGUID");
			break;
		
		/* A DirectDraw object representing this driver has already been created
		   for this process */
		case DDERR_DIRECTDRAWALREADYCREATED:
			lpszError = TEXT("DDERR_DIRECTDRAWALREADYCREATED");
			break;
		
		/* A hardware only DirectDraw object creation was attempted but the driver
		   did not support any hardware */
		case DDERR_NODIRECTDRAWHW:
			lpszError = TEXT("DDERR_NODIRECTDRAWHW");
			break;
		
		/* this process already has created a primary surface */
		case DDERR_PRIMARYSURFACEALREADYEXISTS:
			lpszError = TEXT("DDERR_PRIMARYSURFACEALREADYEXISTS");
			break;
		
		/* software emulation not available */
		case DDERR_NOEMULATION:
			lpszError = TEXT("DDERR_NOEMULATION");
			break;
		
		/* Region passed to Clipper::GetClipList is too small */
		case DDERR_REGIONTOOSMALL:
			lpszError = TEXT("DDERR_REGIONTOOSMALL");
			break;
		
		/* An attempt was made to set a clip list for a clipper objec that
		   is already monitoring an hwnd */
		case DDERR_CLIPPERISUSINGHWND:
			lpszError = TEXT("DDERR_CLIPPERISUSINGHWND");
			break;
		
		/* No clipper object attached to surface object */
		case DDERR_NOCLIPPERATTACHED:
			lpszError = TEXT("DDERR_NOCLIPPERATTACHED");
			break;
		
		/* Clipper notification requires an HWND or
		   no HWND has previously been set as the CooperativeLevel HWND */
		case DDERR_NOHWND:
			lpszError = TEXT("DDERR_NOHWND");
			break;
		
		/* HWND used by DirectDraw CooperativeLevel has been subclassed,
		   this prevents DirectDraw from restoring state */
		case DDERR_HWNDSUBCLASSED:
			lpszError = TEXT("DDERR_HWNDSUBCLASSED");
			break;
		
		/* The CooperativeLevel HWND has already been set.
		   It can not be reset while the process has surfaces or palettes created */
		case DDERR_HWNDALREADYSET:
			lpszError = TEXT("DDERR_HWNDALREADYSET");
			break;
		
		/* No palette object attached to this surface */
		case DDERR_NOPALETTEATTACHED:
			lpszError = TEXT("DDERR_NOPALETTEATTACHED");
			break;
		
		/* No hardware support for 16 or 256 color palettes */
		case DDERR_NOPALETTEHW:
			lpszError = TEXT("DDERR_NOPALETTEHW");
			break;
		
		/* If a clipper object is attached to the source surface passed into a
		   BltFast call */
		case DDERR_BLTFASTCANTCLIP:
			lpszError = TEXT("DDERR_BLTFASTCANTCLIP");
			break;
		
		/* No blter */
		case DDERR_NOBLTHW:
			lpszError = TEXT("DDERR_NOBLTHW");
			break;
		
		/* No DirectDraw ROP hardware */
		case DDERR_NODDROPSHW:
			lpszError = TEXT("DDERR_NODDROPSHW");
			break;
		
		/* Returned when GetOverlayPosition is called on a hidden overlay */
		case DDERR_OVERLAYNOTVISIBLE:
			lpszError = TEXT("DDERR_OVERLAYNOTVISIBLE");
			break;
		
		/* Returned when GetOverlayPosition is called on a overlay that UpdateOverlay
		   has never been called on to establish a destionation */
		case DDERR_NOOVERLAYDEST:
			lpszError = TEXT("DDERR_NOOVERLAYDEST");
			break;
		
		/* returned when the position of the overlay on the destionation is no longer
		   legal for that destionation */
		case DDERR_INVALIDPOSITION:
			lpszError = TEXT("DDERR_INVALIDPOSITION");
			break;
		
		/* Returned when an overlay member is called for a non-overlay surface */
		case DDERR_NOTAOVERLAYSURFACE:
			lpszError = TEXT("DDERR_NOTAOVERLAYSURFACE");
			break;
		
		/* An attempt was made to set the cooperative level when it was already
		   set to exclusive */
		case DDERR_EXCLUSIVEMODEALREADYSET:
			lpszError = TEXT("DDERR_EXCLUSIVEMODEALREADYSET");
			break;
		
		/* An attempt has been made to flip a surface that is not flippable */
		case DDERR_NOTFLIPPABLE:
			lpszError = TEXT("DDERR_NOTFLIPPABLE");
			break;
		
		/* Can't duplicate primary & 3D surfaces, or surfaces that are implicitly
		   created */
		case DDERR_CANTDUPLICATE:
			lpszError = TEXT("DDERR_CANTDUPLICATE");
			break;
		
		/* Surface was not locked.  An attempt to unlock a surface that was not
		   locked at all, or by this process, has been attempted */
		case DDERR_NOTLOCKED:
			lpszError = TEXT("DDERR_NOTLOCKED");
			break;
		
		/* Windows can not create any more DCs */
		case DDERR_CANTCREATEDC:
			lpszError = TEXT("DDERR_CANTCREATEDC");
			break;
		
		/* No DC was ever created for this surface */
		case DDERR_NODC:
			lpszError = TEXT("DDERR_NODC");
			break;
		
		/* This surface can not be restored because it was created in a different
		   mode */
		case DDERR_WRONGMODE:
			lpszError = TEXT("DDERR_WRONGMODE");
			break;
		
		/* This surface can not be restored because it is an implicitly created
		   surface */
		case DDERR_IMPLICITLYCREATED:
			lpszError = TEXT("DDERR_IMPLICITLYCREATED");
			break;
		
		/* The surface being used is not a palette-based surface */
		case DDERR_NOTPALETTIZED:
			lpszError = TEXT("DDERR_NOTPALETTIZED");
			break;
		
		/* The display is currently in an unsupported mode */
		case DDERR_UNSUPPORTEDMODE:
			lpszError = TEXT("DDERR_UNSUPPORTEDMODE");
			break;
		
		/* Operation could not be carried out because there is no mip-map
		   texture mapping hardware present or available */
		case DDERR_NOMIPMAPHW:
			lpszError = TEXT("DDERR_NOMIPMAPHW");
			break;
		
		/* The requested action could not be performed because the surface was of
		   the wrong type */
		case DDERR_INVALIDSURFACETYPE:
			lpszError = TEXT("DDERR_INVALIDSURFACETYPE");
			break;
		
		/* Device does not support optimized surfaces, therefore no video memory optimized surfaces */
		case DDERR_NOOPTIMIZEHW:
			lpszError = TEXT("DDERR_NOOPTIMIZEHW");
			break;
		
		/* Surface is an optimized surface, but has not yet been allocated any memory */
		case DDERR_NOTLOADED:
			lpszError = TEXT("DDERR_NOTLOADED");
			break;
		
		/* A DC has already been returned for this surface. Only one DC can be
		   retrieved per surface */
		case DDERR_DCALREADYCREATED:
			lpszError = TEXT("DDERR_DCALREADYCREATED");
			break;
		
		/* An attempt was made to allocate non-local video memory from a device
		   that does not support non-local video memory */
		case DDERR_NONONLOCALVIDMEM:
			lpszError = TEXT("DDERR_NONONLOCALVIDMEM");
			break;
		
		/* The attempt to page lock a surface failed */
		case DDERR_CANTPAGELOCK:
			lpszError = TEXT("DDERR_CANTPAGELOCK");
			break;
		
		/* The attempt to page unlock a surface failed */
		case DDERR_CANTPAGEUNLOCK:
			lpszError = TEXT("DDERR_CANTPAGEUNLOCK");
			break;
		
		/* An attempt was made to page unlock a surface with no outstanding page locks */
		case DDERR_NOTPAGELOCKED:
			lpszError = TEXT("DDERR_NOTPAGELOCKED");
			break;
		
		/* There is more data available than the specified buffer size could hold */
		case DDERR_MOREDATA:
			lpszError = TEXT("DDERR_MOREDATA");
			break;
		
		/* The video port is not active */
		case DDERR_VIDEONOTACTIVE:
			lpszError = TEXT("DDERR_VIDEONOTACTIVE");
			break;
		
		/* Surfaces created by one direct draw device cannot be used directly by
		   another direct draw device */
		case DDERR_DEVICEDOESNTOWNSURFACE:
			lpszError = TEXT("DDERR_DEVICEDOESNTOWNSURFACE");
			break;
		
		/* An attempt was made to invoke an interface member of a DirectDraw object
		   created by CoCreateInstance() before it was initialized */
		case DDERR_NOTINITIALIZED:
			lpszError = TEXT("DDERR_NOTINITIALIZED");
			break;

#endif /*WIN_TRANSLATE_ERRORS*/

			/* Unknown DD/App Error */
		default:
			wsprintf( szMsg, "Error #%ld", (DWORD)hResult );
			lpszError = szMsg;
			break;
    }
	
    /* Copy DD Error string to buff */
    cLen = strlen( lpszError );
    if( cLen >= cchError )
    {
        cLen = cchError - 1;
    }
    if( cLen )
    {
        strncpy( lpszErrorBuff, lpszError, cLen );
        lpszErrorBuff[ cLen ] = 0;
    }
	return TRUE;
} /* #OF# GetDDErrorString */
