/****************************************************************************
File    : PaletteDlg.cpp
/*
@(#) #SY# Atari800Win PLus
@(#) #IS# CPaletteDlg implementation file
@(#) #BY# Tomasz Szymankowski
@(#) #LM# 23.11.2003
*/

#include "StdAfx.h"
#include "Atari800Win.h"
#include "Helpers.h"
#include "FileService.h"
#include "PaletteDlg.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#define IDC_PALETTE_FIRST	IDC_PALETTE_BLACKLEVEL
#define IDC_PALETTE_LAST	IDC_PALETTE_CANCEL

#define MIN_BLACK			0
#define MAX_BLACK			255
#define MIN_WHITE			0
#define MAX_WHITE			255
#define MIN_INTENSITY		0
#define MAX_INTENSITY		255
#define MIN_SHIFT			0
#define MAX_SHIFT			255

#define BAR_LINES_NO		16
#define BAR_ENTRIES_NO		(PAL_ENTRIES_NO / BAR_LINES_NO)

static int m_colortable[256] = {
 0x2d2d2d, 0x3b3b3b, 0x494949, 0x575757,
 0x656565, 0x737373, 0x818181, 0x8f8f8f,
 0x9d9d9d, 0xababab, 0xb9b9b9, 0xc7c7c7,
 0xd5d5d5, 0xe3e3e3, 0xf1f1f1, 0xffffff,
 0x5c2300, 0x6a3100, 0x783f00, 0x864d0a,
 0x945b18, 0xa26926, 0xb07734, 0xbe8542,
 0xcc9350, 0xdaa15e, 0xe8af6c, 0xf6bd7a,
 0xffcb88, 0xffd996, 0xffe7a4, 0xfff5b2,
 0x691409, 0x772217, 0x853025, 0x933e33,
 0xa14c41, 0xaf5a4f, 0xbd685d, 0xcb766b,
 0xd98479, 0xe79287, 0xf5a095, 0xffaea3,
 0xffbcb1, 0xffcabf, 0xffd8cd, 0xffe6db,
 0x6c0a38, 0x7a1846, 0x882654, 0x963462,
 0xa44270, 0xb2507e, 0xc05e8c, 0xce6c9a,
 0xdc7aa8, 0xea88b6, 0xf896c4, 0xffa4d2,
 0xffb2e0, 0xffc0ee, 0xffcefc, 0xffdcff,
 0x640565, 0x721373, 0x802181, 0x8e2f8f,
 0x9c3d9d, 0xaa4bab, 0xb859b9, 0xc667c7,
 0xd475d5, 0xe283e3, 0xf091f1, 0xfe9fff,
 0xffadff, 0xffbbff, 0xffc9ff, 0xffd7ff,
 0x520789, 0x601597, 0x6e23a5, 0x7c31b3,
 0x8a3fc1, 0x984dcf, 0xa65bdd, 0xb469eb,
 0xc277f9, 0xd085ff, 0xde93ff, 0xeca1ff,
 0xfaafff, 0xffbdff, 0xffcbff, 0xffd9ff,
 0x3a109c, 0x481eaa, 0x562cb8, 0x643ac6,
 0x7248d4, 0x8056e2, 0x8e64f0, 0x9c72fe,
 0xaa80ff, 0xb88eff, 0xc69cff, 0xd4aaff,
 0xe2b8ff, 0xf0c6ff, 0xfed4ff, 0xffe2ff,
 0x1f1e9c, 0x2d2caa, 0x3b3ab8, 0x4948c6,
 0x5756d4, 0x6564e2, 0x7372f0, 0x8180fe,
 0x8f8eff, 0x9d9cff, 0xabaaff, 0xb9b8ff,
 0xc7c6ff, 0xd5d4ff, 0xe3e2ff, 0xf1f0ff,
 0x072e89, 0x153c97, 0x234aa5, 0x3158b3,
 0x3f66c1, 0x4d74cf, 0x5b82dd, 0x6990eb,
 0x779ef9, 0x85acff, 0x93baff, 0xa1c8ff,
 0xafd6ff, 0xbde4ff, 0xcbf2ff, 0xd9ffff,
 0x003e65, 0x034c73, 0x115a81, 0x1f688f,
 0x2d769d, 0x3b84ab, 0x4992b9, 0x57a0c7,
 0x65aed5, 0x73bce3, 0x81caf1, 0x8fd8ff,
 0x9de6ff, 0xabf4ff, 0xb9ffff, 0xc7ffff,
 0x004b38, 0x005946, 0x096754, 0x177562,
 0x258370, 0x33917e, 0x419f8c, 0x4fad9a,
 0x5dbba8, 0x6bc9b6, 0x79d7c4, 0x87e5d2,
 0x95f3e0, 0xa3ffee, 0xb1fffc, 0xbfffff,
 0x005209, 0x006017, 0x0c6e25, 0x1a7c33,
 0x288a41, 0x36984f, 0x44a65d, 0x52b46b,
 0x60c279, 0x6ed087, 0x7cde95, 0x8aeca3,
 0x98fab1, 0xa6ffbf, 0xb4ffcd, 0xc2ffdb,
 0x005300, 0x0b6100, 0x196f00, 0x277d0a,
 0x358b18, 0x439926, 0x51a734, 0x5fb542,
 0x6dc350, 0x7bd15e, 0x89df6c, 0x97ed7a,
 0xa5fb88, 0xb3ff96, 0xc1ffa4, 0xcfffb2,
 0x134e00, 0x215c00, 0x2f6a00, 0x3d7800,
 0x4b8600, 0x59940b, 0x67a219, 0x75b027,
 0x83be35, 0x91cc43, 0x9fda51, 0xade85f,
 0xbbf66d, 0xc9ff7b, 0xd7ff89, 0xe5ff97,
 0x2d4300, 0x3b5100, 0x495f00, 0x576d00,
 0x657b00, 0x738901, 0x81970f, 0x8fa51d,
 0x9db32b, 0xabc139, 0xb9cf47, 0xc7dd55,
 0xd5eb63, 0xe3f971, 0xf1ff7f, 0xffff8d,
 0x463300, 0x544100, 0x624f00, 0x705d00,
 0x7e6b00, 0x8c790b, 0x9a8719, 0xa89527,
 0xb6a335, 0xc4b143, 0xd2bf51, 0xe0cd5f,
 0xeedb6d, 0xfce97b, 0xfff789, 0xffff97
};

/////////////////////////////////////////////////////////////////////////////
// CPaletteDlg dialog

BEGIN_MESSAGE_MAP(CPaletteDlg, CCommonDlg)
	//{{AFX_MSG_MAP(CPaletteDlg)
	ON_NOTIFY(UDN_DELTAPOS, IDC_PALETTE_BLACKSPIN, OnDeltaposBlackSpin)
	ON_EN_KILLFOCUS(IDC_PALETTE_BLACKLEVEL, OnKillfocusBlackLevel)
	ON_NOTIFY(UDN_DELTAPOS, IDC_PALETTE_WHITESPIN, OnDeltaposWhiteSpin)
	ON_EN_KILLFOCUS(IDC_PALETTE_WHITELEVEL, OnKillfocusWhiteLevel)
	ON_NOTIFY(UDN_DELTAPOS, IDC_PALETTE_SATURATIONSPIN, OnDeltaposSaturationSpin)
	ON_EN_KILLFOCUS(IDC_PALETTE_SATURATION, OnKillfocusSaturation)
	ON_NOTIFY(UDN_DELTAPOS, IDC_PALETTE_SHIFTSPIN, OnDeltaposShiftSpin)
	ON_EN_KILLFOCUS(IDC_PALETTE_COLORSHIFT, OnKillfocusColorShift)
	ON_BN_CLICKED(IDC_PALETTE_APPLYADJUSTMENT, OnApplyAdjustment)
	ON_BN_CLICKED(IDC_PALETTE_BROWSE, OnBrowse)
	ON_EN_KILLFOCUS(IDC_PALETTE_EDIT, OnKillfocusEdit)
	ON_BN_CLICKED(IDC_PALETTE_USEEXTERNAL, OnUseExternal)
	ON_WM_PAINT()
	//}}AFX_MSG_MAP
	ON_BN_CLICKED(IDC_PALETTE_OK, OnOK)
	ON_BN_CLICKED(IDC_PALETTE_CANCEL, OnCancel)
END_MESSAGE_MAP()

/*========================================================
Method   : CPaletteDlg::CPaletteDlg
=========================================================*/
/* #FN#
   Standard constructor */
CPaletteDlg::
CPaletteDlg(
	CWnd *pParent /*=NULL*/ /* #IN# Pointer to the parent window */
)
	: CCommonDlg( _IsFlagSet( g_Screen.ulState, SM_MODE_FULL ) ? IDD_PALETTE_SMALL : CPaletteDlg::IDD, pParent)
{
	//{{AFX_DATA_INIT(CPaletteDlg)
		// NOTE: the ClassWizard will add member initialization here
	//}}AFX_DATA_INIT

	m_bExitPass  = FALSE;
	m_nFirstCtrl = IDC_PALETTE_FIRST;
	m_nLastCtrl  = IDC_PALETTE_LAST;

} /* #OF# CPaletteDlg::CPaletteDlg */

/*========================================================
Method   : CPaletteDlg::DoDataExchange
=========================================================*/
/* #FN#
   Dynamic Data Exchange (not used) */
void
/* #AS#
   Nothing */
CPaletteDlg::
DoDataExchange(
	CDataExchange *pDX /* #IN# Pointer to CDataExchange object */
)
{
	CCommonDlg::DoDataExchange( pDX );
	//{{AFX_DATA_MAP(CPaletteDlg)
		// NOTE: the ClassWizard will add DDX and DDV calls here
	//}}AFX_DATA_MAP

} /* #OF# CPaletteDlg::DoDataExchange */


/////////////////////////////////////////////////////////////////////////////
// CPaletteDlg implementation

/*========================================================
Method   : CPaletteDlg::PreparePalette
=========================================================*/
/* #FN#
   Generates new or reads and format an external palette from a file */
BOOL
/* #AS#
   Nothing */
CPaletteDlg::
PreparePalette(
	LPSTR pszPaletteFile
)
{
	BOOL bResult = FALSE;

	if( _IsFlagSet( m_ulMiscState, MS_USE_EXT_PALETTE ) &&
		/* Load an external palette only if there
		   is an appropriate option checked off */
		_IsPathAvailable( pszPaletteFile ) )
	{
		if( !Palette_Read( pszPaletteFile ) )
		{
			DisplayMessage( GetSafeHwnd(), IDS_ERROR_ACT_READ, 0, MB_ICONEXCLAMATION | MB_OK );
			strcpy( pszPaletteFile, FILE_NONE );
		}
		else
		{
			/* Should we generate or format an external palette? */
			if( _IsFlagSet( m_ulMiscState, MS_TRANS_LOADED_PAL ) )
			{
				Palette_Adjust( m_nBlackLevel, m_nWhiteLevel, m_nColorShift );
			}
			bResult = TRUE;
		}
	}

	if ( !bResult )
		memcpy( colortable, m_colortable, 256*sizeof(int));

	/* Palette bar is drawn in windowed modes only */
	if( !m_bSmallMode )
	{
		CWnd *pStatic = GetDlgItem( IDC_PALETTE_BAR );
		RECT  rc;

		ASSERT(NULL != pStatic);
		/* There is no reason to make the whole window invalidated */
		pStatic->GetWindowRect( &rc );
		ScreenToClient( &rc );
		/* This makes the changes visible */
		InvalidateRect( &rc, FALSE );
	}
	return bResult;

} /* #OF# CPaletteDlg::PreparePalette */

/*========================================================
Method   : CPaletteDlg::PaintPalette
=========================================================*/
/* #FN#
   Fills the palette bar */
BOOL
/* #AS#
   TRUE if succeeded, otherwise FALSE */
CPaletteDlg::
PaintPalette(
	CDC *pDC
)
{
	CPalette     palPalette;
	CPalette    *pOldPalette;
	CBrush      *pBrush;
	CBrush      *pOldBrush;
	int          nRGB;
	HANDLE       hLogPal;    /* Handle to a logical palette */
	LPLOGPALETTE lpPal;      /* Pointer to a logical palette */
	BOOL         bResult;

	/* Allocate memory block for logical palette */
	hLogPal = ::GlobalAlloc( GHND/*LPTR*/, sizeof(LOGPALETTE) + sizeof(PALETTEENTRY) * PAL_ENTRIES_NO );
	/* If not enough memory, clean up and return NULL */
	if( !hLogPal )
		return FALSE;

	lpPal = (LPLOGPALETTE)::GlobalLock( (HGLOBAL)hLogPal );
	/* Init palette header info */
	lpPal->palVersion    = 0x300;
	lpPal->palNumEntries = (WORD)PAL_ENTRIES_NO;

	/* Copy entries */
	for( int i = 0; i < PAL_ENTRIES_NO; i++ )
	{
		nRGB = colortable[ i ];
		
		lpPal->palPalEntry[ i ].peRed   = (nRGB & 0x00ff0000) >> 16;
		lpPal->palPalEntry[ i ].peGreen = (nRGB & 0x0000ff00) >> 8;
		lpPal->palPalEntry[ i ].peBlue  = nRGB & 0x000000ff;
		lpPal->palPalEntry[ i ].peFlags = 0;
	}
	/* Create palette */
	bResult = (BOOL)(palPalette.CreatePalette( lpPal ));

	::GlobalUnlock( (HGLOBAL) hLogPal );
	::GlobalFree( (HGLOBAL) hLogPal );

	if( bResult )
	{
		int nWidth   = (m_rcPalBar.Width() - BAR_ENTRIES_NO) / BAR_ENTRIES_NO;
		int nHeight  = (m_rcPalBar.Height() - BAR_LINES_NO) / BAR_LINES_NO;

		int nOffsetX = (m_rcPalBar.Width() - (nWidth + 1) * BAR_ENTRIES_NO) / 2 + 1;
		int nOffsetY = (m_rcPalBar.Height() - (nHeight + 1) * BAR_LINES_NO) / 2 + 1;

		/* Select and realize palette */
		pOldPalette = pDC->SelectPalette( &palPalette, TRUE );
		pDC->RealizePalette();

		for( int i = 0; i < BAR_LINES_NO; i++ ) /* Eight lines x 32 color picks */
		{
			/* Draw each stripe */
			for( int j = 0; j < BAR_ENTRIES_NO; j++ )
			{
				pBrush = new CBrush;
				/* A palette-relative pBrush is used for drawing */
				if( TRUE == pBrush->CreateSolidBrush( PALETTEINDEX( j + (i * BAR_ENTRIES_NO) ) ) )
				{
					/* Setup pBrush */
					pOldBrush = pDC->SelectObject( pBrush );
					pBrush->UnrealizeObject();
					/* Draw stripe */
					pDC->PatBlt( m_rcPalBar.left + nOffsetX + j * nWidth + j,
								 m_rcPalBar.top + nOffsetY + i * nHeight + i,
								 nWidth, nHeight, PATCOPY );
					/* Restore old brush */
					pDC->SelectObject( pOldBrush );
				}
				delete pBrush;
			}
		}
		/* Back to old palette */
		pDC->SelectPalette( pOldPalette, TRUE );
		pDC->RealizePalette();
	}
	return bResult;

} /* #OF# CPaletteDlg::PaintPalette */

/*========================================================
Method   : CPaletteDlg::RestorePalette
=========================================================*/
/* #FN#
   Restores the palette which the dialog was invoked with */
void
/* #AS#
   Nothing */
CPaletteDlg::
RestorePalette()
{
	int nRGB;

	struct ScreenInterParms_t dipInfo;
	dipInfo.dwMask = DIP_BITMAPINFO;

	Screen_GetInterParms( &dipInfo );

	/* Restore old palette */
	for( int i = 0; i < PAL_ENTRIES_NO; i++ )
	{
		nRGB  = (dipInfo.pBitmapInfo->bmiColors[ i ].rgbRed   & 0xff) << 16;
		nRGB |= (dipInfo.pBitmapInfo->bmiColors[ i ].rgbGreen & 0xff) <<  8;
		nRGB |= (dipInfo.pBitmapInfo->bmiColors[ i ].rgbBlue  & 0xff);

		colortable[ i ] = nRGB;
	}
} /* #OF# CPaletteDlg::RestorePalette */

/*========================================================
Method   : CPaletteDlg::SetDlgState
=========================================================*/
/* #FN#
   Sets up the state of the dialog controls */
void
/* #AS#
   Nothing */
CPaletteDlg::
SetDlgState()
{
	CFileStatus fsStatus;

	/* Check if there is the pointed palette file */
	BOOL bPalette = _IsPathAvailable( m_szPaletteFile ) &&
					CFile::GetStatus( m_szPaletteFile, fsStatus );

	if( !bPalette )
		/* Clear "Use external palette" flag if there is no file */
		_ClrFlag( m_ulMiscState, MS_USE_EXT_PALETTE );

	/* Set up check buttons states and activity */
	_SetChkBttn( IDC_PALETTE_APPLYADJUSTMENT,  _IsFlagSet( m_ulMiscState, MS_TRANS_LOADED_PAL ) );
	_EnableCtrl( IDC_PALETTE_APPLYADJUSTMENT,  _IsFlagSet( m_ulMiscState, MS_USE_EXT_PALETTE ) );
	_SetChkBttn( IDC_PALETTE_USEEXTERNAL,      _IsFlagSet( m_ulMiscState, MS_USE_EXT_PALETTE ) );
	_EnableCtrl( IDC_PALETTE_USEEXTERNAL,      bPalette );

	_EnableCtrl( IDC_PALETTE_COLORSHIFT,       !_IsFlagSet( m_ulMiscState, MS_USE_EXT_PALETTE ) );
	_EnableCtrl( IDC_PALETTE_COLORSHIFT_LABEL, !_IsFlagSet( m_ulMiscState, MS_USE_EXT_PALETTE ) );
	_EnableCtrl( IDC_PALETTE_SHIFTSPIN,        !_IsFlagSet( m_ulMiscState, MS_USE_EXT_PALETTE ) );

	SetDlgItemInt( IDC_PALETTE_BLACKLEVEL, m_nBlackLevel, FALSE );
	SetDlgItemInt( IDC_PALETTE_WHITELEVEL, m_nWhiteLevel, FALSE );
	SetDlgItemInt( IDC_PALETTE_SATURATION, m_nSaturation, FALSE );
	SetDlgItemInt( IDC_PALETTE_COLORSHIFT, m_nColorShift, FALSE );

	SetDlgItemText( IDC_PALETTE_EDIT, m_szPaletteFile );

} /* #OF# CPaletteDlg::SetDlgState */


/////////////////////////////////////////////////////////////////////////////
// CPaletteDlg message handlers

/*========================================================
Method   : CPaletteDlg::OnInitDialog
=========================================================*/
/* #FN#
   Performs special processing when the dialog box is initialized */
BOOL
/* #AS#
   TRUE unless you set the focus to a control */
CPaletteDlg::
OnInitDialog()
{
	CCommonDlg::OnInitDialog();
	
	m_bSmallMode  = _IsFlagSet( g_Screen.ulState, SM_MODE_FULL );
	m_ulMiscState = g_Misc.ulState;
	m_nBlackLevel = g_Screen.Pal.nBlackLevel;
	m_nWhiteLevel = g_Screen.Pal.nWhiteLevel;
	m_nSaturation = g_Screen.Pal.nSaturation;
	m_nColorShift = g_Screen.Pal.nColorShift;

	_strncpy( m_szPaletteFile, g_szPaletteFile, MAX_PATH );

	if( !m_bSmallMode )
	{
		GetDlgItem( IDC_PALETTE_BAR )->GetWindowRect( m_rcPalBar );
		ScreenToClient( m_rcPalBar );
		m_rcPalBar.DeflateRect( CX_BORDER, CY_BORDER );

		/* Read the palette from a file or/and format it */
		PreparePalette( m_szPaletteFile );
	}
	SetDlgState();

	return TRUE;  // return TRUE unless you set the focus to a control
	              // EXCEPTION: OCX Property Pages should return FALSE
} /* #OF# CPaletteDlg::OnInitDialog */

/*========================================================
Method   : CPaletteDlg::OnApplyAdjustment
=========================================================*/
/* #FN#
   Sets a state of the object regarding to an appropriate check box */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnApplyAdjustment()
{
	_ClickButton( IDC_PALETTE_APPLYADJUSTMENT, m_ulMiscState, MS_TRANS_LOADED_PAL );
	/* Read the palette from a file or/and format it */
	PreparePalette( m_szPaletteFile );

} /* #OF# CPaletteDlg::OnApplyAdjustment */

/*========================================================
Method   : CPaletteDlg::OnDeltaposBlackSpin
=========================================================*/
/* #FN#
   Sets a state of the object regarding to an appropriate spin control */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnDeltaposBlackSpin(
	NMHDR   *pNMHDR, /* #IN#  */
	LRESULT *pResult /* #OUT# */
)
{
	_DeltaposSpin( pNMHDR, IDC_PALETTE_BLACKLEVEL, m_nBlackLevel, MIN_BLACK, MAX_BLACK );
	/* Read the palette from a file or/and format it */
	PreparePalette( m_szPaletteFile );

	*pResult = 0;

} /* #OF# CPaletteDlg::OnDeltaposBlackSpin */

/*========================================================
Method   : CPaletteDlg::OnKillfocusBlackLevel
=========================================================*/
/* #FN#
   Sets a state of the object regarding to an appropriate edit control */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnKillfocusBlackLevel()
{
	_KillfocusSpin( IDC_PALETTE_BLACKLEVEL, m_nBlackLevel, MIN_BLACK, MAX_BLACK );
	if( !m_bExitPass )
		/* Read the palette from a file or/and format it */
		PreparePalette( m_szPaletteFile );

} /* #OF# CPaletteDlg::OnKillfocusBlackLevel */

/*========================================================
Method   : CPaletteDlg::OnDeltaposWhiteSpin
=========================================================*/
/* #FN#
   Sets a state of the object regarding to an appropriate spin control */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnDeltaposWhiteSpin(
	NMHDR   *pNMHDR, /* #IN#  */
	LRESULT *pResult /* #OUT# */
)
{
	_DeltaposSpin( pNMHDR, IDC_PALETTE_WHITELEVEL, m_nWhiteLevel, MIN_WHITE, MAX_WHITE );
	/* Read the palette from a file or/and format it */
	PreparePalette( m_szPaletteFile );

	*pResult = 0;

} /* #OF# CPaletteDlg::OnDeltaposWhiteSpin */

/*========================================================
Method   : CPaletteDlg::OnKillfocusWhiteLevel
=========================================================*/
/* #FN#
   Sets a state of the object regarding to an appropriate edit control */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnKillfocusWhiteLevel()
{
	_KillfocusSpin( IDC_PALETTE_WHITELEVEL, m_nWhiteLevel, MIN_WHITE, MAX_WHITE );
	if( !m_bExitPass )
		/* Read the palette from a file or/and format it */
		PreparePalette( m_szPaletteFile );

} /* #OF# CPaletteDlg::OnKillfocusWhiteLevel */

/*========================================================
Method   : CPaletteDlg::OnDeltaposSaturationSpin
=========================================================*/
/* #FN#
   Sets a state of the object regarding to an appropriate spin control */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnDeltaposSaturationSpin(
	NMHDR   *pNMHDR, /* #IN#  */
	LRESULT *pResult /* #OUT# */
)
{
	_DeltaposSpin( pNMHDR, IDC_PALETTE_SATURATION, m_nSaturation, MIN_INTENSITY, MAX_INTENSITY );
	/* Read the palette from a file or/and format it */
	PreparePalette( m_szPaletteFile );

	*pResult = 0;

} /* #OF# CPaletteDlg::OnDeltaposSaturationSpin */

/*========================================================
Method   : CPaletteDlg::OnKillfocusSaturation
=========================================================*/
/* #FN#
   Sets a state of the object regarding to an appropriate edit control */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnKillfocusSaturation()
{
	_KillfocusSpin( IDC_PALETTE_SATURATION, m_nSaturation, MIN_INTENSITY, MAX_INTENSITY );
	if( !m_bExitPass )
		/* Read the palette from a file or/and format it */
		PreparePalette( m_szPaletteFile );

} /* #OF# CPaletteDlg::OnKillfocusSaturation */

/*========================================================
Method   : CPaletteDlg::OnDeltaposShiftSpin
=========================================================*/
/* #FN#
   Sets a state of the object regarding to an appropriate spin control */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnDeltaposShiftSpin(
	NMHDR   *pNMHDR, /* #IN#  */
	LRESULT *pResult /* #OUT# */
)
{
	_DeltaposSpin( pNMHDR, IDC_PALETTE_COLORSHIFT, m_nColorShift, MIN_SHIFT, MAX_SHIFT );
	/* Read the palette from a file or/and format it */
	PreparePalette( m_szPaletteFile );

	*pResult = 0;

} /* #OF# CPaletteDlg::OnDeltaposShiftSpin */

/*========================================================
Method   : CPaletteDlg::OnKillfocusColorShift
=========================================================*/
/* #FN#
   Sets a state of the object regarding to an appropriate edit control */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnKillfocusColorShift()
{
	_KillfocusSpin( IDC_PALETTE_COLORSHIFT, m_nColorShift, MIN_SHIFT, MAX_SHIFT );
	if( !m_bExitPass )
		/* Read the palette from a file or/and format it */
		PreparePalette( m_szPaletteFile );

} /* #OF# CPaletteDlg::OnKillfocusColorShift */

/*========================================================
Method   : CPaletteDlg::OnUseExternal
=========================================================*/
/* #FN#
   Sets a state of the object regarding to an appropriate check box */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnUseExternal()
{
	_ClickButton( IDC_PALETTE_USEEXTERNAL, m_ulMiscState, MS_USE_EXT_PALETTE );
	/* Read the palette from a file or/and format it */
	PreparePalette( m_szPaletteFile );

	SetDlgState();

} /* #OF# CPaletteDlg::OnUseExternal */

/*========================================================
Method   : CPaletteDlg::OnBrowse
=========================================================*/
/* #FN#
   Allows selecting a palette file using FileDialog window */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnBrowse()
{
	if( PickFileName( TRUE, m_szPaletteFile, IDS_SELECT_ACT_LOAD, IDS_FILTER_ACT,
					  "act", PF_LOAD_FLAGS, TRUE, DEFAULT_ACT, this ) &&
		*m_szPaletteFile != '\0' )
	{
		SetDlgItemText( IDC_PALETTE_EDIT, m_szPaletteFile );
		/* Read the palette from a file or/and format it */
		PreparePalette( m_szPaletteFile);

		SetDlgState();
	}
} /* #OF# CPaletteDlg::OnBrowse */

/*========================================================
Method   : CPaletteDlg::OnKillfocusEdit
=========================================================*/
/* #FN#
   The framework calls this function before an edit losing input focus */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnKillfocusEdit()
{
	char szPaletteOld[ MAX_PATH + 1 ];

	strcpy( szPaletteOld, m_szPaletteFile );
	GetDlgItemText( IDC_PALETTE_EDIT, m_szPaletteFile, MAX_PATH );

	if( !m_bExitPass &&
		_stricmp( szPaletteOld, m_szPaletteFile ) != 0 )
	{
		/* Read the palette from a file or/and format it */
		PreparePalette( m_szPaletteFile );
		/* Set the dialog controls */
		SetDlgState();
	}
} /* #OF# CPaletteDlg::OnKillfocusEdit */

/*========================================================
Method   : CPaletteDlg::OnPaint
=========================================================*/
/* #FN#
   The framework calls this member function when Windows or an application
   makes a request to repaint a portion of an applications window */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnPaint()
{
	CPaintDC dc( this ); /* Device context for painting */

	if( !m_bSmallMode )
	{
		/* Paint the palette bar */
		PaintPalette( &dc );
	}
} /* #OF# CPaletteDlg::OnPaint */

/*========================================================
Method   : CPaletteDlg::ReceiveFocused
=========================================================*/
/* #FN#
   Receives the edit controls content again. The user could press
   'Enter' or 'Alt-O' and then all changes he's made in the last
   edited control would be lost. */
void
/* #AS#
   Nothing */
CPaletteDlg::
ReceiveFocused()
{
	CWnd *pWnd    = GetFocus();
	UINT  nCtrlID = pWnd ? pWnd->GetDlgCtrlID() : 0;

	switch( nCtrlID )
	{
		case IDC_PALETTE_BLACKLEVEL:
			OnKillfocusBlackLevel();
			break;
		case IDC_PALETTE_WHITELEVEL:
			OnKillfocusWhiteLevel();
			break;
		case IDC_PALETTE_SATURATION:
			OnKillfocusSaturation();
			break;
		case IDC_PALETTE_COLORSHIFT:
			OnKillfocusColorShift();
			break;
		case IDC_PALETTE_EDIT:
			OnKillfocusEdit();
			break;
	}
} /* #OF# CPaletteDlg::ReceiveFocused */

/*========================================================
Method   : CPaletteDlg::OnOK
=========================================================*/
/* #FN#
   Called when the user clicks the OK button */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnOK()
{
	BOOL bChanged = FALSE;

	/* There is a problem with messages routing when the dialog is
	   closed with Enter/Alt-O key. KILLFOCUS message arrives
	   to late and we have to invoke KillFocus handlers in OnOK
	   method by ourselves. That's why we use this member. */
	m_bExitPass = TRUE;

	/* Unfortunately, edit controls do not lose the focus before
	   handling this when the user uses accelerators */
	ReceiveFocused();

	if( m_nBlackLevel != g_Screen.Pal.nBlackLevel )
	{
		g_Screen.Pal.nBlackLevel = m_nBlackLevel;
		WriteRegDWORD( NULL, REG_COLOR_BLACK, g_Screen.Pal.nBlackLevel );
		bChanged = TRUE;
	}
	if( m_nWhiteLevel != g_Screen.Pal.nWhiteLevel )
	{
		g_Screen.Pal.nWhiteLevel = m_nWhiteLevel;
		WriteRegDWORD( NULL, REG_COLOR_WHITE, g_Screen.Pal.nWhiteLevel );
		bChanged = TRUE;
	}
	if( m_nSaturation != g_Screen.Pal.nSaturation )
	{
		g_Screen.Pal.nSaturation = m_nSaturation;
		WriteRegDWORD( NULL, REG_COLOR_SATURATION, g_Screen.Pal.nSaturation );
		bChanged = TRUE;
	}
	if( m_nColorShift != g_Screen.Pal.nColorShift )
	{
		g_Screen.Pal.nColorShift = m_nColorShift;
		WriteRegDWORD( NULL, REG_COLOR_SHIFT, g_Screen.Pal.nColorShift );
		bChanged = TRUE;
	}

	/* Always reload the selected palette 
	if( !PreparePalette( m_szPaletteFile ) )
	{
		if( _IsFlagSet( m_ulMiscState, MS_USE_EXT_PALETTE ) )
		{
			_ClrFlag( m_ulMiscState, MS_USE_EXT_PALETTE );
			DisplayMessage( GetSafeHwnd(), IDS_ERROR_NO_PALETTE, 0, MB_ICONEXCLAMATION | MB_OK, m_szPaletteFile );
		}
		RestorePalette();
	} */

	/* Windows NT doesn't like empty strings in Registry */
	if( '\0' == *m_szPaletteFile )
		strcpy( m_szPaletteFile, DEFAULT_ACT );

	if( _stricmp( g_szPaletteFile, m_szPaletteFile ) != 0 )
	{
		_strncpy( g_szPaletteFile, m_szPaletteFile, MAX_PATH );
		WriteRegString( NULL, REG_FILE_PALETTE, g_szPaletteFile );
		bChanged = TRUE;
	}
	/* Check the miscellanous states */
	if( m_ulMiscState != g_Misc.ulState )
	{
		g_Misc.ulState = m_ulMiscState;
		WriteRegDWORD( NULL, REG_MISC_STATE, g_Misc.ulState );
		bChanged = TRUE;
	}	
	/* Apply the changes to the main window */
	Screen_UseAtariPalette( TRUE );

	/* Prepare interpolation stuff if needed */
	if( bChanged )
	{
		Screen_FreeInterp();
		/* Now when the interpolation stuff was freed check
		   if there is a need for preparing a new one. If
		   not, the stuff will be prepared when later (e.g.
		   when "smooth' stretching will be checked off) */
		Screen_PrepareInterp( FALSE );
	}
	CCommonDlg::OnOK();

} /* #OF# CPaletteDlg::OnOK */

/*========================================================
Method   : CPaletteDlg::OnCancel
=========================================================*/
/* #FN#
   Called when the user clicks the CANCEL button */
void
/* #AS#
   Nothing */
CPaletteDlg::
OnCancel()
{
	/* Restore the original palette */
	RestorePalette();

	CCommonDlg::OnCancel();

} /* #OF# CPaletteDlg::OnCancel */
